#!/bin/bash

############################################################
# script to build a docker container from binary_c
#
# Run this script from binary_c's "docker" directory
############################################################

# initial working directory
WD=$PWD

# set cache date to 'date ...' for a unique (non-cached build)
CACHE_DATE=$(date +%Y-%m-%d:%H:%M:%S) # default
# or today to use the cache
#CACHE_DATE=today

# which git branch do we want?
GIT_BRANCH=master # default
#GIT_BRANCH=izzard-discs

# TMPDIR is our temporary space : usually mktemp is good enough
# but you may want to change this location manually
TMPDIR=$(mktemp -d -t ci-XXXXXXXXXX) # default

############################################################
# get binary_c version string
############################################################
BINARY_C_VERSION=$(grep define\ BINARY_C_VERSION\ \" ../src/binary_c_version.h |gawk "{print \$3}"|sed s/\"//g)
echo "Building docker container for binary_c version $BINARY_C_VERSION"

############################################################
# First, clone the master branch to a temporary directory
############################################################

# make temporary directory
mkdir -p $TMPDIR
echo Cloning binary_c to $TMPDIR
cd $TMPDIR
echo "TMPDIR is $TMPDIR"

# git clone binary_c (master) to $TMP, but only a
# depth=1 "shallow clone"
git clone --depth 1 --branch $GIT_BRANCH gitlab@gitlab.eps.surrey.ac.uk:ri0005/binary_c.git

############################################################
# bind the temporary directory, $TMP/binary_c, here using
# bindfs so we can pretend it's local and hence Docker can
# see/use it
############################################################
cd $WD
mkdir -p ./binary_c_master
bindfs --no-allow-other $TMPDIR/binary_c ./binary_c_master

############################################################
# merge docker lines to reduce image size
############################################################
merge.pl Dockerfile > Dockerfile.merged

############################################################
# build docker container (this takes a while...)
############################################################
docker --log-level=debug build --build-arg CACHE_DATE=$CACHE_DATE -t binary_c -f Dockerfile.merged . |& tee build.log

############################################################
# unbind binary_c git master
############################################################
fusermount -u ./binary_c_master
rmdir binary_c_master

############################################################
# tag docker container
############################################################
echo "Tagging container: robizzard/binary_c:master$BINARY_C_VERSION"
docker tag binary_c robizzard/binary_c:master$BINARY_C_VERSION

echo "Tagging container: robizzard/binary_c:latest"
docker tag binary_c robizzard/binary_c:latest

############################################################
# now you should do "docker push robizzard/binary_c:master<version>
############################################################
echo "Now you should"
echo
echo "Run the following command to update tagged version $BINARY_C_VERSION"
echo
echo "docker push robizzard/binary_c:master$BINARY_C_VERSION"
echo
echo "Run the following command to then update the latest tag"
echo
echo "docker push robizzard/binary_c"
echo

############################################################
# clean up the temporary directory
############################################################
#rm -rf $TMPDIR


