package Histogram;
$|=1;
use warnings;
use strict;
use Carp qw(confess);
use Inline 'Info';
use Sort::Key qw(nsort);
use 5.16.0;
use Hash::RobMerge qw(arithmetically_add_hashes);

# Rob's module to make histograms

#
# usage:
#
# Make a new histogram object:
# my $histogram = Histogram->new();
#
# Or for a shared object so you can multithread (beware: slow!):
# my $histogram = Histogram->new_shared();
#
# Set binwidths:
# $histogram->binwidths(0.1,0.1);
#
# Then add data with co-ordinates $x,$y and value $value
# $histogram->add_data($x,$y,$value);
#
# Set the output format:
# $histogram->format('gnuplot'); # suitable for gnuplotting
# $histogram->format('gnuplot_surface'); # ditto but for pm3d surfaces
# $histogram->format(); # raw data
#
# Print to screen:
# print $histogram->data();
#
# Or dump to a file:
# $histogram->dump('histogram.dat');
#
# You can access a sub-histrogram from its co-ordinates:
# $sub_histogram=$histogram->sub_histogram(1,2);
#
# You can merge two histograms (1 and 2 go into 1, 2 is left untouched):
# $histogram1->merge($histogram2);
#
# You don't have to have Histogram bin the data, you can do it yourself:
# $histogram->pre_binned(1);
#
# Verbose output:
# $histogram->vb(1);
#
# Access an array of data from perl:
# @array = $histogram->data();
# $arrayref = $histogram->dataref();
#

require Exporter;

our @ISA = qw(Exporter);

# Items to export into callers namespace by default. Note: do not export
# names by default without a very good reason. Use EXPORT_OK instead.
# Do not simply export all your public functions/methods/constants.

# This allows declaration	use Histogram ':all';
# If you do not need this, moving things directly into @EXPORT or @EXPORT_OK
# will save memory.

our %EXPORT_TAGS = ( 'all' => [ qw( ) ] );

our @EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );

our @EXPORT = qw(
	&generic_interpolation_wrapper
);

our $VERSION = '0.01';

###### methods and subroutines follow

sub new
{
    # make a new Histogram object
    my ($class,@args) = @_;
    $class //= __PACKAGE__;
    my $self={};
    bless $self,$class;
    $self->initialise(@args);
    if($self->{shared})
    {
	# declare object as shared so it can be shared between threads
	use threads;
	use threads::shared;
	my $shared_self :shared = shared_clone($self);
	bless ($shared_self,$class);
	return $shared_self;
    }
    else
    {
	return $self;
    }
}

sub new_shared
{
    # make a new shared Histogram object
    return Histogram->new(shared=>1);
}

sub initialise
{
    # function to make a Histogram object
    my ($self,@args)=@_;
    
    %$self=(
	_histogram_data => {}, # the histogram (hash)
	_bin_widths => [], # N bin widths (array)
	_format => 'gnuplot', # format : default to gnuplot
	_pre_binned => 0 , # if 1 do not call bin_data

	# verbose option
	_vb => 0,

	# apply args
	@args,

	# when thawing serialized versions of this object, 
	# identify with this (never change it!)
	_is_histogram_object => 1 ,
    );

    return $self;
}

sub destroy
{
    my $self=shift;
    $self = undef;
}

sub sub_histogram
{
    # access a sub histogram from the given coordinates, or undef on failure

    my $self = shift @_; # Histogram object
    my @coords = @_; # n-dimensional coordinates of the point
    my $h = $self->{_histogram_data}; # hash reference base
    foreach my $coord (@coords)
    {
	return undef if(!defined $h->{$coord}); # error case
	$h = $h->{$coord};
    }
    return $h;
}

sub add_data
{
    my $self = shift @_; # Histogram object
    my $datavalue = pop @_; # value of the point
    my @coords = @_; # n-dimensional coordinates of the point
    my $vb = $self->{_vb}; # verbosity
    my $h = $self->{_histogram_data}; # hash reference base
    my $i=0; # coordinate counter
    my $last_coord = pop @coords; # last point coordinate

    print "add_data datavalue=$datavalue coords=(@_) " if($vb);
    
    # build the pointer to the appropriate hash location
    foreach my $coord (@coords)
    {
	# if no bin width is defined, fail
	if(!defined $self->{_bin_widths}->[$i])
	{
	    confess "\nNo bin width for variable $i (coordinate $coord) : you have to set the binwidth with \$histogram->binwidth($i,<width>); or \$histogram->binwidths(<width0>,<width1>,...);"; 
	}

	print "$h [".join(' ',sort keys %$h).'] ' if($vb);

	my $bin = $self->bin_data($coord,$self->{_bin_widths}->[$i++]);
	if(!defined $h->{$bin})
	{
	    # make new hash if required
	    $h->{$bin}= $self->{shared} ? &share({}) : {}; 
	}
	$h=$h->{$bin}; # move hash pointer to the next 'level' 
	print "\{$coord -> $bin\} $h " if($vb);	
    }

    # add the data
    if(!defined $self->{_bin_widths}->[$i])
    {
	confess "\nNo bin width for variable $i (coordinate $last_coord) : you have to set the binwidth with \$histogram->binwidth($i,<width>);"; 
    }

    print '['.join(' ',sort keys %$h).'] ' if($vb);

    my $bin = $self->bin_data($last_coord,$self->{_bin_widths}->[$i]);
    say "\{$last_coord -> $bin\}+=$datavalue" if($vb);
    $$h{$bin} += $datavalue;

}

sub merge
{
    # merge a histogram object into this one
    my $self=shift;
    my $other=shift;
    my $vb=$self->vb || $other->vb;
    
    print "Merge histogram objects $self and $other into $self\n"if($vb);

    # do sanity checks: e.g. bin widths must be the same
    if($#{$self->{_bin_widths}} != $#{$other->{_bin_widths}})
    {
	confess "Tried to merge two Histograms ($self and $other) but one has $#{$self->{_bin_widths}} bins while the other has  $#{$other->{_bin_widths}}! These numbers must match\n";
    }

    for(my $i=0;$i<=$#{$self->{_bin_widths}}; $i++)
    {
	if($self->{_bin_widths}->[$i] ne $other->{_bin_widths}->[$i])
	{
	    confess "Tried to merge two Histograms ($self and $other) but bin $i is discrepent ($self->{_bin_widths}->[$i] vs $other->{_bin_widths}->[$i]\n";
	}
    }

    # merge the data hashes

    print "Histogram->merge add data hashes : $self->{_histogram_data} to $other->{_histogram_data}\n" if($vb);
    arithmetically_add_hashes($self->{_histogram_data},
			      $other->{_histogram_data});

    # return merged object
    return $self;
}

sub binwidth
{
    # set width of bin $n and/or return bin width $n
    my ($self,$n,$width) = (@_);
    $self->{_bin_widths}->[$n] = $width if(defined $width);
    return $self->{_bin_widths}->[$n];
}

sub binwidths
{
    # set/return list of bin widths
    my ($self,@widths)=(@_);
    @{$self->{_bin_widths}} = @widths if(defined $widths[0]);
    return @{$self->{_bin_widths}};
}

sub vb
{
    # set or return vb
    my ($self,$vb) = (@_);
    $self->{_vb} = $vb if(defined $vb); 
    return $self->{_vb};
}

sub format
{
    # set or return format
    my ($self,$format) = (@_);
    $self->{_format} = $format if(defined $format); 
    return $self->{_format};
}

sub pre_binned
{
    # set or return pre_binned
    my ($self,$pre_binned) = (@_);
    $self->{_pre_binned} = $pre_binned if(defined $pre_binned); 
    return $self->{_pre_binned};
}

sub bin_data
{
    # bin $x to within $dx
    my ($self,$x,$dx)=(@_);
    return $self->{_pre_binned} ? $x :  
	(($x>0.0?1.0:-1.0)*0.5+int($x/$dx))*$dx;
}

sub dataref
{
    my ($self,$format)=@_;

    # return a reference to a list of data lines which represent the histogram
    # in a format suitable for plotting

    my $h = $self->{_histogram_data};
    my $array = [];
    $self->_recurse_hash($array,$h);
    return $array;
}

sub data
{
    # return an array based on dataref() above
    my $self=$_[0];
    return @{$self->dataref(@_)};
}

sub dump
{
    # dump data to file
    my ($self,$filename) = @_;
    open(my $fp, '>',$filename)||
	confess("cannot open $filename for writing in Histogram->dump");
    print {$fp} join("\n",$self->data()),"\n"; 
    close $fp;
}

sub _recurse_hash
{
    my $self = shift @_;
    my $array = shift @_; # the output array
    my $h = shift @_; # the hash ref (we're recursing in this)
    my @coords = @_; # @coords now contains the previous hash keys

    foreach my $k (nsort keys %$h)
    {
	if(ref $$h{$k} eq 'HASH')
	{ 
	    # another hash : recurse it
	    $self->_recurse_hash($array,$$h{$k},$k);
 	}
	else
	{
	    # data : add to array, return
	    my $value = $$h{$k}; # data value
	    my @c = (@coords,$k);  # data coordinates

	    if($self->{_format} eq 'gnuplot')
	    {
		# for gnuplot formatting, break lines when x value changes
		state $prev;
		push(@$array,'') if(defined $prev && $coords[0] ne $prev);
		$prev = $coords[0];
		push(@$array,join(' ',@c,$value)); 
	    }
	    elsif($self->{_format} eq 'gnuplot_surface')
	    {
		# output little squares for a gnuplot surface plot
		# NB data must be 2D!		
		state $prev;
		push(@$array,'')if(defined $prev && $coords[0] ne $prev);
		$prev = $coords[0];

		$c[0] -= $self->{_bin_widths}[0]*0.5; 
		$c[1] -= $self->{_bin_widths}[1]*0.5;
		my $l0 = join(' ',@c,$value);
 		push(@$array,$l0);

		$c[1] += $self->{_bin_widths}[1];
		push(@$array,join(' ',@c,$value)); 

		$c[0] += $self->{_bin_widths}[0];
		push(@$array,join(' ',@c,$value)); 

		$c[1] -= $self->{_bin_widths}[1];
		push(@$array,join(' ',@c,$value),$l0,''); 
	    }
	    else
	    {
		# just output lines of data
		push(@$array,join(' ',@c,$value)); 
	    }
	}
    } 
}


sub TO_JSON { return { %{ shift() } }; }



1;
