package IMF;

# Rob's Initial Mass Function (IMF) module
#
# Contains various routines to calculate the probability
# of existence of a star, 
#
# This module was designed to be used by the binary_grid
# module, but of course you can use it elsewhere too.
#
# Some documentation is available in the binary_grid docs,
# see the doc directory of binary_c
#
# V0.01 original version
# V0.04 Better documentation, slight code cleanups and optimisations
# V0.05 Clean up, rewritten to use state variables

our $VERSION = '0.05';

use 5.16.0;
use feature 'state';
use strict;
use warnings;

# rob's modules
use rob_misc;
use binary_stars;
use Math::Trig;

# global variables
use vars qw($imf1 $imf2 $sepdist $plaw_m0 $plaw_m1 $plaw_m2 $plaw_mmax $plaw_p1 $plaw_p2 $plaw_p3 $plaw_p $sepmin $sepmax $idlsep $a1 $a2 $a3 $imf2_gaussian_correction $imf2_minq_gaussian $period_gaussian_correction $log_normal_median $log_normal_dispersion $lognormal_A $imf1_func_pointer);

require Exporter;

our @ISA = qw(Exporter);

our %EXPORT_TAGS = ( 'all' => [ qw(
	&calc_prob &init_IMF &imf1
) ] );

our @EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );

our @EXPORT = qw(
	&calc_prob &init_IMF &imf1
);


# constants
use constant PI => acos(0.0)*2.0;
use constant GAUSSIAN_PREFACTOR => (1.0/sqrt(2.0*PI));
use constant logmin => 1e-30;
use constant value_of_log10 => log(10.0);
use constant log_ln_converter => (1.0/log(10.0));

my $imf2_minq_gaussian=-123456;

sub init_IMF
{
    # get the IMF parameters as passed in
    $imf1=shift;
    $imf2=shift;
    $sepdist=shift;
    $plaw_m0=shift;
    $plaw_m1=shift;
    $plaw_m2=shift;
    $plaw_mmax=shift;
    $plaw_p1=shift;
    $plaw_p2=shift;
    $plaw_p3=shift;
    $plaw_p=shift;
    $sepmin=shift;
    $sepmax=shift;
    if(defined($sepmax) && defined($sepmin))
    {
	$idlsep=1.0/(1e-14+log($sepmax)-log($sepmin));
    }

    # log normal IMF reuses plaw_m0 and plaw_mmax
    $log_normal_median=shift;
    $log_normal_dispersion=shift;

    $binary_grid::grid_options{'imf2_maxq'}=1 if(!defined($binary_grid::grid_options{'imf2_maxq'}));
    $binary_grid::grid_options{'period_dist'}='gaussian' if(!defined($binary_grid::grid_options{'period_dist'}));
    $imf1_func_pointer=undef; # just in case the IMF is redefined
}

sub calc_prob
{ 
    # calculate probability given phase volume, m1,m2,separation (or period)
    #
    # phase volume is dlogm1*dlogm2*dlogsep

    # NB eccentricity is dealt with elsewhere

    # always return 1.0 if prob1==1
    return 1.0 if(defined($binary_grid::grid_options{'prob1'})&&
                  $binary_grid::grid_options{'prob1'}==1);

    # always need phase volume and mass of star 1 
    my ($vol,$m1,$m2,$sep_or_per) = @_;
    state $prevm1;
    state $prevpm1;
    state $prevm2;
    state $prevpm2;
    state $prevq;
    state $prevpq;
    state $prevminq; 

    # first: single star/primary IMF
    my $p=1.0; # probability

    # calculate the M1 part of the probability if required
    if($m1!=$prevm1)
    {
	$prevpm1 = imf1($m1)*$m1;
	$prevm1   = $m1;
    }

    # always need a volume factor and M1 distribution
    $p *= $prevpm1*$vol;

    if(!$_[0])
    {
	# single star only : can return here
	return $p;
    }
    elsif(!$p)
    {
	# if M1 grid has zero probability, so will the system: return 0 here
	return 0.0;
    }
    else
    {
	# binary stars:

	# secondary mass distribution
	my $q=$m2/$m1;

	# if over maxq, return 0
	my $maxq=$binary_grid::grid_options{'imf2_maxq'};
	if($q>$maxq)
	{
	    return 0.0;
	}
	else
	{
            # calculate minimum allowed q = M1/M2
	    my $minq = $binary_grid::grid_options{'imf2_minq'} // 
                $binary_grid::grid_options{'imf2_minm'}/$m1;

	    if($q<$minq)
	    {
		return 0.0;
	    }
	    else
	    {	    
		# if M1==M2 we actually want to simulate HALF a grid cell
                # because M2>M1 is impossible.
		$p*=0.5 if ($q==1.0);
		
		if($imf2 eq 'gaussian')
		{
		    # gaussian distribution based on q
		    if((!defined($imf2_gaussian_correction))||($minq!=$prevminq))
		    {
			$prevminq=$minq;
			$imf2_minq_gaussian = $minq;
			# calculate correction for parts of the gaussian outside
			# the q_min to q_max range
			#
			# The correction will change if $minq changes, which may depend
			# on m1
			my $ptot = 0.0;
			my $dqq  = 0.001; # trial and error suggests this is sufficient, if a
			# little slow
			
			for(my $qq=$minq; $qq<=$maxq; $qq+=$dqq)
			{
			    $ptot+=$dqq*gaussian($qq,$binary_grid::grid_options{'q_mean'},$binary_grid::grid_options{'q_sigma'});
			}
			$imf2_gaussian_correction=1.0/$ptot;
                    }

		    # calculate the gaussian
		    if($q!=$prevq)
		    {
			# note extra factor of q to convert to d/d(lnm2)
			$prevpq=$q*$imf2_gaussian_correction * 
                            gaussian($q,
                                     $binary_grid::grid_options{'q_mean'},
                                     $binary_grid::grid_options{'q_sigma'});
			$prevq=$q;
		    }
		    $p *= $prevpq;
		}
		elsif($imf2 eq 'ktg93')
		{
		    # use imf of KTG93 for secondary
		    if($m2 != $prevm2)
		    {
			$prevpm2 = imf_ktg($m2)*$m2;
			$prevm2  = $m2;
		    }
		    $p *= $prevpm2;
		}
		elsif($imf2 eq 'imf1')
		{
		    # use primary IMF for secondary
		    if($m2 != $prevm2)
		    {
			$prevpm2= imf1($m2)*$m2;
			$prevm2=$m2;
		    }
		    $p*=$prevpm2;
		}
		elsif($imf2 eq 'plawm')
		{
		    if($m2 != $prevm2)
		    {
			my $m2max;
			if($binary_grid::grid_options{'imf2_maxm'} eq 'auto')
			{
			    # automatically select max m2 value to be m1, so q<=1
			    $m2max=$m1;
			}
			else
			{
			    $m2max=$binary_grid::grid_options{'imf2_maxm'};
			}
			
			$prevpm2=general_plaw($binary_grid::grid_options{'imf2_exponent'},
					      $binary_grid::grid_options{'imf2_minm'},
					      $m2max,$m2);
			
			# convert to d/d(lnm2)
			$prevpm2*=$m2;
			$prevm2=$m2;
		    }
		    $p*=$prevpm2;
		}
		elsif($imf2 eq 'plawq')
		{
		    # power law in q (convert to d/d(lnm2))
		    if($q != $prevq)
		    {
			$prevpq=$q*general_plaw($binary_grid::grid_options{'imf2_exponent'},
						$minq,
						$binary_grid::grid_options{'imf2_maxq'},
						$q);
			$prevq=$q;
		    }
		    $p*=$prevpq;
		}
		else
		{		    
		    # use flat Q imf for secondary, between $minq and 1
		    # only recalculate when we have to!
		    if($q != $prevq)
		    {
			$prevq  = $q;
                        $prevpq = $minq ? $q/(1.0-$minq) : $q;
		    }
		    $p *= $prevpq;
		}
	    }
	    

	    # now for the period/separation distribution
	    if(!($p && $sep_or_per))
	    {
		return 0.0;
	    }
	    else
	    {
		# NB there is no point in caching this distribution: it always changes!
                $p *= $binary_grid::grid_options{'sepdist'} ne 'period' ? 
                    sepdist($sep_or_per) : perdist($m1,$m2,$sep_or_per);
	    }
	}
    }
    return $p;
}

sub perdist
{
    # we distribute according to log period, not separation
    my ($m1,$m2,$per)=@_;
    my $logper=log10($per);

    return 0.0 if(($logper<$binary_grid::grid_options{'period_dist_log_min'})||
		  ($logper>$binary_grid::grid_options{'period_dist_log_max'}));

    if($binary_grid::grid_options{'period_dist'} eq 'gaussian')
    {
	if(!defined($period_gaussian_correction))
	{ 
	    # calculate correction for parts of the gaussian outside
	    # the log_period_min to log_period_max
	    my $ptot=0.0;
	    my $dlp=0.001; # sufficient for 4 decimal place accuracy

	    for(my $lp=$binary_grid::grid_options{'period_dist_log_min'}; 
		$lp<=$binary_grid::grid_options{'period_dist_log_max'}; 
		$lp+=$dlp)
	    {
		$ptot+=$dlp*gaussian($lp,
				     $binary_grid::grid_options{'period_dist_mean_log'},
				     $binary_grid::grid_options{'period_dist_sigma'});
	    }
	    $period_gaussian_correction=1.0/$ptot;
	    print "Period: Ptot=$ptot -> correction factor $period_gaussian_correction\n" if($binary_grid::vb);
	}

	# if out of distribution bounds, return 0.0
	if(($logper < $binary_grid::grid_options{'period_dist_log_min'})||
	   ($logper > $binary_grid::grid_options{'period_dist_log_max'}))
	{
	    return 0.0;
	}
	else
	{
	    # the gaussian is d(prob)/d(log_10 P)
	    my $g= $period_gaussian_correction* 
		gaussian($logper,
			 $binary_grid::grid_options{'period_dist_mean_log'},
			 $binary_grid::grid_options{'period_dist_sigma'});
	    
	    # remember we want ln space, not log10 space (as in D+M), so convert
	    return log_ln_converter*$g;
	}
    }
    else
    {
	print "Unknown period distribution $binary_grid::grid_options{'period_dist'}\n";
	exit(0);
    }
}

sub imf1
{
    my ($m)=@_;
    
    # define range of masses by the "plaw" m0 -> mmax
    # (really these are for the power law, but we can 
    # use them for all distributions)
    return (0.0) if($m<$plaw_m0 || $m>$plaw_mmax);

    if(!defined($imf1_func_pointer))
    {
	# single star Initial Mass Function : 
        # Set the function pointer based on the input string.
        my %funcs = (
            ktg93 => \&imf_ktg,
            chabrier => \&imf_chabrier2003,
            plaw => \&imf_plaw,
            tinsley1980 => \&imf_tinsley1980,
            scalo1986 => \&imf_scalo1986,
            lognormal => \&imf_lognormal,
            '1' => \&imf_const
            );

        if(defined $funcs{$imf1})
        {
            $imf1_func_pointer = $funcs{$imf1};
        }
	else
	{
	    print STDERR "Unknown IMF \"$imf1\"\n";
            exit;
	}
    }

    # call the function
    &$imf1_func_pointer($m,$plaw_p);
}

sub imf_const
{
    # constant IMF : can be any number,
    # normalized elsewhere
    return 1.0;
}

sub sepdist
{
    my $sep=shift;
    # check the separation is in the bounds of the distribution
    return 0.0 if($sep<$sepmin || $sep>$sepmax);

    # it is in the bounds...
    if($sepdist eq 'plaw')
    {
	return # power law in separation
	       general_plaw($binary_grid::grid_options{'sepdist_exponent'},
			    $binary_grid::grid_options{'sepdist_min'},
			    $binary_grid::grid_options{'sepdist_max'},
			    $sep)
	       # convert to d/dlna
	       *$sep;
    }
    else
    {
	# flat (in logspace) separation distribution (precalculated in init_IMF)
	return $idlsep;
    }

}

sub general_plaw
{
    # a general power law from min to max with variable exponent
    my ($exponent,$min,$max,$x) = @_;

    return 0.0 if($x<$min || $x>$max);

    my $C;
    if($exponent == -1.0)
    {
	# special case
	$C=1.0/(log($max)-log($min));
    }
    else
    {
	my $gamma=1.0+$exponent;
	$C=$gamma/($max**$gamma-$min**$gamma);
    }
    return $C * $x**$exponent;
}

sub imf_ktg
{
    # three part power law (similar to KTG93)
    return three_part_power_law($_[0], # mass
                                # options
                                $plaw_m0,$plaw_m1,$plaw_m2,
                                $plaw_mmax,
                                $plaw_p1,$plaw_p2,$plaw_p3);
}

sub imf_plaw
{
    # single component power law, like salpeter
    my $m=shift;
    my $alpha=shift;
    my $alpha1=1.0+$alpha;
    return $m**$alpha*($alpha1/($plaw_mmax**$alpha1-$plaw_m0**$alpha1));
}


sub imf_tinsley1980
{
    # from Tinsley 1980
    return three_part_power_law($_[0],
				0.1,2.0,10.0,80.0,
				-2.0,-2.3,-3.3);
}

sub imf_scalo1986
{
    # from Scalo 1986
    return three_part_power_law($_[0],
				0.1,1.0,2.0,80.0,
				-2.35,-2.35,-2.70);
}

sub imf_scalo1998
{
    # from Scalo 1998
    return three_part_power_law($_[0],
				0.1,1.0,10.0,80.0,
				-1.2,-2.7,-2.3);
}

sub three_part_power_law
{
 
    # calculate normalization coefficients
    if(!($a2))
        # only do this once! (no need to heavily optimise)
    {
	my $m=$_[0];
	my $m0=$_[1];
	my $m1=$_[2];
	my $m2=$_[3];
	my $mmax=$_[4];
	my $p1=$_[5];
	my $p2=$_[6];
	my $p3=$_[7];

	$a2=(($m1**$p2)*($m1**(-$p1)))*
            (1.0/(1.0+$p1))*
            ($m1**(1.0+$p1)-$m0**(1.0+$p1))+
            
            (($m2**(1.0+$p2)-$m1**(1.0+$p2)))*
            (1.0/(1.0+$p2))+
	    (($m2**$p2)*($m2**(-$p3)))*
            (1.0/(1.0+$p3))*
            ($mmax**(1.0+$p3)-$m2**(1.0+$p3));
        $a2=1.0/($a2+1e-50);
        $a1=$a2*$m1**$p2*$m1**(-$p1);
        $a3=$a2*$m2**$p2*$m2**(-$p3);
    }
    
    if($_[0]<$_[1])
    {
	return(0.0);
    }
    elsif($_[0]<$_[2])
    {
        return($a1*($_[0]**$_[5]));
    }
    elsif($_[0]<$_[3])
    {
	return($a2*($_[0]**$_[6]));
    }
    elsif($_[0]<$_[4])
    {
	return($a3*($_[0]**$_[7]));
    }
    else
    {
	return(0.0);
    }

}

sub imf_chabrier2003
{
    my ($m) = @_;
    my $p;
    my $A;

    # IMF of Chabrier 2003 PASP 115:763-795

    use constant Chabrier_logmc => log10(0.079);
    use constant Chabrier_sigma2 => (0.69*0.69);
    use constant Chabrier_A1 => 0.158;
    use constant Chabrier_A2 => 4.43e-2;
    use constant Chabrier_x => -1.3;

    # NB the imf given is a function of log10(m)
    if($m<1.0)
    {	
        $A = 0.158;
	my $dm = log10($m)- Chabrier_logmc;
        $p = Chabrier_A1*exp(-($dm*$dm)/(2.0*Chabrier_sigma2));
    }
    else
    {
        $p = Chabrier_A2 * ($m**(Chabrier_x));
    }
    # ... so we must convert back to a function of m
    $p /= 0.1202462 * $m * value_of_log10;
    return $p;
}


sub gaussian
{
    # three args: x, mean, sigma
    my $r=1.0/$_[2]; # 1/sigma
    my $y=($_[0]-$_[1])*$r;
    return(GAUSSIAN_PREFACTOR*$r*exp(-0.5*$y*$y));
}

sub imf_lognormal
{
    my ($m) = @_;

    # log-normal distribution, e.g. Millar and Scalo 1979
    #
    # of the form
    # xi(M) = A/m exp[ -(log10(m/Md))^2 / (2*Dm^2)]
    #
    # Where Md is a "median" mass
    # and Dm is a dispersion
    # 
    # Note: always use log10!

    if(!defined($lognormal_A))
    {
	# normalize
	# the formula from Millar + Scalo 1979 is 
	# xi(log10 M)=c0 exp(-c1 (log10 M - c2)^2)
	# equivalent to
	# xi(M)=c0/(ln10 * m) exp ( ... )
	# where
	# c0=66-242 (this is determined by the normalization)
	# c1~1.1
	# c2~-1.0
	#
	# Now, convert c2 to a dispersion Dm :
	# Dm = (2*c1)**-0.5
	# i.e. Dm ~ 0.67 = $log_normal_dispersion
	#
	# And convert c2 to Md (the median)
	# Md=10^(c2)~0.1 = $log_normal_median
	#
	my $n=10000; # integral usually converges to ~4sf if n~10000
	my $dm=($plaw_mmax-$plaw_m0)/$n;
	my $dm2=0.5*$dm; # half dm
	my $integral=0.0;
	for(my $m=$plaw_m0; $m<=$plaw_mmax; )
	{
	    $m+=$dm2;
	    my $x=(log10($m/$log_normal_median) / $log_normal_dispersion);
	    my $xi=(1.0/$m) * exp ( - 0.5 * $x *$x);
	    $integral += $xi * $dm;
	    $m+=$dm2;
	}
	$lognormal_A=1.0/$integral;
    }

    my $x=(log10($m/$log_normal_median) / $log_normal_dispersion);
    my $xi=(1.0/$m) * exp ( - 0.5 * $x *$x);
    return $lognormal_A*$xi;
}

# Preloaded methods go here.

1;
__END__
# Below is stub documentation for your module. You'd better edit it!

=head1 NAME

IMF - Perl module for calculating the Initial Mass Function of stars 
and perhaps galaxies.

=head1 SYNOPSIS

  use IMF;
  calc_prob(...);

=head1 DESCRIPTION

The initial mass function gives the distribution of masses of stars 
at their birth. This module should be used in conjunction with the
binary_grid Perl module package to evolve grids of single and 
binary stars with the appropriate weightings.

=head2 EXPORT

=head1 SEE ALSO

See the scripts gce.source.pl, gce.pl, yields_vs_time.pl etc.

=head1 AUTHOR

Rob Izzard, r.g.izzard@phys.uu.nl, also at gmail: rob.izzard

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2005-8 by Rob Izzard

If you use this module you promise to pay me 1000000 UK pounds (or equivalent local currency, provided it's not Italian) per line of my code you use. That means a lot of money.

.
.
.
.
.

You believe me, right?

Note that I take NO RESPONSIBILITY for what this code may do, or what you may do with it. I provide it to you in good faith, and you are (of course!) allowed to view and hack all or any of it as you see fit. 


=cut
