#include "../binary_c.h"

/*
 * Calculate the moment of inertia of a star
 */

static double BH_MOI(const struct star_t * RESTRICT const star);
static double neutron_star_MOI(const struct star_t * RESTRICT const star,
                               const double r);
static double normal_star_MOI(const struct star_t * RESTRICT const star,
                              const double r);

double Pure_function moment_of_inertia(const struct star_t * RESTRICT const star,
                                       const double r)
{  
    const double momI =
        star->stellar_type == MASSLESS_REMNANT ? TINY :
        star->stellar_type == BLACK_HOLE ? BH_MOI(star) :
        star->stellar_type == NEUTRON_STAR ? neutron_star_MOI(star,r) :
        normal_star_MOI(star,r);
/*
    if(1)
        printf("momI [star %d type %d has core? %d, M=%g R=%g (RL=%g) L=%g J=%g] = (Menv=%g-%g) * (k2=%g) * (r=%g)^2  + (k3=%g) * (Mc=%g) * (rc=%g)^2 = %g (%g) + %g (%g) = %g\n",
               star->starnum,
               star->stellar_type,
               HAS_BSE_CORE(star->stellar_type),

               star->mass,
               star->radius, 
               star->roche_radius,
               star->luminosity,
               star->angular_momentum,

               star->mass,
               star->core_mass,
               star->k2,
               r,
               K3,
               star->core_mass,
               star->core_radius,
               (star->mass - star->core_mass)*star->k2*Pow2(r),
               envelope_moment_of_inertia(star,r),
               K3* Pow2(star->core_radius)*star->core_mass,
               core_moment_of_inertia(star,star->core_radius),
               momI);
*/

    return momI;
}

static double neutron_star_MOI(const struct star_t * RESTRICT const star,
                               const double r)
{
    /*
     * Based on Ravenhall and Pethick, 1994
     * ApJ 484, 846, Eq. 6.
     * 
     * NB factor x is dimensionless.
     */
    const double x = GRAVITATIONAL_CONSTANT * star->mass * M_SUN /
        (r * R_SUN * Pow2(SPEED_OF_LIGHT));
    return 0.21* star->mass * Pow2(star->radius) / (1.0 - 2.0 * x);
}

static double BH_MOI(const struct star_t * RESTRICT const star)
{
    /*
     * Hard to define a moment of inertia for a neutron star,
     * but we know its angular momentum (code units) and spin rate (/y). 
     */
    return star->angular_momentum / star->omega;
}

static double normal_star_MOI(const struct star_t * RESTRICT const star,
                              const double r)
{
    /*
     * Moment of inertia of a "normal" star
     */

    return
        /* "envelope" */
        envelope_moment_of_inertia(star,r)
        +
        /* "core" */
        core_moment_of_inertia(star,star->core_radius);
}

double Pure_function core_moment_of_inertia(const struct star_t * RESTRICT const star,
                                            const double rc)
{
    return HAS_BSE_CORE(star->stellar_type) ? (K3 * star->core_mass * Pow2(rc)) : 0.0;
}

double envelope_moment_of_inertia(const struct star_t * RESTRICT const star,
                                  const double r)
{
    return star->k2 * (star->mass - star->core_mass) * Pow2(r);
}
