#include "wind.h"

/*
 * Mass-loss rate for AGB stars
 * (EAGB and TPAGB)
 */

static double wind_EAGB(WIND_PROTOTYPE_ARGS);
static double wind_TPAGB(WIND_PROTOTYPE_ARGS);
static double Mira_pulsation_period(const double mass,
                                    const double radius,
                                    struct stardata_t * stardata);
static double Mira_wind_velocity(const double period);

double wind_AGB(WIND_PROTOTYPE_ARGS)
{
    double mdot_AGB;

    if(stellar_type==EAGB)
    {
        mdot_AGB = wind_EAGB(WIND_CALL_ARGS);
    }
    else if(stellar_type==TPAGB)
    {
        mdot_AGB = wind_TPAGB(WIND_CALL_ARGS);
    }
    else
    {
        mdot_AGB = 0.0;
    }
    return (mdot_AGB);
}


static double wind_EAGB(WIND_PROTOTYPE_ARGS)
{
    double mdot_EAGB;
    
    /* wind loss on the Early AGB */

    
#ifdef PN_FAST_WIND
    if(mass-core_mass < PN_FAST_WIND_DM_AGB)
    {
        mdot_EAGB = PN_FAST_WIND_MDOT_AGB;
    }
    else
    {
#endif
        /*
         * Vassiliadis and Wood, 1993, with Max(mass-2.5,0) term
         * as in VW93 and Hurley et al 2002 
         */

        /* calculate Mira pulsation period */
        double p0 = -2.07 - 0.90*log10(mass) + 1.940*log10(radius);
        p0 = pow(10.0,p0);
        p0 = Min(p0,2000.0);

        /* hence mass loss rate */
        mdot_EAGB = -11.4+0.0125*(p0-100.0*Max(mass-2.50,0.0));
        mdot_EAGB = pow(10.0,mdot_EAGB);
        mdot_EAGB = Min(mdot_EAGB,1.36e-9*luminosity); // steady superwind
#ifdef PN_FAST_WIND
    }
#endif

    return(mdot_EAGB);
} 

static double wind_TPAGB(WIND_PROTOTYPE_ARGS)
{
    double mdot_TPAGB;
    
#ifdef PN_FAST_WIND
    if(mass - core_mass < PN_FAST_WIND_DM_AGB)
    {
        mdot_TPAGB = PN_FAST_WIND_MDOT_AGB;
    }
    else
    {
#endif
        if(stardata->preferences->tpagbwind<2)
        {
            /*
             * Vassiliadis and Wood 1993 
             *
             * type 0 is as Karakas et al. 2002
             * type 1 is as Hurley et al. 2002
             */

            /* calculate mira pulsation period */
            double p0 = Mira_pulsation_period(mass,radius,stardata);
            
            /* wind velocity in cgs FAR FROM THE STAR! */ 
            double vw = Mira_wind_velocity(p0);
            Dprint("Mira vw = %g from p0 = %g\n",vw,p0);

#ifdef NUCSYN
            // store period for later logging?
            star->mira=p0;
#endif

            if(stardata->preferences->tpagbwind==AGB_WIND_VW93_KARAKAS)
            {
                /*
                 * Karakas et al. 2002
                 * calculate mdot_TPAGB without limit on p0
                 * and without the mass correction
                 */
                mdot_TPAGB = -11.4+0.0125*(p0); // without -100.0*Max((mt-2.50),0.0));
                mdot_TPAGB = pow(10.0,mdot_TPAGB);

                if(p0>stardata->preferences->tpagb_superwind_mira_switchon)
                {
                    /* Switch on superwind */
                    p0=stardata->preferences->tpagb_superwind_mira_switchon;

                    /* limit wind velocity (cgs) */
                    vw = 1.0e5 * (-13.5+0.056*p0);
                    Clamp(vw,0.0,1.5e6);

                    /* hence mass loss rate */
                    mdot_TPAGB = Min(mdot_TPAGB,
                                     (L_SUN/M_SUN) * (YEAR_LENGTH_IN_SECONDS/SPEED_OF_LIGHT)
                                     * (luminosity/vw));
                }
            }
            else
            {
                /* tpagbwind==1==AGB_WIND_VW93_ORIG
                 * 
                 * : Original Vassiliadis and Wood 1993 */

                /* calculate mdot_TPAGB *with* mass correction */
                mdot_TPAGB = -11.4 + 0.0125 *
                    (p0 - 100.0*Max(mass - 2.50, 0.0));
                mdot_TPAGB = pow(10.0,mdot_TPAGB);
                /* superwind */
                mdot_TPAGB = Min(mdot_TPAGB,1.36e-9 * luminosity);
            }

            star->vwind = vw;
                
            Dprint("WIND pmira=%12.12e l=%12.12e r=%12.12e L/vw=%12.12e m=%12.12e -> mdot_TPAGB=%g\n",p0,
                   luminosity,
                   radius,
                   (L_SUN/M_SUN)*
                   (YEAR_LENGTH_IN_SECONDS/SPEED_OF_LIGHT)*
                   (luminosity/vw),
                   mass,
                   mdot_TPAGB);

#ifdef VW93_MULTIPLIER
            /* Allow VW93 rate to be modulated */
            mdot_TPAGB *= stardata->preferences->vw93_multiplier;
#endif

            Dprint("hence mdot_TPAGB=%g\n",mdot_TPAGB);

        }

        else if (stardata->preferences->tpagbwind==AGB_WIND_REIMERS)
        {
            /* Reimers mass loss on the TPAGB */
            mdot_TPAGB = stardata->preferences->tpagb_reimers_eta*4.0E-13*radius*luminosity/mass;
            Dprint("Reimers TPAGB %g\n",mdot_TPAGB);
        }
        else if (stardata->preferences->tpagbwind==AGB_WIND_BLOECKER)
        {
            /* Bloecker 1995 */
            mdot_TPAGB=4.83e-9*stardata->preferences->tpagb_reimers_eta*
                pow(luminosity,2.7)*pow(mass,-2.1)*
                (4.0E-13*radius*luminosity/mass);
        }
        else if(stardata->preferences->tpagbwind==AGB_WIND_VAN_LOON)
        {
            /* Van Loon 2005 */

            double l4=log10(luminosity*1e-4);
            double teff = 1000.0*pow((1130.0*luminosity/Pow2(radius)),0.25);

            /* general formula from the abstract of his paper */
            //mdot_TPAGB=-5.65+1.05*l4-6.3*log10(teff/3500);

            /* 
             * instead, use the red supergiant formula if we're
             * bright enough, otherwise use the AGB formula, neither
             * may be correct at low luminosity
             */
            if(l4<0.9)
            {
                /* AGB formula */
                mdot_TPAGB=-5.6+1.10*l4-5.2*log10(teff/3500);
            }
            else
            {
                /* Red supergiant formula */
                mdot_TPAGB=-5.3+0.82*l4-10.8*log10(teff/3500);
            }
                
            mdot_TPAGB=pow(10.0,mdot_TPAGB);

            /*
             * Force small envelope to be lost when the AGB star
             * gets hot (i.e. approaches the WD track)
             *
             */
            if(teff>4000) mdot_TPAGB=1e-4;
        }
#ifdef NUCSYN
        else if(stardata->preferences->tpagbwind==AGB_WIND_ROB_CWIND)
        {
            /*
             * Strong wind for C-rich stars only
             */
            double r=(star->Xenv[XC12]/12.0+
                      star->Xenv[XC13]/13.0)/
                (star->Xenv[XO16]/16.0+
                 star->Xenv[XO17]/17.0+
                 star->Xenv[XO18]/18.0);
            /*
              double rn=(star->Xenv[XC12]/12.0+
              star->Xenv[XC13]/13.0)/
              (star->Xenv[XN14]/14.0+
              star->Xenv[XN14]/15.0);
            */
            printf("M=%g Mc=%g C/O=%g\n",star->mass,star->core_mass,r);

            /* 
             * We need more C than O, otherwise CO forms
             * and more C than N, otherwise CN forms
             * ASSUME THESE HAVE LOW OPACITY compared to C-C CCC etc.
             */
            if((r>0.99))//&&(rn>0.99)&&(star->num_thermal_pulses>10))
            {
                // nC/nO > 0.99
                mdot_TPAGB=1e-4;
            }
            else
            {
                mdot_TPAGB=1e-8;
            }
        }

        else if((stardata->preferences->tpagbwind==AGB_WIND_VW93_KARAKAS_CARBON_STARS)||
                (stardata->preferences->tpagbwind==AGB_WIND_VW93_ORIG_CARBON_STARS))
        {
            // deprecated!
            mdot_TPAGB=0.0;
        }

#ifdef MATTSSON_MASS_LOSS
        else if(stardata->preferences->tpagbwind==AGB_WIND_MATTSSON)
        {
            // Lars Mattsson's wind
            Boolean fail=FALSE;
            double teff = 1000.0*pow((1130.0*luminosity/Pow2(radius)),0.25);
            mdot_TPAGB=mattsson_massloss(mass,
                                         star->phase_start_mass,
                                         luminosity,
                                         teff,
                                         star->Xenv[XC12],
                                         star->Xenv[XC13],
                                         star->Xenv[XO16],
                                         stardata->common.XZAMS[XH1],
                                         &fail);
            
        }
#endif // MATTSSON_MASS_LOSS
#endif // NUCSYN

        else
        {
            mdot_TPAGB=0.0;
        }

#ifdef PN_FAST_WIND
    }
#endif

    Dprint("TPAGB wind %g\n",mdot_TPAGB);
    return(mdot_TPAGB);
} 

static double Mira_pulsation_period(const double mass,
                                    const double radius,
                                    struct stardata_t * stardata)
{
    /* calculate Mira pulsation period */
    double p0 = -2.07 - 0.90*log10(mass) + 1.940 * log10(radius);
    p0 = pow(10.0,p0);    

    /* Limit p0 (radius diverges at low Menv) */
    p0 = Min(p0,2000.0);

#ifdef VW93_MIRA_SHIFT
    /* artificial shift VW93 Mira period */
    p0 = Max(p0+stardata->preferences->vw93_mira_shift,0.0);

    Dprint("Mira pulsation period (M=%g, R=%g, shift=%g) = %g\n",mass,radius,stardata->preferences->vw93_mira_shift,p0);
#else
    Dprint("Mira pulsation period (M=%g, R=%g) = %g\n",mass,radius,p0);
#endif // VW93_MIRA_SHIFT

    return(p0);
}

static double Mira_wind_velocity(const double period)
{
    /*
     * calculate wind velocity at infinity for a Mira
     *
     * This formula is problematic if period < 13.5/0.056 = 241 days,
     * hence limit the period min to 300 days.  
     */
    double vw = Max(1.0e5 * (-13.5+0.056*Max(period,300.0)),0.0) ; 
    return vw;
}
