#include "../binary_c.h"
#include "timestep.h"
#if defined NUCSYN && defined NUCSYN_YIELDS_VS_TIME
#include "../nucsyn/nucsyn_yield_macros.h"
#endif

void timestep_logging(Timestep_prototype_args)
{
    /*
     * Limit timestep because of logging requirements
     */

#ifdef HRDIAG
    if(stellar_type<NEUTRON_STAR)
    {
        /* set global hrdiag logging timestep */

        /* guess for star 1: just set directly from the guess */
        if(stardata->model.hrdiag_log_dt < TINY)
        {
            stardata->model.hrdiag_log_dt = stardata->model.hrdiag_dt_guess;
#ifdef HRDIAG_DTLOG    
            printf("HRDT (deltat) guess for star 1 %g\n",stardata->model.hrdiag_dt_guess);
#endif
        }
        else
            /* second guess (star2) : take the minimum of previous dt and this one */
        {
#ifdef HRDIAG_DTLOG 
            printf("HRDT (deltat) guess for star 2 (guess %g, for star 1 %g)\n",
                   stardata->model.hrdiag_dt_guess,
                   stardata->model.hrdiag_log_dt);
#endif
            stardata->model.hrdiag_log_dt = Min(stardata->model.hrdiag_dt_guess,stardata->model.hrdiag_log_dt);
        }
      
        HRDIAG_DT_NEGATIVE_CHECK;
    
        /* force at least 0.05 (in practice 0.049 - Nyquist/Shannon!)
         * resolution in log t 
         */
        double hrdt_sample = 
            pow(10.0,log10(stardata->model.prev_hrdiag_log_time) + 
                HRDIAG_DT_SAMPLE_FRAC)-
            stardata->model.prev_hrdiag_log_time;
        
#ifdef HRDIAG_DTLOG
        printf("HRDT (deltat) time sample limit Min(dt %g, sample %g) takes us to logt=%g\n",
               stardata->model.hrdiag_log_dt,
               hrdt_sample,
               log10(stardata->model.prev_hrdiag_log_time + hrdt_sample)
            );
#endif
        stardata->model.hrdiag_log_dt = Min(stardata->model.hrdiag_log_dt,hrdt_sample);
    }
    else
    {
        /* we're a remnant and the first star : set the timestep to be huge */
        if(stardata->model.hrdiag_log_dt < TINY)
        {
            stardata->model.hrdiag_log_dt = 1e6;
        }
    }

#endif
#ifdef SELMA
    Limit_timestep(*dt,0.1,star,DT_LIMIT_SELMA);
#endif


#if defined NUCSYN && defined NUCSYN_CEMP_LOGGING
    /*
     * Timestep control for CEMP logging
     */ 
    static const double cemp_min_dt[]=CEMP_MIN_DT_ARRAY;
    if(stellar_type<HeWD)
    {
        if(stardata->model.time > stardata->preferences->CEMP_minimum_age)
        {
            if(stellar_type>MAIN_SEQUENCE)
            {
                /* 
                 * Post-MS stars :
                 * force dt to be short for logging:
                 */
                Limit_timestep(*dt,cemp_min_dt[stellar_type],star,DT_LIMIT_CEMP_POSTMS);
            }
            else
            {
                /*
                 * MS stars: be more clever. We only want to resolve the 
                 * turnoff, where logg approaches stardata->preferences->CEMP_logg_maximum,
                 * so if logg is within 0.1 of this value (which we assume we can resolve!)
                 * reduce the timestep
                 */
                double log10g=logg(star);
                //if(star->starnum==0) printf("MS t=%g dt was %g, log10g=%g (cf threshold %g) ",stardata->model.time,*dt,log10g,stardata->preferences->CEMP_logg_maximum);
                if(log10g > stardata->preferences->CEMP_logg_maximum +0.1 )
                {
                    /* not yet an EMP : use default resolution
                     * but modulate by a factor which determines how close we
                     * are to the threshold logg. This reduces the timestep so
                     * evolution can be fast but should resolve the switch to a
                     * EMP star when logg is small enough
                     */
                    double f=log10g-stardata->preferences->CEMP_logg_maximum;
                    f=Max(0.1,Min(1.0,f));
                    double dtcemp = *dt * f;
                    Limit_timestep(*dt,dtcemp,star,DT_LIMIT_CEMP_NOTEMP);
                }
                else
                {
                    /*
                     * Already or nearly an EMP (within 0.1 in log10g): 
                     * use specified resolution. 
                     */
                    Limit_timestep(*dt,cemp_min_dt[stellar_type],star,DT_LIMIT_CEMP_EMP);
                }
            }
        }
        else
        {            
            /*
             * force a timestep very close to the minimum CEMP age,
             * so we can accurately determine when CEMPing starts 
             */
            double dtnear = stardata->preferences->CEMP_minimum_age - cemp_min_dt[stellar_type] - stardata->model.time;
            if(dtnear > 0.0)
            {
                Limit_timestep(*dt,
                               dtnear,
                               star,
                               DT_LIMIT_CEMP_NEARLY);
            }
        }
        /*
         * Make sure the timestep is not *too* small, 1% of the
         * recommended timestep should be ok
         */
        Floor_timestep(*dt, cemp_min_dt[stellar_type]*0.01,star,DT_LIMIT_CEMP_FLOOR);
    }
#endif


#ifdef FABIAN_IMF_LOG
    /*
     * Log at exact times; therefore check whether normal timestep *dt
     * is smaller than time to next logging point. If so change *dt
     */
    if (ON_MAIN_SEQUENCE(stellar_type)) {
        *dt = Min(stardata->model.fabian_imf_log_timestep, *dt);

        /*
         * change next timestep? (i.e. next logging time closer to actual model time than *dt?)
         */
        Limit_timestep(*dt, stardata->model.next_fabian_imf_log_time-stardata->model.time,star,DT_LIMIT_FABIAN_IMF_LOG);
        *time_remaining = tm - age;
    }
#endif


#ifdef HRDIAG
    if(stardata->preferences->hrdiag_output)
    {
        /* constrain timestep to hit the next hrdiag log time */
        double hrdt=
            pow(10.0,stardata->model.next_hrdiag_log_time) - 
            stardata->model.time;

#ifdef HRDIAG_DTLOG
        printf("HRDT (deltat) check next %g now %g : delta = %g\n",
               pow(10.0,stardata->model.next_hrdiag_log_time),
               stardata->model.time,
               hrdt);
#endif   

        if(hrdt < TINY)
        {
            /* 
             * If hrdt = 0 then we're on the timestep and cannot
             * use the time difference to constrain the next timestep.
             * However, we can guess what it should be.
             */
            Limit_timestep(*dt,stardata->model.hrdiag_log_dt,star,DT_LIMIT_HRD1);
#ifdef HRDIAG_DTLOG     
            printf("HRDT (deltat) at log time set dt=%g (%d from %g)\n",*dt,stellar_type,stardata->model.hrdiag_log_dt);
#endif      
        }
        else
        {
            /*
             * If hrdt > 0 then we're not exactly on the timestep
             * and must set the timestep to hit it exactly the next time.
             */
            Limit_timestep(*dt,hrdt,star,DT_LIMIT_HRD2);
#ifdef HRDIAG_DTLOG
            printf("HRDT (deltat) off timestep set dt=%g (%d)\n",*dt,stellar_type);
#endif      
        }

        HRDIAG_DT_NEGATIVE_CHECK;       

        star->tn = tn;    
    }
#endif //HRDIAG



#ifdef BLUE_STRAGGLER_PROJECT
    /* 
     * Limit the timestep if we're in the final 10% of the main sequence
     * because this is when we're likely to have a blue straggler phase, 
     * or (of course) if we're already a blue straggler
     */    
    if((star->blue_straggler==TRUE) &&
       (ON_MAIN_SEQUENCE(star->stellar_type) &&
        (*dt>BLUE_STRAGGLER_MAX_TIMESTEP) &&
        (stardata->model.time > 0.9*star->tms)&&
        (star->phase_start_mass > star->pms_mass + BLUE_STRAGGLER_ACCRETED_MASS_THRESHOLD)))
    {
        Limit_timestep(*dt,BLUE_STRAGGLER_MAX_TIMESTEP,star,DT_LIMIT_BLUE_STRAGGLER);
    }
#endif// BLUE_STRAGGLER_PROJECT



#if defined __DEPRECATED_YVT__ && defined NUCSYN && defined NUCSYN_YIELDS_VS_TIME
    /* 
     * guess next timestep based on required resolution, 
     * also assume massless remnats don't do much
     */
    //if(stellar_type != MASSLESS_REMNANT)
    {
        double ydt = Yields_timestep_linear;
        double t = stardata->model.time;
                
        if(stardata->preferences->yields_logtimes == TRUE &&
           stardata->model.time < stardata->preferences->yields_startlogtime)
        {
            /*
             * When using log time, we need to define behaviour 
             * before the start of logging. This is because log(0) is
             * not defined.
             */
            double x =
                Min(Yields_timestep_linear,
                    stardata->preferences->yields_startlogtime - stardata->model.time);

            
            Limit_timestep(*dt,
                           x,
                           star,
                           DT_LIMIT_YVT);            
        }
        else
        {
            /*
             * Calculate next yield time, which is either linear or log
             *
             * NB: stardata->model.next_yield_out is out of date at this point,
             * need to recalculate.
             *
             * The call through Time_from_yields_time ensures this is 
             * the linear time.
             */
            double next_yield_time = Time_from_yields_time(Next_yields_time);

            /*
             * Calculate timestep, x, so we hit the next yields_timestep exactly. If this is zero,
             * we're *already* on this timestep, and we shouldn't limit
             */
            double x = next_yield_time - stardata->model.time ;
            if(Is_not_zero(x))
            {
                Limit_timestep(*dt,
                               x,
                               star,
                               DT_LIMIT_YVT);
            }
            
            if(0 && star->starnum==0)
                fprintf(stdout,
                        "dt = %g from ydt = %g, timey = %g, model->time = %g -> x = %g\n",
                        *dt,
                        ydt,
                        Yields_time(t),
                        stardata->model.time,
                        x); 
        }
    }
#endif // NUCSYN && NUCSYN_YIELDS_VS_TIME


}
