#include "timestep.h"
#include "../binary_c.h"


#if defined NUCSYN && defined NUCSYN_ANGELOU_LITHIUM
#include "../nucsyn/nucsyn_angelou.h"
#endif

//#define SECOND_DERIVATIVES
#define SECOND_DERIVATIVE_F 0.1

void timestep_limits(Timestep_prototype_args)
{
    /*
     * Perhaps moderate the timestep depending on the solver?
     */
    /*
      const double fsolver =
      stardata->preferences->solver == SOLVER_FORWARD_EULER ? 1.0 :
      stardata->preferences->solver == SOLVER_RK2 ? 1.0 :
      stardata->preferences->solver == SOLVER_RK4 ? 1.0 :
      stardata->preferences->solver == SOLVER_PREDICTOR_CORRECTOR ? 1.0 :
      1.0;
    */
    const double fsolver = stardata->preferences->timestep_solver_factor;

    if(Is_zero(fsolver))
    {
        Exit_binary_c(BINARY_C_OUT_OF_RANGE,
                      "fsolver (stardata->preferences->timestep_solver_factor) must be > 0\n");
    }
    
#ifdef SECOND_DERIVATIVES
    struct star_t * prevstar =
        &stardata->previous_stardata->star[star->starnum];
#endif
    
#define Second_derivative(N) (                  \
        (star->derivative[(N)]                  \
         -                                      \
         prevstar->derivative[(N)])/            \
        stardata->model.dt                      \
        )
    

    /*
     * Limit timestep to prevent too much angular 
     * momentum change (<0.5%) by magnetic braking. 
     *
     * The original BSE code took 3% changes as the maximum,
     * but this causes oscillation with the tides.
     */
    if(Use_timestep(DT_LIMIT_STELLAR_MAGNETIC_BRAKING) &&
       stellar_type<HeWD && 
       fabs(star->derivative[DERIVATIVE_STELLAR_ANGMOM_MAGNETIC_BRAKING])>1e-30 &&
       star->angular_momentum > MIN_ANGMOM_FOR_TIMESTEP_AND_REJECTION_TESTS)
    {
        const double dtmb = 
            fsolver * stardata->preferences->timestep_multipliers[DT_LIMIT_STELLAR_MAGNETIC_BRAKING] * star->angular_momentum * 1e-6 /
            Max(1e-50,fabs(star->derivative[DERIVATIVE_STELLAR_ANGMOM_MAGNETIC_BRAKING]));
        Dprint("dtmb = %g from fsolver=%g * fmult=%g * J=%g / dJ/dt=%g\n",
               dtmb,
               fsolver,
               stardata->preferences->timestep_multipliers[DT_LIMIT_STELLAR_MAGNETIC_BRAKING],
               star->angular_momentum,
               star->derivative[DERIVATIVE_STELLAR_ANGMOM_MAGNETIC_BRAKING]);
        
        Limit_timestep(*dt,dtmb,star,DT_LIMIT_STELLAR_MAGNETIC_BRAKING);
    }
    
    if(Use_timestep(DT_LIMIT_STELLAR_MASS_LOSS)
       &&
       stellar_type<HeWD)
    {
        /*
         * Limit timestep to prevent mass changing by > 1%
         */
        const double mdot = fabs(star->derivative[DERIVATIVE_STELLAR_MASS]);
        if(mdot>1e-14)
        {
            /* representative mass to be resolved */
            const double m = Min(0.1*(star->mass - star->core_mass),0.01*star->mass);

            if(m > 0.01)
            {
                /* hence mass loss timescale */
                double tmdot = fsolver * stardata->preferences->timestep_multipliers[DT_LIMIT_STELLAR_MASS_LOSS] * m/Max(1e-50,mdot) * 1e-6;
                
                /* resolve mass loss */
                Limit_timestep(*dt,tmdot,star,DT_LIMIT_STELLAR_MASS_LOSS);
            }
        }
    }
    
    /*
     * Limit timestep to prevent angular momentum changing by > 1%
     */
    if(Use_timestep(DT_LIMIT_STELLAR_ANGMOM))
    {
        const double jdot = fabs(star->derivative[DERIVATIVE_STELLAR_ANGMOM]);
        if(jdot>1e-20 && star->angular_momentum > MIN_ANGMOM_FOR_TIMESTEP_AND_REJECTION_TESTS)
        {
            const double dtj = fsolver *
                stardata->preferences->timestep_multipliers[DT_LIMIT_STELLAR_ANGMOM] *
                star->angular_momentum/Max(1e-50,jdot) * 1e-6;
            Limit_timestep(*dt,dtj,star,DT_LIMIT_STELLAR_ANGMOM);
        
            Dprint("Jdotstar%d %g %g (wind %g %g RLOF %g %g tides %g MB %g) -> timescale %g y\n",
                   star->starnum,
                   jdot,
                   star->derivative[DERIVATIVE_STELLAR_ANGMOM_WIND_LOSS],
                   star->derivative[DERIVATIVE_STELLAR_ANGMOM_WIND_GAIN],
                   star->derivative[DERIVATIVE_STELLAR_ANGMOM_RLOF_LOSS],
                   star->derivative[DERIVATIVE_STELLAR_ANGMOM_RLOF_GAIN],
                   star->derivative[DERIVATIVE_STELLAR_ANGMOM_TIDES],
                   star->derivative[DERIVATIVE_STELLAR_ANGMOM_MAGNETIC_BRAKING],
                   star->angular_momentum,
                   star->angular_momentum/jdot
                );

            if(0)printf("JDOTDT%d %30.20e %g %g %g %g %g\n",
                        star->starnum,
                        stardata->model.time,
                        fabs(star->angular_momentum/star->derivative[DERIVATIVE_STELLAR_ANGMOM]),
                        fabs(star->angular_momentum/star->derivative[DERIVATIVE_STELLAR_ANGMOM_MAGNETIC_BRAKING]),
                        fabs(star->angular_momentum/star->derivative[DERIVATIVE_STELLAR_ANGMOM_RLOF_GAIN]),
                        fabs(star->angular_momentum/star->derivative[DERIVATIVE_STELLAR_ANGMOM_RLOF_LOSS]),
                        fabs(star->omega/star->derivative[DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES])
                );
            /*
             * Resolve second derivative
             */
#ifdef SECOND_DERIVATIVES
            if(Use_timestep(DT_LIMIT_STELLAR_ANGMOM2))
            {
                const double dtj2 = 1e-6 * fsolver * stardata->preferences->timestep_multipliers[DT_LIMIT_STELLAR_ANGMOM2] * SECOND_DERIVATIVE_F *
                    fabs(star->derivative[DERIVATIVE_STELLAR_ANGMOM] / 
                         Second_derivative(DERIVATIVE_STELLAR_ANGMOM));
                Limit_timestep(*dt,dtj2,star,DT_LIMIT_STELLAR_ANGMOM2);
            }
#endif
        }
    }

    if(System_is_binary)
    {
        /*
         * Resolve stellar tidal changes
         */
        if(fabs(star->omega) > 1e-20)
        {
            /*
             * Estimate the tidal timescale from 
             * omega / domega/dt and e / de/dt
             */
            if(Use_timestep(DT_LIMIT_TIDES))
            {
                const double ttid_omega = 
                    fabs(star->omega) / 
                    Max(1e-50,fabs(star->derivative[DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES]))
#ifdef EXPONENTIAL_TIDES
                    * 1.0 
#else
                    /* ad-hoc term to take into account tidal strength */
                    * Min(1.0,1.0/Pow3d4(stardata->preferences->tidal_strength_factor))
#endif
                    ;
                
                /* 
                 * Hence the tidal timescale in y
                 */
                const double tidal_timescale = Min3(1e100,
                                                    stardata->common.orbit.tcirc,
                                                    ttid_omega);
            
                const double dt_tides =
                    (star->mass - star->core_mass < 0.01) ?
                    1e100 : 
                    (fsolver * stardata->preferences->timestep_multipliers[DT_LIMIT_TIDES] * tidal_timescale * 1e-6);
                Dprint("DT TIDES : omega %g, circ %g -> %g Myr\n",ttid_omega,stardata->common.orbit.tcirc,dt_tides);
            
                Limit_timestep(*dt,dt_tides,star,DT_LIMIT_TIDES);
            }
            
#ifdef SECOND_DERIVATIVES
            if(Use_timestep(DT_LIMIT_TIDES2))
            {
                const double dt_tides2 = fsolver * stardata->preferences->timestep_multipliers[DT_LIMIT_TIDES2] * SECOND_DERIVATIVE_F *
                    fabs(star->derivative[DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES] /
                         Second_derivative(DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES));
                Limit_timestep(*dt,dt_tides2,star,DT_LIMIT_TIDES2);
            }
#endif
            
            /*
             * Also try to resolve changes in the stellar angular velocity
             * becauses of stellar evolution (i.e. changes in stellar radius).
             */
            if(Use_timestep(DT_LIMIT_RADIUS_CHANGES) &&
               Is_not_zero(star->derivative[DERIVATIVE_STELLAR_RADIUS]))
            {
                const double dtR = fabs(star->radius / (star->derivative[DERIVATIVE_STELLAR_RADIUS]));
                const double dt_stellar_radius_changes = fsolver *
                    stardata->preferences->timestep_multipliers[DT_LIMIT_RADIUS_CHANGES] *
                    Max(1e-50,dtR) * 1e-6;
                Limit_timestep(*dt,dt_stellar_radius_changes,star,DT_LIMIT_RADIUS_CHANGES);
            }

#ifdef __LOGGING
            if(star->starnum==1)
            {
                printf("dt_tides %g omega = %g vs omegadot = %g vs omega_orb %g : dt_tides %g\n",
                       stardata->model.time,
                       fabs(star->omega),
                       Max(1e-50,fabs(star->derivative[DERIVATIVE_STELLAR_ANGULAR_VELOCITY_TIDES])),
                       stardata->common.orbit.angular_frequency,
                       dt_tides
                    );
            }
#endif // __LOGGING
        }

        timestep_RLOF(Timestep_call_args);

        {
            double accretion_rate = Mdot_gain(star);
            if(In_range(stardata->model.time,
                        stardata->preferences->accretion_start_time,
                        stardata->preferences->accretion_end_time))
            {
                accretion_rate = Max3(accretion_rate,
                                      stardata->preferences->mass_accretion_rate1,
                                      stardata->preferences->mass_accretion_rate2);
            }
            if(Is_not_zero(accretion_rate))
            {
                if(Use_timestep(DT_LIMIT_MASS_GAIN))
                {
                    const double dtlim = fsolver * 0.1 * star->mass / accretion_rate;
                    Limit_timestep(*dt,
                                   dtlim,
                                   star,
                                   DT_LIMIT_MASS_GAIN);
                }
                
#ifdef SECOND_DERIVATIVES
                if(Use_timestep(DT_LIMIT_MASS_GAIN2))
                {
                    const double dtlim2 =  SECOND_DERIVATIVE_F * Second_derivative(DT_LIMIT_MASS_GAIN) * 1e-6;
                    Limit_timestep(*dt,
                                   dtlim2,
                                   star,
                                   DT_LIMIT_MASS_GAIN2);
                }
#endif//SECOND_DERIVATIVES
            }
        }

        
        if(stardata->preferences->individual_novae == TRUE &&
           WHITE_DWARF(star->stellar_type) &&
           (star->nova != NOVA_STATE_NONE ||
            stardata->model.time < stardata->model.nova_timeout))
        {


            /*
             * When novae are switched on and off because the system
             * becomes occasionally detached it's hard to know 
             * dtnova (because mdot is sometimes zero).
             *
             * If dtnova is zero, this is the first time, then assume
             * a mass transfer rate of 1e-9 Msun/yr if nothing else.
             */
            const double accretion_rate = Mdot_gain(star);
            const Time dtnova = 1e-6 * nova_recurrence_time(accretion_rate,
                                                            star->mass);

            /*
             * Hence the timestep limit in Myr
             */
            const Time dtlim =
                stardata->preferences->timestep_multipliers[DT_LIMIT_NOVAE] *
                dtnova;

            /*
             * Modulate to combine multiple novae
             *
             * The factor f is applied only if we have already had more than
             * nova_timestep_accelerator_num novae.
             *
             * If accelerator_num <= 0, f = 1.0.
             *
             * If Nnovae < accelerator_num, f = 1.0.
             * 
             * Then, if acclerator_max > 0.0 we have:
             *
             * f = Max(accelerator_max, pow(Nnovae, accelerator_index))
             *
             * otherwise
             *
             * f = pow(Nnovae, accelerator_index).
             */
            const double f =

                stardata->preferences->nova_timestep_accelerator_num < +TINY ?
                /* num < 0.0 : f = 1.0 */
                1.0 : 
                (
                    /* the cap is LARGE_FLOAT (i.e. very large) if acclerator_max <= 0.0 */
                    Max(
                        (stardata->preferences->nova_timestep_accelerator_max < +TINY ?
                         LARGE_FLOAT :
                         stardata->preferences->nova_timestep_accelerator_max),

                        /* check we've had sufficient novae */
                        (star->num_novae > stardata->preferences->nova_timestep_accelerator_num ?
                         /* apply the pow(Nnovae, index) */
                         pow(star->num_novae,stardata->preferences->nova_timestep_accelerator_index) :
                         /* or don't */
                         1.0)
                        )
                    );
            
            /* 
             * limit timestep because of novae
             */
            Limit_timestep(*dt,
                           dtlim*f,
                           star,
                           DT_LIMIT_NOVAE);
        }

        if(star->starnum == 0)
        {
            /*
             * Orbital limits
             */
        
            /*
             * Limit timestep to prevent angular momentum changing by > 1%
             */
            if(Use_timestep(DT_LIMIT_ORBITAL_ANGMOM))
            {
                const double jdot = fabs(stardata->model.derivative[DERIVATIVE_ORBIT_ANGMOM]);
                if(jdot>1e-20 && stardata->common.orbit.angular_momentum > MIN_ANGMOM_FOR_TIMESTEP_AND_REJECTION_TESTS)
                {
                    /* 
                     * factor to phase in shrinkage of the timestep
                     * as the separation drops below 1.0
                     */ 
                    const double fclose = stardata->common.orbit.separation < 1.0 ?
                        Max(1e-5,stardata->common.orbit.separation) : 1.0;
                    const double dtj = 
                        stardata->preferences->timestep_multipliers[DT_LIMIT_ORBITAL_ANGMOM] // default = 0.01
                        * fsolver * fclose * stardata->common.orbit.angular_momentum/
                        Max(1e-50,jdot) * 1e-6;
                    Dprint("Jdotorb = %g Jorb=%g -> timescale %g y, fclose %g\n",
                           jdot,
                           stardata->common.orbit.angular_momentum,
                           stardata->common.orbit.angular_momentum/jdot,
                           fclose
                        );
                    Limit_timestep(*dt,dtj,star,DT_LIMIT_ORBITAL_ANGMOM);
                }
            }


            if(Use_timestep(DT_LIMIT_GRAVITATIONAL_WAVE_RADIATION))
            {
                /*
                 * Resolve spiral in and hence gravitational wave mergers.
                 */
                const double dtspiral = Peters_grav_wave_merger_time(
                    stardata,
                    stardata->star[0].mass,
                    stardata->star[1].mass,
                    stardata->common.orbit.separation,
                    stardata->common.orbit.eccentricity) * 1e-6;
                const double dtmerge = fsolver * dtspiral * stardata->preferences->timestep_multipliers[DT_LIMIT_GRAVITATIONAL_WAVE_RADIATION];
                Dprint("DTMERGE GRAV WAVE : spiral time %g -> dtmerge = %g * dtspiral * %g = %g\n",
                       dtspiral,
                       fsolver,
                       stardata->preferences->timestep_multipliers[DT_LIMIT_GRAVITATIONAL_WAVE_RADIATION],
                       dtmerge);
                Limit_timestep(*dt,dtmerge,star,DT_LIMIT_GRAVITATIONAL_WAVE_RADIATION);
            }
        }

    }//Binary check
    
#ifdef CIRCUMBINARY_DISK_DERMINE
    /* 
     * Reduce timestep to resolve circumbinary disk physics
     */
    if(Use_timestep(DT_LIMIT_CIRCUMBINARY_DISC) &&
       stardata->common.m_circumbinary_disk>0.0)
    {
        double dtcbd = 0.1*circumbinary_disk_massloss_timescale(stardata)*1e-6;
        Limit_timestep(*dt,dtcbd,star,DT_LIMIT_CIRCUMBINARY_DISC);
    }
#endif

#ifdef RESOLVE_POSTAGB
    if(stardata->star[0].stellar_type==TPAGB ||
       (WHITE_DWARF(stardata->star[0].stellar_type)
        &&
        stardata->star[0].age < 10.0))
    {
        /*
         * Don't let the temperature change by ~100K, or 
         * 1% of Teff, whichever is greater.
         */
        double x;
        if(Is_not_zero(stardata->star[0].derivative[DERIVATIVE_STELLAR_TEMPERATURE]) &&
           !isnan(stardata->star[0].derivative[DERIVATIVE_STELLAR_TEMPERATURE]))
        {
            x = 1e-6 * //0.01*
                Max(0.01*Teff(0),100.0)/
                Min(1e-3,fabs(stardata->star[0].derivative[DERIVATIVE_STELLAR_TEMPERATURE]));
        }
        else
        {
            x = 1e50;
        }
        Limit_timestep(*dt,x,star,DT_LIMIT_RESOLVE_POSTAGB);

        /*
         * Estimate nuclear burning lifetime
         */
        double dmcdt = 
            Max3(star->derivative[DERIVATIVE_STELLAR_CO_CORE_MASS],
                 star->derivative[DERIVATIVE_STELLAR_HE_CORE_MASS],
                 star->derivative[DERIVATIVE_STELLAR_GB_CORE_MASS]);
        if(dmcdt > REALLY_TINY)
        {
            const double menv = star->mass - star->core_mass;
            if(menv > REALLY_TINY)
            {
                const double dtnuc = 1e-6*(star->mass - star->core_mass) / dmcdt;
                Limit_timestep(*dt,dtnuc*0.01,star,DT_LIMIT_RESOLVE_POSTAGB);
            }
        }
    }
#endif

#ifdef DISCS
    /*
     * limit to 10% of the disc's natural evolutionary timescale.
     */
    {
        int i;
        for(i=0; i<stardata->common.ndiscs; i++)
        {
            if(stardata->common.discs[i].M >
               M_SUN * stardata->preferences->cbdisc_minimum_mass)
            {
                double dtdisc = 0.1 * timestep_disc(stardata,
                                                    &stardata->common.discs[i]);
                Limit_timestep(*dt,
                               dtdisc,
                               star,
                               DT_LIMIT_DISC);
            }
        }
    }
#endif
    
#if defined RESOLVE_PN_FAST_WIND && defined PN_FAST_WIND
    /*
     * Reduce timestep to better resolve the fast wind at the end of the AGB
     * or in post-common envelope.
     * Also reduce the timestep for stars which are new-born white dwarfs
     */

    
    if(/* giant stars : anything with a small envelope */
        ((ON_GIANT_BRANCH(star->stellar_type))
         &&(star->mass - star->core_mass < RESOLVE_PN_FAST_WIND_MAX_MENV)
            )||
        /* white dwarfs : anything bright (i.e. young) */
        (WHITE_DWARF(star->stellar_type)&&
         (star->luminosity > RESOLVE_PN_FAST_WIND_MIN_WD_LUM)))
    {
        /* 
         * For delta t = 1% * delta m / mdot ~ 0.01 * 1e-3 / 1e-8 we 
         * have dt<1e3 years. 
         *
         * Or assume crossing time is ~1e4 years and we want 1000 points in 
         * there i.e. 10 years ... but these may not all be in the horizontal 
         * part of the HRD.
         *
         * Note that we assume a mass loss rate of 1e-8 for these stars:
         * if you're using the PN_FAST_WIND this is the rate in any case.
         */
        double mdot = AGB(star->stellar_type) ? PN_FAST_WIND_MDOT_AGB :PN_FAST_WIND_MDOT_GB;

        /*
         * A menv/mass^n (n=2) dependence ensures we resolve even
         * massive AGBs moving across the HRD
         */

        /*
         * NB the premultiplier used to be 1e-3 but there were times
         * when this crippled performance...
         */
        double dtfastwind = 0.1 * (star->mass - star->core_mass)
            /(Pow2(star->phase_start_mass) * mdot);
        
        dtfastwind *= 1e-6; // convert to MYr

        /*
         * Enhance the number of grid points in the high-luminosity
         * horizontal passage across the HRD, but only in the PN phase
         */
        double teff = Teff_from_star_struct(star);

        if(teff > 0.5*PN_MIN_TEFF)
        {
            dtfastwind /= Min(5.0,Max(1.0,10*(log10(star->luminosity)-1.5)));
        }
                
        Limit_timestep(*dt,dtfastwind,star,DT_LIMIT_FASTWIND);
        
        Dprint("DT < 0.1 because of PN : menv=%g teff=%g L=%g : dt=%g\n",
               star->mass - star->core_mass,teff,star->luminosity,*dt);

    }
#endif //(defined RESOLVE_PN_FAST_WIND && defined PN_FAST_WIND)

    {
        const double decretion_rate = Mdot_loss(star);
        if(Is_not_zero(decretion_rate))
        {
            const double dtlim = -0.1 * star->mass / decretion_rate;
            Limit_timestep(*dt,
                           dtlim,
                           star,
                           DT_LIMIT_MASS_LOSS);
#ifdef SECOND_DERIVATIVES
            const double dtlim2 = SECOND_DERIVATIVE_F * Second_derivative(DT_LIMIT_MASS_LOSS)
                *1e-6;
            Limit_timestep(*dt,
                           dtlim2,
                           star,
                           DT_LIMIT_MASS_LOSS2);
#endif//SECOND_DERIVATIVES            
        }
    }


#if defined NUCSYN && defined NUCSYN_ANGELOU_LITHIUM
    
    {
        double lithium_time = angelou_factor(time,star->stellar_type);
        
        
        if(Is_not_zero(lithium_time))
        {
            double dtli = fabs(lithium_time - stardata->model.time + star->stellar_type_tstart);
            if(Is_not_zero(dtli))
            {
                Limit_timestep(*dt,
                               Max(1e-3,0.1 * dtli),
                               star,
                               DT_LIMIT_NUCSYN_ANGELOU_LITHIUM);
            }
        }

        if(0)
            fprintf(stderr,
                    ">>>TIMESTEP star %d at time %g this %d prev %d XLi7 %g\n",
                    star->starnum,
                    stardata->model.time,
                    star->angelou_lithium_boost_this_timestep,
                    prevstar->angelou_lithium_boost_this_timestep,
                    Observed_abundance(star,XLi7));
               
        if(
            star->angelou_lithium_boost_this_timestep == FALSE
            &&
            (
                prevstar->angelou_lithium_boost_this_timestep == TRUE
                ||
                Observed_abundance(star,XLi7) > (1.0+TINY)*NUCSYN_ANGELOU_LITHIUM_MINIMUM_Li7)
            )
        {
            /*
             * You might think we should only apply this timestep
             * limiter if the Li7 abundance spikes, but it *may* 
             * spike in the next timestep, we just don't know. 
             */
            double decay_time =
                angelou_factor(decay_time,star->stellar_type);
                
            if(Is_not_zero(decay_time))
            {
                Limit_timestep(*dt,
                               (1.0-TINY) * NUCSYN_ANGELOU_LITHIUM_MAX_TIMESTEP_FACTOR * decay_time,
                               star,
                               DT_LIMIT_NUCSYN_ANGELOU_LITHIUM);
            }
            
            if(0)
                fprintf(stderr,
                        "USE DECAY TIMESTEP star %d at time %g use decay time %g -> %g (this %d prev %d >lowerli7 %d [Li7=%g min=%g] reject %d)\n",
                        star->starnum,
                        stardata->model.time,
                        NUCSYN_ANGELOU_LITHIUM_MAX_TIMESTEP_FACTOR * decay_time,
                        *dt,
                        star->angelou_lithium_boost_this_timestep,
                        prevstar->angelou_lithium_boost_this_timestep,
                        (int)(Observed_abundance(star,XLi7) > (1.0+TINY)*NUCSYN_ANGELOU_LITHIUM_MINIMUM_Li7),
                        Observed_abundance(star,XLi7),
                        (1.0+TINY)*NUCSYN_ANGELOU_LITHIUM_MINIMUM_Li7,
                        star->reject
                    );
        }
    }
#endif



    /* 
     * if dt_zoomfac < 1, don't let the timestep be > 
     * 1.2 * dt_zoomfac * (this or previous timestep)
     */
    if(Use_timestep(DT_LIMIT_ZOOMFAC) &&
       stardata->previous_stardata &&
       stardata->model.dt_zoomfac - 1.0 < -TINY)
    {
        const double dtz = stardata->preferences->timestep_multipliers[DT_LIMIT_ZOOMFAC] *
            Min(stardata->model.dtm,
                stardata->previous_stardata->model.dtm) * 1.2 * stardata->model.dt_zoomfac; 
        Limit_timestep(*dt,dtz,star,DT_LIMIT_STELLAR_MAGNETIC_BRAKING);                       
    }
    
    //show_derivatives(stardata);
}
