#include "../binary_c.h"
#include "timestep.h"

void timestep_TPAGB(Timestep_prototype_args)
{
#ifdef BSE   
    double accretion_rate = Mdot_net(star);

    if(accretion_rate > 0.0)
    {
        /*
         * AGB star is accreting, so is likely
         * a converted white dwarf. In this case, use the 
         * limit from the mass transfer, rather than 
         * attempting to calculate the nuclear timescale
         * here (we cannot possibly know it).
         */
        *time_remaining = LONG_TIME_REMAINING;
#ifdef NUCSYN
#ifdef TPAGB_SPEEDUP
        if(star->num_thermal_pulses>TPAGB_SPEEDUP_AFTER)
        {
            Set_timestep(*dt,0.01,star,DT_LIMIT_TPAGB_NUCSYN_SPEEDUP);
        }
        else
#endif
        {
            double tip=star->interpulse_period*stardata->preferences->dtfac*0.9999;
            Set_timestep(*dt,tip,star,DT_LIMIT_TPAGB_NUCSYN_INTERPULSE);
        }
#endif
    }
    else
    {
        int core_algorithm = AGB_Core_Algorithm;
        if(core_algorithm == AGB_CORE_ALGORITHM_HURLEY)
        {
            double dtt = tscls[age<tscls[12] ? 10 : 11] - age;
            if(Use_timestep(DT_LIMIT_TPAGB))
            {
                Set_timestep(*dt,Min(stardata->preferences->timestep_multipliers[DT_LIMIT_TPAGB]*dtt,0.005),star,DT_LIMIT_TPAGB);
            }
            
#ifdef HRDIAG
            stardata->model.hrdiag_dt_guess = HRDIAG_FRAC* dtt;
#ifdef HRDIAG_DTLOG
            printf("HRDT (deltat) TPAGB %g\n",stardata->model.hrdiag_dt_guess);
#endif
            HRDIAG_DT_NEGATIVE_CHECK;
#ifdef HRDIAG_EXTRA_RESOLUTION_AT_END_AGB
            double menv=star->mass - star->core_mass;
            if(menv < 0.1)
            {
                stardata->model.hrdiag_dt_guess = Min(stardata->model.hrdiag_dt_guess,
                                                      0.1*pow(10.0,-log10(Max(1.0,star->luminosity))));
            }
#endif //HRDIAG_EXTRA_RESOLUTION_AT_END_AGB
#endif  //HRDIAG

            *time_remaining = tn - age;

#ifdef DTR_FUDGES
            // Rob's fudge for small envelopes, sometimes things get
            // unstable
            if(*time_remaining<1e-9)
            {
                *time_remaining = Max(*dt,*time_remaining);
            }
#endif //DTR_FUDGES
        }
        else
        {

            /*
             * We know the interpulse period, so we can use it to determine
             * a minimum timestep 
             *
             * If we're at a large thermal pulse number, assume
             * the AGB star has really become asymptotic in all
             * regards, including nucleosynthesis. In this case we 
             * can extend the timestep.
             */
#ifdef TPAGB_SPEEDUP
            if(star->num_thermal_pulses>TPAGB_SPEEDUP_AFTER)
            {
                if(Use_timestep(DT_LIMIT_TPAGB_NUCSYN_SPEEDUP))
                {
                    Set_timestep(*dt,stardata->preferences->timestep_multipliers[DT_LIMIT_TPAGB_NUCSYN_SPEEDUP],star,DT_LIMIT_TPAGB_NUCSYN_SPEEDUP);
                }
            }
            else
#endif
                if(Use_timestep(DT_LIMIT_TPAGB_NUCSYN_INTERPULSE))
            {
                double tip = star->interpulse_period;

                if(tip < 1e-6)
                {
                    /*
                     * recalculate interpulse
                     */
                    tip = Karakas2002_interpulse_period(stardata,
                                                        star->mc_1tp,
                                                        star->mass,
                                                        star->mc_1tp,
                                                        stardata->common.metallicity,
                                                        0.0,
                                                        star);
                }

                tip *= stardata->preferences->dtfac*//0.9999*
                    stardata->preferences->timestep_multipliers[DT_LIMIT_TPAGB_NUCSYN_INTERPULSE];
                        
                
                // NB tip is in Myr
                Set_timestep(*dt,tip,star,DT_LIMIT_TPAGB_NUCSYN_INTERPULSE);
            }

#if defined SLOW_DOWN_PREROCHE_TPAGB && !defined RLOF_REDUCE_TIMESTEP
            /*
             * Slow down as we approach RLOF, we want to make this
             * as accurate as possible.
             */     
            if(stardata->model.sgl == FALSE &&
               Use_timestep(DT_LIMIT_TPAGB_NUCSYN_PREROCHE))
            {
                double tip=star->interpulse_period*stardata->preferences->dtfac*0.9999;
                double rr=star->radius / star->roche_radius;
                rr=1.0-rr;
                rr=Min(1.0,rr);
                rr=Max(0.1,rr);
                Limit_timestep(*dt,tip*stardata->preferences->timestep_multipliers[DT_LIMIT_TPAGB_NUCSYN_PREROCHE],star,DT_LIMIT_TPAGB_NUCSYN_PREROCHE);
            }
#endif// SLOW_DOWN_PREROCHE_TPAGB

            /*
             * Ignore time_remaining : we don't really know this because
             * the core mass determines what happens, not the time
             */

            *time_remaining = LONG_TIME_REMAINING;
    
#ifdef KARAKAS2002_SMOOTH_AGB_RADIUS_TRANSITION
            /* Approach second pulse slowly */ 
            if(Use_timestep(DT_LIMIT_TPAGB_NUCSYN_KARAKAS_SMOOTH) &&
               star->age-TPAGB_start_time(star)<1e-6*KARAKAS2002_SMOOTH_AGB_RADIUS_TRANSITION_SMOOTHING_TIME)
            {
                double newdt= stardata->preferences->timestep_multipliers[DT_LIMIT_TPAGB_NUCSYN_KARAKAS_SMOOTH] * *dt;
                Set_timestep(*dt,newdt,star,DT_LIMIT_TPAGB_NUCSYN_KARAKAS_SMOOTH);
            }
#endif
        }
    }
#endif//BSE
}
