#ifndef TIMESTEP_H
#define TIMESTEP_H
#include "../binary_c_parameters.h"

/*
 * A list of fixed timesteps : 
 * these should be set up in setup_fixed_timesteps.c
 * and are handled in timestep_fixed_timesteps.c
 */
#define FIXED_TIMESTEP_YIELDS 0
#define FIXED_TIMESTEP_TEST 1

#define NUMBER_OF_FIXED_TIMESTEPS 2

/********************************************************************/


#ifdef HRDIAG
#define HRDIAG_FRAC 0.01
#define HRDIAG_DT_NEGATIVE_CHECK if(stardata->model.hrdiag_dt_guess<0.0){Exit_binary_c(BINARY_C_OUT_OF_RANGE,"hrdiag dt guess < 0 error\n");}

/*
 * timestep should be at least half the width of the time bin
 * (1.0 also seems to work but we want to be sure) 
 */
#define HRDIAG_DT_SAMPLE_FRAC (0.5 * HRDIAG_BIN_SIZE_LOGTIME)
#endif

#define Use_timestep(N)                                                 \
    (Is_not_zero(stardata->preferences->timestep_multipliers[(N)]))

/*
 * Compare two times, A and B, to see if they are equal for the 
 * fixed timestepping scheme.
 *
 * Given that the time is incremented, it is liable to integration
 * errors, so comparing the times to within a very small epsilon (like
 * TINY == DBL_EPSILON) will not work, we have to be smarter, 
 * so instead define TIMESTEP_EPSILON here and use Float_same_within_eps
 * to compare the times.
 *
 * The default TIMESTEP_EPSILON is 1e-10 which is four orders of
 * magnitude bigger than DBL_EPSLION (~1e-14). This allows for considerable
 * error in time integration.
 */
#define TIMESTEP_EPSILON 1e-10

#define Times_are_equal(A,B)                            \
    Float_same_within_eps((A),(B),TIMESTEP_EPSILON)

#define Times_more_or_equal(A,B)                                \
    (                                                           \
        Float_same_within_eps((A),(B),TIMESTEP_EPSILON) ||      \
        More_or_equal((A),(B))                                  \
        )

#ifdef BSE
#define Timestep_call_args                      \
    stellar_type,                               \
        age,                                    \
        tm,                                     \
        tn,                                     \
        tscls,                                  \
        dt,                                     \
        time_remaining,                         \
        star,                                   \
        stardata

#define Timestep_prototype_args                 \
    const Stellar_type stellar_type,            \
        const double age,                       \
        const double tm,                        \
        const double tn,                        \
        const double tscls[],                   \
        double * const dt,                      \
        double * const time_remaining,          \
        struct star_t * const star,             \
        struct stardata_t * const stardata
#else

#define Timestep_call_args                      \
    stellar_type,                               \
        dt,                                     \
        time_remaining,                         \
        star,                                   \
        stardata

#define Timestep_prototype_args                 \
    const Stellar_type stellar_type,            \
        double * const dt,                      \
        double * const time_remaining,          \
        struct star_t * const star,             \
        struct stardata_t * const stardata
#endif

/*
 * Macros to set and/or limit the timestep and save which limiter 
 * is responsible for the current timestep. 
 *
 * Limit_timestep reduces the timestep (this is normally the case) while
 * Floor_timestep sets a floor below which it cannot fall.
 *
 * Modulate_timestep is a bit different : it multiplies a timestep
 * and makes the dtlimiter reason negative - although does not save
 * where this modulation is done. Such modulations are rare, so not usually
 * an issue, but if you want to track them you will have to introduce 
 * a new logging variable.
 */

#define Set_timestep(DT,TO,IN,WHY)              \
    {                                           \
        (DT)=(TO);                              \
        Dprint("Set timestep to %g why %d\n",   \
               (TO),                            \
               (WHY));                          \
        if(WHY!=DT_LIMIT_MAXIMUM_TIMESTEP &&    \
           WHY!=DT_LIMIT_MINIMUM_TIMESTEP)      \
        {                                       \
            (IN)->dtlimiter=(WHY);              \
        }                                       \
    }

#define Limit_timestep(DT,TO,IN,WHY)                    \
    {                                                   \
        if((DT)>(TO))                                   \
        {                                               \
            Dprint("Limit timestep to %g why %d\n",     \
                   (TO),                                \
                   (WHY));                              \
            Set_timestep((DT),(TO),(IN),(WHY));         \
        }                                               \
    }

#define Floor_timestep(DT,TO,IN,WHY)                    \
    {                                                   \
        if((DT)<(TO))                                   \
        {                                               \
            Dprint("Floor timestep to %g why %d\n",     \
                   (TO),                                \
                   (WHY));                              \
            Set_timestep((DT),(TO),(IN),(WHY));         \
        }                                               \
    }

#define Modulate_timestep(DT,BY,IN)                     \
    {                                                   \
        if(!Fequal(1.0,(BY)))                           \
        {                                               \
            (DT)*=(BY);                                 \
            Dprint("Modulate timestep by %g to %g\n",   \
                   (BY),                                \
                   (DT));                               \
            (IN)->dtlimiter=-abs((IN)->dtlimiter);      \
        }                                               \
    }
#else



#define DT_LIMIT_NONE 0
#define DT_LIMIT_MS 1
#define DT_LIMIT_PREMS 2
#define DT_LIMIT_PREROCHE_MS 3
#define DT_LIMIT_PREROCHE_HG 4
#define DT_LIMIT_HG 5
#define DT_LIMIT_FGB 6
#define DT_LIMIT_CHeB 7
#define DT_LIMIT_EAGB 8
#define DT_LIMIT_EAGB_AXEL 9
#define DT_LIMIT_EAGB_PREROCHE 10
#define DT_LIMIT_TPAGB 11
#define DT_LIMIT_TPAGB_NUCSYN_INTERPULSE 12
#define DT_LIMIT_TPAGB_NUCSYN_SPEEDUP 13
#define DT_LIMIT_TPAGB_NUCSYN_PREROCHE 14
#define DT_LIMIT_TPAGB_NUCSYN_KARAKAS_SMOOTH 15
#define DT_LIMIT_HeMS 16
#define DT_LIMIT_HeHG_GB 17
#define DT_LIMIT_OTHER_STELLAR_TYPES 18
#define DT_LIMIT_STELLAR_ANGMOM 19
#define DT_LIMIT_STELLAR_MASS_LOSS 20
#define DT_LIMIT_STELLAR_MAGNETIC_BRAKING 21
#define DT_LIMIT_CIRCUMBINARY_DISC 22
#define DT_LIMIT_RESOLVE_POSTAGB 23
#define DT_LIMIT_DISC 24
#define DT_LIMIT_FASTWIND 25
#define DT_LIMIT_SELMA 26
#define DT_LIMIT_CEMP_POSTMS 27
#define DT_LIMIT_CEMP_NOTEMP 28
#define DT_LIMIT_CEMP_EMP 29
#define DT_LIMIT_CEMP_NEARLY 30
#define DT_LIMIT_CEMP_FLOOR 31
#define DT_LIMIT_FABIAN_IMF_LOG 32
#define DT_LIMIT_HRD1 33
#define DT_LIMIT_HRD2 34
#define DT_LIMIT_BLUE_STRAGGLER 35
#define DT_LIMIT_YVT 36
#define DT_LIMIT_MINIMUM_TIMESTEP 37
#define DT_LIMIT_MAXIMUM_TIMESTEP 38
#define DT_LIMIT_NOVAE 39
#define DT_LIMIT_ARTIFICIAL_ACCRETION 40
#define DT_LIMIT_SN 41
#define DT_LIMIT_MASS_GAIN 42
#define DT_LIMIT_MASS_LOSS 43
#define DT_LIMIT_TIDES 44
#define DT_LIMIT_NUCSYN_ANGELOU_LITHIUM 45
#define DT_LIMIT_CARBON_BURNING 46
#define DT_LIMIT_BURN_IN 47
#define DT_LIMIT_RADIUS_CHANGES 48
#define DT_LIMIT_MASSLESS_REMNANT 49
#define DT_LIMIT_ORBITAL_ANGMOM 50
#define DT_LIMIT_STELLAR_ANGMOM2 51
#define DT_LIMIT_TIDES2 52
#define DT_LIMIT_MASS_GAIN2 53
#define DT_LIMIT_MASS_LOSS2 54
#define DT_LIMIT_GRAVITATIONAL_WAVE_RADIATION 55
#define DT_LIMIT_RLOF 56
#define DT_LIMIT_FIXED_TIMESTEP 57
#define DT_LIMIT_TIME_REMAINING 58
#define DT_LIMIT_ZOOMFAC 59
#define DT_NUMBER_OF_TIMESTEP_LIMITERS 60

#define DT_LIMIT_STRINGS                                \
    "DT_LIMIT_NONE",                                    \
        "DT_LIMIT_MS",                                  \
        "DT_LIMIT_PREMS",                               \
        "DT_LIMIT_PREROCHE_MS",                         \
        "DT_LIMIT_PREROCHE_HG",                         \
        "DT_LIMIT_HG",                                  \
        "DT_LIMIT_FGB",                                 \
        "DT_LIMIT_CHeB",                                \
        "DT_LIMIT_EAGB",                                \
        "DT_LIMIT_EAGB_AXEL",                           \
        "DT_LIMIT_EAGB_PREROCHE",   /*10*/              \
        "DT_LIMIT_TPAGB",                               \
        "DT_LIMIT_TPAGB_NUCSYN_INTERPULSE",             \
        "DT_LIMIT_TPAGB_NUCSYN_SPEEDUP",                \
        "DT_LIMIT_TPAGB_NUCSYN_PREROCHE",               \
        "DT_LIMIT_TPAGB_NUCSYN_KARAKAS_SMOOTH",         \
        "DT_LIMIT_HeMS",                                \
        "DT_LIMIT_HeHG_GB",                             \
        "DT_LIMIT_OTHER_STELLAR_TYPES",                 \
        "DT_LIMIT_STELLAR_ANGMOM",                      \
        "DT_LIMIT_STELLAR_MASS_LOSS",     /*20*/        \
        "DT_LIMIT_STELLAR_MAGNETIC_BRAKING",            \
        "DT_LIMIT_CIRCUMBINARY_DISC",                   \
        "DT_LIMIT_RESOLVE_POSTAGB",                     \
        "DT_LIMIT_DISC",                                \
        "DT_LIMIT_FASTWIND",                            \
        "DT_LIMIT_SELMA",                               \
        "DT_LIMIT_CEMP_POSTMS",                         \
        "DT_LIMIT_CEMP_NOTEMP",                         \
        "DT_LIMIT_CEMP_EMP",                            \
        "DT_LIMIT_CEMP_NEARLY",             /*30*/      \
        "DT_LIMIT_CEMP_FLOOR",                          \
        "DT_LIMIT_FABIAN_IMF_LOG",                      \
        "DT_LIMIT_HRD1",                                \
        "DT_LIMIT_HRD2",                                \
        "DT_LIMIT_BLUE_STRAGGLER",                      \
        "DT_LIMIT_YVT",                                 \
        "DT_LIMIT_MINIMUM_TIMESTEP",                    \
        "DT_LIMIT_MAXIMUM_TIMESTEP",                    \
        "DT_LIMIT_NOVAE",                               \
        "DT_LIMIT_ARTIFICIAL_ACCRETION", /* 40 */       \
        "DT_LIMIT_SN",                                  \
        "DT_LIMIT_MASS_GAIN",                           \
        "DT_LIMIT_MASS_LOSS",                           \
        "DT_LIMIT_TIDES",                               \
        "DT_LIMIT_NUCSYN_ANGELOU_LITHIUM",              \
        "DT_LIMIT_CARBON_BURNING",                      \
        "DT_LIMIT_BURN_IN",                             \
        "DT_LIMIT_RADIUS_CHANGES",                      \
        "DT_LIMIT_MASSLESS_REMNANT",                    \
        "DT_LIMIT_ORBITAL_ANGMOM",/*50*/                \
        "DT_LIMIT_STELLAR_ANGMOM2",                     \
        "DT_LIMIT_TIDES2",                              \
        "DT_LIMIT_MASS_GAIN2",                          \
        "DT_LIMIT_MASS_LOSS2", /*54*/                   \
        "DT_LIMIT_GRAVITATIONAL_WAVE_RADIATION",/*55*/  \
        "DT_LIMIT_RLOF",/*56*/                          \
        "DT_LIMIT_FIXED_TIMESTEP",/*57*/                \
        "DT_LIMIT_TIME_REMAINING", /*58*/               \
        "DT_LIMIT_ZOOMFAC"



#define Dt_limit_string(N)                                              \
    (                                                                   \
        (N)== DT_LIMIT_NONE ? "DT_LIMIT_NONE" :                         \
        (N)== DT_LIMIT_MS ? "DT_LIMIT_MS" :                             \
        (N)== DT_LIMIT_PREMS ? "DT_LIMIT_PREMS" :                       \
        (N)== DT_LIMIT_PREROCHE_MS ? "DT_LIMIT_PREROCHE_MS" :           \
        (N)== DT_LIMIT_PREROCHE_HG ? "DT_LIMIT_PREROCHE_HG" :           \
        (N)== DT_LIMIT_HG ? "DT_LIMIT_HG" :                             \
        (N)== DT_LIMIT_FGB ? "DT_LIMIT_FGB" :                           \
        (N)== DT_LIMIT_CHeB ? "DT_LIMIT_CHeB" :                         \
        (N)== DT_LIMIT_EAGB ? "DT_LIMIT_EAGB" :                         \
        (N)== DT_LIMIT_EAGB_AXEL ? "DT_LIMIT_EAGB_AXEL" :               \
        (N)== DT_LIMIT_EAGB_PREROCHE ? "DT_LIMIT_EAGB_PREROCHE" :    /*10*/ \
        (N)== DT_LIMIT_TPAGB ? "DT_LIMIT_TPAGB" :                       \
        (N)== DT_LIMIT_TPAGB_NUCSYN_INTERPULSE ? "DT_LIMIT_TPAGB_NUCSYN_INTERPULSE" : \
        (N)== DT_LIMIT_TPAGB_NUCSYN_SPEEDUP ? "DT_LIMIT_TPAGB_NUCSYN_SPEEDUP" : \
        (N)== DT_LIMIT_TPAGB_NUCSYN_PREROCHE ? "DT_LIMIT_TPAGB_NUCSYN_PREROCHE" : \
        (N)== DT_LIMIT_TPAGB_NUCSYN_KARAKAS_SMOOTH ? "DT_LIMIT_TPAGB_NUCSYN_KARAKAS_SMOOTH" : \
        (N)== DT_LIMIT_HeMS ? "DT_LIMIT_HeMS" :                         \
        (N)== DT_LIMIT_HeHG_GB ? "DT_LIMIT_HeHG_GB" :                   \
        (N)== DT_LIMIT_OTHER_STELLAR_TYPES ? "DT_LIMIT_OTHER_STELLAR_TYPES" : \
        (N)== DT_LIMIT_STELLAR_ANGMOM ? "DT_LIMIT_STELLAR_ANGMOM" :     \
        (N)== DT_LIMIT_STELLAR_MASS_LOSS ? "DT_LIMIT_STELLAR_MASS_LOSS" :      /*20*/ \
        (N)== DT_LIMIT_STELLAR_MAGNETIC_BRAKING ? "DT_LIMIT_STELLAR_MAGNETIC_BRAKING" : \
        (N)== DT_LIMIT_CIRCUMBINARY_DISC ? "DT_LIMIT_CIRCUMBINARY_DISC" : \
        (N)== DT_LIMIT_RESOLVE_POSTAGB ? "DT_LIMIT_RESOLVE_POSTAGB" :   \
        (N)== DT_LIMIT_DISC ? "DT_LIMIT_DISC" :                         \
        (N)== DT_LIMIT_FASTWIND ? "DT_LIMIT_FASTWIND" :                 \
        (N)== DT_LIMIT_SELMA ? "DT_LIMIT_SELMA" :                       \
        (N)== DT_LIMIT_CEMP_POSTMS ? "DT_LIMIT_CEMP_POSTMS" :           \
        (N)== DT_LIMIT_CEMP_NOTEMP ? "DT_LIMIT_CEMP_NOTEMP" :           \
        (N)== DT_LIMIT_CEMP_EMP ? "DT_LIMIT_CEMP_EMP" :                 \
        (N)== DT_LIMIT_CEMP_NEARLY ? "DT_LIMIT_CEMP_NEARLY" :              /*30*/ \
        (N)== DT_LIMIT_CEMP_FLOOR ? "DT_LIMIT_CEMP_FLOOR" :             \
        (N)== DT_LIMIT_FABIAN_IMF_LOG ? "DT_LIMIT_FABIAN_IMF_LOG" :     \
        (N)== DT_LIMIT_HRD1 ? "DT_LIMIT_HRD1" :                         \
        (N)== DT_LIMIT_HRD2 ? "DT_LIMIT_HRD2" :                         \
        (N)== DT_LIMIT_BLUE_STRAGGLER ? "DT_LIMIT_BLUE_STRAGGLER" :     \
        (N)== DT_LIMIT_YVT ? "DT_LIMIT_YVT" :                           \
        (N)== DT_LIMIT_MINIMUM_TIMESTEP ? "DT_LIMIT_MINIMUM_TIMESTEP" : \
        (N)== DT_LIMIT_MAXIMUM_TIMESTEP ? "DT_LIMIT_MAXIMUM_TIMESTEP" : \
        (N)== DT_LIMIT_NOVAE ? "DT_LIMIT_NOVAE" :                       \
        (N)== DT_LIMIT_ARTIFICIAL_ACCRETION ? "DT_LIMIT_ARTIFICIAL_ACCRETION" :  /* 40 */ \
        (N)== DT_LIMIT_SN ? "DT_LIMIT_SN" :                             \
        (N)== DT_LIMIT_MASS_GAIN ? "DT_LIMIT_MASS_GAIN" :               \
        (N)== DT_LIMIT_MASS_LOSS ? "DT_LIMIT_MASS_LOSS" :               \
        (N)== DT_LIMIT_TIDES ? "DT_LIMIT_TIDES" :                       \
        (N)== DT_LIMIT_NUCSYN_ANGELOU_LITHIUM ? "DT_LIMIT_NUCSYN_ANGELOU_LITHIUM" : \
        (N)== DT_LIMIT_CARBON_BURNING ? "DT_LIMIT_CARBON_BURNING" :     \
        (N)== DT_LIMIT_BURN_IN ? "DT_LIMIT_BURN_IN" :                   \
        (N)== DT_LIMIT_RADIUS_CHANGES ? "DT_LIMIT_RADIUS_CHANGES" :     \
        (N)== DT_LIMIT_MASSLESS_REMNANT ? "DT_LIMIT_MASSLESS_REMNANT" : \
        (N)== DT_LIMIT_ORBITAL_ANGMOM ? "DT_LIMIT_ORBITAL_ANGMOM" : /*50*/ \
        (N)== DT_LIMIT_STELLAR_ANGMOM2 ? "DT_LIMIT_STELLAR_ANGMOM2" :   \
        (N)== DT_LIMIT_TIDES2 ? "DT_LIMIT_TIDES2" :                     \
        (N)== DT_LIMIT_MASS_GAIN2 ? "DT_LIMIT_MASS_GAIN2" :             \
        (N)== DT_LIMIT_MASS_LOSS2 ? "DT_LIMIT_MASS_LOSS2" :  /*54*/     \
        (N)== DT_LIMIT_GRAVITATIONAL_WAVE_RADIATION ? "DT_LIMIT_GRAVITATIONAL_WAVE_RADIATION" : /*55*/ \
        (N)== DT_LIMIT_RLOF ? "DT_LIMIT_RLOF" : /*56*/                  \
        (N)== DT_LIMIT_FIXED_TIMESTEP ? "DT_LIMIT_FIXED_TIMESTEP" : /*57*/ \
        (N)== DT_LIMIT_TIME_REMAINING ? "DT_LIMIT_TIME_REMAINING" : /*58*/ \
        (N)== DT_LIMIT_ZOOMFAC ? "DT_LIMIT_ZOOMFAC" : /*59*/            \
        "Unknown"                                                       \
        )


#endif // TIMESTEP_H
