#include "../binary_c.h"
#include "timestep.h"


void stellar_timestep(Timestep_prototype_args)
{
    /*
     * Calculate the stellar timestep.
     *
     * This is set in *dt and has units of Myr. 
     *
     * Note that the stellar type can be MASSLESS_REMNANT.
     */
    Dprint("stellar_timestep at %g in star %d stellar_type=%d=%d, age=%g, tm=%g, tn=%g time_remaining=%g\n",
           stardata->model.time,
           star->starnum,
           stellar_type,
           star->stellar_type,
#ifdef BSE
           age,
           tm,
           tn
#else
           star->age,
           0.0,
           0.0
#endif//BSE
           ,*time_remaining
        );

    /* 
     * First, set no limit
     */
    *dt  = LONG_TIMESTEP;
    star->dtlimiter = DT_LIMIT_NONE;
    
#ifdef HRDIAG
    stardata->model.hrdiag_dt_guess=100.0;
#endif


    if(stardata->preferences->stellar_structure_algorithm !=
       STELLAR_STRUCTURE_ALGORITHM_NONE)
    {
        switch(stellar_type)
        {
        case LOW_MASS_MAIN_SEQUENCE:
        case MAIN_SEQUENCE:
            timestep_MS(Timestep_call_args);
            break;

        case HERTZSPRUNG_GAP:
            timestep_HG(Timestep_call_args);
            break;

        case FIRST_GIANT_BRANCH:
            timestep_FGB(Timestep_call_args);
            break;

        case CHeB:
            timestep_CHeB(Timestep_call_args);
            break;

        case EAGB:
            timestep_EAGB(Timestep_call_args);
            break;

        case TPAGB:
            timestep_TPAGB(Timestep_call_args);
            break;

        case HeMS:
            timestep_HeMS(Timestep_call_args);
            break;
        
        case HeHG:
        case HeGB:
            timestep_HeHG_HeGB(Timestep_call_args);
            break;

        case MASSLESS_REMNANT:
            timestep_massless_remnant(Timestep_call_args);
            break;
            
        default:
            timestep_other(Timestep_call_args);

        }
    }
    
    timestep_limits(Timestep_call_args);
    timestep_logging(Timestep_call_args);
    timestep_fixed_timesteps(Timestep_call_args);
    timestep_modulation(Timestep_call_args);
    timestep_hard_limits(Timestep_call_args);
  
    star->stellar_timestep = *dt;

//#define ___LOGGING
#ifdef ___LOGGING
    fprintf(stdout,
            "%sstellar_timestep at t=%23.15g model %d reject %u out star %d stellar_type %d (m=%g mc=%g L=%g R=%g minit=%g menv=%g SN=%d J*=%g; a=%g e=%g Jorb=%g, ω*=%g ωorb=%g tcirc=%g y), age=%20.12g, tm=%20.12g, tn=%20.12g    >>> %sdt=%20.12g Myr%s <<<    time_remaining=%g : set by %s%s%s (%d) : logtnow=%g logtnext=%g (Fequal? %s Times_are_equal? %s : diff %g) triggerd? %d %s\n",
            YELLOW,
            stardata->model.time,
            stardata->model.model_number,
            stardata->model.reject,
            star->starnum,
            star->stellar_type,
            star->mass,
            star->core_mass,
            star->luminosity,
            star->radius,
            star->phase_start_mass,
            star->mass - star->core_mass,
            star->SN_type,
            star->angular_momentum,
            stardata->common.orbit.separation,
            stardata->common.orbit.eccentricity,
            stardata->common.orbit.angular_momentum,
            star->omega,
            stardata->common.orbit.angular_frequency,
            stardata->common.orbit.tcirc,
#ifdef BSE
            age,
            tm,
            tn,
#else
            star->age,
            0.0,
            0.0,
#endif
            
            BRIGHT_YELLOW,
            *dt,
            YELLOW,
            *time_remaining,
            BRIGHT_MAGENTA,
            Dt_limit_string(abs(star->dtlimiter)),
            YELLOW,
            star->dtlimiter,
            log10(stardata->model.time),
            log10(stardata->model.time + *dt),
            Yesno(Fequal(log10(stardata->model.time),
                         log10(stardata->model.time + *dt))),
            Yesno(Times_are_equal(log10(stardata->model.time),
                                  log10(stardata->model.time + *dt))),
            log10(stardata->model.time) - 
            log10(stardata->model.time + *dt),
            stardata->model.fixed_timestep_triggered,
            COLOUR_RESET
        );
#endif //___LOGGING

    return;
}
 
