#include "../binary_c.h"
#include "supernovae.h"

/*
 * Do supernova : 
 * remove mass, update the stellar type, kick the star
 *
 * The star has the pre-supernova structure in it,
 * while star->new_supernova contains a new_stellar_structure
 * struct containing the post-explosion information.
 */

void supernova(struct stardata_t * const stardata,
               struct star_t * const star)
{
    /*
     * Save information about the pre-explosion object.
     *
     * if star->new_supernova is NULL, then this function
     * has already been called. This is fine: this is to
     * allow star->SN_type to remain (for logging purposes)
     * and then be reset on the next timestep.
     */
    if(star->new_supernova == NULL) return;
    
    Dprint("Supernova in star %d stellar_type %d SN_type %s (%d) m %g, new_supernova = %p\n",
           star->starnum,
           star->stellar_type,
           SN_String(star->SN_type),
           star->SN_type,
           star->mass,
           star->new_supernova
        );


    struct star_t * pre_explosion_star = New_star_from(star);
    struct star_t * companion MAYBE_UNUSED = Other_star_struct(star);
    double mass_ejected MAYBE_UNUSED = star->mass - 
        star->new_supernova->new_stellar_structure->mass;

#ifdef LOG_SUPERNOVAE
    /*
     * Note that log_sn has been deprecated : you
     * should catch supernovae in log_every_timestep, not here
     */
    log_sn(stardata,star->starnum,star->SN_type,PRE_SN);
#endif

    /*
     * Save the pre-explosion state of the system
     */
    struct stardata_t * pre_explosion_stardata = New_stardata_from(stardata);
    
    /*
     * Update the stellar structure to become
     * a post-SN object (NS, BH or MASSLESS_REMNANT).
     * Its mass and stellar type certainly change.
     * We update its radius and luminosity below.
     */
    if(star->new_supernova == NULL)
    {
        Exit_binary_c(BINARY_C_POINTER_FAILURE,
                      "new_supernova pointer is NULL, it should be filled with data.");
    }
    else if(star->new_supernova->new_stellar_structure == NULL)
    {
        Exit_binary_c(BINARY_C_POINTER_FAILURE,
                      "new_supernova->new_stellar_structure pointer is NULL, it should be filled with data.");
    }

    set_star_struct_from_stellar_structure(
        star->new_supernova->new_stellar_structure,
        star);
    star->core_mass = star->mass;
    star->core_radius = star->radius;
    star->phase_start_core_mass = star->core_mass;
    star->stellar_timestep = star->stellar_type==MASSLESS_REMNANT ? 1e3 : 0.01;
    star->phase_start_mass = star->mass;
    star->age = 0.0;
    star->epoch = stardata->model.time - star->age;
    star->mass_at_RLOF_start = star->phase_start_mass;
    star->SN_type = star->new_supernova->new_stellar_structure->SN_type;


    /*
     * Update stellar structure
     */

#ifdef BSE
    
    /*
     * Set L,R,Rc, etc.
     */
    double GB[GB_ARRAY_SIZE];
    double lums[LUMS_ARRAY_SIZE];
    double tscls[TSCLS_ARRAY_SIZE];
    double tm,tn;
        
    stellar_timescales(star->stellar_type,
                       star->phase_start_mass,
                       star->mass,
                       &tm,
                       &tn,
                       tscls,
                       lums,
                       GB,
                       stardata,
                       star);
    double m0 = star->phase_start_mass;
    
    double age = star->age;
    double mt = star->mass;
    double rm = star->radius;
    double lum = star->luminosity;
    Stellar_type stellar_type = star->stellar_type;
    double mc = star->core_mass;
    double mcCO = star->CO_core_mass;
    double mcGB = star->GB_core_mass;
    double mcmaxMS = star->max_MS_core_mass;
    double rc = star->core_radius;
    double time_first_pulse = star->time_first_pulse;
    double num_thermal_pulses = star->num_thermal_pulses;
    double time_prev_pulse = star->time_prev_pulse;
    double prev_tagb = star->prev_tagb;
    double menv_1tp = star->menv_1tp;
    double mc_1tp = star->mc_1tp;
    double core_mass_no_3dup = star->core_mass_no_3dup;
    double interpulse_period = star->interpulse_period;
    double time_next_pulse = star->time_next_pulse;
    double lambda_3dup = star->lambda_3dup;
    double num_thermal_pulses_since_mcmin = star->num_thermal_pulses_since_mcmin;
    double spiky_luminosity = star->spiky_luminosity;

    stellar_structure_longcall(&m0,
                               &mc_1tp,
                               &age,
                               &mt,
                               &tm,
                               &tn,
                               tscls,
                               lums,
                               GB,
                               &rm,
                               &lum,
                               &stellar_type,
                               &mc,
                               &mcCO,
                               &mcGB,
                               &mcmaxMS,
                               &rc,
                               &star->menv,
                               &star->renv,
                               &star->k2,
                      
                               &time_first_pulse,
                               &num_thermal_pulses,
                               &time_prev_pulse,
                               &prev_tagb,
                               &menv_1tp,
                               &mc_1tp,
                               &core_mass_no_3dup,
                               &interpulse_period,
                               &time_next_pulse,
                               &lambda_3dup,
                               &num_thermal_pulses_since_mcmin,
                               &spiky_luminosity,

                               &star->SN_type,
                               TRUE,
                               star,
                               stardata,
                               STELLAR_STRUCTURE_CALLER_supernova);
    star->radius = rm;
    star->core_radius = rc;
    star->luminosity = lum;
    Dprint("post-explosion structure: m0=%g mc=%g mc_1tp=%g stellar_type = %d\n",
           m0,mc,mc_1tp,stellar_type);

#else
    stellar_structure(star,stardata,TRUE,STELLAR_STRUCTURE_CALLER_supernova);
#endif
    
    star->SN_type = star->new_supernova->new_stellar_structure->SN_type;

    Dprint("pre-explosion (type %d) structure: m0=%g mc=%g mc_1tp=%g stellar_type = %d\n",
           star->SN_type,
           pre_explosion_star->phase_start_mass,
           pre_explosion_star->core_mass,
           pre_explosion_star->phase_start_core_mass,
           pre_explosion_star->stellar_type);
    
    
    if(POST_SN_OBJECT(star->stellar_type))
    {
        /*
         * Spin rate of the post-SN object.
         *
         * Assume it has the same specific angular momentum
         * as the pre-explosion stellar core.
         */
        double l = pre_explosion_star->angular_momentum/
            pre_explosion_star->mass;


        /*
         * Maximum angular momentum a star can have, from General
         * Relativity (Kerr solution assuming no charge) in code units
         */
        const double Jmax =
            Angular_momentum_cgs_to_code(
                GRAVITATIONAL_CONSTANT * Pow2(star->mass * M_SUN) /
                SPEED_OF_LIGHT
                );
        
        star->angular_momentum = Min(star->angular_momentum,
                                     Jmax);
        
        /*
         * Hence the new total angular momentum
         */
        star->angular_momentum = l * star->mass;

        if(star->stellar_type == BLACK_HOLE)
        {
            star->omega = bh_angular_velocity(star->mass,
                                              star->angular_momentum);
        }
        else
        {
            const double I = moment_of_inertia(star,star->radius);
            star->omega = star->angular_momentum / I;
        }


        /*
        const double fKerr = star->angular_momentum / Jmax;
        Dprint("SNomega %g from pre explosion M=%g Mc=%g R=%g kw=%d omega=%g : remNaNt J = %g, Jmax = %g, fKerr = %g, omega = %g, R = %g\n",
               star->omega,
               pre_explosion_star->mass,
               pre_explosion_star->core_mass,
               pre_explosion_star->radius,
               pre_explosion_star->stellar_type,
               pre_explosion_star->omega,
               star->angular_momentum,
               Jmax,
               fKerr,
               star->omega,
               star->radius
            );
        fflush(NULL);
        */        


    }

    /*
     * Kick the remnant
     */
    supernova_kick(stardata,pre_explosion_stardata,star,pre_explosion_star);

    /* update in case of supernova */
    /*
      if(EITHER_STAR_MASSLESS || 
      Less_or_equal(stardata->common.orbit.separation,0.0) ||
      stardata->common.orbit.eccentricity > 1.0 ||
      stardata->common.orbit.eccentricity < -TINY) stardata->model.sgl = TRUE;
    */
    stardata->model.sgl = System_is_single;

    /*
     * Nucleosynthesis
     */
#if defined NUCSYN && defined NUCSYN_SUPERNOVAE
    Dprint("Yield SN type %d (pre explosion stellar types %d %d)\n",
           star->SN_type,
           pre_explosion_star->stellar_type,
           star->new_supernova->new_companion_structure->stellar_type
        );

    Append_logstring(LOG_SN_AND_KICK,
                     ", dm(exploder) = %g, dm(companion) = %g",
                     mass_ejected,
                     companion->dm_companion_SN);

    nucsyn_sn_yield(stardata,
                    star,
                    pre_explosion_star,
                    mass_ejected,
                    0.0, // force automatic set of menv, mc
                    0.0,
                    star->new_supernova->new_companion_structure->stellar_type
        );
#endif // NUCSYN && NUCSYN_SUPERNOVAE

    /*
     * Logging
     */
    Dprint("Logging\n");
#ifdef LOG_SUPERNOVAE
    log_sn(stardata,star->starnum,star->SN_type,POST_SN);
#endif
#if (defined SHORT_SUPERNOVA_LOG ||             \
     defined SUPERNOVA_COMPANION_LOG)           \
    && defined FILE_LOG
    /* 
     * Before we do anything, log the supernova
     * Note that this happens EVEN IF NUCSYN IS NOT DEFINED
     */
    log_supernova(star->SN_type,
                  stardata->model.log_fp,
                  star->starnum,
                  mass_ejected,
                  Max(0.0,mass_ejected - pre_explosion_star->menv),
                  stardata);
#endif 

#if defined DISCS && defined DISC_LOG_POPSYN
    Dprint("Logging (popsyn)\n");
    int i;
    for(i=0; i<stardata->common.ndiscs; i++)
    {
        struct disc_t * disc = & stardata->common.discs[i];
        struct binary_system_t binary;
        disc_init_binary_structure(stardata,&binary,disc);
        if(disc->n_thermal_zones > 0 &&
           Is_not_zero(disc->M))
        {
            Dprint("DISC_SN %g %g %d %d %d %g %g\n",
                   disc->M/M_SUN,
                   disc->J,
                   pre_explosion_star->stellar_type,
                   star->stellar_type,
                   star->SN_type,
                   pre_explosion_star->mass,
                   pre_explosion_star->mass - star->core_mass
                   
                );
        }
    }
#endif // DISCS
    
    /*
     * Turn off the supernovae flag
     */
//    star->SN_type = SN_NONE;

    /*
     * Free the memory associated with the new_supernova
     */
    Dprint("call free supernova\n");
    free_supernova(stardata,star);
    Safe_free(pre_explosion_star);
    Safe_free(pre_explosion_stardata);
    Dprint("Done\n");
}
