#include "../binary_c.h"
#include "supernovae.h"

void setup_binary_at_kick(struct stardata_t * stardata,
                          const double separation,
                          const double eccentricity,
                          const double system_mass,
                          struct kick_system_t * kick_system)
{
    Dprint("sgl = %d\n",stardata->model.sgl);
    if(separation < TINY ||
       eccentricity < -TINY ||
       eccentricity > 1.0 ||
       stardata->model.sgl == TRUE 
        )
    {
        /*
         * System is already disrupted or merged : 
         * pretend it's just at very large separation.
         */
        kick_system->bound_before_explosion = FALSE;
        kick_system->mean_anomaly =
            kick_system->eccentric_anomaly = 
            kick_system->sinbeta =
            kick_system->orbital_speed = 
            kick_system->orbital_speed_squared = 
            0.0;
        kick_system->cosbeta = 1.0;        
        kick_system->separation = 1e50;
    }
    else
    {
        kick_system->bound_before_explosion = TRUE;
        /*
         * Find the separation at the moment of explosion
         */
        Random_seed * RESTRICT random_seed = & stardata->common.random_seed;
        double xx = (double) random_number(stardata,random_seed);

        /* choose mean anomaly, NB cannot be zero */
        double mean_anomaly = Max(TINY,xx*TWOPI); 
        double eccentric_anomaly = mean_anomaly;
        
        /* Newton-Raphson search for eccentric anomaly, solves Eq. A4 */
        double dif = 0.1 * (eccentric_anomaly - eccentricity * sin(eccentric_anomaly) - mean_anomaly);
        int newton_count = 0;
    
        while(Abs_more_than(dif/mean_anomaly,(MEAN_ANOMALY_TOLERANCE)))
        {
            if(newton_count++ > (MEAN_ANOMALY_MAX_ITERATIONS))
            {
                Exit_binary_c(
                    BINARY_C_NR_FAILED_IN_KICK,
                    "Newton-Raphson in monte_carlo_kick failed to converge after %d iterations: consider setting the initial guess to be smaller\n",
                    newton_count-1);
            }
            eccentric_anomaly -= dif / (1.0 - eccentricity * cos(eccentric_anomaly));
            dif = eccentric_anomaly - eccentricity * sin(eccentric_anomaly) - mean_anomaly;
        }

        kick_system->mean_anomaly = mean_anomaly;
        kick_system->eccentric_anomaly = eccentric_anomaly;
        Dprint("mean anomaly %g, eccentric anomaly %g\n",mean_anomaly,eccentric_anomaly);

        /*
         * Separation at the moment of the kick
         */
        kick_system->separation = separation * (1.0 - eccentricity * cos(eccentric_anomaly));
        Dprint("separation at kick %g (a = %g)\n",kick_system->separation,separation);
    
    
        /*
         * Find the orbital speed and relative
         * velocity angle, beta (as cos and sin beta)
         */

        // Hurley+ 2002 Eq. A2
        double ecc2 = Pow2(eccentricity);
        kick_system->sinbeta = sqrt((Pow2(separation)*(1.0-ecc2))/
                                    (kick_system->separation*(2.0*separation-kick_system->separation)));  
        kick_system->cosbeta = -eccentricity*sin(eccentric_anomaly)/
            sqrt(1.0-ecc2*Pow2(cos(eccentric_anomaly)));

        // Eq. A5 in km^2 s^-2
        kick_system->orbital_speed_squared = GMRKM * system_mass *
            (2.0/kick_system->separation - 1.0/separation);
    
        // km s^-1
        kick_system->orbital_speed = sqrt(kick_system->orbital_speed_squared);
    }
    Dprint("Orbital speed at kick %20.12g (m1+m2=%g r=%g sep=%g bound? %s), Initial e=%g\n",
           kick_system->orbital_speed,
           system_mass,
           kick_system->separation,
           separation,
           (kick_system->bound_before_explosion == TRUE ? "Y" : "N"),
           eccentricity);
        
}
