#include "../binary_c.h"


double Pure_function ns_bh_mass(const double m,
                                const double mc_CO,
                                struct stardata_t * RESTRICT const stardata,
                                const struct star_t * RESTRICT const star)
{
    /*
     * Function to evaluate the NS/BH remnant mass given 
     * total mass m and CO Core mass mc_CO 
     */
    double mbh; 

    /* 
     * If we are a TPAGB star, assume the core collapses to
     * a NS in an electron capture supernova. This should
     * only happen in unusual circumstances, or as an STPAGB star. 
     */
    if(star->stellar_type==TPAGB) return star->core_mass;
    
    if(stardata->preferences->BH_prescription==BH_HURLEY2002)
    {
        /* 
         * Chris' NS/BH formula as in Hurley+ 2000, 2002
         */
        mbh = 1.170 + 0.090*mc_CO;
    }
    else if(stardata->preferences->BH_prescription==BH_BELCZYNSKI)
    {
        /*  
         * Use NS/BH mass given by Belczynski et al. 2002, ApJ, 572, 407. 
         */
        // First set mcx
        double mcx;
        if(mc_CO<2.5)
        {
            mcx = 0.1617670*mc_CO + 1.0670550;
        }
        else
        {
            mcx = 0.3141540*mc_CO + 0.6860880;
        }
        // Then set the remnant mass
        if(mc_CO<=5.0)
        {
            mbh = mcx;
        }
        else if(mc_CO<7.60)
        {
            mbh = mcx + (mc_CO - 5.0)*(m - mcx)/2.60;
        }
        else
        {
            mbh = m;
        }
        //printf("Mc_CO=%g, Mcx=%g, Mbh=%g\n", mc_CO, mcx, mbh);
    }
    else if(stardata->preferences->BH_prescription==BH_SPERA2015)
    {
        /*
         * Black hole mass from Spera, Mapelli and Bressan 2015
         * (MNRAS 451, 4086, astro-ph 1505.05201) as a function
         * of the CO core mass and the metallicity.
         * 
         * The following follows their Appendix C fitting
         * formulae.
         */
        double p;
        if(stardata->common.metallicity<=5e-4)
        {
            /* Z < 5e-4 */
            p = -2.333 + 0.1559 * mc_CO + 0.2700 * mc_CO * mc_CO; // Eq. C2
            if(mc_CO <= 5.0)
            {
                mbh = Max(p,1.27); // Eq. C1 MCO < 5
            }
            else if(mc_CO < 10.0)
            {
                mbh = p; // Eq. C1 5 < MCO < 10
            }
            else
            {
                double mZ = -6.476e2 * stardata->common.metallicity + 1.911; // Eq. C3
                double qZ = 2.300e3 * stardata->common.metallicity + 11.67; // Eq. C3
                double f = mZ * mc_CO + qZ; // Eq. C2
                mbh = Min(p,f); // Eq. C1 MCO > 10
            }
        }
        else
        {
            /* Z > 5e-4 */
            double A1,A2,L,eta;
            if(stardata->common.metallicity >= 1e-3)
            {
                /* Z >= 1e-3 : Eq. C6 */
                A1 = 1.340 - 29.49 /
                    (1.0 + pow(stardata->common.metallicity/1.110e-3,2.361));
                A2 = 80.22 - 74.73 * pow(stardata->common.metallicity,0.965) /
                    (2.720e-3 + pow(stardata->common.metallicity,0.965));
                L = 5.683 + 3.533 /
                    (1.0 + pow(stardata->common.metallicity/7.430e-3,1.993));
                eta = 1.066 - 1.121 /
                    (1.0 + pow(stardata->common.metallicity/2.558e-2,0.609));
            }
            else
            {
                /* Z < 1e-3 : Eq. C7 */
                A1 = 1.105e5 * stardata->common.metallicity - 1.258e2;
                A2 = 91.56 - 1.957e4 * stardata->common.metallicity - 1.558e7 * Pow2(stardata->common.metallicity);
                L = 1.134e4 * stardata->common.metallicity - 2.143;
                eta = 3.090e-2 - 22.30 * stardata->common.metallicity + 7.363e4 * Pow2(stardata->common.metallicity);
            }

            /* Eq. C5 */
            double h = A1 + (A2 - A1) / (1.0 + pow(10.0, (L - mc_CO) * eta)); 
            
            if(mc_CO <= 5.0)
            {
                mbh = Max(h,1.27); // Eq. C4 MCO < 5
            }
            else if(mc_CO < 10.0)
            {
                mbh = h;  // Eq. C4 5 < MCO < 10
            }
            else
            {
                double mZ,qZ;
                
                if(stardata->common.metallicity>=2e-3)
                {
                    /* Eq. C8, Z >= 2e-3 */
                    mZ = 1.217;
                    qZ = 1.061;
                }
                else if(stardata->common.metallicity>=1e-3)
                {
                    /* Eq. C9, 1e-3 < Z < 2e-3 */
                    mZ = -43.82 * stardata->common.metallicity + 1.304;
                    qZ = -1.296e4 * stardata->common.metallicity + 26.98;
                }
                else
                {
                    /* Eq. C10, Z < 1e-3 */
                    mZ = -6.476e2 * stardata->common.metallicity + 1.911;
                    qZ = 2.300e3 * stardata->common.metallicity + 11.67;
                }

                double f = mZ * mc_CO + qZ; // Eq. C5
                mbh = Max(h,f); // Eq. C4 MCO > 10
            }
        }

        /* BH mass cannot exceed the total stellar mass */
        mbh = Min(m, mbh);
        
    }
    else if(stardata->preferences->BH_prescription==BH_FRYER12_RAPID)
    {
	
	/*
	  Function to put together the various functions above based on users choice of SN engine.
	  
	  Parameters
	  -----------
	  MpreSN : double
	  Pre-supernova total mass of the star in Msol
	  COCoreMass : double
	  CO Core mass at time of the SN in Msol
	  fallbackFraction : double
	  Fraction of mass [0,1] falling back onto proto compact object
	  engine : int
	  Which supernova engine to use
	  
	  Returns
	  --------
	  Mrem : double
	  Remnant mass in Msol
	  
	  -- Simon Stevenson
	*/
	
	
	double Mproto    = 0.0;
	double fb        = 0.0;
	double Mfb       = 0.0;
	double Mrem_grav MAYBE_UNUSED  = 0.0;
	
	// Calculate Mproto
        Mproto = COCoreToProtoMassRapid(mc_CO);
        
        // Calculate fb, Mfb
        fb = fallbackRapid(m, Mproto, mc_CO);
        
        Mfb = massFallbackGeneral(m, Mproto, fb);
        
        // Calculate Mremnant, baryonic
        mbh = baryonicRemnantMassGeneral(Mproto, Mfb);
	
	//MATHIEU_WARNING:DEBUG
        //printf("fb=%e Mfb=%e\n",fb,Mfb);
	/*
          Mrem_grav = gravitationalRemnantMassGeneral(mbh);
          mbh = Mrem_grav
	*/
    }
    else if(stardata->preferences->BH_prescription==BH_FRYER12_DELAYED)
    {
	double Mproto    = 0.0;
	double fb        = 0.0;
	double Mfb       = 0.0;
	double Mrem_bary MAYBE_UNUSED = 0.0;
	        
        // Calculate Mproto
        Mproto = COCoreToProtoMassDelayed(mc_CO);
        
        // Calculate fb, Mfb
        fb = fallbackDelayed(m, Mproto, mc_CO);
        
        Mfb = massFallbackGeneral(m, Mproto, fb);
        
        // Calculate Mremnant, baryonic, Mremnant, gravitational
        mbh = baryonicRemnantMassGeneral(Mproto, Mfb);
        
        /* Mrem_grav = gravitationalRemnantMassGeneral(mbh); */
	//MATHIEU_WARNING:DEBUG
        //printf("fb=%e Mfb=%e\n",fb,Mfb);
    }
    else
    {
        // no prescription!
        mbh=0;
    }

    Dprint("BH_MASS M = %g MCO = %g Z = %g -> MBH = %g\n",m,mc_CO,stardata->common.metallicity,mbh);

    return(mbh);
}


