
#include "../binary_c.h"


/*
 * Function to be called to trigger a supernova.
 *
 * oldstar is the pre-supernova stellar structure, 
 * e.g. to be called from within stellar_structure_*
 *
 * newstar is the stellar structure at the point just prior to the 
 * supernova being triggered. If this is NULL, the stellar structure
 * in *star is used instead (it is copied
 * using the same routine that is used in stellar structure 
 * calculation).
 * 
 * This routine allocates memory to store the appropriate structures,
 * and sets up the new structure ready to be modified by the
 * stellar structure routine.
 *
 * companion stores the companion stellar structure in cases where
 * this is required. It is ignored if it is NULL.
 *
 * returns the new star structure, or NULL if supernovae are not allowed.
 */

struct star_t * new_supernova(struct stardata_t * const stardata,
                              struct star_t * const oldstar,
                              struct star_t * const companion,
                              struct star_t * const newstar)
{
    struct star_t * retstar;

    Dprint("NEW SUPERNOVA star %d st=%d M=%g Mc=%g t=%g dt=%g : new_supernova (passed in) is %p, s=%p\n",
           oldstar->starnum,
           oldstar->stellar_type,
           oldstar->mass,
           oldstar->core_mass,
           stardata->model.time,
           stardata->model.dt,
           oldstar->new_supernova,
           newstar);
        
    /*
     * Use binary_c's event structure to trigger the supernova
     */
    struct binary_c_new_supernova_event_t * new_sn =
        Malloc(sizeof(struct binary_c_new_supernova_event_t));

    new_sn->oldstar = oldstar;
    new_sn->companion = companion;
    new_sn->newstar = newstar;

    if(Add_new_event(stardata,
                     BINARY_C_EVENT_SUPERNOVA,
                     &supernova_event_handler,
                     &supernova_erase_event_handler,
                     new_sn,
                     NORMAL_EVENT)
       != BINARY_C_EVENT_DENIED)
    {

        if(oldstar->new_supernova)
        {
            /*
             * Pending SNe should just be allowed to happen.
             * In theory, we should improve the evolution algorithm,
             * but if a supernova has been detected previously, 
             */
            Dprint("Using previous stellar structure in oldstar %p oldstar->new_supernova=%p\n",oldstar,oldstar->new_supernova);
            Dprint("This has new_stellar_structure = %p\n",
                   oldstar->new_supernova->new_stellar_structure);

            retstar = oldstar->new_supernova->new_stellar_structure;
        }
        else
        {
            Dprint("malloc new stellar structure\n");
            oldstar->new_supernova = Malloc(sizeof(struct new_supernova_t));
            oldstar->new_supernova->new_stellar_structure = New_star;
            oldstar->new_supernova->new_companion_structure = NULL;
            Dprint("new supernova is %p, new structure is %p\n",
                   oldstar->new_supernova,
                   oldstar->new_supernova->new_stellar_structure);

            if(newstar==NULL)
            {
                Dprint("Setting stellar structure from star structure");
                set_stellar_structure_struct_from_star(
                    oldstar->new_supernova->new_stellar_structure,
                    oldstar);
                Dprint("Set explosion type %d\n",
                       oldstar->new_supernova->new_stellar_structure->SN_type);
            }
            else
            {
                Dprint("Using provided s struct with stellar type %d SN type %d",
                       newstar->stellar_type,
                       newstar->SN_type);
                Copy_star(newstar,
                          oldstar->new_supernova->new_stellar_structure);
            }

            if(companion != NULL)
            {
                /*
                 * Add details about the companion
                 */
                oldstar->new_supernova->new_companion_structure = New_star;
                set_stellar_structure_struct_from_star(
                    oldstar->new_supernova->new_companion_structure,
                    companion);
            }
    
            Dprint("Made new SN %p -> new structure %p, explosion type %d\n",
                   oldstar->new_supernova,
                   oldstar->new_supernova->new_stellar_structure,
                   oldstar->new_supernova->new_stellar_structure->SN_type);

            retstar = oldstar->new_supernova->new_stellar_structure;
        }

        return retstar;
    }
    else
    {
        Safe_free(new_sn);
        return NULL;
    }

}
