#include "../binary_c.h"
#ifdef BSE

/*
 * Stellar timescale caclulations in hydrogen-rich stars.
 *
 * The following depend only on phase_start_mass:
 * GB:
 * GB_EFFECTIVE_AH,GB_A_H_HE,GB_A_HE,GB_B,GB_D,GB_p,GB_q

 * timescales:
 * T_MS, T_BGB, T_GIANT_TX, T_GIANT_TINF_1, T_GIANT_TINF_2,
 * T_HE_IGNITION, T_HE_BURNING, 
 * 
 *
 * luminosities: 
 * L_BGB, L_HE_IGNITION, L_HE_BURNING
 * 

 * Note that the giant_branch_parameters, main_sequence_parameters
 * and metallicity parameters only depend on metallicity,
 * so can be considered constants in this subroutine.
 */

int stellar_timescales_hydrogen_stars(const double phase_start_mass, 
                                      const double mass,
                                      double * RESTRICT tm, 
                                      double * RESTRICT tn, 
                                      double * RESTRICT const timescales, 
                                      double * RESTRICT const luminosities, 
                                      double * RESTRICT const GB,
                                      struct stardata_t * const stardata,
                                      struct star_t * const star,
                                      const Stellar_type stellar_type
    )
{
    /* speed improvement (dummy) variables */
    double GBp,GBq,ip,iq,ipq,mp,mq;
    const double *giant_branch_parameters=
        stardata->common.giant_branch_parameters;
    const double *main_sequence_parameters=
        stardata->common.main_sequence_parameters;
    const double *metallicity_parameters=
        stardata->common.metallicity_parameters;

    Dprint("Stellar timescales not helium : phase_start_mass=%g MSP=%p\n",phase_start_mass,main_sequence_parameters);

#ifdef USE_BSE_TIMESCALES_H
    {
        int i;
        double params[2] = {
            log10(stardata->common.metallicity),
            log10(phase_start_mass)
        };
        double result[TSCLS_ARRAY_SIZE + LUMS_ARRAY_SIZE + GB_ARRAY_SIZE];
        Interpolate(stardata->store->BSE_TIMESCALES_H,params,result,FALSE);
        for(i=0;i<TSCLS_ARRAY_SIZE;i++)
        {
            timescales[i] = pow(10.0,result[i]);
        }

#undef _OFFSET
#define _OFFSET (TSCLS_ARRAY_SIZE)
        /*
         * Set the luminosities that depend only on 
         * the phase_start_mass
         */
        luminosities[L_BGB] = pow(10.0,result[L_BGB + _OFFSET]);
        luminosities[L_LMX] = pow(10.0,result[L_LMX + _OFFSET]);
        luminosities[L_HE_IGNITION] = pow(10.0,result[L_HE_IGNITION + _OFFSET]);
        luminosities[L_HE_BURNING] = pow(10.0,result[L_HE_BURNING + _OFFSET]);
        luminosities[L_BAGB] = pow(10.0,result[L_BAGB + _OFFSET]);
        luminosities[L_TPAGB_FIRST_PULSE] = pow(10.0,result[L_BAGB + _OFFSET]);

#undef _OFFSET
#define _OFFSET (TSCLS_ARRAY_SIZE + LUMS_ARRAY_SIZE)        
        for(i=0;i<GB_ARRAY_SIZE;i++)
        {
            GB[i] = pow(10.0,result[i + _OFFSET]);
        }
    }
#undef _OFFSET
#else
    /*             
     * MS and BGB times             
     */
    timescales[T_BGB] = tbgbf(stardata,
                              phase_start_mass,
                              main_sequence_parameters,
                              stardata->common.metallicity);
    timescales[T_MS] = Max(metallicity_parameters[8],
                           thookf(phase_start_mass,
                                  main_sequence_parameters))
        *timescales[T_BGB];
#endif

    /* store *tm for legacy code */
    *tm = timescales[T_MS];
    
    Dprint("TM CALC zpar[6]=%30.22e thookf(%30.22e)=%30.22e, tscls[T_BGB]=%30.22e hence tm=%30.22e\n",
           metallicity_parameters[8],
           phase_start_mass,
           thookf(phase_start_mass,main_sequence_parameters),
           timescales[T_BGB],*tm);

  
    Dprint("Calc T BGB =%g hence tm=%30.22e (mult=Max(%g,%g))\n",
           timescales[T_BGB],*tm,
           metallicity_parameters[8],
           thookf(phase_start_mass,main_sequence_parameters));
    /*
     * Zero- and terminal age main sequence luminosity
     */
#ifdef USE_BSE_MS_LR
    {
        double params[3] = {
            log10(stardata->common.metallicity),
            log10(phase_start_mass),
            0.0
        };
        double result[2];

        /* start of main sequence = ZAMS */
        params[2] = 0.0;
        Interpolate(stardata->store->BSE_MS_LR,params,result,FALSE);
        luminosities[L_ZAMS] = pow(10.0,result[0]);

        /* end of main sequence =  T(A)MS */
        params[2] = 1.0;
        Interpolate(stardata->store->BSE_MS_LR,params,result,FALSE);
        luminosities[L_END_MS] = pow(10.0,result[0]);        
    }
#else
    luminosities[L_ZAMS] = lzamsf(phase_start_mass,main_sequence_parameters);
    luminosities[L_END_MS] = ltmsf(phase_start_mass,main_sequence_parameters);
#endif
    
    /* 
     * very low-mass stars: approximate "evolution"
     */
    if(stellar_type<HERTZSPRUNG_GAP && phase_start_mass<0.1)
    {
#ifndef USE_BSE_TIMESCALES_H
        timescales[T_HE_IGNITION] = 1.1 * timescales[T_BGB];
        timescales[T_HE_BURNING]  = 0.1 * timescales[T_BGB];
        luminosities[L_BGB] = lbgbf(phase_start_mass,giant_branch_parameters);
#endif
        *tn = 1.0e10;
        Dprint("calc_lum (Low mass star <0.1) out st=%d m=%g mass=%g tm=%g tn=%g\n",
               stellar_type,phase_start_mass,mass,*tm,*tn);
        return(1);
    }
    else
    {
#ifndef USE_BSE_TIMESCALES_H
        stellar_timescales_GB_parameters(phase_start_mass,
                                         metallicity_parameters[6],
                                         GB);
#endif

        mp=1.0-GB[GB_p];
        mq=1.0-GB[GB_q];
        GBp=-mp/GB[GB_p];
        GBq=-mq/GB[GB_q];
        ip=-1.0/mp;
        iq=-1.0/mq;

#ifndef USE_BSE_TIMESCALES_H
        //ipq=1.0/(-(1.0-GB[GB_p])+1.0-GB[GB_q]);
        ipq = 1.0/(mq-mp);
        GB[GB_Mx] = pow(GB[GB_B]/GB[GB_D], ipq);

        Dprint("Set GB_Mx(star=%d) from %g/%g, ipq=%g -> %g\n",star->starnum,GB[GB_B],GB[GB_D],ipq,GB[GB_Mx]);


        /* 
         * Change in slope of giant L-Mc relation.
         */
        luminosities[L_LMX] = GB[GB_D]*pow(GB[GB_Mx],GB[GB_p]);

        /* 
         * HeI ignition luminosity
         */
        luminosities[L_HE_IGNITION] = lheif(phase_start_mass,
                                            metallicity_parameters[ZPAR_MASS_HE_FLASH],
                                            giant_branch_parameters); 
        luminosities[L_BAGB] = lbagbf(phase_start_mass,
                                      metallicity_parameters[ZPAR_MASS_HE_FLASH],
                                      giant_branch_parameters);

        Dprint("compare phase_start_mass=%g vs metallicity_parameter[3]=%g\n",phase_start_mass,metallicity_parameters[ZPAR_MASS_FGB]);



        if(Less_or_equal(phase_start_mass,
                         metallicity_parameters[ZPAR_MASS_FGB]))
	{
	    stellar_timescales_low_mass_GB(GBp,
                                           GBq,
                                           ip,iq,
                                           phase_start_mass,
                                           timescales,
                                           luminosities,
                                           GB,
                                           giant_branch_parameters,
                                           metallicity_parameters,
                                           stardata);

	}
        else
        {
            stellar_timescales_high_mass_GB(phase_start_mass,
                                            metallicity_parameters[ZPAR_MASS_HE_FLASH],
                                            timescales,
                                            luminosities,
                                            giant_branch_parameters);
        }
#endif // USE_BSE_TIMESCALES_H

        Dprint("Done compare stellar_type %d\n",stellar_type);
        
        {
            double mcbagb = mcagbf(phase_start_mass,giant_branch_parameters);
            double tbagb = timescales[T_HE_IGNITION] + timescales[T_HE_BURNING];

            if(stellar_type<EAGB && Fequal(mass,mcbagb))
            {
                Dprint("Set tn=tbagb=%g\n",*tn);
                *tn = tbagb;
                return(1);
            }
            else
            {
                Dprint("Set post HG tscls stellar_type=%d\n",stellar_type);
                int kp=
                    stellar_timescales_post_HG(tbagb,
                                               ip,iq,GBp,GBq,mp,mq,
                                               mcbagb,
                                               phase_start_mass,
                                               mass,
                                               tn,
                                               GB,
                                               timescales,
                                               luminosities,
                                               giant_branch_parameters,
                                               metallicity_parameters,
                                               stardata,
                                               star,
                                               stellar_type);
                return (kp);
            }
        }
    }
}
#endif
