
#include "../binary_c.h"
#ifdef BSE
/*
 *       BSE library function 
 *
 *       Computes the characteristic luminosities and timescales 
 *       at different stages of stellar evolution.
 * 
 *       Ref: P.P. Eggleton, M.J. Fitchett & C.A. Tout (1989) Ap.J. 347, 998.
 *
 *       Revised 27th March 1995 by C. A. Tout
 *       and 24th October 1995 to include metallicity
 *       and 13th December 1996 to include naked helium stars
 *
 *       Revised 5th April 1997 by J. R. Hurley
 *       to include Z=0.001 as well as Z=0.02, convective overshooting,
 *       MS hook and more elaborate CHeB. It now also sets the Giant
 *       Branch parameters relevant to the mass of the star.
 *
 *       ------------------------------------------------------------
 *       Times: 0: Main sequence T_MS
 *              1: BGB  T_BGB     
 *              2: He ignition T_HE_IGNITION   
 *              3: He burning T_HE_BURNING
 *              4: Giant t(inf1) T_GIANT_TINF_1    
 *              5: Giant t(inf2) T_GIANT_TINF_2
 *              6: Giant t(Mx) T_GIANT_TX
 *              7: FAGB t(inf1) T_EAGB_TINF_1
 *              8: FAGB t(inf2) T_EAGB_TINF_2
 *              9: FAGB  t(Mx) T_EAGB_T
 *             10: SAGB t(inf1) T_TPAGB_TINF_1
 *             11: SAGB t(inf2) T_TPAGB_TINF_2
 *             12: SAGB  t(Mx) T_TPAGB_TX
 *             13: TP T_TPAGB_FIRST_PULSE
 *             14: t(Mcmax) T_TMCMAX
 *
 *       Luminosities:  1: ZAMS L_ZAMS
 *                      2: End MS L_END_MS
 *                      3: BGB L_BGB
 *                      4: He ignition L_HE_IGNITION
 *                      5: He burning L_HE_BURNING
 *                      6: L(Mx) L_LMX
 *                      7: BAGB L_BAGB
 *                      8: TP L_TPAGB_FIRST_PULSE
 *
 *       GB:    1: effective A(H) GB_EFFECTIVE_AH   
 *              2: A(H,He) GB_A_H_HE
 *              3: B GB_B
 *              4: D GB_D
 *              5: p GB_p
 *              6: q GB_q
 *              7: Mx GB_Mx
 *              8: A(He) GB_A_HE
 *              9: Mc,BGB GB_MC_BGB
 *             10: Axel parameter 1 GB_AXEL_1
 *             11: Axel parameter 2 GB_AXEL_2
 *
 *       ------------------------------------------------------------
 */
 
#undef Cprint
#if (DEBUG==1)
#define Cprint(...)                                       \
    if((DEBUG) && (Debug_expression))                     \
        debug_fprintf(__FILE__,__LINE__,__VA_ARGS__);
//#undef STELLAR_TIMESCALES_CACHE
#else
#define Cprint(...) /**/;
#endif

#define OUTLOOP(A,B) {                                                  \
        unsigned int i;                                                 \
        for(i=1;i<=A;i++)                                               \
        {                                                               \
            Dprint_no_newline("%u=%12.12e %s",i,B[i],i==A?"":":");      \
        }                                                               \
        Dprint("%s","");                                                \
    }
#define OUTLOOP_STRING(A,S,B) {                                         \
        unsigned int i;                                                 \
        for(i=1;i<=A;i++)                                               \
        {                                                               \
            Dprint_no_newline("%10u = %20s = %s%12.12e%s %s",           \
                              i,                                        \
                              S[i],                                     \
                              ((i>0 && B[i]<B[i-1]) ? BRIGHT_YELLOW : ""), \
                              B[i],                                     \
                              COLOUR_RESET,                             \
                              i==A?"":":");                             \
        }                                                               \
        Dprint("%s","");                                                \
    }

#ifdef STELLAR_TIMESCALES_CACHE_DEBUG
static void stellar_timescales_dump(double *GB,
                                    double *lums,
                                    double *tscls,
                                    double tm,double tn);
#endif // STELLAR_TIMESCALES_CACHE_DEBUG

void stellar_timescales(const Stellar_type stellar_type,
                        double mass, 
                        double mt,
                        double * RESTRICT const tm, 
                        double * RESTRICT const tn, 
                        double * RESTRICT const timescales, 
                        double * RESTRICT const luminosities, 
                        double * RESTRICT const GB,
                        struct stardata_t * RESTRICT const stardata,
                        struct star_t * RESTRICT const star
    )
{
    Dprint("stellar timescales star %d in st=%d m0=%g mt=%g age=%g\ntimescales = %p, luminosities = %p, GB = %p\n",
           star->starnum,
           stellar_type,
           mass,
           mt,
           star->age,
           timescales,
           luminosities,
           GB);

    /* 
     * set arrays to 0 to prevent errors during debugging
     */
    memset(timescales,0,sizeof(double)*TSCLS_ARRAY_SIZE);
    memset(luminosities,0,sizeof(double)*LUMS_ARRAY_SIZE);
    memset(GB,0,sizeof(double)*GB_ARRAY_SIZE);

    mass=fabs(mass);
    mt=fabs(mt);
#ifdef MINIMUM_STELLAR_MASS
    mass = Max(MINIMUM_STELLAR_MASS,mass);
    mt = Max(MINIMUM_STELLAR_MASS,mt);
#endif

#ifdef STELLAR_TIMESCALES_CACHE
    /*
     * The STELLAR_TIMESCALES cache caches the parameters into the function
     * and returns saved results (saving recalculation)...
     *
     * NB only activate the cache for non-remnants
     */
  
#define STELLAR_TIMESCALES_CACHE_SIZE 2
#define CACHE_LEN (GB_ARRAY_SIZE+LUMS_ARRAY_SIZE+TSCLS_ARRAY_SIZE)

#define stellar_timescales_cache_stellar_type (stardata->tmpstore->stellar_timescales_cache_stellar_type)
#define stellar_timescales_cache_mass (stardata->tmpstore->stellar_timescales_cache_mass)
#define stellar_timescales_cache_mt (stardata->tmpstore->stellar_timescales_cache_mt)
#define stellar_timescales_cache_mc (stardata->tmpstore->stellar_timescales_cache_mc)
#define stellar_timescales_cache_GB (stardata->tmpstore->stellar_timescales_cache_GB)
#define stellar_timescales_cache_luminosities (stardata->tmpstore->stellar_timescales_cache_luminosities)
#define stellar_timescales_cache_timescales (stardata->tmpstore->stellar_timescales_cache_timescales)
#define stellar_timescales_cache_tm (stardata->tmpstore->stellar_timescales_cache_tm)
#define stellar_timescales_cache_tn (stardata->tmpstore->stellar_timescales_cache_tn)
  
    int i;
#endif // STELLAR_TIMESCALES_CACHE

#ifdef STELLAR_TIMESCALES_SANITY
    if(tm==tn)
    {
        Exit_binary_c(BINARY_C_POINTER_FAILURE,
                      "tm = tn = %p pointer error in calc_lum_and_evol_time\n",
                      tm);
    }
#endif

    if(stellar_type<HeWD)
    {

#ifdef STELLAR_TIMESCALES_CACHE
        Cprint("STELLAR_TIMESCALES args %d %g %g %g\n",stellar_type,mass,mt,star->core_mass); 

        /* check cache */
        for(i=0;i<STELLAR_TIMESCALES_CACHE_SIZE;i++)
        {
            if((stellar_timescales_cache_stellar_type[i]==stellar_type) &&
               Fequal(stellar_timescales_cache_mt[i],mt) &&
               Fequal(stellar_timescales_cache_mc[i],star->core_mass) &&
               Fequal(stellar_timescales_cache_mass[i],mass))
            {

                Cprint("STELLAR_TIMESCALES from cache line %d\n",i);
#ifdef STELLAR_TIMESCALES_CACHE_DEBUG
                stellar_timescales_dump(GB,luminosities,timescales,*tm,*tn);
#endif //STELLAR_TIMESCALES_CACHE_DEBUG

                /* return cache-matched results */
                memcpy(GB,&(stellar_timescales_cache_GB[i][0]),sizeof(double)*GB_ARRAY_SIZE);
                memcpy(luminosities,&(stellar_timescales_cache_luminosities[i][0]),sizeof(double)*LUMS_ARRAY_SIZE);
                memcpy(timescales,&(stellar_timescales_cache_timescales[i][0]),sizeof(double)*TSCLS_ARRAY_SIZE);
                *tm = stellar_timescales_cache_tm[i];
                *tn = stellar_timescales_cache_tn[i];
#ifdef NUCSYN
                star->tm=*tm;
                star->tn=*tn;
#endif
                Dprint("cache return age=%g\n",star->age);
                return;
            }
        }
        /* no cache match, do calculations */
#endif // STELLAR_TIMESCALES_CACHE

        Dprint("Check stellar type %d\n",stellar_type);

        /*
         * Decide the rest of the timescales based on 
         * whether the star is hydrogen or helium rich
         */
        if(NAKED_HELIUM_STAR(stellar_type))
        {
            stellar_timescales_helium_stars(mass,
                                            mt,
                                            tm,
                                            tn,
                                            timescales,
                                            luminosities,
                                            GB,
                                            stardata,
                                            star);
        }
        else
        {
            stellar_timescales_hydrogen_stars(mass,
                                              mt,
                                              tm,
                                              tn, 
                                              timescales, 
                                              luminosities,
                                              GB,
                                              stardata,
                                              star,
                                              stellar_type);
        }

#ifdef STELLAR_TIMESCALES_CACHE
        /* save results to the cache */

        /* shift cache and arg list down one */
        memmove(stellar_timescales_cache_mass+1,stellar_timescales_cache_mass,sizeof(double)*(STELLAR_TIMESCALES_CACHE_SIZE-1));
        memmove(stellar_timescales_cache_mt+1,stellar_timescales_cache_mt,sizeof(double)*(STELLAR_TIMESCALES_CACHE_SIZE-1));
        memmove(stellar_timescales_cache_mc+1,stellar_timescales_cache_mc,sizeof(double)*(STELLAR_TIMESCALES_CACHE_SIZE-1));
        memmove(stellar_timescales_cache_stellar_type+1,stellar_timescales_cache_stellar_type,sizeof(Stellar_type)*(STELLAR_TIMESCALES_CACHE_SIZE-1));
      
        memmove(stellar_timescales_cache_tm+1,stellar_timescales_cache_tm,sizeof(double)*(STELLAR_TIMESCALES_CACHE_SIZE-1));
        memmove(stellar_timescales_cache_tn+1,stellar_timescales_cache_tn,sizeof(double)*(STELLAR_TIMESCALES_CACHE_SIZE-1));
        memmove(&(stellar_timescales_cache_GB[1][0]),stellar_timescales_cache_GB,sizeof(double)*GB_ARRAY_SIZE*(STELLAR_TIMESCALES_CACHE_SIZE-1));
        memmove(&(stellar_timescales_cache_luminosities[1][0]),stellar_timescales_cache_luminosities,sizeof(double)*LUMS_ARRAY_SIZE*(STELLAR_TIMESCALES_CACHE_SIZE-1));
        memmove(&(stellar_timescales_cache_timescales[1][0]),stellar_timescales_cache_timescales,sizeof(double)*TSCLS_ARRAY_SIZE*(STELLAR_TIMESCALES_CACHE_SIZE-1));
      
        /* add new args to arg list */
        stellar_timescales_cache_mass[0]=mass;
        stellar_timescales_cache_mt[0]=mt;
        stellar_timescales_cache_mc[0]=star->core_mass;
        stellar_timescales_cache_stellar_type[0]=stellar_type;
      
        /* save current result in the cache */
        memcpy(stellar_timescales_cache_GB,GB,sizeof(double)*GB_ARRAY_SIZE);
        memcpy(stellar_timescales_cache_luminosities,luminosities,sizeof(double)*LUMS_ARRAY_SIZE);
        memcpy(stellar_timescales_cache_timescales,timescales,sizeof(double)*TSCLS_ARRAY_SIZE);
        stellar_timescales_cache_tm[0]=*tm;
        stellar_timescales_cache_tn[0]=*tn;

        Cprint("STELLAR_TIMESCALES save to cache\n");
#ifdef STELLAR_TIMESCALES_CACHE_DEBUG      
        stellar_timescales_dump(GB,luminosities,timescales,*tm,*tn);
#endif //STELLAR_TIMESCALES_CACHE_DEBUG
        Cprint("STELLAR_TIMESCALES set cache %g (%g) %g (%g) %g (%g), %g (%g), %g (%g)\n",
               stellar_timescales_cache_GB[0][1],GB[1],
               stellar_timescales_cache_luminosities[0][1],luminosities[1],
               stellar_timescales_cache_timescales[0][1],timescales[1],
               stellar_timescales_cache_tm[0],*tm,
               stellar_timescales_cache_tn[0],*tn
            );
#endif //STELLAR_TIMESCALES_CACHE

    }
    else
    {
        *tn = 1e10;
        *tm = 1e10;
    }

    Dprint("Check sanity\n");

#ifdef STELLAR_TIMESCALES_SANITY
    if((stellar_type < 7)&&(timescales[T_BGB] < *tm))
    {
        Exit_binary_c(BINARY_C_TIMESCALE_ERROR,
                      "tms < tbgb error : Timescales MS=%g BGB=%g\n",
                      *tm,
                      timescales[T_BGB]);
    }
#endif

#ifdef NUCSYN
    /*
     * Save timescales for this star: useful for nucsyn_WR
     */
    star->tm=*tm;
    star->tn=*tn;
#endif

#if (DEBUG==1)
    Dprint("stellar timescales out star %d st=%d m=%g mt=%g tm=%30.22e tn=%g\n",
           star->starnum,stellar_type,mass,mt,*tm,*tn);
    Dprint("timescales : \n");
    const static char tstrings[15][20]=TIMESCALES_STRINGS;
    OUTLOOP_STRING(14,tstrings,timescales);
    Dprint("EAGB starts at T_HE_IGNITION + T_HE_BURNING = %g + %g = %g\n",
           timescales[T_HE_IGNITION],
           timescales[T_HE_BURNING],
           timescales[T_HE_IGNITION]+timescales[T_HE_BURNING]);
    Dprint_no_newline("luminosities : ");
    OUTLOOP(8,luminosities);
    Dprint_no_newline("GB : ");
    OUTLOOP(11,GB);
           
#endif
           
}

#ifdef STELLAR_TIMESCALES_CACHE_DEBUG
void stellar_timescales_dump(double *GB,
                             double *lums,
                             double *tscls,
                             double tm,
                             double tn)
{
    int i;
    printf("STELLAR_TIMESCALES dump: tm=%g tn=%g ",tm,tn);
    for(i=0;i<GB_ARRAY_SIZE;i++)
    {
        printf("%g ",GB[i]);
    }
    for(i=0;i<LUMS_ARRAY_SIZE;i++)
    {
        printf("%g ",lums[i]);
    }
    for(i=0;i<TSCLS_ARRAY_SIZE;i++)
    {
        printf("%g ",tscls[i]);
    }
    printf("\n");
}
#endif
#endif//BSE
