#include "../binary_c.h"
#ifdef BSE

#include "stellar_structure_debug.h"

/*
 * Calculate the core radius and the luminosity and radius of the
 * remnant that the star will become.
 *
 * Also calculates k2 (radius of gyration) in some cases.
 */


void stellar_structure_remnant_and_perturbations(struct star_t * newstar,
                                                 double remnant_luminosity,
                                                 double remnant_radius,
                                                 double tbagb,
                                                 double mtc,
                                                 double mcx,
                                                 struct stardata_t * stardata)
{
    const double * metallicity_parameters=stardata->common.metallicity_parameters; 
  
    Dprint("Calculating core radius, lum and rad of remn a nt star, stellar type %d.\n",newstar->stellar_type);
  
    double tau = 0.0;

    /*
     * Main sequence 
     */
    if(ON_EITHER_MAIN_SEQUENCE(newstar->stellar_type))
    {
        newstar->core_radius = 0.0;
        remnant_luminosity = 0.0;
        remnant_radius = 0.0;
    }
    /*
     * HG and RGB
     */
    else if(newstar->stellar_type==HERTZSPRUNG_GAP
            ||
            newstar->stellar_type==FIRST_GIANT_BRANCH)
    {
        Boolean degenerate_core = Less_or_equal(newstar->phase_start_mass,
                                                metallicity_parameters[ZPAR_MASS_HE_FLASH]); 
        
        double LWD = lwd(newstar->core_mass,0.0,HeWD,stardata);
        if(degenerate_core)
        {
            remnant_luminosity = LWD;
            if(0)printf("LLL remnant L = %g (degenerate core, lwd) Mc=%g\n",
                   remnant_luminosity,
                   newstar->core_mass);
            remnant_radius = rwd(newstar->core_mass,stardata);

#ifdef HALL_TOUT_2014_RADII
            newstar->core_radius =
                Hall_Tout_2014_low_mass_HG_RGB_radius(newstar->core_mass,
                                                      stardata);
#else
            newstar->core_radius = 5.0*remnant_radius;
#endif // HALL_TOUT_2014_RADII
        }
        else
        {
            remnant_luminosity = Max(LWD,lzhef(newstar->core_mass));
            remnant_radius = rzhef(newstar->core_mass);
            newstar->core_radius = remnant_radius; 
            if(0)printf("LLL remnant L = %g (non-degenerate core, lzhef) Mc=%g\n",
                   remnant_luminosity,
                   newstar->core_mass);
        }
    }
    /*
     * CHeB
     */
    else if(newstar->stellar_type==CORE_HELIUM_BURNING)
    {
        tau = (newstar->age - newstar->timescales[T_HE_IGNITION])/newstar->timescales[T_HE_BURNING];
        Dprint("Call calc_lum as HeMS although the star is really CHeB (tau=%g core_mass=%g)\n",
               tau,
               newstar->core_mass);
        call_calc_lt2(HeMS,newstar->core_mass,newstar->core_mass);

        double am = Max(0.0, 0.850 - 0.080*newstar->core_mass);
        remnant_luminosity = newstar->luminosities[L_ZAMS]*
            (1.0+0.450*tau+am*Pow2(tau));
        remnant_radius = rzhef(newstar->core_mass);
        am = Max(0.0, 0.40 - 0.220*log10(newstar->core_mass));
        remnant_radius *= (1.0 + am * (tau-Pow6(tau)));
        Dprint("Call lt2 for core radius with stellar_type=%d phase_start_mass=%g mass=%g\n",
               newstar->stellar_type,
               newstar->phase_start_mass,
               newstar->mass);
        call_calc_lt2(newstar->stellar_type,
                      newstar->phase_start_mass,
                      newstar->mass);
        newstar->core_radius = remnant_radius;
    }
    /*
     * EAGB
     */
    else if (newstar->stellar_type==EARLY_ASYMPTOTIC_GIANT_BRANCH)
    {
        if(newstar->tn > tbagb)
        {
            tau = 3.0 * (newstar->age - tbagb) / (newstar->tn - tbagb);
        }
        
        Dprint("EAGB tau = %12.12e from age=%12.12e tbagb=%12.12e tn=%12.12e m=%g mc=%g\n",
               tau,
               newstar->age,
               tbagb,
               newstar->tn,
               newstar->mass,
               newstar->core_mass
            );

        call_calc_lt2(HeGB,
                      newstar->core_mass,
                      newstar->core_mass);
        if(Is_zero(newstar->age))
        {
            Exit_binary_c(2,"Age should not be zero");
        }
        remnant_luminosity = lmcgbf(mcx,newstar->GB);

        Dprint("LMCGBF gave %g, tau = %g, endMS=%g\n",
               remnant_luminosity,
               tau,
               newstar->luminosities[L_END_MS]);

        if(tau<1.0)
        { 
            remnant_luminosity = newstar->luminosities[L_END_MS]*
                pow(remnant_luminosity/newstar->luminosities[L_END_MS] ,tau);
        }
        
        Dprint("REMNA NT_LUMINOSITY remna nt_luminosity = %12.12e\n",
               remnant_luminosity);


        remnant_radius = rzhef(newstar->core_mass);
        Dprint("rzhef(%g) gave %g\n",newstar->core_mass,remnant_radius)

        double r1 = rhehgf(newstar->core_mass,
                           remnant_luminosity,
                           remnant_radius,
                           newstar->luminosities[L_END_MS]);

        Dprint("rhehgf(%g,%g,%g,%g) = %g\n",
               newstar->core_mass,
               remnant_luminosity,
               remnant_radius,
               newstar->luminosities[L_END_MS],
               r1);

        double r2 = rhegbf(remnant_luminosity);
        
        Dprint("EAGB set remna nt_radius=%12.12e from Min(%12.12e,%12.12e)\n",
               Min(r1,r2),r1,r2);

        /*
         * Fudge here to catch negative radii, which are 
         * sometimes given by rhehgf.
         * If both radii are positive, use the minimum,
         * but ensure it's > 1e-3 Rsun.
         * If either is negative, use the other.
         * Otherwise, just use 1e-3 Rsun so that
         * there are no nans.
         *
         * This solution is not ideal.
         */
        if(r1>0.0 && r2>0.0)
        {
            /* the original BSE algorithm */
            remnant_radius = Max(1e-3,Min(r1,r2));
        }
        else if(r1>0.0)
        {
            remnant_radius = r1;
        }
        else if(r2>0.0)
        {
            remnant_radius = r2; 
        }
        else
        {
            /* at least something that will not cause a crash */
            remnant_radius = 1e-3;
        }

        call_calc_lt2(newstar->stellar_type,
                      newstar->phase_start_mass,
                      newstar->mass);
        newstar->core_radius = remnant_radius;
        Dprint("EAGB Rc = %12.12e (remn ant_radius=%12.12e)\n",
               newstar->core_radius,
               remnant_radius);
    }
    else if(newstar->stellar_type==TPAGB)
    {
        remnant_luminosity = lwd(newstar->core_mass,0.0,COWD,stardata);
        remnant_radius = rwd(newstar->core_mass,stardata);

#ifdef HALL_TOUT_2014_RADII
        newstar->core_radius = Hall_Tout_2014_AGB_radius(newstar->core_mass,stardata);
#else
        newstar->core_radius = 5.0*remnant_radius;
#endif //HALL_TOUT_2014_RADII
    }
    else if(newstar->stellar_type<=NAKED_HELIUM_STAR_GIANT_BRANCH)
    {
        remnant_luminosity = lwd(newstar->core_mass,0.0,COWD,stardata);
        remnant_radius = rwd(newstar->core_mass,stardata);
#ifdef HALL_TOUT_2014_RADII
        if(newstar->stellar_type==HeHG)
        {
            newstar->core_radius =
                Hall_Tout_2014_HeHG_radius(newstar->core_mass);
        }
        else if(newstar->stellar_type==HeGB)
        {
            newstar->core_radius =
                Hall_Tout_2014_HeGB_radius(newstar->core_mass,
                                           stardata);
        }
#else
        newstar->core_radius = 5.0*remnant_radius;
#endif
        newstar->core_radius = Max(1.4e-5, newstar->core_radius);
        
        Dprint("Helium giant remn ant: L=%g R=%g -> Rc=%g\n",
               remnant_luminosity,
               remnant_radius,
               newstar->core_radius);
    }
    else
    {
        newstar->core_radius = newstar->radius;
        newstar->menv = 1E-10;
        newstar->renv = 1E-10;
        newstar->k2 = 0.21;
    }

    /*
     * Perturb the luminosity and radius due to small envelope mass.
     *
     * Note: only do this if we're not about to explode as a SN.
     */
    //if(newstar->SN_type == SN_NONE)
    {
        stellar_structure_small_envelope_perturbations(
            stardata,
            newstar,
            mtc,
            remnant_luminosity,
            remnant_radius);
    }
}

#endif//BSE
