#ifndef STELLAR_STRUCTURE_PROTOTYPES_H
#define STELLAR_STRUCTURE_PROTOTYPES_H

#include "stellar_structure_macros.h"

void stellar_structure(struct star_t * const oldstar,
                       struct stardata_t * const stardata,
#ifdef BSE
                       double *  const timescales,
                       double *  const luminosities,
                       double *  const GB,
#endif
                       const Boolean allow_SN,
                       const Caller_id caller_id);

#ifdef BSE
void stellar_structure_longcall(double * const   mass,
                                double * const  phase_start_core_mass,
                                double * const   age, /** current age in Myr **/
                                double * const   mt, /** current mass in solar units **/
                                double * const   tm, /** double in, main sequence time **/
                                double * const   tn, /** double in, nuclear burning time **/
                                double * const   timescales, /** array in, time scale for different stages **/
                                double * const   luminosities, /** array in, characteristic luminosity **/
                                double * const   GB, /** array in, giant branch parameters **/
                                double * const   r, /** stellar radius in solar units **/
                                double * const   lum,/** characteristic luminosity **/
                                Stellar_type * const stellar_type, /** stellar type (1-10) **/
                                double * const   core_mass, /** core mass **/
                                double * const   CO_core_mass, 
                                double * const   GB_core_mass, 
                                double * const  max_MS_core_mass,
                                double * const   rc /** core radius **/,
                                double * const   menv, /* envelope mass */
                                double * const   renv, /* envelope radius */
                                double * const   k2, /* radius of gyration of the envelope */

                                /* TPAGB parameters */
                                double * const  time_first_pulse,
                                double * const  num_thermal_pulses,
                                double * const  time_prev_pulse,
                                double * const  prev_tagb,
                                double * const  menv_1tp,
                                double * const  mc_1tp,
                                double * const  core_mass_no_3dup,
                                double * const  interpulse_period,
                                double * const  time_next_pulse,
                                double * const  lambda_3dup,
                                double * const  num_thermal_pulses_since_mcmin,
                                double * const  spiky_luminosity,

                                Supernova_type * const SN_type, /* SN type, if one happens */
                                const Boolean allow_SN,
                                struct star_t * const star,
                                struct stardata_t * const  stardata,
                                const Caller_id caller_id
    );

Stellar_type stellar_structure_MS(struct star_t * RESTRICT const newstar,
                                  struct stardata_t * RESTRICT const stardata);

Stellar_type stellar_structure_HG(struct star_t * RESTRICT const newstar,
                                  const double rg,
                                  struct stardata_t * RESTRICT stardata,
                                  const Caller_id caller_id);

Stellar_type stellar_structure_RG(struct star_t * const newstar,
                                  double * const rg,
                                  struct stardata_t * RESTRICT const stardata,
                                  const Caller_id caller_id);

Stellar_type stellar_structure_CHeB(struct star_t * RESTRICT const newstar,
                                    double * RESTRICT const rg,
                                    struct stardata_t * RESTRICT const stardata,
                                    const Caller_id caller_id);
Stellar_type stellar_structure_AGB(struct star_t * const oldstar,
                                   struct star_t * RESTRICT const newstar,
                                   double * RESTRICT const mc_CO,
                                   double * RESTRICT const rg,
                                   const double tbagb,
                                   struct stardata_t * RESTRICT const stardata,
                                   const Caller_id caller_id);

Boolean stellar_structure_EAGB(struct star_t * oldstar,
                               struct star_t * RESTRICT const newstar,
                               const double tbagb,
                               double * RESTRICT const mc_CO,
                               const double mcbagb,
                               struct stardata_t * RESTRICT const stardata,
                               const Caller_id caller_id,
                               const Boolean force_to_explosion);

void stellar_structure_TPAGB(struct star_t * RESTRICT const newstar,
                             double * RESTRICT const CO_core_mass,
                             double * RESTRICT const mcmax,
                             const double mcbagb,
                             struct stardata_t * RESTRICT const stardata,
                             const Caller_id caller_id);


double lambda_3dup(struct stardata_t * RESTRICT const stardata,
                   struct star_t * RESTRICT const newstar,
                   Boolean * const above_mcmin);
double Pure_function lambda_3dup_Hurley(Boolean * RESTRICT const above_mcmin,
                                        const double m);
double lambda_3dup_Karakas(
    Boolean * RESTRICT const above_mcmin,
    struct star_t * RESTRICT const newstar,
    struct stardata_t * RESTRICT const stardata
    );
double lambda_3dup_stancliffe(Boolean * above_mcmin,
                              struct star_t * newstar,
                              const double n,
                              struct stardata_t * stardata);
void stellar_structure_HeStar(struct star_t * newstar,
                              double *mtc,
                              double *rg,
                              double *remnant_radius,
                              struct stardata_t * stardata,
                              const Caller_id caller_id);

void stellar_structure_WD(struct star_t * newstar,
                          struct stardata_t * stardata,
                          const Caller_id caller_id);

void stellar_structure_NS(struct star_t * newstar,
                          struct stardata_t * stardata);

void stellar_structure_BH(struct stardata_t * stardata,
                          struct star_t * newstar);

void stellar_structure_remnant_and_perturbations(struct star_t * newstar,
                                                 double remnant_luminosity,
                                                 double remnant_radius,
                                                 double tbagb,
                                                 double mtc,
                                                 double mcx,
                                                 struct stardata_t * stardata);
void stellar_structure_small_envelope_perturbations(struct stardata_t * staradata,
                                                    struct star_t * newstar,
                                                    const double mtc,
                                                    const double remnant_luminosity,
                                                    const double remnant_radius);
void stellar_structure_nucsyn(Stellar_type stellar_type,
                              Stellar_type stellar_typein,
                              struct star_t * star,
                              struct stardata_t * stardata);
double Pure_function remnant_radius(int stellar_type,
                                    double mass,
                                    double mc,
                                    double r,
                                    double *metallicity_parameters,
                                    struct stardata_t * stardata
    ) ;


void convective_envelope_mass_and_radius(struct stardata_t * stardata,
           struct star_t * newstar,
	   const double rg,
	   double z);

void stellar_structure_BSE(struct star_t * newstar,
                           struct star_t * oldstar,
                           struct stardata_t *stardata,
                           const Caller_id caller_id);

#endif // BSE


void stellar_structure_algorithm(struct star_t * newstar,
                                 struct star_t * oldstar,
                                 struct stardata_t *stardata,
                                 const Caller_id caller_id);


void stellar_structure_make_massless_remnant(struct star_t * newstar);
void set_stellar_structure_struct_from_star(struct star_t * newstar,
                                            struct star_t * oldstar);

void set_star_struct_from_stellar_structure(struct star_t * newstar,
                                            struct star_t * oldstar);
double TPAGB_luminosity(const Stellar_type stellar_type,
                        const double m,
                        const double mc,
                        const double mc1tp,
                        const double mcnodup,
                        const double age,
                        const double Z,
                        const double phase_start_mass,
                        const double num_thermal_pulses,
                        const double interpulse_period,
                        const double time_first_pulse,
                        const double time_next_pulse,
                        struct stardata_t * RESTRICT const stardata,
                        double * RESTRICT const GB
    );


void stellar_structure_modified_BSE(
    struct stardata_t * const stardata,
    struct star_t * const star,
    struct star_t * RESTRICT const newstar,
    double *  timescales,
    double *  luminosities,
    double *  GB,
    const Caller_id caller_id,
    Supernova_type * const new_SN_type
    );

void stellar_structure_small_envelope_miller_bertolami(struct stardata_t * const stardata,
                                                       struct star_t * const star,
                                                       const double remnant_radius,
                                                       const double remnant_luminosity
    );


double Constant_function bh_angular_velocity(const double M,
                                             const double J);

Boolean Pure_function stellar_structure_giant_branch_helium_ignition(struct stardata_t * const stardata,
                                                                     struct star_t * const star);


#endif /*STELLAR_STRUCTURE_PROTOTYPES_H*/
