#include "../binary_c.h"

#ifdef NUCSYN
void stellar_structure_nucsyn(Stellar_type stellar_type,
                              Stellar_type stellar_typein,
                              struct star_t * star,
                              struct stardata_t * stardata)
{
    /*
     * A function to do some nucleosynthesis at the end of stellar_structure
     */
    Dprint("STELLAR_STRUCTURE nucleosynthesis : st in %d out %d\n",
           stellar_typein,
           stellar_type
        );

    /*
     * If the star is not on the AGB set the spiky luminosity
     * to the actual luminosity (for logging purposes)
     */
    if(stellar_type!=TPAGB) star->spiky_luminosity = star->luminosity;

    /*
     * Check the new stellar type, if it's not the same as the stellar
     * type that was passed in, we might need to change the surface
     * abundance and the abundance of the accretion layer (which is mixed
     * in if the star is convective)
     */
    if(stellar_type!=stellar_typein)
    {
#ifdef NUCSYN_FIRST_DREDGE_UP_ACCRETION_CORRECTION_FROM_TAMS
        if(stellar_type==HG && stellar_typein<HG)
        {
            /* MS->HG conversion : copy the *envelope* abundances to XTAMS */
            Dprint("MEMCPY TAMS star %d (env C12=%g acc %g\n",star->starnum,
                   star->Xenv[XC12],star->Xacc[XC12]);
            /* save TAMS abundance for 1st DUP routine */
            Copy_abundances(star->Xenv,star->XTAMS);
        }
#endif // NUCSYN_FIRST_DREDGE_UP_ACCRETION_CORRECTION_FROM_TAMS

        if(stellar_type==EAGB && stellar_typein==TPAGB)
        {
            /* This should never happen! */
            Exit_binary_c(BINARY_C_ALGORITHM_BRANCH_FAILURE,
                          "AGB WENT THE WRONG WAY!\n");
        }

        if(stellar_type>=NS)
        {
            /* BH,NS or massless */
            nucsyn_set_remnant_abunds(&star->Xenv[0]);
        }
        else if(WHITE_DWARF(stellar_type))
        {
            /* all white dwarfs */
            nucsyn_set_WD_abunds(stardata,
                                 &star->Xenv[0],
                                 stellar_type);
        }
        else if(NAKED_HELIUM_STAR(stellar_type))
        {
            /* all helium stars */
            if(stellar_typein==HeMS && 
               stellar_type==HeHG)
            {
                Dprint("Convert HeMs -> HeHG (mix in dmacc)\n");

                /* star has been converted from a radiative
                 * star into a convective giant star so 
                 * mix the envelope (defined by the core mass)
                 * and the accretion layer, set
                 * the accretion layer size to zero
                 */
                if(can_use_strip_and_mix(stardata,star)==FALSE)
                {
                    if(star->dmacc>0.0)
                    {
                        nucsyn_mix_shells(star->mass - star->core_mass,
                                          star->Xenv,
                                          star->dmacc,
                                          star->Xacc);
                        star->dmacc=0.0;
                    }
                }
            }
        }

        if(CONVECTIVE_ENVELOPE(stellar_type) &&
           star->dmacc > 0.0 &&
           can_use_strip_and_mix(stardata,star)==FALSE)
        {
            Dprint("CONVERT to convective star at t=%g kw=%d convective=%d dmacc=%g\n",
                   stardata->model.time,
                   star->stellar_type,
                   CONVECTIVE_ENVELOPE(star->stellar_type),
                   star->dmacc);
            /*
             * convective star : mix envelope and accretion layer
             */
            nucsyn_mix_shells(star->mass - star->core_mass,
                              star->Xenv, // the envelope
                              star->dmacc,
                              star->Xacc);
            star->dmacc=0.0;
        }

        Dprint("Surface abunds now H1=%12.12e He4=%12.12e C12=%12.12e C13=%12.12e N14=%12.12e\n",
               star->Xenv[XH1],
               star->Xenv[XHe4],
               star->Xenv[XC12],
               star->Xenv[XC13],
               star->Xenv[XN14]);
    }

#ifdef LITHIUM_TABLES
    if(NUCLEAR_BURNING(stellar_type))
    {
        /* prevent call in final thermal pulse */
        nucsyn_lithium(star,stardata);
    }
    else
    {
        star->Xenv[XLi7]=0.0;
    }
#endif

#ifdef NUCSYN_ANGELOU_LITHIUM
    nucsyn_angelou_lithium(stardata,star);
#endif // NUCSYN_ANGELOU_LITHIUM
    
#ifdef NUCSYN_STRIP_AND_MIX
    /*
     * First dredge up and/or envelope stripping
     * using TAMS abundances
     */
    nucsyn_strip_and_mix(stardata,star);
#endif // NUCSYN_STRIP_AND_MIX


    /* enforce black hole/massless remnant accretion layer removal */
    if(stellar_type>=BH) star->dmacc=0.0;

#ifdef NUCSYN_WR
    if(stellar_type < HeWD &&
       stellar_type > LOW_MASS_MS &&
       stellar_type != TPAGB &&
       // applies to helium stars and stars with ZAMS mass > 8.0
       (stellar_type > TPAGB ||
        star->effective_zams_mass > 8.0))
    {
        nucsyn_WR(star,stardata,star->mass);
    }

#ifdef NUCSYN_WR_LOG
    // Wolf-Rayet star logging
    // or if either star is not a WD
    if(((stellar_type<COWD)||(stardata->star[Other_star(star->starnum)].stellar_type<COWD))
       &&(More_or_equal(stardata->model.time,last_wr_log_time))
       &&(stardata->model.time>0.1))
    {
        last_wr_log_time=stardata->model.time;

        mc = star->core_mass;
        if(star->stellar_type==HeMS) mc=star->mass;

        // zcno may have changed, recalculate
        zcno=star->Xenv[XC12]+star->Xenv[XC13]+
            star->Xenv[XN14]+star->Xenv[XO16]+
            star->Xenv[XO17]+star->Xenv[XO18]+1e-40;
     
        if((star->stellar_type<HeWD)&&(star->mass>0.1)&&(star->Xenv[XC12]<0.75))
        {

            printf("WRCF%d %12.12e %12.12e %12.12e %12.12e %12.12e %12.12e %12.12e %12.12e %12.12e %d %12.12e %12.12e %12.12e %12.12e %12.12e %12.12e %12.12e %12.12e %12.12e %d %12.12e %12.12e %12.12e %12.12e %12.12e %12.12e\n", 
                   star->starnum,
                   star->mass, //0
                   star->Xenv[XH1],
                   star->Xenv[XHe4], 
                   star->Xenv[XC12]/zcno,
                   star->Xenv[XC13]/zcno,
                   star->Xenv[XN14]/zcno,
                   star->Xenv[XO16]/zcno,
                   star->Xenv[XO17]/zcno,
                   star->Xenv[XO18]/zcno,
                   star->stellar_type, //9
                   star->Xenv[XC12],//10
                   star->Xenv[XC13],
                   star->Xenv[XN14],//12
                   star->Xenv[XO16],
                   star->Xenv[XO17],
                   star->Xenv[XO18],
                   stardata->model.time*1e6, //16
                   log10(Max(star->luminosity,TINY)),
                   // effective temperature
                   log10(Max(1000.0*pow((1130.0*(star->luminosity)/Pow2(*r)),
                                        0.25)
                             ,TINY)),
                   nucsyn_WR_type(star), //19
                   mc,
                   star->Xenv[XNe20],
                   star->Xenv[XNe22],
                   star->Xenv[XMg24],
                   (star->Xenv[XC13]+star->Xenv[XC12])/Max(1e-30,star->Xenv[XO16])*4.0/3.0,
                   star->Xenv[XC12]/Max(1e-30,star->Xenv[XC13])*12.0/13.0
                );
             
         

        }
    }
#endif // NUCSYN_WR_LOG
#endif // NUCSYN_WR
}
#endif
