#include "../binary_c.h"
#ifdef BSE
void stellar_structure_modified_BSE(
    struct stardata_t * const stardata,
    struct star_t * const star,
    struct star_t * RESTRICT const newstar,
    double *   timescales,
    double *   luminosities,
    double *   GB,
    const Caller_id caller_id,
    Supernova_type * const new_SN_type
    )
{
    
    /*
     * The timescales, luminosities and GB arrays 
     * are usually passed in. However, they could 
     * also be passed in as NULL. If this is the
     * case, allocate appropriate memory and recalculate
     * the various timescales and luminosities. This 
     * memory will be freed on exit. 
     */
    Boolean alloc_timescales, alloc_luminosities, alloc_GB;

    if(timescales != NULL)
    {
        newstar->timescales = timescales;
        alloc_timescales = FALSE;
    }
    else
    {
        newstar->timescales = timescales =
            Malloc(sizeof(double) * TSCLS_ARRAY_SIZE);
        alloc_timescales = TRUE;
    }

    if(luminosities != NULL)
    {
        newstar->luminosities = luminosities;
        alloc_luminosities = FALSE;
    }
    else
    {
        newstar->luminosities = luminosities =
            Malloc(sizeof(double) * LUMS_ARRAY_SIZE);
        alloc_luminosities = TRUE;
    }

    if(GB != NULL)
    {
        alloc_GB = FALSE;
        newstar->GB = GB;
    }
    else
    {
        newstar->GB = GB =
            Malloc(sizeof(double) * GB_ARRAY_SIZE);
        alloc_GB = TRUE;
    }
     
    /*
     * If any of the above is newly allocated, recalculate
     * the star's timescales and luminosities
     */
    if(alloc_luminosities == TRUE ||
       alloc_timescales == TRUE ||
       alloc_GB == TRUE)
    {
        stellar_timescales(newstar->stellar_type,
                           newstar->phase_start_mass,
                           newstar->mass,
                           &newstar->tm,
                           &newstar->tn,
                           timescales,
                           luminosities,
                           GB,
                           stardata,
                           newstar);
    }

    /* calculate new stellar structure */
    stellar_structure_algorithm(newstar,star,stardata,caller_id);

    /* 
     * if there is a supernova and we're on an intermediate solution
     * step, restore the structure to the pre-SN state. The final step
     * should also have a supernova, and will trigger.
     */
    if(0 &&
       newstar->SN_type != SN_NONE &&
       stardata->preferences->disable_events == FALSE &&
       stardata->model.intermediate_step == TRUE)
    {
        printf("RETURN STAR TO PREV (radius %g)\n",star->radius);
        Copy_star(star,newstar);
        printf("R now %g\n",star->radius);
    }
    
#ifdef BSE
    star->tbgb = timescales[T_BGB];
    star->tms = star->tm;
#endif // BSE
    
    *new_SN_type = newstar->SN_type;

    if(alloc_timescales)
    {
        Safe_free(newstar->timescales);
    }
    if(alloc_luminosities)
    {
        Safe_free(newstar->luminosities);
    }
    if(alloc_GB)
    { 
        Safe_free(newstar->GB);
    }
}
#endif//BSE
