#include "../binary_c.h"
#ifdef BSE
#include "stellar_structure_debug.h"

void stellar_structure_WD(struct star_t * newstar,
                          struct stardata_t * stardata,
                          const Caller_id caller_id)
{
    
    Dprint("White dwarf... (star %d type in %d) with core mass %12.12e\n",
           newstar->starnum,
           newstar->stellar_type,
           newstar->core_mass);


    
    /*
     * White dwarf structure
     */
    newstar->core_mass  = newstar->mass;
    newstar->CO_core_mass = newstar->stellar_type == HeWD ? 0.0 : newstar->mass;
    newstar->GB_core_mass = 0.0;
    newstar->luminosity = lwd(newstar->mass,
                              newstar->age,
                              newstar->stellar_type,
                              stardata);
    newstar->radius     = Min(rwd(newstar->mass,stardata),0.10);
    if(newstar->mass < 5e-4) newstar->radius = (newstar->mass < 5e-6) ? 0.009 : 0.09;
    newstar->derivative[DERIVATIVE_STELLAR_GB_CORE_MASS] = 0.0;
    newstar->derivative[DERIVATIVE_STELLAR_HE_CORE_MASS] = 0.0;
    newstar->derivative[DERIVATIVE_STELLAR_CO_CORE_MASS] = 0.0;
    newstar->derivative[DERIVATIVE_STELLAR_HE_CORE_MASS_NO_TDUP] = 0.0;
#define MCh (stardata->preferences->chandrasekhar_mass)
    Stellar_type companion_type = Other_star_struct(newstar)->stellar_type;
    
    /*
     * Check for supernova
     */ 
    if(More_or_equal(newstar->mass,MCh))
    {
        /*
         * Accretion induced supernova with no remnant
         * unless WD is ONe in which case we assume a NS 
         * is the remnant. Jarrod's 2007 code sets the 
         * mass to 1.3Msun, but in this version we let the
         * mass be set by the ns_bh_mass function.
         */
        Dprint("above Chandrasekhar Mass, M = %g, Mch = %g\n", 
               newstar->mass,
               MCh);

        struct star_t * news;
    
        if(newstar->stellar_type==ONeWD)
        {
            Dprint("accretion induced sn with core mass %g\n",newstar->mass);
            newstar->SN_type = SN_AIC;
            news = new_supernova(stardata,
                                 newstar,
                                 Other_star_struct(newstar),
                                 newstar);
            if(news)
            {
                newstar->stellar_type = NEUTRON_STAR;
                newstar->mass = ns_bh_mass(newstar->mass,
                                           newstar->mass,
                                           stardata,newstar);
            }
        }
        else
        {               
            Dprint("STELLAR_STRUCTURE CHAND\n");
            /*
             * When a (He/CO) WD reaches 
             * Mch by wind, blow it up in a Type Ia 
             * (JC: 13/12/2010)
             */
            newstar->SN_type = SN_IA_CHAND;
            news = new_supernova(stardata,
                                 newstar,
                                 Other_star_struct(newstar),
                                 newstar);
            if(news)
            {
                stellar_structure_make_massless_remnant(news);
            }
            Append_logstring(LOG_ROCHE,
                             "SNIa M=%g>MCh=%g (%g %g) stellar type %d companion type %d",
                             newstar->mass,
                             stardata->preferences->chandrasekhar_mass,
                             stardata->star[0].mass,
                             stardata->star[1].mass,
                             newstar->stellar_type,
                             companion_type
                );
        }
    }
    else
    {

        Dprint("below Chandrasekhar Mass M = %g, MCh = %g\n",
               newstar->mass,
               MCh);
        Dprint("r=%.12g\n",newstar->radius);

        /*
         * Check for Edge Lit Detonation: 
         * This is a COWD that accretes ~0.15Msun
         * of hydrogen or helium
         * and then blows up in a sub-Mch explosion.
         * 
         * Note the mass_accretion_for_eld = 0.15 in the BSE
         * model. If negative or zero, there is no ELD.
         */

        Boolean ELD =
            stardata->preferences->mass_accretion_for_eld > +TINY &&
            newstar->stellar_type==COWD &&
            companion_type<=HeWD &&
            More_or_equal(newstar->mass - newstar->phase_start_mass,
                          stardata->preferences->mass_accretion_for_eld);
        
        Dprint("ELD check stellar type COWD? %s, Companion<=HeWD? %s, DMacc = %g > %g -> %s\n",
               Yesno(newstar->stellar_type==COWD),
               Yesno(companion_type<=HeWD),
               newstar->mass - newstar->phase_start_mass,
               stardata->preferences->mass_accretion_for_eld,
               Yesno(ELD));
        Dprint("new stellar type %d : mass %g cf. max %g\n",
               newstar->stellar_type,
               newstar->mass,
               stardata->preferences->max_HeWD_mass);


        if(ELD)           
        {
            Dprint("COWD ELD");
            newstar->SN_type=SN_IA_ELD;
            struct star_t * news = new_supernova(stardata,
                                                 newstar,
                                                 Other_star_struct(newstar),
                                                 newstar);
            
            Append_logstring(LOG_ROCHE,
                             "SNIa (ELD) M=%g; DM=%g > %g (companion type %d)",
                             newstar->mass,
                             newstar->mass - newstar->phase_start_mass,
                             stardata->preferences->mass_accretion_for_eld,
                             companion_type);
            if(news)
            {
                stellar_structure_make_massless_remnant(news);
            }
        }
#ifdef ALLOW_HeWD_SUPERNOVAE
        else if(newstar->stellar_type==HeWD &&
                companion_type<=HeWD &&
                More_or_equal(newstar->mass,HeWD_HeWD_IGNITION_MASS))
        {
            Dprint("HeWD SN");
            newstar->SN_type = SN_IA_He;
            struct star_t * news = 
                new_supernova(stardata,
                              newstar,
                              Other_star_struct(newstar),
                              newstar);
            Append_logstring(LOG_ROCHE,
                             "SNI? (Helium) M=%g > %g (companion type %d)\n",
                             newstar->mass,
                             HeWD_HeWD_IGNITION_MASS,
                             companion_type);
            if(news)
            {
                stellar_structure_make_massless_remnant(news);
            }
        }
#endif
        else if(newstar->stellar_type == HeWD &&
                newstar->mass > stardata->preferences->max_HeWD_mass)
        {
            /*
             * Maximum mass for a HeWD : ignite helium now
             */
            Dprint("mass exceeds max HeWD mass\n");
            newstar->stellar_type = HeMS;
            newstar->phase_start_mass = newstar->mass;
            newstar->core_mass = 0.0;
            call_calc_lt2(newstar->stellar_type,
                          newstar->phase_start_mass,
                           newstar->mass);
        }
    }

#if defined NUCSYN && defined NUCSYN_FIRST_DREDGE_UP_PHASE_IN
    newstar->mc_gb_was=0.0;
#endif//NUCSYN_FIRST_DREDGE_UP_PHASE_IN
}
#endif//BSE
