#include "../binary_c.h"
#ifdef BSE
#include "stellar_structure_debug.h"

static Constant_function double Karakas_lamaxf(double m,
                                               const Abundance z);
                           

static double Karakas_mcmin_for_dredgeup(const double m,
                                         const double phase_start_mass,
                                         struct stardata_t * RESTRICT const stardata);

                         
static double Pure_function core_growth_q(const double Z,
                                          const double X);



/************************************************************/

void stellar_structure_TPAGB(struct star_t * RESTRICT const newstar,
                             double * RESTRICT const CO_core_mass,
                             double * RESTRICT const mcmax,
                             const double mcbagb,
                             struct stardata_t * RESTRICT const stardata,
                             const Caller_id caller_id)
{  
    Dprint("stellar_structure_TPAGB (caller_id=%d, num_thermal_pulses=%g)\n",
           caller_id,
           newstar->num_thermal_pulses);
    Boolean above_mcmin = FALSE;
    /*
     * TPAGB star stellar structure
     */
    newstar->stellar_type = TPAGB;
    newstar->GB_core_mass = 0.0;
    newstar->derivative[DERIVATIVE_STELLAR_GB_CORE_MASS] = 0.0;
    newstar->derivative[DERIVATIVE_STELLAR_HE_CORE_MASS] = 0.0;
    newstar->derivative[DERIVATIVE_STELLAR_CO_CORE_MASS] = 0.0;
    newstar->derivative[DERIVATIVE_STELLAR_HE_CORE_MASS_NO_TDUP] = 0.0;

    
    /*
     * Choose the AGB structure method
     *
     * This is only relevant if we're running 
     * the 'default' method : then use Hurley without
     * NUCSYN, and Karakas with NUCSYN.
     *
     * This is really a backwards compatibility option, 
     * it's how binary_c worked for a decade or more.
     */
    int core_algorithm = AGB_Core_Algorithm;
    int radius_algorithm = AGB_Radius_Algorithm;

#if defined NUCSYN && defined NUCSYN_SECOND_DREDGE_UP 
    nucsyn_check_for_second_dredge_up(stardata,newstar,mcbagb);
#endif


    /* 
     * compare to the previous stardata to determine
     * whether we were a TPAGB star in the past. Because previous_stardata
     * does not change between multi-step solver calls, this 
     * always works.
     */
    const Boolean first_pulse = stardata->previous_stardata->star[newstar->starnum].stellar_type != TPAGB;

    double menv = Max(0.0,newstar->mass - newstar->core_mass);
    Dprint("first pulse? %d\n",first_pulse);

    if(unlikely(first_pulse))
    {
        /*
         * Many things must be initialised at the 
         * first thermal pulse
         */
        newstar->time_first_pulse = newstar->age;
        Dprint("time first pulse set to %g\n",newstar->time_first_pulse);
        newstar->num_thermal_pulses = 0.0;
        newstar->time_prev_pulse = 0.0; 
        newstar->prev_tagb = 0.0;
        const double mc1tp = guess_mc1tp(newstar->phase_start_mass,
                                         newstar->phase_start_mass,
                                         newstar,
                                         stardata);
        const double mcnew = Max(newstar->CO_core_mass,
                                 Min(mc1tp,newstar->core_mass));
        
        newstar->core_mass = mcnew;
        newstar->core_mass_no_3dup = mcnew;
        newstar->mc_1tp = mcnew;
        newstar->menv_1tp = newstar->mass - mcnew;
        newstar->Mc_prev_pulse = mcnew;
        newstar->interpulse_period =
            Karakas2002_interpulse_period(stardata,
                                          newstar->mc_1tp,
                                          newstar->mass,
                                          newstar->mc_1tp,
                                          stardata->common.metallicity,
                                          0.0,
                                          newstar);

        /*
         * newstar->time_next_pulse is relative to the start
         * of the TPAGB
         */
        newstar->time_next_pulse = newstar->interpulse_period*1e6;
    }
    else
    {
        /* if star accretes, increase menv_1tp */
        newstar->menv_1tp = Max(menv, newstar->menv_1tp);
    }

    /* 
     * age on the TPAGB (years)
     */
    double tagb = 1e6 * (newstar->age - newstar->time_first_pulse);
    Dprint(
            "tagb = %g from age = %g, first pulse at %g\n",
            tagb,
            newstar->age,
            newstar->time_first_pulse
        );

    /*
     * Lower limit to the interpulse period is 0.01 years
     * (this is arbitrary but prevents numerical issues).
     */
    newstar->interpulse_period = Max(1e-8,
                                     newstar->interpulse_period);


    /*
     * Calculate the third dredge up efficiency, lambda,
     * also sets above_mcmin : a Boolean which tells us whether
     * the core mass exceeds the minimum core mass for third
     * dredge up
     */
    newstar->lambda_3dup = lambda_3dup(stardata,newstar,&above_mcmin);
    
    if(core_algorithm == AGB_CORE_ALGORITHM_KARAKAS)
    {   
        /*
         * Grow the core because of hydrogen shell burning
         */
        
        /*
         * Requires the luminosity. This is prior to 
         * dredge up, so is only approximately correct.
         */
        newstar->luminosity = TPAGB_luminosity(newstar->stellar_type,
                                               newstar->mass,
                                               newstar->core_mass,
                                               newstar->mc_1tp,
                                               newstar->core_mass_no_3dup,
                                               newstar->age,
                                               stardata->common.metallicity,
                                               newstar->phase_start_mass,
                                               newstar->num_thermal_pulses,
                                               newstar->interpulse_period,
                                               newstar->time_first_pulse,
                                               newstar->time_next_pulse,
                                               stardata,
                                               newstar->GB);
                
        double qh6 = core_growth_q(stardata->common.metallicity,
#ifdef NUCSYN
                                   newstar->Xenv[XH1]
#else
                                   0.7
#endif
            );
        
        double lmax = newstar->mass < 3.0 ? 3e4 : 4e4;
        double dmcdt = Min(newstar->luminosity, lmax) * qh6 ; // per year
        
        /*
         * Hence the core growth rates
         */
        newstar->derivative[DERIVATIVE_STELLAR_GB_CORE_MASS] = 0.0;
        newstar->derivative[DERIVATIVE_STELLAR_HE_CORE_MASS] += dmcdt * (1.0 - newstar->lambda_3dup);
        newstar->derivative[DERIVATIVE_STELLAR_HE_CORE_MASS_NO_TDUP] += dmcdt;
        newstar->derivative[DERIVATIVE_STELLAR_CO_CORE_MASS] += dmcdt * (1.0 - newstar->lambda_3dup);

        /*
         * Core growth in this timestep (dt / years = dtm * 1e6)
         */
        double dmc = dmcdt * stardata->model.dtm * 1e6;

        /*
         * Core cannot grow outside the star
         */
        dmc = Min(newstar->mass - newstar->core_mass, dmc);

        /*
         * Hence the change in core mass
         */
        double ddmc = dmc;
        
        /*
         * Note: dt, and hence dmc, could be negative.
         */
        Dprint("dmc = %g from dmc/dt = %g, qh6 = %g, L = %g, dt = %g, newstar->dm_3dup = %g ::: ddmc = %g\n",
               dmc,
               dmcdt,
               qh6,
               newstar->luminosity,
               stardata->model.dt,
               newstar->dm_3dup,
               ddmc
            );


        
        Dprint("H-shell burn grown core to %g\n",newstar->core_mass);
        Dprint("Check for next pulse at tagb=%g : dntp = %g, Next_pulse = %d, Interpulse period %g years\n",
               tagb,
               newstar->dntp,
               Next_pulse(newstar),
               1e6*newstar->interpulse_period);
        
    }
    else if(core_algorithm == AGB_CORE_ALGORITHM_HURLEY)
    {
        /* 
         * BSE core-mass growth algorithm
         *
         *
         * Do not let lambda_3dup drop : 
         * this causes problems when the envelope 
         * mass is small because lambda -> 0 (formally)
         * because Menv < Menv_min_for_third_dup.
         * This causes the core to grow wildly, 
         * which is wrong and gives a SN. Leaves lambda
         * as is is equivalent to the Hurley algorithm.
         */
        newstar->lambda_3dup = Max(newstar->lambda_3dup,
                                   lambda_3dup(stardata,
                                               newstar,
                                               &above_mcmin));
        
        /*
         * The original BSE algorithm calculates
         * core mass changes by integrating the 
         * Mc(t) function manually. This requires
         * an exact solution for the time at the first
         * thermal pulses.
         *
         * Instead of this, we calculate dMc/dt 
         * and integrate this (with respect to time)
         * numerically. This is more stable against
         * non-BSE radii, luminosities, etc. which
         * give a different time at the first pulse.
         * Now, just the time *relative* to the first
         * thermal pulse time is important.
         */
        
        /*
         * calculate raw dMc/dt because of shell burning
         */
        double dmcdt = dmcgbtf_dt(
            newstar->age,
            newstar->GB[GB_A_H_HE],
            newstar->GB,
            newstar->timescales[T_TPAGB_TINF_1],
            newstar->timescales[T_TPAGB_TINF_2],
            newstar->timescales[T_TPAGB_TX]
            );
        newstar->derivative[DERIVATIVE_STELLAR_GB_CORE_MASS] = 0.0;
        newstar->derivative[DERIVATIVE_STELLAR_HE_CORE_MASS] = dmcdt * 1e-6;
        newstar->derivative[DERIVATIVE_STELLAR_CO_CORE_MASS] = dmcdt * 1e-6;
        double dmc = stardata->model.dtm * dmcdt;
  
        /*
         * Core cannot grow outside the star
         */
        dmc = Min(newstar->mass - newstar->core_mass, dmc);

        /*
         * Increase core mass, (eventually) taking into account 
         * any third dredge up
         */
        newstar->derivative[DERIVATIVE_STELLAR_GB_CORE_MASS] = 0.0;
        newstar->derivative[DERIVATIVE_STELLAR_HE_CORE_MASS] += dmcdt * 1e-6 * (1.0 - newstar->lambda_3dup);
        newstar->derivative[DERIVATIVE_STELLAR_HE_CORE_MASS_NO_TDUP] += dmcdt * 1e-6;
        newstar->derivative[DERIVATIVE_STELLAR_CO_CORE_MASS] += dmcdt * 1e-6 * (1.0 - newstar->lambda_3dup);
        
#ifdef ORIGINAL_BSE_TPAGB
        /*
         * Original BSE algorithm.
         * Note, will not always work well especially
         * when combined with other formalisms!
         */
        newstar->core_mass = mcgbtf(
            newstar->age,
            newstar->GB[GB_A_H_HE],
            newstar->GB,
            newstar->timescales[T_TPAGB_TINF_1],
            newstar->timescales[T_TPAGB_TINF_2],
            newstar->timescales[T_TPAGB_TX]
            );

        /*
         * Approximate 3rd Dredge-up on AGB by limiting (CO) Core_Mass.
         */ 
        *CO_core_mass = mcgbtf(
            newstar->timescales[T_TPAGB_FIRST_PULSE],
            newstar->GB[2],
            newstar->GB,
            newstar->timescales[T_TPAGB_TINF_1],
            newstar->timescales[T_TPAGB_TINF_2],
            newstar->timescales[T_TPAGB_TX]
            );
        newstar->core_mass -= newstar->lambda_3dup * (newstar->core_mass - *CO_core_mass);
#endif // ORIGINAL_BSE_TPAGB
    }
    else
    {
        Exit_binary_c(BINARY_C_ALGORITHM_BRANCH_FAILURE,
                      "AGB core algorithm %d is unknown",
                      core_algorithm);
    }



    /*
     * Calculate luminosity
     */
    newstar->luminosity = TPAGB_luminosity(newstar->stellar_type,
                                           newstar->mass,
                                           newstar->core_mass,
                                           newstar->mc_1tp,
                                           newstar->core_mass_no_3dup,
                                           newstar->age,
                                           stardata->common.metallicity,
                                           newstar->phase_start_mass,
                                           newstar->num_thermal_pulses,
                                           newstar->interpulse_period,
                                           newstar->time_first_pulse, 
                                           newstar->time_next_pulse,
                                           stardata,
                                           newstar->GB);
    
    if(TPAGB_LUMTYPE==TPAGB_LUMINOSITY_DIPS &&
       tagb<MAX_TPAGB_TIME &&
       newstar->num_thermal_pulses<PULSE_LUM_DROP_N_PULSES)
    {
        /*
         * crude attempt to parameterise the post-pulse luminosity drop
         */
        newstar->spiky_luminosity = newstar->luminosity 
            * (1.0-Min(THERMAL_PULSE_LUM_DROP_FACTOR,
                       THERMAL_PULSE_LUM_DROP_FACTOR*
                       exp(-THERMAL_PULSE_LUM_DROP_TIMESCALE*
                           (tagb-newstar->time_prev_pulse)/(1e6*newstar->interpulse_period))
                   ));
    }

    Dprint("TPAGB L = %g\n",newstar->luminosity);

    /*
     * Hence radius
     */
    double r_Hurley =
        ragbf(newstar->mass,
              newstar->luminosity,
              stardata->common.metallicity_parameters[ZPAR_MASS_HE_FLASH],
              stardata->common.giant_branch_parameters
#ifdef AXEL_RGBF_FIX
              ,stardata->common.metallicity
#endif
            );

    if(radius_algorithm == AGB_RADIUS_ALGORITHM_KARAKAS)
    {
        newstar->radius = Karakas2002_ragb_teff(newstar->mass,
                                                stardata->common.metallicity,
                                                newstar->luminosity,
                                                newstar->phase_start_mass,
                                                newstar->mc_1tp,
                                                newstar->mass - newstar->core_mass,
                                                newstar->core_mass_no_3dup,
                                                newstar,
                                                stardata);
        /*
         * If M > TPAGB_MASS_BREAK, interpolate into the
         * Hurley+ (2002) radius fit.
         */
        if(newstar->mass > TPAGB_MASS_BREAK)
        {
            double f8 = 1.0 / (1.0 + pow(0.01, newstar->mass - 8.0));
            newstar->radius = f8*r_Hurley + (1.0-f8)*newstar->radius;
        }
    }
    else if(radius_algorithm == AGB_RADIUS_ALGORITHM_HURLEY)
    {
        newstar->radius = r_Hurley;
    }
    else
    {
        Exit_binary_c(BINARY_C_ALGORITHM_BRANCH_FAILURE,
                      "AGB radius algorithm %d is unknown",
                      radius_algorithm);
    }
    Dprint("TPAGB R = %g\n",newstar->radius);


    /*
     * CO core mass = He core mass - intershell mass
     */
    *CO_core_mass = newstar->core_mass - dm_intershell(newstar->core_mass);

    /*
     * Maximum core mass cannot exceed the total mass
     */
    *mcmax = Min(newstar->mass,*mcmax);

#ifdef WD_KICKS
    if(
        /* WD kick at a given pulse number */    
        (stardata->preferences->wd_kick_when == WD_KICK_AT_GIVEN_PULSE &&
         newstar->already_kicked_WD==FALSE &&
         newstar->num_thermal_pulses+1 > stardata->preferences->wd_kick_pulse_number)
        ||
        /* WD kick every pulse */
        (stardata->preferences->wd_kick_when == WD_KICK_AT_EVERY_PULSE &&
         ( (int)(newstar->num_thermal_pulses+0.01) - newstar->prev_kick_pulse_number > 0))
        )
    {
        newstar->kick_WD=TRUE;
    }
#endif // WD_KICKS

#ifdef NUCSYN

    /*
     * Require core radius for logging
     */
#ifdef HALL_TOUT_2014_RADII
    newstar->core_radius = Hall_Tout_2014_low_mass_HG_RGB_radius(newstar->core_mass,
                                                                 stardata);
#else
    newstar->core_radius = 5.0*rwd(newstar->core_mass,stardata);
#endif // HALL_TOUT_2014_RADII

    /*
     * Apply 3DUP/HBB changes to surface abundances only 
     * on the final step of the solver.
     */
    if(stardata->model.intermediate_step == FALSE)
    {
        nucsyn_tpagb(&newstar->phase_start_mass,
                     &newstar->age,
                     &newstar->core_mass,
                     &newstar->mass,
                     &newstar->luminosity,
                     &newstar->radius,
                     &newstar->core_radius,
                     newstar,
                     stardata,
                     newstar->GB);
    }
#endif // NUCSYN

}


double lambda_3dup(struct stardata_t * RESTRICT const stardata,
                   struct star_t * RESTRICT const newstar,
                   Boolean * const above_mcmin)
{
    double lambda;

    /*
     * Lambda = 0 if third dup is deliberately switched off
     * or if menv < threshold (if NOT using the Hurley formalism). 
     */
    Dprint("Calc lambda : tdup on? %d : menv = %g > menv,min = %g ?\n",
           stardata->preferences->third_dup,
           newstar->mass - newstar->core_mass,
           Minimum_envelope_mass_for_3dup);
    

    if(stardata->preferences->third_dup == FALSE ||
       newstar->mass - newstar->core_mass < Minimum_envelope_mass_for_3dup)
    {
        *above_mcmin = FALSE;
        lambda = 0.0;
    }
    else
    {
        /*
         * Determine algorithm to use
         */
        int third_dup_algorithm = AGB_Third_Dredge_Up_Algorithm;

        if(third_dup_algorithm == AGB_THIRD_DREDGE_UP_ALGORITHM_KARAKAS)
        {
            lambda = lambda_3dup_Karakas(above_mcmin,newstar,stardata);
        }
        else if(third_dup_algorithm == AGB_THIRD_DREDGE_UP_ALGORITHM_STANCLIFFE)
        {
            /*
             * Richard Stancliffe's fits for LMC/SMC
             */
            lambda = lambda_3dup_stancliffe(above_mcmin,
                                            newstar,
                                            newstar->num_thermal_pulses_since_mcmin,
                                            stardata);
        }
        else if(third_dup_algorithm == AGB_THIRD_DREDGE_UP_ALGORITHM_HURLEY)
        {
            lambda = lambda_3dup_Hurley(above_mcmin,newstar->phase_start_mass);
        }
        else
        {
            lambda = 0.0;
            Exit_binary_c(BINARY_C_ALGORITHM_BRANCH_FAILURE,
                          "AGB 3dup algorithm %d is unknown",
                          third_dup_algorithm);
    
        }

        /*
         * Artificial modulation
         */
        lambda *= stardata->preferences->lambda_multiplier;
        lambda = Max(stardata->preferences->lambda_min, 
                     lambda);
    }
    
    Dprint("Calc lambda : Boolean %d : menv %g > %g ? -> above mcmin %d, lambda = %g\n",
           stardata->preferences->third_dup,
           Max(0.0,newstar->mass - newstar->core_mass),
           Minimum_envelope_mass_for_3dup,
           * above_mcmin,
           lambda);
    return lambda;
}

double lambda_3dup_stancliffe(Boolean * above_mcmin,
                              struct star_t * newstar,
                              const double n,
                              struct stardata_t * stardata)
{
    /*
     * Third dredge up efficiency based on Richard Stancliffe's models
     * of SMC and LMC metallicity stars. Valid for 1.5<M<6 only.
     * Fit form is like Karakas et al. except with a drop after about 15 pulses
     */
    double l=0.0;
    double m = Limit_range(newstar->phase_start_mass,1.5,6.0);
    double mcmin = 
        stardata->common.metallicity > 0.006 ? 
        // high metallicity : use LMC
        (9.65900e-01+
         2.63090e-01*exp(-Pow2(m-1.07240)/1.26290)+
         -7.37630e-01/(1+pow(3.34400e-01,2.86280-m)))
        : 
        // low metallicity : use SMC
        (5.80420e-01+
         6.53760e-02*m/(1+pow(9.80490e-02,m-2.66910)));
    
    mcmin += stardata->preferences->delta_mcmin;
    mcmin = Max(0.0,mcmin);

    if(newstar->core_mass < mcmin)
    {
        l = 0.0;
        * above_mcmin = FALSE;
    }
    else
    {
        double z = stardata->common.metallicity;
        double c1=15.0*z +
            Quartic(m,2.03690,-2.87750,1.62060,-3.35270e-01,2.32110e-02);
        double c2 = 
            6.83500-4.44870/(1.0+pow(2.0e-03,m-2.5));
        l = Max(stardata->preferences->lambda_min, c1) 
            * (1.0 - exp(-n/c2) * (1.0 - 0.42*n));
        Clamp(l,0.0,1.0);
        * above_mcmin = TRUE;
    }

    return(l);
}

double Pure_function lambda_3dup_Hurley(Boolean * RESTRICT const above_mcmin,
                                        const double m)
{
    /*
     * Third dredge up efficiency based on Hurley et al. (2002)
     */
    double lambda = Min(0.90,(0.30+0.0010*Pow5(m)));
    * above_mcmin = (lambda > 0.0); 
    return lambda;
}

double lambda_3dup_Karakas(
    Boolean * RESTRICT const above_mcmin,
    struct star_t * RESTRICT const newstar,
    struct stardata_t * RESTRICT const stardata
    )
{
    /*
     * Table of NCAL fits to Karakas' new (2007) data tables
     */
    double lambda;
    double mcmin = Karakas_mcmin_for_dredgeup(newstar->phase_start_mass,
                                              newstar->phase_start_mass,
                                              stardata);
    mcmin += stardata->preferences->delta_mcmin;
    mcmin = Max(0.0, mcmin);

    if(newstar->core_mass < mcmin)
    {
        lambda = 0.0;
        * above_mcmin = FALSE;
    }
    else
    {
        double x[2]={
            stardata->common.metallicity,
            Limit_range(newstar->phase_start_mass,1.0,6.5)
        };
        double rr[1]; // stores the resulting NCAL
        Interpolate(stardata->store->Karakas_ncal,
                    x,rr,2);
        double lammax = Min(1.0,
                            Karakas_lamaxf(newstar->phase_start_mass,
                                           stardata->common.metallicity));
                                         
        Dprint("lammax = %g\n",lammax);
        Clamp(lammax,stardata->preferences->lambda_min,1.0);
        lambda = (1.0-exp(-newstar->num_thermal_pulses_since_mcmin/rr[0]))
            *lammax;
        * above_mcmin = TRUE;
    }

    Dprint("Lambda 3dup Karakas lambda = %g, above mcmin? %d (NTP since mcmin %g)\n",
           lambda,
           *above_mcmin,
           newstar->num_thermal_pulses_since_mcmin
        );

    return lambda;
}


static Constant_function double Karakas_lamaxf(double m,
                                               const Abundance z)
{
    /*
     * Asymptotic value of dredge-up parameter lambda, according to the fitting
     * formula by Amanda Karakas (for non-overshooting models!).
     */
    double l=0.0;
    const double m3 = Pow3(m);
    
    /* 
     * Maximum dredge-up parameter (asymptotic value) as a function
     * of total mass M and metallicity Z.
     *  Fitting formula from Karakas, Lattanzio & Pols 2002.
     */
    Clamp(m,1.0,6.5);

#define ASIZE 4

    /*
     * Asymptotic value of dredge-up parameter lambda, according to the fitting
     * formula by Amanda Karakas (for non-overshooting models!). NB: the
     * Z-dependence has not yet been included, m is the mass at the beginning of
     * the AGB (NOT the current mass!)
     */
    const double zz1 = 0.02;
    const double zz2 = 0.008; 
    const double zz3 = 0.004;
    const double zz4 = 0.0001;
    double h1,h2;
    const double a1[ASIZE] = {-1.17696,0.76262,0.026028,0.041019}; 
    const double a2[ASIZE] = {-0.609465,0.55430,0.056878,0.069227}; 
    const double a3[ASIZE] = {-0.764199,0.70859,0.058833,0.075921};
    //const double a4[ASIZE] = {-0.82367052137017,0.855045797056254,0.075969726978165,0.094591645793495};
  
    /* better fit from Rob */
    const double a4[ASIZE] = {-8.90480e-01,8.84020e-01,1.12030e-01,1.41860e-01};

    /* new fit based on Amanda's new "radius tables":
     *  This is a smoother transition from amanda's functions to the newer fit
     * and neglects her pulses which have lambda>1.1 (the 'degenerate' pulses - I guess!)
     *
     * Note: this breaks at 4Msun Z=0.0001 - oh dear! 
     * Best avoided then, use the "better fit" above.
     */
    //const double a4[ASIZE]={-5.62410e-01,7.32720e-01,1.63250e-02,3.45080e-02};

    double a[ASIZE];
    double zz = z; //log10(z/0.02);
    unsigned int i;

    zz=z;
    if(zz>zz2) 
    {
        h1 = (zz-zz2)/(zz1-zz2);
        h2 = 1.0 - h1;
        for(i=0;i<ASIZE;i++)
        {
            a[i] = a1[i]*h1 + a2[i]*h2;
        }
    }
    else if(zz>zz3)
    {
        h1 = (zz-zz3)/(zz2-zz3);
        h2 = 1.0 - h1;
        for(i=0;i<ASIZE;i++)
        {
            a[i] = a2[i]*h1 + a3[i]*h2;
        }
    }
    else
    {
        h1 = (zz-zz4)/(zz3-zz4);
        h2 = 1.0 - h1;
        for(i=0;i<ASIZE;i++)
        {
            a[i] = a3[i]*h1 + a4[i]*h2;
        }
    }




    /* hence the fit */
    l = (a[0] + a[1]*m + a[2]*m3)/(1.0 + a[3]*m3);
    Clamp(l,0.0,1.0);
    return (l);

#undef ASIZE

}

static double Karakas_mcmin_for_dredgeup(const double m,
                                         const double phase_start_mass,
                                         struct stardata_t * RESTRICT const stardata)
{

    /* Updated version from Onno (thanks Onno!) */
#define ASIZE 4
    const Abundance zz1 = 0.02; 
    const Abundance zz2 = 0.008; 
    const Abundance zz3 = 0.004;
    const Abundance zz4 = 0.0001;
    const double a1[ASIZE] = {0.732759,-0.0202898,-0.0385818,0.0115593};
    const double a2[ASIZE] = {0.672660,0.0657372,-0.1080931,0.02754823};
    const double a3[ASIZE] = {0.516045,0.2411016,-0.1938891,0.0446382};
    //double a4[ASIZE] = {0.516045,0.2411016,-0.1938891,0.0446382};
    /* updated values from amanda */
    //const double a4[ASIZE]={0.78887270271463,-0.43719446893414,0.24178487467924,-0.031176683714852};
    /* updated fit from rob */
    const double a4[ASIZE]={2.18500e-01,6.21750e-01,-3.85190e-01 ,8.84560e-02};

    double h1,h2;
    double y;
    Abundance zz = stardata->common.metallicity;
    Clamp(zz,0.0001,0.02);

    
    /*
     * Choose to interpolcate the coefficients or the result.
     * Really, you should interpolate the result, such that the
     * expressions are calculated at metallicities at which they
     * are known to be correct.
     */
    //#define INTERPOLATE_COEFFS
#define INTERPOLATE_MCMIN

    const double *al , *ar;
#ifdef INTERPOLATE_COEFFS
    double a[ASIZE];
    unsigned int i;

    if(zz>zz2) 
    {
        h1 = (zz-zz2)/(zz1-zz2);
        al = a1;
        ar = a2;
    }
    else if(zz>zz3) 
    {
        h1 = (zz-zz3)/(zz2-zz3);
        al = a2;
        ar = a3;
    }
    else 
    {
        h1 = (zz-zz4)/(zz3-zz4);
        al = a3;
        ar = a4;
    }
    h2 = 1.0 - h1;
    for(i=0;i<ASIZE;i++)
    {
        a[i] = al[i]*h1 + ar[i]*h2; 
    }
    y = Cubic_array(m,a);
#endif

#ifdef INTERPOLATE_MCMIN
    double y1,y2;
    if(zz>zz2)
    {
        h1 = (zz-zz2)/(zz1-zz2);
        al = a1;
        ar = a2;
    }
    else if(zz>zz3)
    {
        h1 = (zz-zz3)/(zz2-zz3);
        al = a2;
        ar = a3;
    }
    else
    {
        h1 = (zz-zz4)/(zz3-zz4);
        al = a3;
        ar = a4;
    }
    y1 = Cubic_array(m,al);
    y2 = Cubic_array(m,ar);
    h2 = 1.0 - h1;
    y = h1*y1 + h2*y2;
#endif

    /* Now to fix for M>Msdu-0.5 */
    y = Max3(0.0,
             Karakas2002_mc1tp(m,phase_start_mass,stardata),
             Min(0.7,y));
    
    return(y);
}


static double Pure_function core_growth_q(const double Z,
                                          const double X)
{
    /*
     * Core growth factor where
     * dMc/dt = Q L
     *
     * This is based on Karakas' models, see
     * Izzard et al. (2004) for details.
     *
     * The factor 0.5 is to fit the detailed models
     * better with the new evolution algorithm. 
     */
    return 
        0.5 * 1e-11 * (2.72530 - 1.8629 * Z)
        * 
        Max(Min(1.0,7.94870e-01 + 5.12820e+01 * Z),0.8);
}


double TPAGB_luminosity(const Stellar_type stellar_type,
                        const double m,
                        const double mc,
                        const double mc1tp,
                        const double mcnodup,
                        const double age,
                        const double Z,
                        const double phase_start_mass,
                        const double num_thermal_pulses,
                        const double interpulse_period,
                        const double time_first_pulse,
                        const double time_next_pulse,
                        struct stardata_t * RESTRICT const stardata,
                        double * RESTRICT const GB
    )
{
    /*
     * Generic wrapper for the current TPAGB luminosity.
     */

    double luminosity;
    int luminosity_algorithm = AGB_Luminosity_Algorithm;

    if(luminosity_algorithm == AGB_LUMINOSITY_ALGORITHM_KARAKAS)
    {
        luminosity = Karakas2002_lumfunc(stellar_type,
                                         m,
                                         mc1tp,
                                         m - mc,
                                         mcnodup,
                                         phase_start_mass,
                                         num_thermal_pulses,
                                         interpulse_period,
                                         time_first_pulse,
                                         time_next_pulse,
                                         age,
                                         Z,
                                         stardata,
                                         GB);
    }
    else if(luminosity_algorithm == AGB_LUMINOSITY_ALGORITHM_HURLEY)
    {
        luminosity = lmcgbf(mc,GB);
    }
    else
    {
        luminosity = 0.0;
        Exit_binary_c(BINARY_C_ALGORITHM_BRANCH_FAILURE,
                      "AGB luminosity algorithm %d is unknown",
                      luminosity_algorithm);
    }

    return luminosity;
}
#endif//BSE
