#include "../binary_c.h"
#ifdef BSE

#include "stellar_structure_debug.h"
#define LOW -1
#define HIGH 1
#define p (newstar->GB[GB_p])
#define q (newstar->GB[GB_q])
#define B (newstar->GB[GB_B])
#define D (newstar->GB[GB_D])
#define A (newstar->GB[GB_EFFECTIVE_AH])

Stellar_type stellar_structure_RG(struct star_t * RESTRICT const newstar,
                                  double * RESTRICT const rg,
                                  struct stardata_t * RESTRICT const stardata,
                                  const Caller_id caller_id)
{
    /*
     * Red Giant.
     */
    const double * metallicity_parameters=stardata->common.metallicity_parameters; 
    const double * giant_branch_parameters=stardata->common.giant_branch_parameters;
    Stellar_type stellar_type=FIRST_GIANT_BRANCH;

    /*
     * BSE works by using t as the proxy parameter
     * a) finding L(t+dt)
     * b) finding Mc(L)
     *
     * We want to use Mc as the parmeter instead.
     * a) find L(Mc) 
     * b) hence dMc/dt and the new Mc(t+dt)
     *
     * Equation numbers are from Hurley+2000
     */

    newstar->luminosity = lgbtf(newstar->age,
                                newstar->GB[GB_EFFECTIVE_AH],
                                newstar->GB,
                                newstar->timescales[T_GIANT_TINF_1],
                                newstar->timescales[T_GIANT_TINF_2],
                                newstar->timescales[T_GIANT_TX]);
    Dprint("L(GB,age=%g) = %g\n",
           newstar->age,
           newstar->luminosity);

    /*
     * Account for growth at the end of the HG.
     * This is often missed because of finite time resolution,
     * and while it is a small effect it can be annoying.
     */
    newstar->core_mass = Max(mcbgb(newstar,stardata),
                             newstar->core_mass);
    
    Boolean degenerate_core = Less_or_equal(newstar->phase_start_mass,
                                            metallicity_parameters[ZPAR_MASS_HE_FLASH]); 

    
    /*
     * BSE algorithm for Mc(L(t))
     */
    if(degenerate_core)
    {
        /*                
         * Star has a degenerate He core which grows on the GB
         */
        Dprint("Degenerate He Core with L=%g and core_mass was %g\n",
               newstar->luminosity,
               newstar->core_mass);
        newstar->core_mass = mcgbf(newstar->luminosity,
                                   newstar->GB,
                                   newstar->luminosities[L_LMX]);
        Dprint("Hence core mass %g\n",newstar->core_mass);
    }
    else
    {
        /*
         * Star has a non-degenerate He core which may grow, but
         * only slightly, on the GB
         */
        Dprint("Non-Degenerate He Core\n");
        double tbranch = newstar->timescales[T_HE_IGNITION] - newstar->timescales[T_BGB];
        double tau = tbranch>TINY ? (newstar->age - newstar->timescales[T_BGB])/tbranch : 1.0;
        double mcx = mcheif(newstar->phase_start_mass,
                            metallicity_parameters[ZPAR_MASS_HE_FLASH],
                            metallicity_parameters[9],
                            giant_branch_parameters);
        double mcy = mcheif(newstar->phase_start_mass,
                            metallicity_parameters[ZPAR_MASS_HE_FLASH],
                            metallicity_parameters[10],
                            giant_branch_parameters);
        newstar->core_mass = mcx + (mcy - mcx)*tau;    
        Dprint("core_mass = %g = %g+(%g-%g)*%g\n",newstar->core_mass,mcx,mcy,mcx,tau);
        Nancheck(newstar->core_mass);
    }
    
    Dprint("setting r from rgbf to ");
    newstar->radius = rgbf(newstar->mass,
                           newstar->luminosity,
                           giant_branch_parameters);
    Dprint("%12.12e\n",newstar->radius);

    *rg = newstar->radius;

    /*
     * Check if the star contains only the helium core
     */
    if(More_or_equal(newstar->core_mass, newstar->mass))
    {
        newstar->age = 0.0;

        const Boolean ignite =
            Fequal(
                stardata->preferences->minimum_helium_ignition_core_mass,
                HELIUM_IGNITION_ALGORITHM_BSE
                ) ?
            /* use BSE algorithm based on the phase start mass */
            newstar->phase_start_mass > metallicity_parameters[ZPAR_MASS_HE_FLASH]
            :
            /* use helium core mass algorithm */
            newstar->core_mass > min_He_ignition_core_mass(stardata,newstar);

        Dprint("CF%d phase start mass %g to zpar %g\n",
               newstar->starnum,
               newstar->phase_start_mass,
               metallicity_parameters[ZPAR_MASS_HE_FLASH]);
        
        Dprint("CF Mc = %g to He ig Mc %g -> ignite? %s\n",
               newstar->core_mass,
               min_He_ignition_core_mass(stardata,newstar),
               Yesno(ignite));
        
        if(ignite == TRUE)
        {
            /*
             * Zero-age helium star
             */
            Dprint("Zero-age helium star\n");
            newstar->core_mass=0.0;
            newstar->phase_start_mass = newstar->mass;
            stellar_type = HeMS;
        }
        else
        {
            /*
             * Zero-age helium white dwarf.
             */
            Dprint("Zero-age helium white dwarf\n");
            newstar->core_mass = newstar->mass;
            newstar->phase_start_mass = newstar->mass;
            stellar_type = HeWD;
        }
    }
    else
    {
        /* star is a normal red giant : we can do first DUP etc. */


        
#if defined NUCSYN && defined NUCSYN_FIRST_DREDGE_UP
        if(!RLOF_CALL)
        {
          
            Dprint("1DUP star %d check mc=%12.12e vs mc1dup(evert)=%12.12e\n",
                   newstar->starnum,
                   newstar->core_mass,
                   mc_1DUP(stardata,newstar->phase_start_mass,stardata->common.metallicity));
#ifdef  NUCSYN_FIRST_DREDGE_UP_PHASE_IN

            /* 1st DUP is a process */
            if(newstar->first_dredge_up==FALSE &&
               newstar->stellar_type==FIRST_GIANT_BRANCH &&
               newstar->effective_zams_mass < NUCSYN_WR_MASS_BREAK)
            {
                nucsyn_set_1st_dup_abunds(newstar->Xenv,
                                          newstar->mass,
                                          stardata->common.metallicity,
                                          FIRST_GIANT_BRANCH,
                                          newstar,
                                          stardata,
                                          FALSE);
            }
        
#else
            /* 1st DUP is an event */
            if(newstar->first_dredge_up==FALSE
               && stellar_type==FIRST_GIANT_BRANCH
               && newstar->effective_zams_mass<NUCSYN_WR_MASS_BREAK
               && newstar->core_mass > mc_1DUP(newstar->phase_start_mass,
                                               stardata->common.metallicity))
            {
                nucsyn_set_1st_dup_abunds(newstar->Xenv,
                                          newstar->mass,
                                          stardata->common.metallicity,
                                          FIRST_GIANT_BRANCH,
                                          newstar,
                                          stardata,
                                          TRUE);
                newstar->first_dredge_up=TRUE;
            }
#endif // NUCSYN_FIRST_DREDGE_UP_PHASE_IN
        }
#endif /* NUCSYN_FIRST_DREDGE_UP && NUCSYN */
    }

#ifdef NUCSYN
    if(!RLOF_CALL)
    {
        /*
         * Save the core mass for the first dredge up routine
         */
        if(newstar->mc_bgb<TINY) newstar->mc_bgb = newstar->core_mass;
    }
#endif

    newstar->stellar_type = stellar_type;

    if(newstar->stellar_type==HeMS) 
    {
        call_calc_lt2(newstar->stellar_type,
                      newstar->phase_start_mass,
                      newstar->mass);
    }
    return(stellar_type);
}
#endif//BSE
