#include "../binary_c.h"
#ifdef BSE

#include "stellar_structure_debug.h"

#define MSR_DEC_ARGS double mass,double age,double tau,double tau1,double tau2,double tp1,double tp2,double lum,double rzams,double rtms,double zp1,const double *main_sequence_parameters,const double *metallicity_parameters,struct star_t *newstar,struct stardata_t * stardata

#define MSR_ARGS newstar->age,tau,tau1,tau2,tp1,tp2,newstar->luminosity,newstar->rzams,newstar->rtms,zp1,main_sequence_parameters,metallicity_parameters,newstar,stardata

static double main_sequence_radius(MSR_DEC_ARGS);

/*
 * BSE function to calculate main sequence 
 * relative age, hence luminosity and
 * radius. Also sets core mass to zero.
 *
 * See Hurley et al 2000 (H00) for details.
 */


Stellar_type stellar_structure_MS_BSE(struct star_t * RESTRICT const newstar,
                                      struct stardata_t * RESTRICT const stardata)
{
    const double * main_sequence_parameters=stardata->common.main_sequence_parameters; 
    const double * metallicity_parameters=stardata->common.metallicity_parameters; 

    /************************/
    /** Main sequence star **/
    /************************/
    double tau,tau1,tau2,ithook,tp,tp1,tp2;
    double alpha,logl,zp1=metallicity_parameters[ZPAR_MASS_MS_HOOK];
    Dprint("MS star\n");

    /* MS stars have no He, CO or GB core */
    newstar->core_mass=0.0;
    newstar->CO_core_mass = 0.0;
    newstar->GB_core_mass = 0.0;

    newstar->derivative[DERIVATIVE_STELLAR_HE_CORE_MASS] = 0.0;
    newstar->derivative[DERIVATIVE_STELLAR_GB_CORE_MASS] = 0.0;
    newstar->derivative[DERIVATIVE_STELLAR_CO_CORE_MASS] = 0.0;
  
    
    /*
     * Calculate the maximum convective core
     * mass on the MS. 
     * In low-mass stars this is zero.
     */
    double fconvcore = 
        effective_core_mass_fraction_of_MS_stars(newstar->mass,
                                                  stardata->common.metallicity);
    
    newstar->max_MS_core_mass = Max(newstar->max_MS_core_mass,
                              newstar->mass * fconvcore);


#if defined NUCSYN && defined NUCSYN_FIRST_DREDGE_UP_ACCRETION_CORRECTION
    /* save the main sequence mass, which will remain at the terminal MS value */
    newstar->MS_mass = newstar->phase_start_mass;
#endif
    
    /* convert to "low mass main sequence" star if mass is low enough */
    newstar->stellar_type = (newstar->phase_start_mass < zp1 - 0.3) ?
        LOW_MASS_MAIN_SEQUENCE : MAIN_SEQUENCE;
  
    /* 
     * tau is the fractional age (tm=MS lifetime)
     * H00 Eq. 11
     */
    tau = Is_zero(newstar->tm) ? 0.0 : newstar->age/newstar->tm;

#ifdef USE_BSE_MS_LR
    /*
     * Use table to look up L,R instead of formulae 
     */
    {
        double params[3] = {
            log10(stardata->common.metallicity),
            log10(newstar->phase_start_mass),
            tau
        };
        double result[2];
        Interpolate(stardata->store->BSE_MS_LR,params,result,FALSE);
        newstar->luminosity = pow(10.0,result[0]);
        newstar->radius = pow(10.0,result[1]);
    }
#else // USE_BSE_MS_LR

    /*
     * MS "hook" time, thook, H00 Eq. 7
     */
    double thook = newstar->timescales[T_BGB]*thookf(newstar->phase_start_mass,
                                               main_sequence_parameters);

    /* calculate ratio of age to time to the MS hook */
    ithook=newstar->age/thook;
  
    /*
     * Times relative to thook, H00 Eqs. 14 and 15
     */
    tau1 = Min(1.0,ithook);
    tau2 = (ithook-1.0)*100.0 + 1.0;
    tau2 = Min(1.0,tau2);
    tau2 = Max(0.0,tau2);
    tp = Pow2(tau);
    tp1 = Pow2(tau1);
    tp2 = Pow2(tau2);

    /*
     * MS luminosity H00 Eq. 12.
     * alpha is calculted using H00 Eq.19 
     */
    alpha=lalphf(newstar->phase_start_mass,main_sequence_parameters);
    logl = alpha*tau-lhookf(newstar->phase_start_mass,zp1,main_sequence_parameters)*(tp1-tp2)+
        (log10(newstar->luminosities[L_END_MS]/newstar->luminosities[L_ZAMS])-alpha)*tp;
    if(tau>TAUMIN)
    {
        /*
         * beta is calculated using H00 Eq. 20 
         */
        logl += lbetaf(newstar->phase_start_mass,main_sequence_parameters)
            *(pow(tau,lnetaf(newstar->phase_start_mass,main_sequence_parameters))-tp);
    }

    /* hence calculate luminosity */
    newstar->luminosity = newstar->luminosities[L_ZAMS]*pow(10.0,logl);

    /* calculate radius */
    Dprint("Radius(phase_start_mass = %g, L = %g) ",
           newstar->phase_start_mass,
           newstar->luminosity
        )
    newstar->radius = main_sequence_radius(newstar->phase_start_mass,
                                           MSR_ARGS);

    Dprint("Fitted R=%g L=%g\n",newstar->radius,newstar->luminosity);
    
#endif // USE_BSE_MS_LR

#ifdef MAIN_SEQUENCE_STRIP
    /*
     * Calculate the radius the star would have in the absence of mass loss
     * and modify it
     */
    {
        double fr,r_no_massloss = 
            main_sequence_radius(newstar->pms_mass,MSR_ARGS);

        if(stardata->model.in_RLOF==TRUE && 
           Is_zero(newstar->tms_at_start_caseA))
        {
            newstar->tms_at_start_caseA=tm;
#ifdef MAIN_SEQUENCE_STRIP_DEBUG
            printf("CASE A START AT t=%g (tms=%g)\n",
                   stardata->model.time,
                   newstar->tms_at_start_caseA);
#endif
        }

        double taur = stardata->model.time / newstar->tms_at_start_caseA;
        taur = Min(1.0,taur);

        fr= radius_stripped(newstar->pms_mass,
                            taur,
                            newstar->mass,
                            *r,
                            star,
                            stardata);

        *r = r_no_massloss * fr;

        /*
         * save the radius for the HG interpolation function
         */
        newstar->main_sequence_radius = *r;
    }
#endif
  

    Dprint("MS radius %g\n",newstar->radius);

    return newstar->stellar_type;
}

static double main_sequence_radius(MSR_DEC_ARGS)
{
    /* main-sequence radius from BSE */
    double r;
    const double alpha = ralphf(mass,
                                main_sequence_parameters);
    
    const double tp3 = Pow3(tau);
    const double dtau3 = tp1*tau1-tp2*tau2;
    double logr = alpha*tau;
    logr += - rhookf(mass,zp1,main_sequence_parameters) * dtau3;
    logr += (log10(rtms/rzams)-alpha)*tp3;
    Dprint("logr(tau = %g) = %g\n",
           tau,
           logr);
    
    if(tau > TAUMIN)
    {
        const double xtmp  = pow(tau,10.0);
        const double beta  = rbetaf(mass,main_sequence_parameters); 
        const double gamma = rgammf(mass,main_sequence_parameters);
        Dprint("xtmp = %g, beta = %g, gamma = %g\n",xtmp,beta,gamma);
        logr +=  xtmp*(beta + gamma*Pow3(xtmp)) - (beta + gamma)*tp3 ;
        Dprint("tau = %g > taumin = %g -> logr = %g\n",tau,TAUMIN,logr);
    }
    
    r = rzams * pow(10.0,logr);
    Dprint("r = (rzams = %g) * pow(10,%g) = %g\n",
           rzams,
           logr,
           r);

#ifdef PRE_MAIN_SEQUENCE
    if(stardata->preferences->pre_main_sequence == TRUE)
    {
        double f= preMS_radius_factor(mass,1e6*age); 
        r *= f;
        newstar->PMS = Boolean_(f>1.00001);
	Dprint("preMS f=%g r=%g roche_radius=%g roche_radius_at_periastron=%g PMS? %d\n",
	       f,
               r,
               newstar->roche_radius,
               newstar->roche_radius_at_periastron,
               newstar->PMS);
    }
#endif

    if(mass<zp1-0.3)
    {
        /*
         * This following is given by Chris for low mass MS stars which will be 
         * substantially degenerate. We need the Hydrogen abundance, X, which we 
         * calculate from Z assuming that the helium abundance, Y, is calculated
         * according to Y = 0.24 + 2*Z. 
         *
         * RGI: If we're following nucleosynthesis, we can use X from that.
         */
#ifdef NUCSYN
        Abundance X= newstar->Xenv[XH1];
#else
        Abundance X= metallicity_parameters[11];
#endif
        r = Max(r,0.0258*pow(1.0+X,5.0/3.0)/cbrt(mass));
    }

    return(r);
}
#endif//BSE
