#include "../binary_c.h"
#ifdef BSE

#include "stellar_structure_debug.h"

void stellar_structure_HeStar(struct star_t * newstar,
                              double *mtc,
                              double *rg,
                              double *remnant_radius,
                              struct stardata_t * stardata,
                              const Caller_id caller_id)
{
    /*
     * Naked Helium Star
     */
    Dprint("Naked He Star (stellar type %d, M = %g, Menv = %g, age = %g, epoch = %g, remna nt_radius = %g)\n",
           newstar->stellar_type,
           newstar->mass,
           newstar->mass - newstar->core_mass,
           newstar->age,
           newstar->epoch,
           * remnant_radius);

    double tau;
    const double * metallicity_parameters=stardata->common.metallicity_parameters;
    Stellar_type stellar_type_in = newstar->stellar_type;
    newstar->core_radius = 0.0;
    
#ifdef NUCSYN
    if(!RLOF_CALL)
    {
        if(newstar->he_t<TINY)  newstar->he_t = newstar->age;
      
        // save in here the fraction of the HeMS we have completed
        newstar->he_f=Max(0.0,
                       Min((newstar->age - newstar->he_t)/(newstar->tm-newstar->he_t),
                           1.0));
        
        if(newstar->stellar_type>7) newstar->he_f=1.0;

        Dprint("he MS fraction = %g\n",newstar->he_f);
    }
#endif

    Dprint("Compare age=%20.12g to tm=%20.12g (menv = %g) really less? %s diff = %g\n",
           newstar->age,
           newstar->tm,
           newstar->mass - newstar->core_mass,
           Yesno(Really_less_than(newstar->age,newstar->tm)),
           Signed_diff(newstar->age , newstar->tm)
        );

    /* core mass above which He burning continues */
    double mc_Heburn = mc_heburn(newstar->mass);

    if(Signed_diff(newstar->age,
                   newstar->tm) < -1e-10)
    {
        /*
         * Main Sequence
         */
        Dprint("MS star, massless core\n");

        newstar->stellar_type = HeMS;

        call_calc_lt2(newstar->stellar_type,newstar->mass,newstar->mass);

        tau = newstar->age / newstar->tm;
        double am = Max(0.0, 0.850 - 0.080*newstar->phase_start_mass);
        newstar->luminosity = newstar->luminosities[L_ZAMS]*(1.0+0.450*tau+am*Pow2(tau));
        am = Max(0.0,0.40-0.220*log10(newstar->mass));
        newstar->radius = *remnant_radius * (1.0 + am*(tau - Pow6(tau)));
        *rg = *remnant_radius;

        Dprint("Helium star radius = %g * %g = %g\n",
               *remnant_radius,
               1.0 + am*(tau-Pow6(tau)),
               newstar->radius);

#if defined NUCSYN && defined NUCSYN_WR
        if(!RLOF_CALL)
        {
            /*
             * Guess the terminal core mass for this star, this 
             * uses the shell-burning formulae given below. Of course
             * it is wrong but trial and error suggests a 10% factor 
             * compensates nicely and by the start of shell burning
             * the match will be pretty good.
             */
            newstar->he_mcmax = 1.1* Min(Min(newstar->mass,mc_Heburn),
                                      Max(stardata->preferences->chandrasekhar_mass,(0.7730*(newstar->phase_start_mass)-0.350)));
        }
#endif

        /*
         * Star has no core mass and hence no memory of its past
         * which is why we subject mass and mt to mass loss for
         * this phase.
         */  
        newstar->core_mass = 0.0;
            
        /* Rob adds : this is not quite true. 
         * Lynnette's models of (WR i.e. massive) helium
         * stars shows that the MS time is NOT just a function of
         * the initial stellar mass. We have to take into account the 
         * previous history of the star! 
         */

        /* evaporation of star if too much mass is lost */
        if(newstar->mass < metallicity_parameters[10] &&
           newstar->mass < stardata->preferences->max_HeWD_mass)
        {
            newstar->stellar_type = HeWD;
        }
    }
    else
    {
        /*
         * Helium Shell Burning
         */
        Dprint("He Shell Burning m = %g, menv = %g\n",
               newstar->mass,
               newstar->mass - newstar->core_mass
            );

        newstar->stellar_type = HeHG;
        
        Dprint("lgbtf : branch %d  : ti-nf-t = %g A=GB8=%g\n",
               newstar->age < newstar->timescales[T_GIANT_TX] ? 0 : 1,
               (newstar->age < newstar->timescales[T_GIANT_TX] ? 
                newstar->timescales[T_GIANT_TINF_1] : 
                newstar->timescales[T_GIANT_TINF_2]) - newstar->age,
               newstar->GB[8]);
        
        newstar->luminosity = lgbtf(newstar->age,
                                    newstar->GB[8],
                                    newstar->GB,
                                    newstar->timescales[T_GIANT_TINF_1],
                                    newstar->timescales[T_GIANT_TINF_2],
                                    newstar->timescales[T_GIANT_TX]); 

        Dprint("Luminosity = %g from age = %g\n",
               newstar->luminosity,
               newstar->age);
        newstar->radius = rhehgf(newstar->mass,
                                 newstar->luminosity,
                                 *remnant_radius,
                                 newstar->luminosities[L_END_MS]);
        *rg = rhegbf(newstar->luminosity);

        /* 
         * Perhaps we're now a HeGB star?
         */
        if(More_or_equal(newstar->radius,*rg)
            ||
           stellar_type_in == HeGB
            )
        {
            newstar->stellar_type=HeGB;
            newstar->radius = *rg;
        }
        Dprint("Radius %g (L=%g rg=%g stellar_type=%d mass=%g Teff=%g)\n",
               newstar->radius,
               newstar->luminosity,
               *rg,
               newstar->stellar_type,
               newstar->mass,
               Teff_from_star_struct(newstar)
            );

        /* CO core mass */
        newstar->core_mass = mcgbf(newstar->luminosity,
                                   newstar->GB,
                                   newstar->luminosities[L_LMX]);

        /* cannot exceed the total mass */
        newstar->core_mass = Min(newstar->mass, newstar->core_mass);
        newstar->CO_core_mass = newstar->core_mass;
  
        *mtc = Min(newstar->mass, mc_Heburn);

        /*
         * what is the maximum possible CO core mass?
         * this is when carbon ignites, or burning stops
         */
        double mcmax = Max(stardata->preferences->chandrasekhar_mass,
                           0.7730 * newstar->phase_start_mass - 0.350);
      
        mcmax = Min(*mtc-TINY,mcmax);

#if defined NUCSYN && defined NUCSYN_WR
        if(!RLOF_CALL)
            newstar->he_mcmax=mcmax; // save for use in nucsyn_WR
#endif

        Dprint("Helium star mc=%g m=%g menv=%g mtc=%g mcmax=%g\n",
               newstar->core_mass,
               newstar->mass,
               newstar->mass - newstar->core_mass,
               *mtc,
               mcmax);


        /*
         * Check if the burning shell has reached the surface
         * or the maximum core mass before carbon ignition
         */
        Dprint("CF Mc = %g to mcmax = %g (m = %g)\n",
               newstar->core_mass,
               mcmax,
               newstar->mass);
        
        if(More_or_equal(newstar->core_mass,mcmax))
        {
            Dprint("core mass > mcmax, cf mc %g vs mcmax = %g and Mch = %g\n",
                   newstar->core_mass,
                   mcmax,
                   stardata->preferences->chandrasekhar_mass
                );

            newstar->core_mass = Min(newstar->mass, mcmax); 
            newstar->CO_core_mass = newstar->core_mass;

            if(newstar->core_mass < stardata->preferences->chandrasekhar_mass)
            {
                /*
                 * Star becomes a new white dwarf
                 */
                newstar->age = 0.0;

                
                Dprint("Core mass %g < MCh, but is it > M_(Cignition)? =%g\n",
                       newstar->core_mass,stardata->preferences->minimum_mass_for_carbon_ignition);

                if(newstar->phase_start_mass < stardata->preferences->minimum_mass_for_carbon_ignition)
                {
                    /*     
                     * Form a new COWD ?
                     */
                    Dprint("Shell burning stops : Zero-age CO WD mt=%g\n",newstar->mass);

                    /* 
                     * If M<0.7 the detailed models (Pols "in prep") show 
                     * that shell He-burning stops before the whole envelope 
                     * is converted into C and O.
                     *
                     * This is modelled by letting a He star become a COWD
                     * when its core mass reaches the value 
                     * Min(1.45M-0.31,M)
                     * (Hurley+ 2002 Eq. 89)
                     *
                     * This calculation has been abstracted into the 
                     * mc_heburn function (see above for call).
                     *
                     * It has been reported that this formula
                     * fails at low Z, but the helium star models
                     * are anyway only valid for Z=0.02.
                     */
                    /*
                    double mlim = Min(mc_Heburn,newstar->mass);
                    printf("Mc = %g : mcx = %g : stops He shell burning at %g\n",
                           newstar->core_mass,
                           newstar->mcx_EAGB,
                           mlim);
                           if(More_or_equal(newstar->core_mass, mlim)) 
                    {
                     
                    }
                    */
                    newstar->stellar_type = COWD;
                    if(!RLOF_CALL)
                    {
                        /*
                         * Reset the TPAGB parameters for this star
                         */
                        newstar->num_thermal_pulses=-1.0; /* resets */
                        newstar->num_thermal_pulses_since_mcmin=0.0;
                    }
                    Dprint("Stellar type now %d\n",newstar->stellar_type);
                }
                else
                {
                    /*     
                     * Zero-age Oxygen/Neon White Dwarf
                     */
                    Dprint("Zero-age O Ne WD\n");

                    newstar->stellar_type = ONeWD;
                    if(!RLOF_CALL)
                    {
                        /*
                         * Reset the TPAGB parameters for this star
                         */ 
                        newstar->num_thermal_pulses_since_mcmin=0.0;
                        newstar->num_thermal_pulses=-1.0; /* resets */
                    }
                }

                /*
                 * If there is > 1e-3Msun of helium on top of the
                 * carbon-oxygen core, call the star a "hybrid" COWD
                 */
                if(newstar->core_mass < newstar->mass - MIN_MASS_He_FOR_HYBRID_COWD &&
                   newstar->stellar_type == COWD)
                {
                    newstar->hybrid_HeCOWD = TRUE;
#ifdef LOG_HYBRID_WHITE_DWARFS
                    Printf("HYBRIDHeCO %g %g %g\n",
                           stardata->model.probability,
                           newstar->mass,
                           newstar->core_mass);
#endif
                }


                newstar->phase_start_mass = newstar->mass;
                newstar->core_mass = newstar->CO_core_mass = newstar->mass;
                Dprint("Stellar type now %d\n",newstar->stellar_type);
            }
            else
            {
                /*
                 * Core mass exceeds MCh : 
                 * some kind of collapse and/or explosion follows
                 */ 
                Dprint("Check C ig m=%g at min mass for C ig=%g, conditions %d %d (lower =%g upper =%g)\n",
                       newstar->phase_start_mass,
                       stardata->preferences->minimum_mass_for_carbon_ignition,
                       ((stardata->preferences->mass_for_Hestar_Ia_upper>TINY)&&
                        (stardata->preferences->mass_for_Hestar_Ia_lower>TINY))
                       ,
                       (newstar->phase_start_mass>=stardata->preferences->mass_for_Hestar_Ia_lower)&&
                       (newstar->phase_start_mass<=stardata->preferences->mass_for_Hestar_Ia_upper),
                       stardata->preferences->mass_for_Hestar_Ia_lower,
                       stardata->preferences->mass_for_Hestar_Ia_upper 
                    );

                if(newstar->phase_start_mass < stardata->preferences->minimum_mass_for_carbon_ignition
                   /* allow ignition in designated mass range */
                   ||
                   ( stardata->preferences->mass_for_Hestar_Ia_upper>TINY &&
                     stardata->preferences->mass_for_Hestar_Ia_lower>TINY &&
                     newstar->phase_start_mass>stardata->preferences->mass_for_Hestar_Ia_lower &&
                     newstar->phase_start_mass<stardata->preferences->mass_for_Hestar_Ia_upper ))
                {
                    /*
                     * Star is not massive enough to ignite C burning.
                     * so no remnant is left after the SN
                     */
                    Dprint("No SN remn-ant\n");

                    newstar->SN_type = SN_HeStarIa;
                    struct star_t * news = 
                        new_supernova(stardata, 
                                      newstar,
                                      Other_star_struct(newstar),
                                      newstar);
                    if(news)
                    {
                        stellar_structure_make_massless_remnant(news);
                    }
                }
                else
                {
                    Dprint("STELLAR_STRUCTURE SN 2 stellar type %d\n",newstar->stellar_type);
                    struct star_t * news;
                    
                    if(newstar->core_mass < stardata->preferences->minimum_mass_for_neon_ignition)
                    {
                        /*
                         * Mass is too low to ignite neon, hence 
                         * the supernova is an electron capture, 
                         * not a canonical core collapse which 
                         * ignites neon, ..., silicon and forms
                         * an iron core. 
                         */
                        newstar->SN_type = SN_ECAP;
                        news = new_supernova(stardata,
                                             newstar,
                                             Other_star_struct(newstar),
                                             newstar); 
                        if(news)
                        {
                            news->mass = ns_bh_mass(newstar->mass,
                                                    newstar->core_mass,
                                                    stardata,
                                                    newstar);
                            news->core_mass = newstar->mass;
                        }
                    }
                    else
                    {
                        newstar->SN_type = SN_IBC;
                        
                        /* 
                         * Check for collapsar!
                         */
                        
                        /*
                         * First calculate angular momentum J
                         */
                        double J=
                            // envelope
                            newstar->k2*M_SUN*(newstar->mass-newstar->core_mass)*
                            Pow2(newstar->radius*R_SUN)*(
                                newstar->omega
                                /YEAR_LENGTH_IN_SECONDS)
                            +
                            // core
                            K3*(newstar->core_mass*M_SUN)*Pow2(newstar->core_radius*R_SUN)*(
                                newstar->omega
                                /YEAR_LENGTH_IN_SECONDS);
                 
                        /*
                         * Calculate specific angular momentum for the whole star
                         */
                        double j=J/(M_SUN*newstar->mass); // (cgs)
                      
                        /*
                         * Calculate BH mass
                         */
                        double mbh=ns_bh_mass(stardata->star[0].mass,
                                              stardata->star[0].core_mass,
                                              stardata,
                                              &(stardata->star[0]))*M_SUN;
                        
                        /*
                         * Calculate *innermost* last stable orbit radius (cm)
                         * 
                         * Schwarzschild
                         * r_LSO=6.0*GRAVITATIONAL_CONSTANT*mbh/(SPEED_OF_LIGHT*SPEED_OF_LIGHT);
                         *
                         * Kerr
                         */
                        double r_LSO=1.23*GRAVITATIONAL_CONSTANT*mbh/(SPEED_OF_LIGHT*SPEED_OF_LIGHT);
                 
  
                        // specific angular moment limit (cgs)
                        // according to the paper
                        //jlim=GRAVITATIONAL_CONSTANT*mbh*sqrt(r_LSO);
                        // rob's calculation which agrees with Heger and Woosley
                        double jlim=sqrt(GRAVITATIONAL_CONSTANT*mbh*r_LSO);

                        Dprint("Collapsar : J=%g j=%g r_LSO=%g mbh=%g jlim=%g collapsar? %d\n",
                               J,j,r_LSO,mbh/M_SUN,jlim,(j>jlim));
                        /*
                         * We have a collapsar
                         */
                        if(j>jlim)
                        {
                            newstar->SN_type = SN_GRB_COLLAPSAR;
                        }
                        news = new_supernova(stardata,
                                             newstar,
                                             Other_star_struct(newstar),
                                             newstar);
                        if(news)
                        {
                            news->mass = ns_bh_mass(newstar->mass,
                                                    newstar->core_mass, 
                                                    stardata,
                                                    newstar);
                            news->core_mass = news->mass;
                        }
                    }

                    if(news)
                    {
                        if(Less_or_equal(news->mass,
                                         stardata->preferences->max_neutron_star_mass))
                        {
                            /*
                             * Zero-age Neutron star
                             */
                            Dprint("Zero-age NS\n");
                            news->stellar_type = NEUTRON_STAR;
                        }
                        else
                        {
                            /*
                             * Zero-age Black hole
                             */
                            Dprint("Zero-age Black Hole\n");
                            news->stellar_type = BLACK_HOLE;
                        }
                    }
                }
            }
        }
    }
    Dprint("Stellar type now %d\n",newstar->stellar_type);
    Dprint("Naked He Star out (stellar type %d, M = %g, Menv = %g, age = %g, epoch = %g)\n",
           newstar->stellar_type,
           newstar->mass,
           newstar->mass - newstar->core_mass,
           newstar->age,
           newstar->epoch);
    
}
#endif//BSE
