#include "../binary_c.h"
#ifdef BSE

#include "stellar_structure_debug.h"

/*
 * Evolve HG stars according to the BSE algorithm
 */

Stellar_type stellar_structure_HG(struct star_t * RESTRICT const newstar,
                                  const double rg,
                                  struct stardata_t * RESTRICT const stardata,
                                  const Caller_id caller_id)
{
    const double * metallicity_parameters=stardata->common.metallicity_parameters; 
    const double * giant_branch_parameters=stardata->common.giant_branch_parameters; 
    double mcx,rho,rx,tau,ry,texp,rmin,tau2;
    double thg = newstar->timescales[T_BGB] - newstar->tm;
  
    Dprint("Hertzsprung Gap Star\n");


#ifdef MAIN_SEQUENCE_STRIP
    /*
     * Use the final MS radius instead of BSE's.
     */
    rtms = newstar->main_sequence_radius;
#endif


    mcx = newstar->core_mass;
    rho = mctmsf(newstar->phase_start_mass); // Hurley+2000 Eq.30
    tau = (newstar->age - newstar->tm)/thg;
    
    double mcbgbf = mcbgb(newstar,stardata);
    newstar->core_mass = mcbgbf * ((1.0-tau)*rho+tau);
    newstar->core_mass = Max(newstar->core_mass,mcx);
    
#ifdef NUCSYN
    if(!RLOF_CALL)
    {
#ifdef NUCSYN_FIRST_DREDGE_UP_PHASE_IN
        newstar->mc_gb_was=0.0;
#endif//NUCSYN_FIRST_DREDGE_UP_PHASE_IN
    }
#endif//NUCSYN
    /*
     * Test whether core mass has reached total mass.
     */
    Dprint("coremass = %12.12g >=? mt= %12.12g; m0=%g zpar12=%g\n",
           newstar->core_mass,
           newstar->mass,
           newstar->phase_start_mass,
           metallicity_parameters[ZPAR_MASS_HE_FLASH]);

    if(unlikely(More_or_equal(newstar->core_mass,newstar->mass)))
    {
	newstar->age = 0.0;
 	if(newstar->phase_start_mass > metallicity_parameters[ZPAR_MASS_HE_FLASH])
	{
	    /*
	     * Zero-age helium star
	     */
	    Dprint("Zero-age helium star\n");
	    
	    newstar->core_mass=0.0;
	    newstar->phase_start_mass=newstar->mass;
	    newstar->stellar_type=HeMS;
            call_calc_lt2(newstar->stellar_type,newstar->phase_start_mass,newstar->mass);

	    Dprint("\ntm is now %12.12g\n",newstar->tm);
	}
	else
	{
	    /*
	     * Zero-age helium white dwarf.
	     */
	    Dprint("Zero-age helium white dwarf\n");
	    newstar->core_mass = newstar->phase_start_mass = newstar->mass;
	    newstar->stellar_type = HeWD;
	}
    }
    else
    {
        newstar->luminosity = newstar->luminosities[L_END_MS]*
            pow(newstar->luminosities[L_BGB]/newstar->luminosities[L_END_MS], tau);
        
        if(Less_or_equal(newstar->phase_start_mass,
                         metallicity_parameters[ZPAR_MASS_FGB]))
	{
	    rx= rg;
        }
        else
        {
            /*                        
             * He-ignition and end of HG occur at Rmin
             */
            rmin = rminf(newstar->phase_start_mass,giant_branch_parameters);
            ry = ragbf(newstar->mass,
                       newstar->luminosities[L_HE_IGNITION],
                       metallicity_parameters[ZPAR_MASS_HE_FLASH],
                       giant_branch_parameters
#ifdef AXEL_RGBF_FIX
                       ,stardata->common.metallicity
#endif
                );
            rx = Min(rmin,ry);
            if(Less_or_equal(newstar->phase_start_mass,MLP))
	    {
		texp = log(newstar->phase_start_mass/MLP)/
                    log(metallicity_parameters[ZPAR_MASS_FGB]/MLP);
		rx = rg;
		rx = rmin*pow(rx/rmin,texp);
	    }
            tau2 = tblf(newstar->phase_start_mass,
                        metallicity_parameters[ZPAR_MASS_HE_FLASH],
                        metallicity_parameters[ZPAR_MASS_FGB],
                        giant_branch_parameters,
                        stardata->common.metallicity);
            if(tau2<TINY) rx = ry;

            Dprint("He-ignition at end of HG : tau2=%g\n",tau2);
        } 
        newstar->radius = newstar->rtms * pow(rx/newstar->rtms,tau);

#ifdef NUCSYN
        if(!RLOF_CALL)
        {
            /* save mass at the start of the HG */
            if(newstar->start_HG_mass < 0) newstar->start_HG_mass=newstar->mass;
        }
#endif

        newstar->stellar_type = HERTZSPRUNG_GAP;

        Dprint("HG : l=%g r=%g (rtms=%g rx=%g tau=%g)\n",newstar->luminosity,newstar->radius,newstar->rtms,rx,tau);
    }

    newstar->CO_core_mass = 0.0;
    newstar->GB_core_mass = 0.0;
    
    return newstar->stellar_type;
}
#endif//BSE
