
#include "../binary_c.h"
#ifdef BSE

#include "stellar_structure_debug.h"

#define LOW -1
#define HIGH 1
#define p (newstar->GB[GB_p])
#define q (newstar->GB[GB_q])
#define B (newstar->GB[GB_B])
#define D (newstar->GB[GB_D])
#define A (newstar->GB[GB_A_HE])

Boolean stellar_structure_EAGB(struct star_t * oldstar,
                               struct star_t * newstar,
                               const double tbagb,
                               double * RESTRICT const mc_CO,
                               const double mcbagb,
                               struct stardata_t * stardata,
                               const Caller_id caller_id,
                               const Boolean force_to_explosion)
{
    Boolean convert_to_helium_star=FALSE;
    Dprint("stellar_structure_EAGB newstar->age=%12.12e LESS than T_TPAGB_FIRST_PULSE=%12.12e\n",
           newstar->age,newstar->timescales[T_TPAGB_FIRST_PULSE]);

    /* 
     * estimate core mass at the first thermal pulse
     */
#if defined NUCSYN && defined NUCSYN_FIRST_DREDGE_UP_PHASE_IN
    newstar->mc_gb_was=0.0;
#endif // NUCSYN && NUCSYN_FIRST_DREDGE_UP_PHASE_IN

    if(force_to_explosion == FALSE)
    {
        *mc_CO = mcgbtf(newstar->age,
                        newstar->GB[GB_A_HE],
                        newstar->GB,
                        newstar->timescales[T_EAGB_TINF_1],
                        newstar->timescales[T_EAGB_TINF_2],
                        newstar->timescales[T_EAGB_T]);
    Dprint("mc_CO = mcx = %g from age = %g tscls %g %g %g GB_A_HE =%g\n",
           *mc_CO,
           newstar->age,
           newstar->timescales[T_EAGB_TINF_1],
           newstar->timescales[T_EAGB_TINF_2],
           newstar->timescales[T_EAGB_T],
           newstar->GB[GB_A_HE]
        );
    }
    else
    {
        Dprint("keep mc_CO = mcx = %g because force_to_explosion is TRUE\n",
               *mc_CO);
    }

    /*
     * Set the initial CO core mass and its 
     * rate of growth
     */
    double CO_dMc_dt MAYBE_UNUSED = 0.0;

    if(Is_zero(newstar->CO_core_mass))
    {
        newstar->CO_core_mass = Min(newstar->core_mass,
                                    mcgbtf(newstar->age,
                                           newstar->GB[GB_A_HE],
                                           newstar->GB,
                                           newstar->timescales[T_EAGB_TINF_1],
                                           newstar->timescales[T_EAGB_TINF_2],
                                           newstar->timescales[T_EAGB_T]));
        CO_dMc_dt = 0.0;
    }
    else
    {
        const int use = newstar->CO_core_mass < newstar->GB[GB_Mx] ? LOW : HIGH;
        CO_dMc_dt =
            A * (use == LOW ? D : B) *
            pow(newstar->CO_core_mass, use == LOW ? p : q);
    }

    /*
     * The helium core mass is always the helium core mass at the base of the AGB:
     * this does not change until the CO core catches up (or there is second
     * dredge up) when the star ascends the TPAGB
     */
    newstar->core_mass = mcbagb;
    newstar->derivative[DERIVATIVE_STELLAR_GB_CORE_MASS] = 0.0;
    newstar->derivative[DERIVATIVE_STELLAR_HE_CORE_MASS] = 0.0;
    newstar->derivative[DERIVATIVE_STELLAR_CO_CORE_MASS] = 0.0; // set above
    
    Dprint("Set core from Max(mcbagb=%12.12e,mcx=%12.12e) -> %g\n",
           mcbagb,newstar->mcx_EAGB,newstar->core_mass);

    /* 
     * mcx is the CO core mass, which is always < the actual core mass
     * but is this check required any more?
     * (RGI : It cannot hurt!)
     */
    *mc_CO = Min(newstar->core_mass,*mc_CO);
    
    Dprint("STELLAR_STRUCTURE type=%d mc=mcbagb=%12.12e mcx=%12.12e\n",
           newstar->stellar_type,
           newstar->core_mass,
           *mc_CO);

    /*
     * Hence the luminosity
     */
    newstar->luminosity = lmcgbf(*mc_CO,newstar->GB);

    newstar->mcx_EAGB = *mc_CO;
    Dprint("Set newstar->mcx_EAGB = %g\n",newstar->mcx_EAGB);
    
    Dprint("mt=%12.12e core=%12.12e (if mt<core > TPAGB)\n",
           newstar->mass,newstar->core_mass);
/*
    printf("EAGB t=%g M=%g McHe=%g McCO=%g\n",
           stardata->model.time,
           newstar->mass,
           newstar->core_mass,
           newstar->mcx_EAGB
        );
*/    
    if(Less_or_equal(newstar->mass, 
                     newstar->core_mass)) 
    {
        /*
         * Evolved naked helium star or white dwarf
         * as the envelope is lost but the
         * star has not completed its interior burning. 

         * The star becomes a post-HeMS star if its core is 
         * sufficiently massive, otherwise a white dwarf.
         */
        Dprint("post HeMS star\n");

        newstar->stellar_type = HeGB;
        newstar->phase_start_mass = newstar->mass;

        /* 
         * The core mass cannot shrink if we're post-AGB as
         * the CO core has already been built up!
         */
        newstar->core_mass = oldstar->stellar_type>TPAGB ? Max(newstar->core_mass,*mc_CO) : *mc_CO;
        
        /*
         * Limit the core mass tot he total mass
         */
        newstar->core_mass = Min(newstar->mass,
                                 newstar->core_mass);

        newstar->CO_core_mass = Min(newstar->CO_core_mass,
                                    newstar->core_mass);
        
        call_calc_lt2(newstar->stellar_type,
                      newstar->phase_start_mass,
                      newstar->mass);

        Dprint("SetAJ tm=%g mc=%g\nSetAJ g=%g\n",newstar->tm,newstar->core_mass,newstar->GB[7]);


        {
            Stellar_type st = newstar->stellar_type;
            double mc = newstar->core_mass; 
            double m0 = newstar->phase_start_mass;
            double age = newstar->age;
            giant_age(&mc,
                      newstar->mass,
                      &st,
                      &m0,
                      &age,
                      stardata,
                      newstar);
        }
                  
        if(Less_or_equal(newstar->core_mass,newstar->GB[GB_Mx]))
        {
            newstar->age = newstar->timescales[T_GIANT_TINF_1] - (1.0/((newstar->GB[GB_p]-1.0)*newstar->GB[GB_A_HE]*newstar->GB[GB_D]))*
                pow(newstar->core_mass,1.0-newstar->GB[GB_p]);
        }
        else
        {
            newstar->age = newstar->timescales[T_GIANT_TINF_2] - (1.0/((newstar->GB[GB_q]-1.0)*newstar->GB[GB_A_HE]*newstar->GB[GB_B]))*
                pow(newstar->core_mass,1.0-newstar->GB[GB_q]);
        }
        newstar->age = Max(newstar->age, newstar->tm);
        
        Dprint("SetAJ=%g\n",newstar->age);

        convert_to_helium_star=TRUE;

        /*
        printf("EAGB t=%g hestar\n",
        stardata->model.time
        );
        */
    }
    else
    { 
        /* 
         * if convert_to_helium_star==TRUE (i.e. star is an
         * EAGB star), we can never get here
         */

        Dprint("Star is EAGB\n");
        Dprint("EAGB core mass %12.12e (CO core %12.12e)\n",
               newstar->core_mass,*mc_CO);
        newstar->max_EAGB_He_core_mass = Max(newstar->max_EAGB_He_core_mass,
                                             newstar->core_mass);

#ifdef NUCSYN
        if(newstar->stellar_type!=TPAGB)
        {
            /* because mc1tp is different from mcbagb
             * sometimes we get confused and try to de-evolve
             * back to an EAGB star from the TPAGB. Don't
             * let this happen!!!
             */
#endif
            newstar->stellar_type = EAGB;
#ifdef NUCSYN
        }
#endif

        /*
         * Set the radius
         */
        newstar->radius = ragbf(newstar->mass,
                                newstar->luminosity,
                                stardata->common.metallicity_parameters[ZPAR_MASS_HE_FLASH],
                                stardata->common.giant_branch_parameters
#ifdef AXEL_RGBF_FIX
                                ,stardata->common.metallicity
#endif // AXEL_RGBF_FIX
            );
        newstar->core_mass = Min(newstar->mass,
                                 newstar->core_mass);
        newstar->CO_core_mass = Min(newstar->CO_core_mass,
                                    newstar->core_mass);

    }
    Dprint("AJ set to %12.12e stellar_type is now %d with M=%g Menv=%g\n",
           newstar->age,
           newstar->stellar_type,
           newstar->mass,
           newstar->mass - newstar->core_mass
        );
    
    return(convert_to_helium_star);
}
#endif//BSE
