#include "../binary_c.h"
#ifdef BSE

#include "stellar_structure_debug.h"

Stellar_type stellar_structure_CHeB(struct star_t *newstar,
                                    double *rg,
                                    struct stardata_t * stardata,
                                    const Caller_id caller_id)
{
    /*
     * Core helium burning star.
     */
    const double * metallicity_parameters=stardata->common.metallicity_parameters; 
    const double * giant_branch_parameters=stardata->common.giant_branch_parameters; 
    double mcx,tau,lx,ly,rx,ry,rmin,texp,taul,tauh,tau2,tloop;
    
    if(0 &&
       newstar->stellar_type == FIRST_GIANT_BRANCH &&
       Less_or_equal(newstar->phase_start_mass,
                     metallicity_parameters[ZPAR_MASS_HE_FLASH]))
    {
        newstar->phase_start_mass = newstar->mass;
        call_calc_lt2(newstar->stellar_type,
                      newstar->phase_start_mass,
                      newstar->mass);
        newstar->age = newstar->timescales[T_HE_IGNITION];
        Dprint("GB -> CHeB transition : new age = %g from T_HE_IGNITION = %g, phase start mass %g\n",
               newstar->age,
               newstar->timescales[T_HE_IGNITION],
               newstar->phase_start_mass
            );
    }

#if defined NUCSYN && defined NUCSYN_FIRST_DREDGE_UP

    /*
     * Check if we have done 1st dredge up
     */
    if(!RLOF_CALL &&
       newstar->first_dredge_up == FALSE &&
       newstar->stellar_type == GIANT_BRANCH)
    {
        Dprint("1DUP (CHeB) check mc=%12.12e vs mc1dup(evert)=%12.12e (newstar->stellar_type=%d)\n",
               newstar->core_mass,
               mc_1DUP(stardata,
                       newstar->phase_start_mass,
                       stardata->common.metallicity),
               newstar->stellar_type);

        /*
         * Enforce first dredge up if it has not yet happened
         */
        if(newstar->first_dredge_up==FALSE
           && newstar->stellar_type==FIRST_GIANT_BRANCH 
           && newstar->effective_zams_mass<NUCSYN_WR_MASS_BREAK)
        {

            Dprint("NUCSYNs__ DUP1\n");
            newstar->first_dredge_up=TRUE;
            nucsyn_set_1st_dup_abunds(newstar->Xenv,
                                      newstar->mass,
                                      stardata->common.metallicity,
                                      newstar->stellar_type,
                                      newstar,
                                      stardata,
                                      TRUE);
        }
#if (DEBUG==1)
        else
        {
            Dprint("(CHeB) Star already had first DUP?\n");
        }
#endif
    }
#ifdef NUCSYN_FIRST_DREDGE_UP_PHASE_IN
    newstar->mc_gb_was=0.0; // reset just in case
#endif
#endif /* NUCSYN_FIRST_DREDGE_UP and NUCSYN */

    double mcbagb = mcagbf(newstar->phase_start_mass,
                           giant_branch_parameters);
    
    if(Less_or_equal(newstar->phase_start_mass,
                     metallicity_parameters[ZPAR_MASS_HE_FLASH]))
    {
        mcx = mcgbf(newstar->luminosities[L_HE_IGNITION],
                    newstar->GB,
                    newstar->luminosities[L_LMX]);
    }
    else
    {
        mcx = mcheif(newstar->phase_start_mass,
                     metallicity_parameters[ZPAR_MASS_HE_FLASH],
                     metallicity_parameters[10],
                     giant_branch_parameters);
    }


    tau = (newstar->age - newstar->timescales[T_HE_IGNITION])/newstar->timescales[T_HE_BURNING];
    Clamp(tau,0.0,1.0);

    /*
     * Core growth 
     */
    newstar->core_mass = mcx + (mcbagb - mcx)*tau;

    Dprint("CHEB mc=%g\n",newstar->core_mass);

    newstar->GB_core_mass = 0.0;
    newstar->CO_core_mass = 0.0;

    Dprint("Core mass set %12.12e from mcx=%12.12e + bit=%12.12e (tau=%12.12e) : mass = %20.12e phase_start_mass = %20.12e\n",
           newstar->core_mass,
           mcx,
           (mcbagb-mcx)*(tau),
           tau,
           newstar->mass,
           newstar->phase_start_mass
        );

    if(Less_or_equal(newstar->phase_start_mass,
                     metallicity_parameters[ZPAR_MASS_HE_FLASH]))
    {
        lx = newstar->luminosities[L_HE_BURNING];
        ly = newstar->luminosities[L_BAGB];
        rx = rzahbf(newstar->mass,
                    newstar->core_mass,
                    metallicity_parameters[ZPAR_MASS_HE_FLASH],
                    giant_branch_parameters);
        
        *rg = rgbf(newstar->mass,
                   lx,
                   giant_branch_parameters);
        rmin=(*rg)*pow(metallicity_parameters[13],newstar->phase_start_mass/metallicity_parameters[ZPAR_MASS_HE_FLASH]);
        
        
        texp = Min(Max(0.40,rmin/rx),2.5);
        ry = ragbf(newstar->mass,
                   ly,
                   metallicity_parameters[ZPAR_MASS_HE_FLASH],
                   giant_branch_parameters
#ifdef AXEL_RGBF_FIX
                   ,stardata->common.metallicity
#endif
            );

        if(rmin<rx)
        {
            taul = cbrt(log(rx/rmin));
        }
        else
        {
            rmin = rx;
            taul = 0.0;
        }
        
        tauh = cbrt(log(ry/rmin));
        tau2 = taul*(tau - 1.00) + tauh*tau;
        newstar->radius = rmin*exp(Pow3(fabs(tau2)));
        Dprint("r=%12.12e\n",newstar->radius);

        *rg += tau*(ry - *rg);
        newstar->luminosity = lx*pow((ly/lx),pow(tau,texp));
    }
    else if (newstar->phase_start_mass > metallicity_parameters[ZPAR_MASS_FGB])
    {
        /*
         * For HM stars He-ignition takes place at Rmin in the HG, and CHeB
         * consists of a blue phase (before tloop) and a RG phase (after tloop).
         */
        Dprint("Core Helium Burning High-mass Blue Phase and RG Phase\n");

        tau2 = tblf(newstar->phase_start_mass,
                    metallicity_parameters[ZPAR_MASS_HE_FLASH],
                    metallicity_parameters[ZPAR_MASS_FGB],
                    giant_branch_parameters,
                    stardata->common.metallicity);

        tloop = newstar->timescales[T_HE_IGNITION] + 
            tau2*newstar->timescales[T_HE_BURNING];

        Dprint("TAU2 %g HE_BURNING %g\n",
               tau2,
               newstar->timescales[T_HE_BURNING]);

        rmin = rminf(newstar->phase_start_mass,
                     giant_branch_parameters);
        *rg = rgbf(newstar->mass,
                   newstar->luminosities[L_HE_IGNITION],
                   giant_branch_parameters);
        rx = ragbf(newstar->mass,
                   newstar->luminosities[L_HE_IGNITION],
                   metallicity_parameters[ZPAR_MASS_HE_FLASH],
                   giant_branch_parameters
#ifdef AXEL_RGBF_FIX
                   ,stardata->common.metallicity
#endif
            );
   
        rmin = Min(rmin, rx);
        /* WARNING: for Z=0.004 rmin << rx, so rmin is used, when the detailed
         * models actually suggest rx should be used (cf M=15 Z=0.004, to 
         * detailed models)
         */
        // This is correct!
        //rmin = rx;
        
        if(Less_or_equal(newstar->phase_start_mass,MLP))
        {
            texp = log(newstar->phase_start_mass/MLP)/log(metallicity_parameters[ZPAR_MASS_FGB]/MLP);
            rx = rmin*pow(*rg/rmin,texp);
        }
        else
        {
            rx = rmin;
        }
              
        texp = Min(Max(0.40,rmin/rx),2.50);
        newstar->luminosity = newstar->luminosities[L_HE_IGNITION]*
            pow(newstar->luminosities[L_BAGB]/newstar->luminosities[L_HE_IGNITION],
                pow(tau,texp)) ;

        Dprint("age = %g (helium ignites at %g) cf tloop = %g : %s CHeB phase\n",
               newstar->age,
               newstar->timescales[T_HE_IGNITION],
               tloop,
               (newstar->age < tloop ? "Blue" : "Red"));

        if(newstar->age < tloop)
        {
            ly = newstar->luminosities[L_HE_IGNITION]*pow(newstar->luminosities[L_BAGB]/newstar->luminosities[L_HE_IGNITION],pow(tau2,texp));
            ry = ragbf(newstar->mass,
                       ly,
                       metallicity_parameters[ZPAR_MASS_HE_FLASH],
                       giant_branch_parameters
#ifdef AXEL_RGBF_FIX
                       ,stardata->common.metallicity
#endif
                );
                            
            taul = 0.0;
            if(Is_not_zero(rmin-rx)) taul = cbrt(log(rx/rmin));
            tauh = ry>rmin ? cbrt(log(ry/rmin)) : 0.0;
            tau = (newstar->age - newstar->timescales[T_HE_IGNITION])/(tau2*newstar->timescales[T_HE_BURNING]);
            tau2 = taul*(tau - 1.00) + tauh*tau;

            /* ROB: this newstar->radius is not large enough to match Z=0.004 models? */
            newstar->radius = rmin*exp(Pow3(fabs(tau2)));
            Dprint("setting r (2) to %12.12e\n",newstar->radius);

            *rg += tau*(ry-*rg);
        }
        else
        {
            newstar->radius = ragbf(newstar->mass,
                              newstar->luminosity,
                              metallicity_parameters[ZPAR_MASS_HE_FLASH],
                              giant_branch_parameters
#ifdef AXEL_RGBF_FIX
                              ,stardata->common.metallicity
#endif
                );

            *rg = newstar->radius;
            Dprint("Set r=rg=%12.12e\n",newstar->radius);
        }
    }
    else
    {
        /*
         * For IM stars CHeB consists of a RG phase (before tloop) and a blue
         * loop (after tloop).
         */
        Dprint("Core Helium burning red giant phase then blue phase\n");


        tau2 = 1.0 - tblf(newstar->phase_start_mass,
                          metallicity_parameters[ZPAR_MASS_HE_FLASH],
                          metallicity_parameters[ZPAR_MASS_FGB],
                          giant_branch_parameters,
                          stardata->common.metallicity);
        tloop = newstar->timescales[T_HE_IGNITION] + tau2*newstar->timescales[T_HE_BURNING];

        if(newstar->age<tloop)
        {
            tau = (tloop - newstar->age)/(tau2*newstar->timescales[T_HE_BURNING]);
                    
            newstar->luminosity = newstar->luminosities[L_HE_BURNING]*
                pow(newstar->luminosities[L_HE_IGNITION]/newstar->luminosities[L_HE_BURNING],Pow3(tau));



                              
            newstar->radius = rgbf(newstar->mass,
                             newstar->luminosity,
                             giant_branch_parameters);
            Dprint("set r (3) from rgbf %12.12e\n",newstar->radius);

            *rg = newstar->radius;
        }
        else
        {
            lx = newstar->luminosities[L_HE_BURNING];
            ly = newstar->luminosities[L_BAGB];
            rx = rgbf(newstar->mass,lx,giant_branch_parameters);
            rmin = rminf(newstar->mass,giant_branch_parameters);
                              
            texp = Min(Max(0.40,(rmin/rx)),2.50);
            ry = ragbf(newstar->mass,ly,metallicity_parameters[ZPAR_MASS_HE_FLASH],giant_branch_parameters
#ifdef AXEL_RGBF_FIX
                       ,stardata->common.metallicity
#endif
                );
                            
            if(rmin<rx)
            {
                taul = cbrt(log(rx/rmin));
            }
            else
            {
                rmin = rx;
                taul = 0.0;
            }
            tauh = cbrt(log(ry/rmin));
            tau = (newstar->age - tloop)/(newstar->timescales[T_HE_BURNING] - (tloop - newstar->timescales[T_HE_IGNITION]));
            tau2 = taul*(tau - 1.0) + tauh*tau;

            newstar->radius = rmin*exp(Pow3(fabs(tau2)));
            Dprint("set r (4) %12.12e\n",newstar->radius);

            *rg = rx + tau*(ry - rx);
            newstar->luminosity = lx*pow(ly/lx,(pow(tau,texp)));
        }
    }
 
    Dprint("Test whether core mass (%12.12e) exceeds total mass (%12.12e)\n",
           newstar->core_mass,newstar->mass);

    /* 
     * Test whether core mass exceeds total mass.
     */

    if(More_or_equal(newstar->core_mass,newstar->mass))
    {
        /*
         * Evolved MS naked helium star.
         */
        Dprint("Core exceeds total mass, Evolved MS naked He Star\n");
        newstar->stellar_type = HeMS;
    }
    else
    {
        Dprint("Setting newstar->stellar_type=4 here\n");
        newstar->stellar_type = CHeB;
    } 


    if(newstar->stellar_type==HeMS)
    {
        Dprint("Convert to HeMS : newstar->age = %g, T_HE_IGNITION = %g\n",
               newstar->age,
               newstar->timescales[T_HE_IGNITION]);
        double xx = (newstar->age - newstar->timescales[T_HE_IGNITION])/
            newstar->timescales[T_HE_BURNING];
        xx = Max(0.0,xx); // prevent minor floating point inaccuracies
        newstar->phase_start_mass = newstar->mass;
        call_calc_lt2(newstar->stellar_type,newstar->phase_start_mass,newstar->mass);
        newstar->age = xx * newstar->tm;
        Dprint("CHeB -> HeMS transition : new age = %g * %g = %g\n",
               xx,
               newstar->tm,
               newstar->age);
    }

    /* no longer require newstar->GB_core_mass */
    newstar->GB_core_mass = 0.0;
    
    Dprint("exit CHEB age = %g\n",newstar->age);

    return newstar->stellar_type;
}
#endif
