#include "../binary_c.h"
#ifdef BSE
#include "stellar_structure_debug.h"

Stellar_type stellar_structure_AGB(struct star_t * const oldstar,
                                   struct star_t * RESTRICT const newstar,
                                   double * RESTRICT const mc_CO,
                                   double * RESTRICT const rg,
                                   const double tbagb,
                                   struct stardata_t * RESTRICT const stardata,
                                   const Caller_id caller_id)
{
    double mcmax;
    const double * giant_branch_parameters =
        stardata->common.giant_branch_parameters; 
    Boolean convert_to_helium_star=FALSE;
    /*
     * Asymptotic Red Giant.
     *
     * On the AGB the He core mass remains constant until at Ltp it
     * is caught by the C core mass and they grow together.
     */

    Dprint("Asymptotic Red Giant\n");
    
    double mcbagb = mcagbf(newstar->phase_start_mass,giant_branch_parameters);
    Boolean ignite_carbon = 
        mcbagb > stardata->preferences->minimum_mass_for_carbon_ignition;

    if(newstar->starnum==0)
    Dprint("Mc1TP0 mcbagb = %g, ignite carbon? %d\n",
           mcbagb,
           ignite_carbon);
    
    
    Dprint("Set mcbagb %12.12e from newstar->phase_start_mass %12.12e\n",mcbagb,newstar->phase_start_mass);

    /* 
     * mc_CO here is the CO core mass
     * at the start of the AGB, i.e. at
     * the end of core helium burning
     */
    double mc_CO_start_AGB
        = Min(newstar->mass,
              mcgbtf(tbagb,
                     newstar->GB[GB_A_HE],
                     newstar->GB,
                     newstar->timescales[T_EAGB_TINF_1],
                     newstar->timescales[T_EAGB_TINF_2],
                     newstar->timescales[T_EAGB_T]));
    
    Dprint("MCX(%g,%d) = %g (tbagb=%g mcbagb=%g) Mc = %g\n",
           newstar->age,
           newstar->stellar_type,
           mc_CO_start_AGB,
           tbagb,
           mcbagb,
           newstar->core_mass
        );
    
    /*
     * NB change due to Rob: use stardata->preferencenewstar->max_tpagb_core_mass here instead
     * of stardata->preferencenewstar->chandrasekhar_mass. Of course these might be exactly the
     * same thing...
     */
    
    /*
     * Equivalent to Hurley et al. 2000's Eq. (75)
     */
    double mcSN = Max(0.7730 * mcbagb - 0.350,
                      stardata->preferences->max_tpagb_core_mass);


    /*
     * How much must the AGB core grow before
     * it is allowed to explode? In Msun.
     *
     * This is a fudge in BSE: see below for implementation.
     */
    const double min_AGB_core_growth = 0.05;

    /*
     * CO core masses in excess of this value (in Msun) are not
     * allowed to have thermal pulses. Such stars will,
     * in any case, soon explode.
     *
     * The Hurley et al. 2000 (based on Pols) models estimate
     * that core with mass > 2.25Msun are non-degenerate, so these
     * should not have a TPAGB.
     */
    const double maximum_CO_core_mass_for_thermal_pulses = 2.25;
    
    /*
     * The following is in BSE: it makes the core 
     * grow somewhat up the AGB before there is an explosion.
     *
     * This is not mentioned in the Hurley et al. (2000) paper. 
     *
     * Presumably this is just a fudge of some sort to 
     * make sure there is no immediate explosion after core helium
     * burning finishes, perhaps to simulate the time of subsequent 
     * burning (e.g. He-shell, C-core, C-shell, etc.) before there
     * is an explosion.
     */
    mcmax = Max(mcSN,
                (1.0 + min_AGB_core_growth) * mc_CO_start_AGB);

    Dprint("Compare mcx=%g to mcxmax=%g (m=%g)\n",
           mc_CO_start_AGB,mcmax,newstar->mass);
    
    Dprint("age %14.14g c.f. tscls 13 (first pulse) %14.14g\n",
           newstar->age,
           newstar->timescales[T_TPAGB_FIRST_PULSE]); 
    
    /*
     * Set the CO core mass at this point in the evolution (in mc_CO),
     * luminosity, radius and age.
     */
    if(newstar->age < newstar->timescales[T_TPAGB_FIRST_PULSE]
       &&
       newstar->stellar_type!=TPAGB)
    {
        /* EAGB star : updates mc_CO */
        convert_to_helium_star=
            stellar_structure_EAGB(oldstar,
                                   newstar,
                                   tbagb,
                                   mc_CO,
                                   mcbagb,
                                   stardata,
                                   caller_id,
                                   FALSE);
    }
    /*
     * RGI
     *
     * Only allow stars with CO core mass less than the 
     * maximum_CO_mass_for_thermal_pulses to have thermal pulses, or
     * those that are already thermally pulsing.
     *
     * This is because more massive stars should explode
     * soon after the AGB begins, and probably don't ever
     * have thermal pulses.
     */
    else if(newstar->stellar_type == TPAGB ||
            (Is_not_zero(newstar->CO_core_mass) &&
             newstar->CO_core_mass < maximum_CO_core_mass_for_thermal_pulses))
    {
        /* TPAGB star */
        stellar_structure_TPAGB(newstar,
                                mc_CO,
                                &mcmax,
                                mcbagb,
                                stardata,
                                caller_id);
    } 
    else
    {
        /*
         * The star wants to start thermal pulses,
         * but we should prevent it from happening. The core
         * should be set to mcmax so there is a supernova as
         * soon as possible.
         */
        Dprint("Force EAGB to explosion\n");
        newstar->stellar_type = EAGB;
        *mc_CO = mcmax;

        /* 
         * call stellar structure to set up the star, otherwise
         * the pre-SN star will not be set, and L=R=0 will cause errors
         */
        stellar_structure_EAGB(oldstar,
                               newstar,
                               tbagb,
                               mc_CO,
                               mcbagb,
                               stardata,
                               caller_id,
                               TRUE);
        *mc_CO = mcmax;
    }

    Dprint("MC_CO now %g start %g mcmax %g core_mass %g stellar_type %d (L=%g R=%g)\n",
           *mc_CO,
           mc_CO_start_AGB,
           mcmax,
           newstar->core_mass,
           newstar->stellar_type,
           newstar->luminosity,
           newstar->radius
        );
    
    /*
     * Check that the core does not collapse or explode
     */
    if(convert_to_helium_star==FALSE)
    {
       *rg = newstar->radius;

        /*
         * mc_CO represents the CO core mass and we now test whether it
         * exceeds either the total mass or the maximum allowed core mass.
         */
        Dprint("COMPARE st=%d mc_CO=%12.12e vs mcmax=%12.12e (Mch=%g) r=%12.12e\n",
               newstar->stellar_type,
               *mc_CO,
               mcmax,
               stardata->preferences->chandrasekhar_mass,
               newstar->radius);


        Dprint("MC_CO=%g mcmax=%g mdot=%g dt=%g\n",
               *mc_CO,
               mcmax,
               newstar->derivative[DERIVATIVE_STELLAR_MASS_WIND_LOSS],
               stardata->model.dt);

        Dprint("Mc1TP0 st=%d mc_CO=%12.12e (McCO start AGB = %g) vs mcmax=%12.12e (Mch=%g McSN=%g) r=%12.12e\n",
               newstar->stellar_type,
               *mc_CO,
               mc_CO_start_AGB,
               mcmax,
               mcSN,
               stardata->preferences->chandrasekhar_mass,
               newstar->radius);

        double envelope_mass = newstar->mass - newstar->core_mass;

        /*
         * Mass accretion rate: usually this is from the 
         * previous timestep, so may not be accurate even if
         * it's the best we can do at this point.
         */
        double accretion_rate = Mdot_net(newstar);

        /*
         * If accreting, do not become a WD. In this case,
         * the slow accretion will lead to flashes on the surface.
         * However, the surface will always be hydrogen or helium
         * rich, not a COWD or ONeWD.
         */
        if(Less_or_equal(envelope_mass, 1e-10) && 
           *mc_CO < mcmax &&
           accretion_rate < REALLY_TINY)
        {
            /* 
             * Envelope lost : nuclear burning stops, convert to WD
             */
            newstar->stellar_type =
                (ignite_carbon==FALSE || newstar->compact_core_type>COWD) ? COWD : ONeWD;

#ifdef WD_KICKS
            if(stardata->preferences->wd_kick_when==WD_KICK_END_AGB)
            {
                /* activate WD kick */
                newstar->kick_WD=TRUE;
            }
#endif // WD_KICKS

            newstar->num_thermal_pulses=-1.0; /* resets */
            newstar->num_thermal_pulses_since_mcmin=0.0;
            newstar->phase_start_mass = newstar->mass;
            newstar->age = 0.0;
        }
        else if(More_or_equal(*mc_CO,mcmax))
        {
            /* 
             * Core mass exceeds the maximum for carbon ignition
             * or the star has no envelope left:
             * either form a WD or collapse the core
             */

            
            /*
             * Always check the core cannot exceed the total mass
             */
            newstar->core_mass = Min(newstar->core_mass,newstar->mass);

            /*
             * And check that the CO core mass does not exceed the total core mass
             * or the CO core mass at carbon ignition
             */
            *mc_CO = Min3(*mc_CO, newstar->core_mass, mcmax);

            /* 
             * max_tpagb_core_mass is the maximum mass a WD can have:
             * this is usually the same as the Chandrasekhar mass 
             * (in stardata->preferences->chandrasekhar_mass) but can be
             * set to something different
             * (in stardata->preferences->max_tpagb_core_mass) to take into
             * account chemical changes, rotation, personal preference...
             */
            if(*mc_CO < stardata->preferences->max_tpagb_core_mass)
            {
                /*
                 * Form a new WD from the CO core.
                 */
                newstar->age = 0.0;
                newstar->mass = *mc_CO;
#ifdef WD_KICKS
                if(stardata->preferences->wd_kick_when==WD_KICK_END_AGB)
                {
                    /* activate WD kick */
                    newstar->kick_WD = TRUE;
                }
#endif
                if(ignite_carbon == FALSE)
                {
                    /*     
                     * Zero-age Carbon/Oxygen White Dwarf
                     */
                    newstar->stellar_type = COWD;
                    Dprint("Zero-age CO WD mt=%g\n",newstar->mass);
                }
                else
                {
                    /*     
                     * Zero-age Oxygen/Neon White Dwarf
                     */
                    newstar->stellar_type = ONeWD;
                    Dprint("Zero-age O Ne White Dwarf\n");
                }
                /*
                 * Reset the TPAGB parameters for this star
                 */
                newstar->num_thermal_pulses=-1.0; /* resets */
                newstar->num_thermal_pulses_since_mcmin=0.0;
                newstar->phase_start_mass = newstar->mass;
            }
            else
            {
                /*
                 * Core mass is greater than stardata->preferences->max_tpagb_core_mass
                 * so there is a supernova here.
                 */
                Dprint("EXPLODE! mc_CO=%g mc=%g\n",*mc_CO,newstar->core_mass);
                if(mcbagb < stardata->preferences->minimum_mass_for_carbon_ignition)
                {
                    /*
                     * Star is not massive enough to start C burning,
                     * the core is degenerate carbon and oxygen,
                     * so no remnant is left after the SN (type "IIa" SN)
                     */
                    Dprint("STELLAR_STRUCTURE SN IIa\n");
                    newstar->SN_type = SN_IIa;
                    struct star_t * news = 
                        new_supernova(stardata,
                                      newstar,
                                      Other_star_struct(newstar),
                                      newstar);
                    if(news)
                    {
                        news->stellar_type = MASSLESS_REMNANT;
                        news->age = 0.0;
                        news->mass = 0.0;
                        news->core_mass = 0.0;
                        news->luminosity = REMNANT_LUMINOSITY;
                        news->radius = REMNANT_RADIUS;
                    }
                }
                else
                {
                    /*
                     * Star ignites carbon, goes on to core collapse 
                     */


                    /*
                     * If the star *was* a TPAGB star, and still is,
                     * then we're fine. However, if the star was a CHeB
                     * star then pretend it's still CHeB for the 
                     * purpose of calculating the remnant mass;
                     */
                    int stype = newstar->stellar_type; /* save */
                    if(oldstar->stellar_type == CHeB &&
                       newstar->stellar_type == TPAGB)
                    {
                        newstar->stellar_type = CHeB;
                    }

                    /*
                     * Use the CO core mass to calculate the remnant
                     * mass.
                     */
                    double remnant_mass = 
                        ns_bh_mass(newstar->mass,
                                   *mc_CO,
                                   stardata,
                                   newstar);
                    newstar->stellar_type = stype; /* restore */
                    
                    Dprint("STELLAR_STRUCTURE core collapse SN, McHe = %g, McCO = %g -> M_remna nt = %g\n",
                           newstar->core_mass,
                           *mc_CO,
                           remnant_mass);
                    newstar->SN_type = SN_II;
                    struct star_t * news = 
                        new_supernova(stardata,
                                      newstar,
                                      Other_star_struct(newstar),
                                      newstar);
                    if(news)
                    {
                        news->mass = remnant_mass;
                        news->core_mass = newstar->mass;
                        news->age = 0.0;
                        if (Less_or_equal(news->mass,
                                          stardata->preferences->max_neutron_star_mass))
                        {
                            /*
                             * Zero-age Neutron star
                             */                            
                            Dprint("Zero-age NS\n");
                            news->stellar_type = NEUTRON_STAR;
                        }
                        else
                        {
                      
                            /*
                             * Zero-age Black hole
                             */
                            Dprint("Zero-age Black Hole\n");
                            news->stellar_type = BLACK_HOLE;
                        }
                    }
                }
            }
        }
    }

#ifdef NUCSYN
    Dprint("stellar_structure_AGB return stellar type %d\n",
           newstar->stellar_type);
#endif
    return newstar->stellar_type;
}
#endif//BSE
