/*
 * stellar_structure() : 
 *
 * Updates the star struct for its age with the new stellar structure,
 * according to binary_c's modified BSE algorithm.
 *
 * NB this function creates a "newstar" struct (in tmpstore), which is copied
 * from the "star" struct passed in. Changes are made to "newstar" and then,
 * if all is ok, this is copied over "star".
 */
#include "../binary_c.h"
#include "stellar_structure_debug.h"


void stellar_structure(struct star_t * const star,
                       struct stardata_t * const stardata,
#ifdef BSE
                       double *  const timescales,
                       double *  const luminosities,
                       double *  const GB,
#endif
                       const Boolean allow_SN,
                       const Caller_id caller_id)
{
    Dprint("stellar structure called from %s with age %g (star %d, type %d)\n",
           Stellar_structure_caller_string(caller_id),
           star->age,
           star->starnum,
           star->stellar_type);

    /*
     * Make the "newstar" struct in the tmpstore if required.
     * Copy the old star on top of it.
     */
    if(stardata->tmpstore->stellar_structure_newstar == NULL)
    {
        stardata->tmpstore->stellar_structure_newstar = New_star;
    }
    struct star_t * newstar = stardata->tmpstore->stellar_structure_newstar;
    Copy_star(star,newstar);
    
    Supernova_type new_SN_type = SN_NONE;
    
    if(stardata->preferences->stellar_structure_algorithm ==
       STELLAR_STRUCTURE_ALGORITHM_MODIFIED_BSE)
    {
#ifdef BSE
        stellar_structure_modified_BSE(stardata,
                                       star,
                                       newstar,
                                       timescales,
                                       luminosities,
                                       GB,
                                       caller_id,
                                       &new_SN_type);
#else
        Exit_binary_c(BINARY_C_ALGORITHM_OUT_OF_RANGE,
                      "Call to modified BSE stellar structure, but BSE is not built in\n");
#endif//BSE
    }
#ifdef BINT
    else if(stardata->preferences->stellar_structure_algorithm ==
            STELLAR_STRUCTURE_ALGORITHM_BINT)
    {
        BINT_stellar_structure(newstar,
                               star,
                               stardata,
                               caller_id);
    }
#endif // BINT
    else
    {
        Exit_binary_c(BINARY_C_ALGORITHM_OUT_OF_RANGE,
                      "Unknown or unsupported stellar structure algorithm %d \n",
                      stardata->preferences->stellar_structure_algorithm);
    }

    /*
     * Propagate SN type
     */
    newstar->SN_type = new_SN_type;

    if(0 && newstar->SN_type != SN_NONE)
    {
        printf("supernova star %d at %p TYPE %d new_supernova %p new_stellar_structure %p\n",
               star->starnum,
               star,
               star->SN_type,
               star->new_supernova,
               star->new_supernova ? star->new_supernova->new_stellar_structure : NULL);
        printf("supernova newstar %d at %p TYPE %d new_supernova %p new_stellar_structure %p\n",
               newstar->starnum,
               newstar,
               newstar->SN_type,
               newstar->new_supernova,
               newstar->new_supernova ? newstar->new_supernova->new_stellar_structure : NULL);
    }

    if(allow_SN == FALSE &&
       newstar->SN_type != SN_NONE)
    {
        /*
         * We have a SN, but are not allowed to have one.
         * Set the stellar structure from the new supernova
         * struct and free the associated memory.
         */
        Dprint("Remove new supernova : SNe not allowed\n");
        free_supernova(stardata,newstar);
        newstar->SN_type = new_SN_type = SN_NONE;
    }

    /*
     * Derived variables 
     */
    newstar->effective_radius = 
        (newstar->roche_radius > TINY &&
         newstar->radius > newstar->roche_radius) ? 
        Max(newstar->roche_radius,
            newstar->core_radius) : newstar->radius;

#ifdef XRAY_LUMINOSITY
    newstar->Xray_luminosity = Xray_luminosity(star);
#endif

#ifdef GIANT_CRICTICAL_Q_CHEN_HAN
    // save base of the giant branch luminosity and radius
    // as well as the A0 parameter
    if(Is_zero(newstar->A0) && (newstar->stellar_type>HERTZSPRUNG_GAP))
    {
        newstar->A0 = log10(newstar->radius);
    }
#endif// GIANT_CRICTICAL_Q_CHEN_HAN

    /*
     * Set the new structure in place : this also copies
     * event structures that have been allocated.
     */
    Copy_star(newstar,star);
    Dprint("return M = %g, R = %g : intermediate %d\n",
           star->mass,
           star->radius,
           stardata->model.intermediate_step
        );

    return;

#ifdef OLDCODE


    /* calculate new stellar structure */
    stellar_structure_algorithm(newstar,star,stardata,caller_id);

    if(allow_SN == FALSE)
    {
        /*
         * We have a SN, but are not allowed to have one.
         * Set the stellar structure from the new supernova
         * struct and free the associated memory.
         */        
        free_supernova(stardata,newstar);
    }

    /* TODO set AGB parameters here */
#endif
}
