#include "../binary_c.h"
#ifdef BSE
#undef K2FUDGE

void convective_envelope_mass_and_radius(struct stardata_t * stardata,
           struct star_t * newstar,
           const double rg,
           double z) // z= metallicity
{
    Dprint("Star %d : M=%g M0=%g Mc=%g Menv=%g st=%d L=%g R=%g J=%g omega=%g v_eq=%g v_crit_eq=%g : k2 was %g\n",
           newstar->starnum,
           newstar->mass,
           newstar->phase_start_mass,
           newstar->core_mass,
           newstar->mass - newstar->core_mass,
           newstar->stellar_type,
           newstar->luminosity,
           newstar->radius,
           newstar->angular_momentum,
           newstar->omega,
           newstar->v_eq,
           newstar->v_crit_eq,
           newstar->k2);
           
    double menvg,menvt,menvz,renvg,renvt,renvz,y,u,k2bgb,k2g,k2z;
    double tebgb=0.0,tauenv,tautms;
    const double ajtm=newstar->age/newstar->tm;
#ifdef K2FUDGE
    double k2e_in=newstar->k2;
#endif
    /*
     * A function to estimate the mass and radius of the convective envelope,
     * as well as the gyration radius of the envelope.
     * N.B. Valid only for Z=0.02!
     *
     * The following input is needed from HRDIAG:
     *   newstar->stellar_type = stellar type
     *   newstar->phase_start_mass = zero-age stellar mass
     *   newstar->mass = actual mass
     *   newstar->core_mass = core mass (not really needed, can also be done outside subroutine)
     *   newstar->luminosity = luminosity
     *   newstar->radius = radius
     *   newstar->core_radius = core radius 
     *   newstar->age = age
     *   newstar->tm = main-sequence lifetime
     *   newstar->luminosities[L_END_MS] = luminosity at TMS, lums(2)
     *   newstar->luminosities[L_BGB] = luminosity at BGB, lums(3)
     *   newstar->luminosities[L_HE_IGNITION] = luminosity at He ignition, lums(4)
     *   newstar->rzams = radius at ZAMS
     *   newstar->rtms = radius at TMS
     *   rg = giant branch or Hayashi track radius, approporaite for the type. 
     *        For kw=1 or 2 this is radius at BGB, and for kw=4 either GB or 
     *        AGB radius at present luminosity.
     *
     * RGI: I also limit the low-mass limit used below to calculate logm to 
     *      0.1 Msun. Failing to do this results in negative k2bgb because the
     *      fit is (presumably) not valid for very low-mass stars/planets.
     * 
     * This was the BSE function "mrenv".
     */
    const double logm = log10(Max(0.1,newstar->phase_start_mass));
    const double A = Min(0.81,Max(0.68,0.68+0.4*logm));

    
 
    //D = -0.1;
    //E = 0.025;
    /*
     * Zero-age and BGB values of k^2.
     */
    k2z = Min(0.21,Max(0.09-0.27*logm,0.037+0.033*logm));
    if(logm>1.3) k2z -= 0.055*Pow2(logm-1.3);
    k2bgb = Min3(0.15,0.147+0.03*logm,0.162-0.04*logm);

    Dprint("k2bgb = %g\n",k2bgb);
  
    switch(newstar->stellar_type)
    {
    case GIANT_BRANCH:
    case CHeB:
    case EAGB:
    case TPAGB:
        /*
         * Envelope k^2 for giant-like stars; this will be modified for non-giant
         * CHeB stars or small envelope mass below.
         * Formula is fairly accurate for both FGB and AGB stars if M <= 10, and
         * gives reasonable values for higher masses. Mass dependence is on actual
         * rather than ZA mass, expected to work for mass-losing stars (but not
         * tested!). The slightly complex appearance is to insure continuity at 
         * the BGB, which depends on the ZA mass.
         */
    {
        /*
         * Define L_BGB if required
         */
        if(Is_zero(newstar->luminosities[L_BGB]))
        {
            newstar->luminosities[L_BGB] = lbgbf(newstar->phase_start_mass,
                                           stardata->common.giant_branch_parameters);
        }

        const double logmass = log10(newstar->mass);
        const double F = 0.208 + 0.125*logmass - 0.035*Pow2(logmass);
        double w = Pow1p5(newstar->mass);
        const double B = 1e-4*(1.0/w+0.1);
        w = (newstar->luminosity - newstar->luminosities[L_BGB])*B;
        w *= w;
        y = (F - 0.033*log10(newstar->luminosities[L_BGB]))/k2bgb - 1.0;

        y = Max(0.0,y); // should never be < 0! causes fatal errors...

        double numerator = (F - 0.033*log10(newstar->luminosity) + 0.4*w); 
        double denominator = 1.0+y*(newstar->luminosities[L_BGB]/newstar->luminosity)+w;
        k2g = numerator / denominator;

        Dprint("numerator = %g - 0.033 * %g + 0.4 * %g = %g\n",
               F,
               log10(newstar->luminosity),
               w,
               numerator);
        Dprint("denominator 1.0+%g*(%g/%g)+%g = %g\n",
               y,
               newstar->luminosities[L_BGB],
               newstar->luminosity,
               w,
               denominator);
        Dprint("k2g = numerator/denominator = %g\n",k2g);
        Dprint("convective_envelope_mass_and_radius newstar->stellar_type=%d k2 mid newstar->k2=%g k2g=%g from newstar->luminosity=%g newstar->luminosities[L_BGB]=%g newstar->mass=%g k2bgb=%g w=%g y=%g numerator=%g denominator=%g\n",
               newstar->stellar_type,
               newstar->k2,
               k2g,
               newstar->luminosity,
               newstar->luminosities[L_BGB],newstar->mass,k2bgb,w,y,
               (F - 0.033*log10(newstar->luminosity) + 0.4*w),
               (1.0+y*(newstar->luminosities[L_BGB]/newstar->luminosity)+w));
    }

    break;
        
    case HeGB:
        /*
         * Rough fit for for HeGB stars...
         */
    {
        const double B = 3.0e+04*Pow1p5(newstar->mass);
        double w = Max(0.0,newstar->luminosity/B-0.5);
        w *= w;
        const double w04=0.4*w;
        k2g = (k2bgb + w04)/(1.0 + w04);
    }
    break;

    default:
        Dprint("Set k2g from k2bgb = %g\n",k2bgb);
        k2g = k2bgb;
    }

    if(newstar->stellar_type<GIANT_BRANCH)
    {
        menvg = 0.5;
        renvg = 0.65;
    }
    else if(newstar->stellar_type==GIANT_BRANCH &&
            newstar->luminosity < 3.0*newstar->luminosities[L_BGB])
    {
        /*
         * FGB stars still close to the BGB do not yet have a fully developed CE.
         */
        double x = Min(3.0,newstar->luminosities[L_HE_IGNITION] / newstar->luminosities[L_BGB]);
        double tau = Max(0.0,Min(1.0,(x-newstar->luminosity / newstar->luminosities[L_BGB])/(x-1.0))); 
        tau *= tau;
        menvg = 1.0 - 0.5*tau;
        renvg = 1.0 - 0.35*tau;
    }      
    else
    {
        menvg = 1.0;
        renvg = 1.0;
    }

    Dprint("CF Radius = %g to rg = %g (RG HE %g)\n",
           newstar->radius,
           rg,
           rhegbf(newstar->luminosity)
        );
    if(newstar->radius < rg)
    {
        /*
         * Stars not on the Hayashi track: MS and HG stars, non-giant CHeB stars,
         * HeMS and HeHG stars, as well as giants with very small envelope mass.
         */
        if(newstar->stellar_type<HeMS)
        {
            /*
             * Envelope k^2 fitted for MS and HG stars.
             * Again, pretty accurate for M <= 10 but less so for larger masses.
             * [Note that this represents the whole star on the MS, so there is a 
             * discontinuity in stellar k^2 between MS and HG - okay for stars with a 
             * MS hook but low-mass stars should preferably be continous...]
             *
             * For other types of star not on the Hayashi track we use the same fit as 
             * for HG stars, this is not very accurate but has the correct qualitative 
             * behaviour. For CheB stars this is an overestimate because they appear
             * to have a more centrally concentrated envelope than HG stars.
             */
            const double tau=newstar->radius/newstar->rzams;
            Dprint("tau = %g from radius=%g / rzams=%g\n",
                   tau,newstar->radius,newstar->rzams);
            const double C = Max(-2.5,Min(-1.5,-2.5+5.0*logm));
            newstar->k2 = (k2z-0.025)*pow(tau,C) + 0.025*pow(tau,-0.1);
            Dprint("k2z = %g tau = %g C = %g \n",k2z,tau,C);
            Dprint("stellar type = %d < HeMS : newstar->k2 set to %g\n",
                   newstar->stellar_type,newstar->k2);
        }
        else if(newstar->stellar_type==HeMS)
        {
            /*
             * Rough fit for naked He MS stars.
             */
            newstar->k2 = 0.08 - 0.03 * ajtm;
            Dprint("HeMS : newstar->k2 = %g\n",newstar->k2);
        }
        else if(newstar->stellar_type<HeWD)
        {
            /*
             * Rough fit for HeHG stars.
             */
            double rzams = rzhef(newstar->phase_start_mass);
            newstar->k2 = 0.08 * newstar->rzams / newstar->radius;
            Dprint("<HeWD : newstar->k2 = %g from RZAMS = %g / R=%g (recalc %g)\n",
                   newstar->k2,
                   newstar->rzams,
                   newstar->radius,
                   rzams
                );

            /* 
             * K2 can become very small when the rough fit
             * breaks down: don't let this happen! Really
             * we need a better solution to fix this ...
             */
            newstar->k2 = Max(0.005,newstar->k2);
        }

        /*
         * tauenv measures proximity to the Hayashi track in terms of Teff.
         * If tauenv>0 then an appreciable convective envelope is present, and
         * k^2 needs to be modified.
         */
        double iw=1.0/(1.0-A);
        if(newstar->stellar_type<GIANT_BRANCH)
        {
            tebgb = sqrt(sqrt(newstar->luminosities[L_BGB])/rg);
            tauenv = tebgb/(sqrt(sqrt(newstar->luminosity)/newstar->radius));
        }
        else
        {
            tebgb = 0.0;
            tauenv = sqrt(newstar->radius/rg);
        }
        tauenv = Max(0.0,Min(1.0,iw*(tauenv-A)));
        Dprint("tauenv (proximity to Hayashi track) = %g\n",tauenv);

        // NB iw=1/(1-A)
        if(tauenv > 0.0)
        {
            newstar->menv = menvg*Pow5(tauenv);
            newstar->renv = renvg*Pow5d4(tauenv); 
            double xx=Pow3(tauenv)*(k2g-newstar->k2);

            Dprint("Near Hayashi track: menv=%g renv=%g\n",
                   newstar->menv,
                   newstar->renv);

            if(newstar->stellar_type<HERTZSPRUNG_GAP)
            {
                /* Zero-age values for CE mass and radius.
                 */
                double x = Max(0.0,Min(1.0,(0.1-logm)/0.55));
                u=Pow5(x);

                {
                    double a,b;

//#undef ZAMS_MENV_METALLICITY_CORRECTION
#ifdef ZAMS_MENV_METALLICITY_CORRECTION
                    /*
x                     * fit is valid for low-mass MS stars (m<3)
                     * and anyway, m~3 has zero convective envelope
                     */
                    if((newstar->phase_start_mass<3.0)&&(newstar->stellar_type<HERTZSPRUNG_GAP))
                    {
                        /*
                         * Scale the zams menv with Z according to Rob's fit
                         */
                        z=Max(1e-4,Min(0.02,z)); // limit to range of fit
                        a=(-1.24090e-04)+(6.66010e+00)*z;
                        b=(3.02190e-01)+(-1.50960e+00)*z;
                    }
                    else
                    {
                        a=0.18;
                        b=0.82;
                    }
#else
                    a=0.18,b=0.82;  // Jarrod's fit
#endif
                    menvz = a*x + b*u;
                }

                renvz = 0.4*Pow1d4(x) + 0.6*Pow2(u);
                double y = 2.0 + 8.0*x;
                /* Values for CE mass and radius at start of the HG.
                 */           
                tautms = Max(0.0,Min(1.0,(tebgb/(sqrt(sqrt(newstar->luminosities[L_END_MS])/newstar->rtms))-A)*iw));
                menvt = menvg*Pow5(tautms);
                renvt = renvg*Pow5d4(tautms);
                /* Modified expressions during MS evolution.
                 */            
                u=pow(ajtm,y);
                if(tautms>0.0)
                {
                    newstar->menv = menvz + u*(newstar->menv)*(1.0 - menvz/menvt);
                    newstar->renv = renvz + u*(newstar->renv)*(1.0 - renvz/renvt);
                }
                else
                {
                    newstar->menv = 0.0;
                    newstar->renv = 0.0;
                }
                xx *= u;
            }//<HERTZSPRUNG_GAP
            newstar->k2 += xx;
            Dprint("Correction xx = %g : newstar->k2 = %g\n",xx,newstar->k2);
        }
        else
        {
            newstar->menv = 0.0;
            newstar->renv = 0.0;
        }
    }
    else
    {
        /*
         * All other stars should be true giants.
         */
        newstar->menv = menvg;
        newstar->renv = renvg;
        newstar->k2 = k2g;
        Dprint("Set k2 = k2g = %g\n",k2g);
    }

    newstar->menv = Max(newstar->menv * (newstar->mass - newstar->core_mass),1.0E-10);
    newstar->renv = Max(newstar->renv * (newstar->radius-newstar->core_radius) ,1.0E-10);

#ifdef K2FUDGE
    /* 
     * For non-compact stars, do not let the moment of inertia
     * change by a fraction > K2FUDGE_MAX in any timestep.
     *
     * This is a fudge : in reality k2 should be a smooth fit!
     */
    if(newstar->stellar_type<HeWD)
    {
        double k2e_change=fabs(k2e_in-newstar->k2)/(k2e_in);
        double k2e_wanted=newstar->k2;
        Dprint("k2e change %3.2f %%\n",100.0newstar->k2_change);
        if(Is_not_zero(k2e_in)&&(k2e_change>K2FUDGE_MAX))
        {
            newstar->k2 = Min(1.1newstar->k2_in,Max(0.9newstar->k2_in,newstar->k2));
#ifdef K2FUDGE_WARNING
            printf("Warning : K2 changed by > %g %% : wants to be %g, was %g but capped at %g\n",K2FUDGE_MAX*100.0,k2e_wanted,k2e_in,newstar->k2);
#endif
        }
    }
#endif


    Dprint("MRENV result menv=%12.12e renv=%12.12e k2e=%g R=%g\n",
           newstar->menv,
           newstar->renv,
           newstar->k2,
           newstar->radius);
    
    if(newstar->k2 < 0.0)
    {
        Exit_binary_c(BINARY_C_OUT_OF_RANGE,
                      "k2e<0 error at time %15.12e : INPUT : newstar->stellar_type=%d mass=%g newstar->mass=%g newstar->core_mass=%g newstar->luminosity=%g newstar->radius=%g newstar->core_radius=%g age=%g tm=%g newstar->luminosities[L_END_MS]=%g newstar->luminosities[L_BGB]=%g newstar->luminosities[L_HE_IGNITION]=%g rzams=%g newstar->rtms=%g rg=%g menv=%g renv=%g k2e=%g k2g=%g\n",
                      stardata->model.time,
                      newstar->stellar_type,newstar->phase_start_mass,newstar->mass,newstar->core_mass,newstar->luminosity,newstar->radius,newstar->core_radius,newstar->age,newstar->tm,newstar->luminosities[L_END_MS],newstar->luminosities[L_BGB],newstar->luminosities[L_HE_IGNITION],newstar->rzams,newstar->rtms,rg,
                      newstar->menv,newstar->renv,newstar->k2,k2g);
    }

}
#endif//BSE
