
#include "../binary_c.h"

#ifdef STELLAR_COLOURS

/*
 * Wrapper function to calculate all the stellar 
 * colours that are available.
 *
 * The **magnitudes_p is NULL the memory is allocated here.
 *
 * If *magnitudes_p is non-NULL but *magnitudes_p is NULL,
 * the memory is allocated here.
 *
 * If both **magnitudes_p and *magnitudes_p are non-NULL,
 * we assume the space is already initialized.
 *
 * Please note that all magntiudes are Vega magnitudes.
 *
 * To understand the difference, please see this website:
 * https://www.astro.umd.edu/~ssm/ASTR620/mags.html
 */

void stellar_magnitudes(struct stardata_t * const stardata,
                        struct star_t * const star,
                        double ** RESTRICT magnitudes_p)
{
    if(magnitudes_p == NULL ||
       *magnitudes_p == NULL)
    {
        /*
         * Allocate space
         */
        *magnitudes_p = Calloc(NUMBER_OF_STELLAR_MAGNITUDES,
                               sizeof(double));
    }
    double * const magnitudes = *magnitudes_p;
    
    /* 
     * Get basis set from Eldridge's numbers, 
     * based on the Basel library
     */
    eldridge2015_colours(stardata,star,magnitudes);
    
    /*
     * Make Gaia colours
     */
    gaia_magnitudes(stardata,magnitudes,GAIA_CONVERSION_USE_DEFAULT);

    Dprint("MAGS%d L = %g,  U = %g, B = %g, V = %g, R = %g, I = %g\n",
           star->starnum,
           star->luminosity,
           magnitudes[STELLAR_MAGNITUDE_U],
           magnitudes[STELLAR_MAGNITUDE_B],
           magnitudes[STELLAR_MAGNITUDE_V],
           magnitudes[STELLAR_MAGNITUDE_R],
           magnitudes[STELLAR_MAGNITUDE_I]);
}
#endif // STELLAR_COLOURS
