#include "../binary_c.h"

static double cubic(const double x,
                    const double a,
                    const double b,
                    const double c,
                    const double d);

static double crosscubic(const double x,
                         const double y,
                         const double a,
                         const double b,
                         const double c,
                         const double d,
                         const double e,
                         const double f,
                         const double g,
                         const double h);

void gaia_magnitudes(struct stardata_t * stardata,
                  double * RESTRICT magnitudes,
                  int method)
{
    /*
     * Given a populated magntiudes array,
     * convert to Gaia magntiudes based on 
     * the conversion formulae of 
     * Jordi et al 2010 (A&A 523,A48)
     *
     * We can do this from either UBVRI or ugriz,
     * but UBVRI fit better (Jordi et al 2010, sec.5.2
     * page 7).
     *
     * You can use single-magnitude fits (cubics), or from 
     * two colours.
     *
     * If you want extinction, you should do it elsewhere.
     */

    /*
     * Default method (defined in stellar_colour_macros.h).
     */
    if(method == GAIA_CONVERSION_USE_DEFAULT)
        method = GAIA_CONVERSION_DEFAULT_METHOD;

    if(method==GAIA_CONVERSION_UBVRI_SINGLE)
    {
        /*
         * Johnson-Cousins UBVRI conversion
         * assumes R=Rc and I=Ic.
         *
         * Fitting coefficients are from page 7
         * in Jordi et al 2010, Table 3, top panel
         */
        double vic = magnitudes[STELLAR_MAGNITUDE_V] - magnitudes[STELLAR_MAGNITUDE_I];
        magnitudes[STELLAR_MAGNITUDE_GAIA_G] = 
            magnitudes[STELLAR_MAGNITUDE_V] +
            cubic(vic, -0.0257, -0.0924, -0.1623, 0.0090);

        magnitudes[STELLAR_MAGNITUDE_GAIA_GBP] = 
            magnitudes[STELLAR_MAGNITUDE_V] -
            cubic(vic, 0.0643, -0.3266, 0.0887, -0.0050);
        
        magnitudes[STELLAR_MAGNITUDE_GAIA_GRP] = 
            magnitudes[STELLAR_MAGNITUDE_V] -
            cubic(vic, -0.0017,0.8794,0.0273,-0.0008);
  
        magnitudes[STELLAR_MAGNITUDE_GAIA_GRVS] = 
            magnitudes[STELLAR_MAGNITUDE_V] -
            cubic(vic, 0.0119, 1.2092, -0.0188, -0.0005);
    }
    else if(method==GAIA_CONVERSION_UBVRI_DOUBLE)
    {
        /*
         * Johnson-Cousins UBVRI conversion
         * assumes R=Rc and I=Ic.
         *
         * Fitting coefficients are from page 11
         * in Jordi et al 2010, Table 7, top panel
         *
         * These are the two-magnitude cross-cubic fits
         * which are supposedly the best.
         */
        double vic = magnitudes[STELLAR_MAGNITUDE_V] - magnitudes[STELLAR_MAGNITUDE_I];
        double bv =  magnitudes[STELLAR_MAGNITUDE_B] - magnitudes[STELLAR_MAGNITUDE_V];
        magnitudes[STELLAR_MAGNITUDE_GAIA_G] = 
            magnitudes[STELLAR_MAGNITUDE_V] +
            crosscubic(vic, bv, -0.0099,-0.2116,-0.1387,0.0060,0.1485,-0.0895,0.0094,0.0327);

        magnitudes[STELLAR_MAGNITUDE_GAIA_GBP] = 
            magnitudes[STELLAR_MAGNITUDE_V] -
            crosscubic(vic, bv, 0.0188,0.0504,-0.0181,0.0021,-0.4728,0.1586,-0.0176,0.0255);
        
        magnitudes[STELLAR_MAGNITUDE_GAIA_GRP] = 
            magnitudes[STELLAR_MAGNITUDE_V] -
            crosscubic(vic, bv, 0.0024,0.8328,0.0572,-0.0020,0.0722,-0.0126,0.0037,-0.0380);
  
        magnitudes[STELLAR_MAGNITUDE_GAIA_GRVS] = 
            magnitudes[STELLAR_MAGNITUDE_V] -
            crosscubic(vic, bv, -0.0188,1.4535,-0.0873,0.0038,-0.2977,0.0701,-0.0120,0.0383);
    }
    else if(method==GAIA_CONVERSION_ugriz_SINGLE)
    {
        /*
         * SDSS ugriz conversion
         *
         * Fitting coefficients are from page 9,
         * Table 5, using g-i
         */
        double gi = magnitudes[STELLAR_MAGNITUDE_g] - magnitudes[STELLAR_MAGNITUDE_i];
        magnitudes[STELLAR_MAGNITUDE_GAIA_G] = 
            magnitudes[STELLAR_MAGNITUDE_g] +
            cubic(gi,-0.0940, -0.5310, -0.0974, 0.0052);

        magnitudes[STELLAR_MAGNITUDE_GAIA_GBP] = 
            magnitudes[STELLAR_MAGNITUDE_GAIA_G] -
            cubic(gi,-0.1235, -0.3289, -0.0582, 0.0033);
        
        magnitudes[STELLAR_MAGNITUDE_GAIA_GRP] = 
            magnitudes[STELLAR_MAGNITUDE_GAIA_G] -
            cubic(gi,0.2566, 0.5086, -0.0678, 0.0032);
  
        magnitudes[STELLAR_MAGNITUDE_GAIA_GRVS] = 
            magnitudes[STELLAR_MAGNITUDE_GAIA_G] -
            cubic(gi,0.3931, 0.7250, -0.0927, 0.0032);
        
    }

    else if(method==GAIA_CONVERSION_ugriz_DOUBLE)
    {
        /*
         * SDSS ugriz conversion
         * assumes R=Rc and I=Ic.
         *
         * Fitting coefficients are from page 11
         * in Jordi et al 2010, Table 7, top panel
         *
         * These are the two-magnitude cross-cubic fits
         * which are supposedly the best.
         */
        double gi = magnitudes[STELLAR_MAGNITUDE_g] - magnitudes[STELLAR_MAGNITUDE_i];
        double gr =  magnitudes[STELLAR_MAGNITUDE_g] - magnitudes[STELLAR_MAGNITUDE_r];
        magnitudes[STELLAR_MAGNITUDE_GAIA_G] = 
            magnitudes[STELLAR_MAGNITUDE_g] +
            crosscubic(gi, gr, -0.1005,-0.5358,-0.1207,0.0082,-0.0272,0.1270,-0.0205,-0.0176);

        magnitudes[STELLAR_MAGNITUDE_GAIA_GBP] = 
            magnitudes[STELLAR_MAGNITUDE_g] -
            crosscubic(gi, gr, -0.0428,0.0158,0.0122,0.0005,0.2382,0.1855,-0.0096,-0.0493);
        
        magnitudes[STELLAR_MAGNITUDE_GAIA_GRP] = 
            magnitudes[STELLAR_MAGNITUDE_g] -
            crosscubic(gi, gr, 0.3731,1.3335,-0.0242,-0.0047,-0.4086,-0.2729,0.0230,0.2166);
        
        magnitudes[STELLAR_MAGNITUDE_GAIA_GRVS] = 
            magnitudes[STELLAR_MAGNITUDE_g] -
            crosscubic(gi, gr, 0.5345,2.3063,-0.2919,0.0049,-1.6562,-0.1306,-0.0257,0.5419);
    }
    else
    {
        Exit_binary_c(BINARY_C_WRONG_ARGUMENT,"Converting from magnitudes to Gaia colours failed with unknown method %d. Please see src/stellar_magnitudes/stellar_colour_macros.h for available methods, or use method=0 for the default (currently %d)\n",method,GAIA_CONVERSION_DEFAULT_METHOD);
    }
}
                    
static double cubic(const double x,
                    const double a,
                    const double b,
                    const double c,
                    const double d)
{
    return a+b*x+c*x*x+d*x*x*x;
}

static double crosscubic(const double x,
                         const double y,
                         const double a,
                         const double b,
                         const double c,
                         const double d,
                         const double e,
                         const double f,
                         const double g,
                         const double h)
{
    return a+b*x+c*x*x+d*x*x*x + 
        e*y+f*y*y+g*y*y*y+
        h*x*y;
}
