/*
 *       Determine the age of a giant from its core mass and type.
 *
 *       Author : C. A. Tout
 *       Date   : 24th September 1996
 *       Revised: 21st February 1997 to include core-helium-burning stars
 *
 *       Rewritten: 2nd January 1998 by J. R. Hurley to be compatible with
 *                  the new evolution routines and to include new stellar
 *                  types.
 *       Rewritten: early 2001, Rob Izzard.
 *
 */ 
#include "../binary_c.h"
#ifdef BSE

#define JMAX 30 /* Please make JMAX > 1 */
#define MACC 0.00001
#define LACC 0.0001 

/* dummy stellar types */
#define COWD_TO_CHeB -1
#define EAGB_TO_CHeB -2

void giant_age(double * RESTRICT const mc, /** core mass in/out **/
               const double mt, /* current mass */
               Stellar_type * RESTRICT const stellar_type, /* stellar type */
               double * RESTRICT const m0, /* mass at start of this stellar phase */
               double * RESTRICT const age, /* stellar age */
               struct stardata_t * RESTRICT const stardata,
               struct star_t * RESTRICT const star)
{
    /** some dummy variables **/
    double mmin,mmax,mmid,dm,f,fmid,dell,derl,lum,mcx,mcy=0;
    const double *metallicity_parameters=stardata->common.metallicity_parameters;
    const double *giant_branch_parameters=stardata->common.giant_branch_parameters;
    double GB[GB_ARRAY_SIZE];
    double luminosities[LUMS_ARRAY_SIZE];
    double timescales[TSCLS_ARRAY_SIZE];
    double tm,tn;

    /* callstar is used to fix up the GOTO crap from the fortran program */
    Boolean callstar=TRUE; 
    unsigned int j; /** counter - presumably < JMAX **/
    Boolean breakout=FALSE; /** dummy variable, used to break out of loops **/


#if (DEBUG==1)
    Stellar_type stellar_typein=*stellar_type;
#endif /*DEBUG*/
    Dprint("giant_age in, stellar type %d, mc=%g, M now=%g, m0=%g, menv = %g \n",
           *stellar_type,
           *mc,
           mt,
           *m0,
           mt - *mc
        );

    /* First we check that we don't have a CheB star 
     * with too small a core mass. */
    if(*stellar_type==CHeB)
    {
        /** Set the minimum CHeB core mass using M = Mflash **/
        mcy = mcheif(metallicity_parameters[ZPAR_MASS_HE_FLASH],
                     metallicity_parameters[ZPAR_MASS_HE_FLASH],
                     metallicity_parameters[10],
                     giant_branch_parameters);

        Dprint("Set mcy = %g cf. mc = %g\n",mcy,*mc);
        if (Less_or_equal(*mc,mcy))
        {
            *stellar_type = FIRST_GIANT_BRANCH;
            Dprint("Set stellar type to GB\n");
        }
    }
    
    /* Next we check that we don't have a GB star for M => Mfgb */
    if(*stellar_type == FIRST_GIANT_BRANCH)
    {
        /* Set the maximum GB core mass using M = Mfgb */
        mcy = mcheif(metallicity_parameters[ZPAR_MASS_FGB],
                     metallicity_parameters[ZPAR_MASS_HE_FLASH],
                     metallicity_parameters[9],
                     giant_branch_parameters);
        
        Dprint("On GB : set mcy = %g cf. mc = %g\n",mcy,*mc)
        if(More_or_equal(*mc,mcy))
        {
            *stellar_type=CHeB;
            *age=0.0;
            Dprint("Set Zero-age CHeB\n");
        } 
    }

    /* TPAGB stars */
    if(*stellar_type==TPAGB)
    {
        /*
         * We try to start the star from the start of the SAGB by
         * setting Mc = Mc,TP.
         */
        mcy = 0.44*stardata->common.max_mass_for_second_dredgeup + 0.448;
        Dprint("start AGB mcy = %g, cf. mc now %g\n",mcy,*mc);
        
        if(*mc>mcy)
        {
            /* 
             * A TPAGB with this sized core mass cannot exist as it should
             * already have become a NS or BH as an EAGB star. 
             * We set it up so that it will.
             */           
            mcx = (*mc + 0.35)/0.773;
        }
        else if  (More_or_equal(*mc,0.80))
        {
            mcx = (*mc - 0.448)/0.44;
        }
        else
        {
            mcx = *mc;
        } 
        Dprint("mcx = %g\n",mcx);
        
        *m0 = mbagbf(mcx,giant_branch_parameters);

        Dprint("new m0 = %g\n",*m0);
        
        if (Less_or_equal(*m0,0.0))
        {
            /* 
             * Carbon core mass is less then the minimum for the start of SAGB.
             * This must be the case of a low-mass C/O or O/Ne WD with only a
             * very small envelope added or possibly the merger of a helium star
             * with a main sequence star. We will set (*m0) = mt and then reset the
             * core mass to allow for some helium to be added to the C/O core.
             */
            *stellar_type = COWD_TO_CHeB;
            Dprint("CO core mass < start of SAGB : low-mass COWD or ONeWD with thin envelope\n");
        }
        else
        {
            stellar_timescales(*stellar_type,
                               *m0,
                               mt,
                               &tm,
                               &tn,
                               timescales,
                               luminosities,
                               GB,
                               stardata,star);
            *age = timescales[T_TPAGB_FIRST_PULSE];
            Dprint("Hence age = T_TPAGB_FIRST_PULSE = %g\n",*age);
        }
    }
    
    if(*stellar_type==EAGB)
    {
        /*
         * We fit a Helium core mass at the base of the AGB.
         */
        *m0 = mbagbf(*mc,giant_branch_parameters);
        Dprint("EAGB : set m0 = %g from *mc = %g\n",*m0,*mc);
                
        if(Less_or_equal(*m0,TINY))
        {
            /*
             * Helium core mass is less then the BAGB minimum.
             * 
             * What has happened here is that a helium star has
             * accreted hydrogen, but it is of such low mass that 
             * it would never normally reach the AGB.
             * 
             * The BSE algorithm converts it to a CHeB star instead.
             */
            //*stellar_type = EAGB_TO_CHeB;

            /*
             * experimental : 
             * we shouldn't convert to a CHeB, just
             * leave as an EAGB / He-star 
             */
            *m0 = mt;
            *stellar_type = EAGB;
            stellar_timescales(*stellar_type,
                               *m0,
                               mt,
                               &tm,
                               &tn,
                               timescales,
                               luminosities,
                               GB,
                               stardata,star);
            
            *age = timescales[T_HE_IGNITION] + timescales[T_HE_BURNING];
            
            Dprint("set to EAGB_TO_CHeB\n");
        }
        else
        { 
            stellar_timescales(*stellar_type,
                               *m0,
                               mt,
                               &tm,
                               &tn,
                               timescales,
                               luminosities,
                               GB,
                               stardata,star);
            *age = timescales[T_HE_IGNITION] + timescales[T_HE_BURNING];
            Dprint("Set new age %g\n",*age);
        }
    }

    if(*stellar_type==CHeB)
    {
        /*
         * The supplied age is actually the fractional age, fage, of CHeB lifetime
         * that has been completed, ie. 0 <= age <= 1.
         */
        Dprint("giant_age : CHeB : fage=%g\n",*age);
        if(*age<0.0 || *age>1.0)
        {
            /** we cannot have a fractional age that's not between 0 and 1 ! **/
            Clamp(*age,0.0,1.0);
#ifdef GIANT_AGE_FATAL_ERRORS
            Exit_binary_c(BINARY_C_GIANT_AGE_FRACTIONAL_AGE_OUT_OF_BOUNDS,
                          " FATAL ERROR! giant_age: fractional age out of bounds : FAGE %f\n",
                          *age); 
#endif
        }
      
        /*
         * Get the minimum, fage=1, and maximum, fage=0, allowable masses
         */
        mcy = mcagbf(metallicity_parameters[ZPAR_MASS_HE_FLASH],giant_branch_parameters);
        Dprint("mcy = %g\n",mcy);
        
        if (More_or_equal(*mc,mcy))
        {
            mmin = mbagbf(*mc,giant_branch_parameters);
        }
        else
        {
            mmin = metallicity_parameters[ZPAR_MASS_HE_FLASH];
        }
        mmax = mheif(*mc,
                     metallicity_parameters[ZPAR_MASS_HE_FLASH],
                     metallicity_parameters[10],
                     giant_branch_parameters);
        Dprint("giant_age : CHeB : mmin=%g mmax=%g\n",mmin,mmax);

          
        /** Set the callstar flag **/
        /** this makes sure we call stellar_timescales **/
        /** unless we _specifically_ say "no, don't do it!" **/
        callstar = TRUE;

        Dprint("age = %g\n",*age);
        if (Fequal(*age,0.0))
        {          
            *m0 = mmax;
        }
        else if (Fequal(*age,1.0))
        {
            *m0 = mmin;
        }
        else
        {
            /* Use the bisection method to find m0 */
            double x = 1.0 - *age;
            Dprint("x = 1 - age = %g\n",x);
            
            fmid = x * mcheif(mmax,
                              metallicity_parameters[ZPAR_MASS_HE_FLASH],
                              metallicity_parameters[10],
                              giant_branch_parameters) + 
                (*age)*mcagbf(mmax,giant_branch_parameters) - *mc;
            Dprint("fmid = %g\n",fmid);
            
            f = x *mcheif(mmin,
                          metallicity_parameters[ZPAR_MASS_HE_FLASH],
                          metallicity_parameters[10],
                          giant_branch_parameters) +
                (*age)*mcagbf(mmin,giant_branch_parameters) - *mc;
            Dprint("f = %g : f*fmid = %g\n",f,f*fmid);

            if(More_or_equal(f*fmid,0.0))
            {
                /*
                 * This will probably occur if mc is just greater 
                 * than the minimum allowed mass
                 * for a CHeB star and fage > 0.
                 *
                 * Rob : this isn't good. The star moves backwards
                 * in its stellar type, when it shouldn't!
                 */ 
                *stellar_type = FIRST_GIANT_BRANCH;
                callstar = FALSE;
                Dprint("Convert to FGB\n");
            }
            else
            {
                *m0 = mmin;
                dm = mmax - mmin;

                for (j=1; j<=JMAX; j++)
                {
                    dm *= 0.5;
                    mmid = *m0 + dm;
                    fmid = (1.0-(*age))*mcheif(mmid,
                                              metallicity_parameters[ZPAR_MASS_HE_FLASH],
                                              metallicity_parameters[10],
                                              giant_branch_parameters) +
                        (*age)*mcagbf(mmid,giant_branch_parameters) - *mc;
                    if(fmid<0.0) 
                    {
                        *m0 = mmid;
                    }
                    if((Abs_less_than(dm,MACC))||(Is_zero(fmid)))
                    {
                        /* Set j=JMAX+2 to break out of the loop */
                        j=JMAX+2;
                        callstar=TRUE;
                    }
                    else if(j==JMAX)
                    {
#ifdef GIANT_AGE_FATAL_ERRORS                   
                        Exit_binary_c(BINARY_C_GIANT_AGE_ROOT_NOT_FOUND,"FATAL ERROR! giant_age: root not found\n");
#endif
                    }
                } /* loop over j */
                Dprint("converged on dm = %g\n",dm);
            } /* fmid >= 0.0 if  */
        }   /* age is not 0 or 1 if  */

        Dprint("giant_age : CHeB : m0=%g, callstar = %d\n",*m0,callstar);

        if(callstar==TRUE)
        {
            stellar_timescales(*stellar_type,
                               *m0,
                               mt,
                               &tm,
                               &tn,
                               timescales,
                               luminosities,
                               GB,
                               stardata,star);
            *age = timescales[T_HE_IGNITION] + *age * timescales[T_HE_BURNING];

            Dprint("giant_age : CHeB : callstar==TRUE so get age=%g\n",*age);
            Dprint("This is from %g+%g*%g\n",
                   timescales[T_HE_IGNITION],
                   *age,
                   timescales[T_HE_BURNING]);
        }
    }

    /*
     * RGI assume HG stars behave like RGB stars, 
     * then evolution is on the timescale of the core mass
     * of the HG star rather than the timescale corresponding
     * to the total mass of the new star.
     */
    if(*stellar_type == FIRST_GIANT_BRANCH ||
       *stellar_type == HERTZSPRUNG_GAP)
    {     
        /*
         * First we double check that we don't have a GB star for M => Mfgb
         */
        mcy = mcheif(metallicity_parameters[ZPAR_MASS_FGB],
                     metallicity_parameters[ZPAR_MASS_HE_FLASH],
                     metallicity_parameters[9],
                     giant_branch_parameters);
        Dprint("FGB or HG : mcy = %g\n",mcy);
        
        if(More_or_equal(*mc,mcy))
        {
#ifdef GIANT_AGE_FATAL_ERRORS   
            Exit_binary_c(BINARY_C_GIANT_AGE_STAR_TOO_BIG_FOR_GB,"giant_age3: star too big for GB\n");
#endif
            /* 2007 change */
            *mc = Min(mt,0.99*mcy);
        }
        
        /* Next we find an m0 so as to place the star at the BGB */
        mcx = mcheif(metallicity_parameters[ZPAR_MASS_HE_FLASH],
                     metallicity_parameters[ZPAR_MASS_HE_FLASH],
                     metallicity_parameters[9],
                     giant_branch_parameters);
        Dprint("mcx = %g\n",mcx);
        
        if(*mc>mcx)
        {
            *m0 = mheif(*mc,
                        metallicity_parameters[ZPAR_MASS_HE_FLASH],
                        metallicity_parameters[9],
                        giant_branch_parameters);
            Dprint("m0 = %g\n",*m0);
        }
        else
        {
            /* Use Newton-Raphson to find m0 from Lbgb */
            *m0 = metallicity_parameters[ZPAR_MASS_HE_FLASH];
            Dprint("NR for Lbgb, m0 = %g\n",*m0);
            stellar_timescales(*stellar_type,
                               *m0,
                               mt,
                               &tm,
                               &tn,
                               timescales,
                               luminosities,
                               GB,
                               stardata,star);
            lum = lmcgbf(*mc,GB);
            Dprint("lum(%g) = %g\n",*mc,lum);
            j = 0;
            
            dell = lbgbf(*m0,giant_branch_parameters) - lum;
            Dprint("dell = %g\n",dell);
            
            double m=*m0;
            Dprint("GB NR in m = %g, looking for L = %g\n",*m0,lum);
            while(Abs_more_than(dell/lum,LACC))
            {
                dell = lbgbf(m,giant_branch_parameters) - lum;
                derl = lbgbdf(m,giant_branch_parameters);
                Dprint("GB NR(m=%g) dell=%g derl=%g\n",m,dell,derl); 
                m -= dell/derl;
                Dprint("m now %g\n",m);
                j++;
                if(j==JMAX)
                {
#ifdef GIANT_AGE_FATAL_ERRORS                       
                    Exit_binary_c(BINARY_C_NEWTON_RAPHSON_FAILURE,
                                  "Fatal error! giant_age (stellar_type==GIANT_BRANCH): root not found in Newton-Raphson\n");
#endif
                }
            }
            *m0=m;
            Dprint("NR gave m0 = %g\n",*m0);
        }

        Dprint("Final GB timescales");
        stellar_timescales(*stellar_type,
                           *m0,
                           mt,
                           &tm,
                           &tn,
                           timescales,
                           luminosities,
                           GB,
                           stardata,star);
            
        *age = timescales[T_BGB] + 1.0E-06*(timescales[T_HE_IGNITION] - timescales[T_BGB]);
        Dprint("new age %g\n",*age);
    }/** if(stellar_type==GIANT_BRANCH) **/
    
    if(POST_MS_NAKED_He_STAR(*stellar_type))
    {
        /*
         * We make a post-MS naked helium star.
         * To make things easier we put the star at the TMS point
         * so it actually begins as type 8.
         */
        *stellar_type = HeHG;
        mmin = *mc;
        Dprint("post-MS naked He star : stellar type %d, mmin = %g from mc = %g\n",
               *stellar_type,
               mmin,
               *mc);
        
        stellar_timescales(*stellar_type,
                           mmin,
                           *mc,
                           &tm,
                           &tn,
                           timescales,
                           luminosities,
                           GB,
                           stardata,
                           star);

        Dprint("call mcgbf L = %g %g\n",
               luminosities[2],
               luminosities[6]);
        
        mcx = mcgbf(luminosities[2],
                    GB,
                    luminosities[6]);

        Dprint("mcx = %g vs mc = %g\n",
               mcx,
               *mc);
        
        if(More_or_equal(mcx,*mc))
        {
#ifdef GIANT_AGE_FATAL_ERRORS
            Exit_binary_c(BINARY_C_GIANT_AGE_MMIN_TOO_BIG,"Fatal error: gntage9: mmin too big\n");
#endif
            /* 2007 change */
            *m0 = mt;
            goto breakloop;
        }
        f = mcx - (*mc);
        mmax = mt;

        Dprint("mmax = %g\n",mmax);
        
        /** use the breakout==0 condition as a **/
        /** way of getting out of this loop **/
        breakout=FALSE;

        for (j=1; ((j<=JMAX)&&(breakout==FALSE)); j++)
        {
            stellar_timescales(*stellar_type,
                               mmax,
                               *mc,
                               &tm,
                               &tn,
                               timescales,
                               luminosities,
                               GB,
                               stardata,star);
            mcy = mcgbf(luminosities[2],
                        GB,
                        luminosities[6]);

            Dprint("mcy = %g\n",mcy);
            
            if(mcy > *mc) 
            {
                breakout = TRUE; /** use this to get out of the loop **/
            }
            else
            {
                mmax *= 2.0;
                if(j==JMAX)
                {
#ifdef GIANT_AGE_FATAL_ERRORS    
                    Exit_binary_c(BINARY_C_GIANT_AGE_MMAX_NOT_FOUND,
                                  "Fatal error! Gntage9: mmax not found\n");
#endif
                    goto breakloop;
                }
            }
            Dprint("converged on mmax = %g\n",mmax);
        }/** j loop **/
        
        fmid = mcy - (*mc);
/*
 * Use the bisection method to find m0
 */

        if(More_or_equal((f*fmid),0.0))
        {
#ifdef GIANT_AGE_FATAL_ERRORS               
            Exit_binary_c(BINARY_C_GIANT_AGE_ROOT_NOT_BRACKETED,
                          "Fatal error! giant_age 9: root not bracketed\n");
#endif
            goto breakloop;
        }
        *m0 = mmin;
        dm = mmax - mmin;

        /** set the loop breakout variable **/
        breakout=FALSE;
        
        for(j=1; ((j<=JMAX)&&(breakout==FALSE)); j++)
        {
            dm *= 0.5;
            mmid = *m0 + dm;
            stellar_timescales(*stellar_type,
                               mmid,
                               *mc,
                               &tm,
                               &tn,
                               timescales,
                               luminosities,
                               GB,
                               stardata,star);
            mcy = mcgbf(luminosities[2],
                        GB,
                        luminosities[6]);
                       
            fmid = mcy - *mc;
            if(fmid<0.0)
            {
                *m0 = mmid;
            }            
            if(Abs_less_than(dm,MACC)||(Fequal(fmid,0.0)))
            {
                goto breakloop;
            }
            else if(j==JMAX)
            {
#ifdef GIANT_AGE_FATAL_ERRORS   
                Exit_binary_c(BINARY_C_GIANT_AGE_ROOT_NOT_FOUND,
                              "Fatal error! GNTAGE9: root not found\n");
#endif
                goto breakloop;
            }
        } /** end of j loop **/
        Dprint("conveged on mmid = %g\n",mmid);

    breakloop:       
        
        stellar_timescales(*stellar_type,
                           *m0,
                           mt,
                           &tm,
                           &tn,
                           timescales,
                           luminosities,
                           GB,
                           stardata,star);

        /*
         * We want to be slightly after the end
         * of the helium main sequence. I tried 1e-10
         * but that failed because of numerics, so lets try
         * 1e-6 instead.
         */
        *age = tm * (1.0 + 1.0e-6);
        Dprint("new age %g\n",*age);
    }
      
    if(*stellar_type==COWD_TO_CHeB ||
       *stellar_type==EAGB_TO_CHeB)
    {
        /*
         * we are converting from an EAGB or COWD to a CHeB star
         */
        Dprint("%s converted to CHeB\n",
               *stellar_type==COWD_TO_CHeB ? "COWD" : "EAGB");
        *stellar_type = CHeB;
        *m0 = mt;
        mcy = mcagbf(*m0,giant_branch_parameters);
        *age = *mc/mcy;
        Dprint("New age %g, mcy = %g\n",*age,mcy);
        stellar_timescales(*stellar_type,
                           *m0,
                           mt,
                           &tm,
                           &tn,
                           timescales,
                           luminosities,
                           GB,
                           stardata,star);

        if(Less_or_equal(*m0,metallicity_parameters[ZPAR_MASS_HE_FLASH]))
        {
            mcx = mcgbf(luminosities[4],
                        GB,
                        luminosities[6]);
                       
        }
        else
        {
            mcx = mcheif(*m0,
                         metallicity_parameters[ZPAR_MASS_HE_FLASH],
                         metallicity_parameters[10],
                         giant_branch_parameters);
        }
        *mc = Min(mt, mcx + (mcy - mcx)* *age);
        Dprint("mcx = %g -> mc = %g\n",mcx,*mc);
        
        *age = timescales[T_HE_IGNITION] + *age * timescales[T_HE_BURNING];
        Dprint("new age %g\n",*age);
    }

    Dprint("Finished in giant_age\n");
#if (DEBUG==1)
    Dprint("It got new stellar type: %d (in was %d)\nInitial Mass: %f\naj: %f\n",
           *stellar_type,
           stellar_typein,
           *m0,*age);
#endif
    Dprint("giant age : out stellar type %d, m0=%g mt=%g, mc=%g, age=%g\n",
           *stellar_type,*m0,mt,*mc,*age);
    
    return;
}


    
 
#endif//BSE
