#include "../binary_c.h"
#include "../nucsyn/nucsyn_yield_macros.h"
#include <sys/resource.h>
#include <stdlib.h>
#include <stdint.h>
#undef __HAVE_LIBBSD__
#ifdef __HAVE_LIBBSD__
#include <bsd/bsd.h>
#endif // __HAVE_LIBBSD__
static float __test_seconds(void);
#include <unistd.h>

void version(struct stardata_t * RESTRICT const stardata)
{
    /***********************/
    /* Version information */
    /***********************/

    /* prevent output to the logfile */
    snprintf(stardata->preferences->log_filename,
             10,
            "/dev/null");

    Printf("\nBinary_c/nucsyn by Robert Izzard, originally based on BSE of Hurley, Pols, Tout, Aarseth, but with many updates and rewrites.\nEmail izzard@astro.uni-bonn.de, rgi@ast.cam.ac.uk, rob.izzard@gmail.com\nVersion %s\nBuild: %s %s\n",
           BINARY_C_VERSION,
           __DATE__,
           __TIME__);

    /* binary_c version strings */
#ifdef BINARY_C_MAJOR_VERSION
    Show_string_macro(BINARY_C_MAJOR_VERSION);
#endif//BINARY_C_MAJOR_VERSION
#ifdef BINARY_C_MINOR_VERSION
    Show_string_macro(BINARY_C_MINOR_VERSION);
#endif//BINARY_C_MINOR_VERSION
#ifdef BINARY_C_PATCH_VERSION
    Show_string_macro(BINARY_C_PATCH_VERSION);
#endif//BINARY_C_PATCH_VERSION
#ifdef BINARY_C_PRE_VERSION
    Show_string_macro(BINARY_C_PRE_VERSION);
#endif//BINARY_C_PRE_VERSION
    
    /* SVN : deprecated from early 2018 but will do no harm if left in */
#if defined SVN_REVISION && (SVN_REVISION+0)
    Printf("SVN revision %d\n",SVN_REVISION+0);
#else
    Printf("SVN revision unknown\n");
#endif
#ifdef SVN_URL
    Printf_deslash("SVN URL \"%s\"\n",Stringof(SVN_URL));
#else
    Printf("SVN URL unknown\n");
#endif

    /* git : replaces SVN from 2018 */
#ifdef GIT_REVISION
    Printf_deslash("git revision \"%s\"\n",Stringof(GIT_REVISION));
#else
    Printf("git revision unknown\n");
#endif

#ifdef GIT_URL
    Printf_deslash("git URL %s\n",Stringof(GIT_URL));
#else
    Printf("git URL unknown\n");
#endif
#ifdef GIT_BRANCH
    Printf_deslash("git branch %s\n",Stringof(GIT_BRANCH));
#else
    Printf("git branch unknown\n");
#endif

    Macrotest(__VERSION__);
#ifdef __VERSION__
    Show_string_macro(__VERSION__);
#endif
    Macrotest(__OPTIMIZE__);
#ifdef __OPTIMIZE__
    Show_int_macro(__OPTIMIZE__);
#endif

    /* CFLAGS, CC, LD etc. are set by configure */
    Macrotest(CFLAGS);
#ifdef CFLAGS
    Show_string_macro(CFLAGS);
#endif
    Macrotest(CC);
#ifdef CC
    Show_string_macro(CC);
#endif
    Macrotest(LD);
#ifdef LD
    Show_string_macro(LD);
#endif
    Macrotest(LIBS);
#ifdef LIBS
    Show_string_macro(LIBS);
#endif
    Macrotest(INCDIRS);
#ifdef INCDIRS
    Show_string_macro(INCDIRS);
#endif
    Macrotest(LIBDIRS);
#ifdef LIBDIRS
    Show_string_macro(LIBDIRS);
#endif
    
    Macrotest(__OPTIMIZE_SIZE__);
    Macrotest(__NO_INLINE__);
    Macrotest(__GNUC__);
#ifdef __GNUC__
    Show_int_macro(__GNUC__);
    Show_int_macro(__GNUC_MINOR__);
    Show_int_macro(__GNUC_PATCHLEVEL__);
#endif
    Macrotest(__clang__);
    Macrotest(__SUNPRO_C);
    Macrotest(__INTEL_COMPILER);
    Macrotest(__STDC__);
    Macrotest(__STDC_VERSION__);
#ifdef __STDC_VERSION__
    Show_long_int_macro(__STDC_VERSION__);
#endif
    Macrotest(__STRICT_ANSI__);
    Macrotest(__ELF__);
    Macrotest(_POSIX_C_SOURCE);
#ifdef _POSIX_C_SOURCE
    Show_long_int_macro(_POSIX_C_SOURCE);
#endif	
    Macrotest(_XOPEN_SOURCE);
#ifdef _XOPEN_SOURCE
    Show_long_int_macro(_XOPEN_SOURCE);
#endif
    Macrotest(_ISOC11_SOURCE);
#ifdef _ISOC11_SOURCE
    Show_long_int_macro(_ISOC11_SOURCE);
#endif
    Macrotest(_ISOC99_SOURCE);
#ifdef _ISOC11_SOURCE
    Show_long_int_macro(_ISOC99_SOURCE);
#endif
    Macrotest(_C89);
    Macrotest(_C90);
    Macrotest(_C94);
    Macrotest(_C99);
    Macrotest(_C11);
#ifdef __BYTE_ORDER__
    Printf("Byte order is %s endian\n",
#if __BYTE_ORDER__ == __ORDER_LITTLE_ENDIAN__
    "little"
#endif
#if __BYTE_ORDER__ == __ORDER_BIG_ENDIAN__
    "big"
#endif
#if __BYTE_ORDER__ == __ORDER_PDP_ENDIAN__
    "PDP"
#endif
        );
#else
    Printf("Byte order is unknown (__BYTE_ORDER__ undefined)\n");
#endif
    Macrotest(__FLOAT_WORD_ORDER__);
#ifdef __FLOAT_WORD_ORDER__
    Printf("Float order is %s endian\n",
#if __FLOAT_WORD_ORDER__ == __ORDER_LITTLE_ENDIAN__
           "little"
#endif
#if __FLOAT_WORD_ORDER__ == __ORDER_BIG_ENDIAN__
           "big"
#endif
        );
#endif // __FLOAT_WORD_ORDER__
    Macrotest(ACML);
    Macrotest(__NO_MATH_ERRNO__,
              "(probably via -ffast-math)");
    Macrotest(__FAST_MATH__,
              "(probably via -ffast-math)");
    Macrotest(__SUPPORT_SNAN__,"(signalling NaNs)");

    Show_string_macro(FPU_PRECISION);
    Macrotest(FPU_CAPTURE_INVALID);
    Macrotest(FPU_CAPTURE_INEXACT);
    Macrotest(FPU_CAPTURE_UNDERFLOWS);
    Macrotest(FPU_CAPTURE_OVERFLOWS);
    Macrotest(FPU_CAPTURE_DIVIDE_BY_ZERO);
    Macrotest(NO_FPU_CONTROL);
    Macrotest(CHECK_ISNAN_WORKS);
    
    Show_float_macro(ZERO);
    Show_float_macro(TINY);
    Show_float_macro(VERY_TINY);
    Show_float_macro(REALLY_TINY);
    Show_float_macro(LARGE_FLOAT);
    Show_float_macro(PI);
    Show_float_macro(TWOPI);
    
    Show_string_macro(Pure_function);
    Show_string_macro(Constant_function);
    Show_string_macro(RESTRICT);
    Show_string_macro(inline);
    Show_string_macro(FASTCALL);
    Show_string_macro(Hot_function);
    Show_string_macro(Nonnull_all_arguments);
    Show_string_macro(Nonnull_some_arguments);
    Show_string_macro(Alloc_size_first);
    Show_string_macro(Alloc_size_second);
    Show_string_macro(Alloc_size_product);
    Show_string_macro(Returns_nonnull);
    Show_string_macro(Deprecated);
    Show_string_macro(Stringify_macro(Autotype));
    Show_string_macro(Stringify(Autotype));
    Macrotest(__USE_C99_BOOLEAN__);
    
    Show_string_macro(MAYBE_UNUSED);

    Macrotest(__HAVE_LIBRINTERPOLATE__);
    Macrotest(__HAVE_LIBRINTERPOLATE_BINARY_C__);
    Show_string_macro(RINTERPOLATE_VERSION);
    Show_int_macro(RINTERPOLATE_MAJOR_VERSION);
    Show_int_macro(RINTERPOLATE_MINOR_VERSION);
    Show_string_macro(GSL_VERSION);
    Show_int_macro(GSL_MAJOR_VERSION);
    Show_int_macro(GSL_MINOR_VERSION);
    
    Macrotest(USE_GSL);
    Show_string_macro(GSL_ROOT_FSOLVER);
    Show_int_macro(GSL_INTEGRATOR_WORKSPACE_SIZE);

    Show_string_macro(__int__);
    Show_string_macro(__double__);
    Show_string_macro(__long__);
    Show_string_macro(__short__);
    Show_string_macro(__unsigned__);
    Show_string_macro(__const__);
    Show_string_macro(__static__);

    Printf("Size of : short int %zu, unsigned short int %zu, int %zu, unsigned int %zu, long int %zu, unsigned long int %zu, long long int %zu, unsigned long long int %zu, size_t %zu, float %zu, double %zu, long double %zu, char %zu, Boolean %zu, stardata_t %zu, preferences_t %zu, star_t %zu, common_t %zu, model_t %zu, diffstats_t %zu, probability_distribution_t %zu, RLOF_orbit_t %zu, store_t %zu, tmpstore_t %zu, data_table_t %zu, stardata_dump_t %zu, GSL_args_t %zu, envelope_t %zu, envelope_shell_t %zu, equation_of_state_t %zu, opacity_t %zu, kick_system_t %zu, coordinate_t %zu, binary_system_t %zu, power_law_t %zu, disc_thermal_zone_t %zu, disc_loss_t %zu, disc_t %zu, mersenne_twister_t %zu, binary_c_random_buffer_t %zu, binary_c_file_t %zu, difflogitem_t %zu, difflogstack_t %zu, binary_c_fixed_timestep_t %zu, new_supernova_t %zu, splitinfo_t %zu, derivative_t %zu, bint_t %zu, orbit_t %zu, Random_seed %zu, Random_buffer %zu, FILE %zu, void* %zu, short int* %zu, unsigned short int* %zu, int* %zu, unsigned int* %zu, long int* %zu, unsigned long int* %zu, long long int* %zu, unsigned long long int* %zu, float* %zu, double* %zu, long double* %zu, char* %zu, Boolean* %zu, stardata_t* %zu, star_t* %zu, FILE* %zu, __int__ %zu, __double__ %zu, __unsigned__ __int__ %zu, __short__ __int__ %zu, __long__ __int__ %zu\n",
           sizeof(short int),
           sizeof(unsigned short int),
           sizeof(int),
           sizeof(unsigned int),
           sizeof(long int),
           sizeof(unsigned long int),
           sizeof(long long int),
           sizeof(unsigned long long int),
           sizeof(size_t),
           sizeof(float),
           sizeof(double),
           sizeof(long double),
           sizeof(char),
           sizeof(Boolean),
           sizeof(struct stardata_t),
           sizeof(struct preferences_t),
           sizeof(struct star_t),
           sizeof(struct common_t),
           sizeof(struct model_t),
           sizeof(struct diffstats_t),
           sizeof(struct probability_distribution_t),
           sizeof(struct RLOF_orbit_t),
           sizeof(struct store_t),
           sizeof(struct tmpstore_t),
           sizeof(struct data_table_t),
           sizeof(struct stardata_dump_t),
           sizeof(struct GSL_args_t),
           sizeof(struct envelope_t),
           sizeof(struct envelope_shell_t),
           sizeof(struct equation_of_state_t),
           sizeof(struct opacity_t),
           sizeof(struct kick_system_t),
           sizeof(struct coordinate_t),
#ifdef DISCS
           sizeof(struct binary_system_t),
           sizeof(struct power_law_t),
           sizeof(struct disc_thermal_zone_t),
           sizeof(struct disc_loss_t),
           sizeof(struct disc_t),
#else
           (size_t)0,
           (size_t)0,
           (size_t)0,
           (size_t)0,
           (size_t)0,
#endif//DISCS
#ifdef USE_MERSENNE_TWISTER
           sizeof(struct mersenne_twister_t),
#else
           (size_t)0,
#endif//USE_MERSENNE_TWISTER
           sizeof(struct binary_c_random_buffer_t),
           sizeof(struct binary_c_file_t),
           sizeof(struct difflogitem_t),
           sizeof(struct difflogstack_t), 
           sizeof(struct binary_c_fixed_timestep_t),
           sizeof(struct new_supernova_t),
#ifdef EVOLUTION_SPLITTING
           sizeof(struct splitinfo_t),
#else
           (size_t)0,
#endif//EVOLUTION_SPLITTING
           sizeof(struct derivative_t),
#ifdef BINT
           sizeof(struct bint_t),
#else
           (size_t)0,
#endif//BINT
           sizeof(struct orbit_t),
           sizeof(Random_seed),
           sizeof(Random_buffer),
           sizeof(FILE),
           sizeof(void *),
           sizeof(short int *),
           sizeof(unsigned short int *),
           sizeof(int *),
           sizeof(unsigned int *),
           sizeof(long int *),
           sizeof(unsigned long int *),
           sizeof(long long int *),
           sizeof(unsigned long long int *),
           sizeof(float *),
           sizeof(double *),
           sizeof(long double *),
           sizeof(char *),
           sizeof(Boolean *),
           sizeof(struct stardata_t *),
           sizeof(struct star_t *),
           sizeof(FILE *),
           sizeof(__int__),
           sizeof(__double__),
           sizeof(__unsigned__ __int__),
           sizeof(__short__ __int__),
           sizeof(__long__ __int__)
        );


        
    Show_string_macro(stardata_t);
    Show_string_macro(star_t);
    Show_string_macro(preferences_t);
    Show_string_macro(store_t);
    Show_string_macro(tmpstore_t);
    Show_string_macro(model_t);
    Show_string_macro(common_t);
    Show_string_macro(printf);
    Show_string_macro(_printf);

#ifdef EVOLUTION_SPLITTING
    Printf(", splitinfo_t %zu",
           sizeof(struct splitinfo_t));
#endif
#ifdef DISCS
    Printf(", disc_t %zu, disc_thermal_zone_t %zu, power_law_t %zu",
           sizeof(struct disc_t),
           sizeof(struct disc_thermal_zone_t),
           sizeof(struct power_law_t)
        );
#endif
    Printf("\n");

    /*
     * Speed tests
     */
#if defined TIMER && defined CPUFREQ && defined millisecond
    if(stardata!=NULL && stardata->preferences->speedtests==1)
    {
        double * p = New_stardata;
        double * q = New_stardata;

#define Swap_stardatas_with_copy(STARDATA1,STARDATA2)   \
        {                                               \
            struct stardata_t * _s = New_stardata;      \
            copy_stardata((const struct stardata_t*)(STARDATA2),(struct stardata_t*)_s,COPY_STARDATA_PREVIOUS_NONE); \
            copy_stardata((const struct stardata_t*)(STARDATA1),(struct stardata_t*)(STARDATA2),COPY_STARDATA_PREVIOUS_NONE); \
            copy_stardata((const struct stardata_t*)_s,(struct stardata_t*)(STARDATA1),COPY_STARDATA_PREVIOUS_NONE); \
            Safe_free(_s);                              \
        }
#define Swap_stardatas_with_move(STARDATA1,STARDATA2)           \
        {                                                       \
            struct stardata_t * _s = New_stardata;              \
            struct stardata_t * _dummy_s MAYBE_UNUSED;          \
            _dummy_s = Move_stardata((STARDATA2),_s);           \
            _dummy_s = Move_stardata((STARDATA1),(STARDATA2));  \
            _dummy_s = Move_stardata(_s,(STARDATA1));           \
            Safe_free(_s);                                      \
        }

        /*
         * nms is the number of milliseconds for which
         * each speed test is carried out.
         * 
         * This defaults to 100ms, but can be set using the
         * NMS environment variable.
         */
        long int nms;
        char * nmsstring = getenv("NMS");
        nms = nmsstring==NULL ? 100 : (strtol(nmsstring,NULL,10));
        Printf("Speed tests for %ld milliseconds\n",nms);
        fflush(stdout);

            /*
             * Time function call using counter.
             * Usage:
             * Timeme(n,{...code block...});
             */            
#define Timeme(counter,...)                                             \
            {                                                           \
                ticks t1 = getticks();                                  \
                if(t1!=0)                                               \
                {                                                       \
                    t1 += (ticks)(nms * millisecond);                   \
                    while(getticks()<t1)                                \
                    {                                                   \
                        {                                               \
                            __VA_ARGS__;                                \
                        }                                               \
                        (counter)++;                                    \
                    }                                                   \
                }                                                       \
            }
#define Speed_increase_pc(n1,n2)                        \
        100.0*(1.0/n1 - 1.0/n2)/(1.0/Max(n1,n2))

        /*
         * compare speed of memcpy to memmove
         */
        {
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wunused-parameter"
            long int copy = 0, move = 0;
            Timeme(copy,{Swap_stardatas_with_copy(p,q)});
            Timeme(move,{Swap_stardatas_with_move(p,q)});
            Printf("Speed of memcpy/speed of memmove = %g\n",
                   (double)move/(double)copy);
#pragma GCC diagnostic pop
        }

        /*
         * Compare speed of strcat, strlcat, sprintf, asprintf
         */
#undef strcat
#undef strncat
#undef sprintf
        {
#define Random_letter  ("ABCDEFGHIJKLMNOPQRSTUVWXYZ"[random () % 26])
#define Reset_strings_12                        \
            {                                   \
                for(i=0;i<n1;i++)               \
                {                               \
                    s1[i] = Random_letter;      \
                    s2[i] = Random_letter;      \
                }                               \
                s1[n1-2] = '\0';                \
                s2[n1-2] = '\0';                \
            }
#define Reset_strings_3                         \
            {                                   \
                for(i=0;i<n1;i++)               \
                {                               \
                    s3[i] = Random_letter;      \
                }                               \
                s3[n1-2] = '\0';                \
            }

#define Alloc_strings_12                        \
            s1 = Calloc(n1+1,sizeof(char));     \
            s2 = Calloc(n1+1,sizeof(char));
#define Alloc_strings_3                         \
            s3 = Calloc((n1+1)*2,sizeof(char));
            
#define Free_strings                            \
            Safe_free(s1);                      \
            Safe_free(s2);                      \
            Safe_free(s3);
            
#define String_Timeme(_alloc,_reset,counter,...)        \
            {                                           \
                Timeme(counter,                         \
                       {                                \
                           char * s1;                   \
                           char * s2;                   \
                           char * MAYBE_UNUSED s3;      \
                           Alloc_strings_12;            \
                           if(_alloc)                   \
                           {                            \
                               Alloc_strings_3;         \
                           }                            \
                           if(_reset)                   \
                           {                            \
                               Reset_strings_12;        \
                               if(_alloc)               \
                               {                        \
                                   Reset_strings_3;     \
                               }                        \
                           }                            \
                           __VA_ARGS__;                 \
                           Free_strings;                \
                       }                                \
                    );                                  \
            }

            long int _overhead, _strcat, _strncat, _strlcat;
            long int _sprintf, _snprintf, _asprintf;
            _overhead = _strcat = _strlcat = _strncat =
                _sprintf = _snprintf = _asprintf = 0;
            const size_t n1 = 1000;
            unsigned int i;
            String_Timeme(TRUE,TRUE,_overhead,{
                    /* do nothing */
                });
            String_Timeme(TRUE,TRUE,_strcat,{
                    strcat(s3,s1);
                });
            String_Timeme(TRUE,TRUE,_strncat,{
                    strncat(s3,s1,n1*2-1);
                });
            String_Timeme(TRUE,TRUE,_strlcat,{
                    strlcat(s3,s1,n1*2);
                });
            String_Timeme(TRUE,TRUE,_sprintf,{
                    sprintf(s3,"%s%s",s1,s2);
                });
            String_Timeme(TRUE,TRUE,_snprintf,{
                    snprintf(s3,n1*2-1,"%s%s",s1,s2);
                });
            
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wunused-result"
            String_Timeme(FALSE,TRUE,_asprintf,{
                    ((void)asprintf(&s3,"%s%s",s1,s2));
                });
#pragma GCC diagnostic pop

            Printf("String concatenation speed (more is better, n1 = %zu, overhead = %ld) : strcat %ld : strncat %ld : strlcat %ld : sprintf %ld : snprintf %ld : asprintf %ld\n",
                   n1,
                   _overhead,
                   _strcat - 1*_overhead,
                   _strncat - 1*_overhead,
                   _strlcat - 1*_overhead,
                   _sprintf - 1*_overhead,
                   _snprintf - 1*_overhead,
                   _asprintf - 1*_overhead);
        }
        
        /*
         * Compare native memcpy to other versions of memcpy 
         * when copying stardatas
         */
        {
            /*
             * Define the length of the test in microseconds.
             *
             * Set the environment variable BINARY_C_SPEEDTEST_SECONDS
             * or default to 1 second.
             */
            
#define TIME_LEN (1e6 * (float)__test_seconds())

            /*
             * If we have libbsd, use it to set random data,
             * otherwise use /dev/urandom or set to zero.
             */
#ifdef __HAVE_LIBBSD__
#define Reset_buffers                           \
            arc4random_buf(s1, size);           \
            memcpy(s2,s1,size);
#else
#define Reset_buffers                                                   \
            {                                                           \
                if(access("/dev/urandom",F_OK))                         \
                {                                                       \
                    int __rnd = open("/dev/urandom",O_RDONLY);          \
                    if(read(__rnd,s1,size)&&                            \
                       read(__rnd,s2,size)){}                           \
                    close(__rnd);                                       \
                }                                                       \
                else                                                    \
                {                                                       \
                    memset(s1,0,size);                                  \
                    memset(s2,0,size);                                  \
                }                                                       \
            }
#endif // __HAVE_LIBBSD__
            
            const size_t size = sizeof(struct stardata_t);
            const size_t malloc_size = size + 10 * sizeof(int);
            struct stardata_t * s1 = Malloc(malloc_size);
            struct stardata_t * s2 = Malloc(malloc_size);
            size_t i;
            printf("sizeof stardata %zu\n",size);
            Reset_buffers;
            
            /*
             * Time native memcpy
             */
            i = 0;
            ticks start_tick = getticks();
            while(Timer_microseconds(getticks()-start_tick) < TIME_LEN)
            {
                memcpy(s2,s1,size);
                i++;
            }
            Printf("Speed of native memcpy %g GiByte/s\n",
                   sizeof(struct stardata_t)*((float)i)/( GIGABYTE * (TIME_LEN * 1e-6) ));
            fflush(NULL);
            if(!Memory_equal(s1,s2,size))
            {
                Exit_binary_c(1,
                              "s1 and s2 differ :( memcpy failed\n");       
            }
            
            /*
             * Time strided memcpy
             *
             * First, the two areas with identical data
             */
            i = 0;
            Reset_buffers;
            const size_t blocksize = 2048*sizeof(char);
            start_tick = getticks();
            while(Timer_microseconds(getticks()-start_tick) < TIME_LEN)
            {
                strided_memcpy(s2,s1,size,blocksize);
                i++;
            }
            Printf("Speed of strided_memcpy (blocksize = %zu, all same) %g GiByte/s\n",
                   blocksize,
                   sizeof(struct stardata_t)*((float)i)/( GIGABYTE * (TIME_LEN * 1e-6) ));
            fflush(NULL);
            if(!Memory_equal(s1,s2,size))
            {
                Exit_binary_c(1,
                              "s1 and s2 differ :( memcpy failed\n");       
            }
           
            /*
             * Now the two areas with 10% difference
             */
            Reset_buffers;
            int * s1int = (int*) s1;
            int j = 1;
            for(i=0; i<size; i+=size/10)
            {
                *( s1int + i/sizeof(int) ) = j;
            }
            i = 0;
            start_tick = getticks();
            while(Timer_microseconds(getticks()-start_tick) < TIME_LEN)
            {
                strided_memcpy(s2,s1,size,blocksize);
                i++;
            }
            Printf("Speed of strided_memcpy (blocksize = %zu, 10%% different) %g GiByte/s\n",
                   blocksize,
                   sizeof(struct stardata_t)*((float)i)/( GIGABYTE * (TIME_LEN * 1e-6) ));
            fflush(NULL);
            if(!Memory_equal(s1,s2,size))
            {
                Exit_binary_c(1,
                              "s1 and s2 differ :( memcpy failed\n");       
            }
           
            
            Reset_buffers;
            j = 1;
            for(i=0; i<size; i+=size/5)
            {
                *( s1int + i/sizeof(int) ) = j;
            }
            i = 0;
            start_tick = getticks();
            while(Timer_microseconds(getticks()-start_tick) < TIME_LEN)
            {
                strided_memcpy(s2,s1,size,blocksize);
                i++;
            }
            Printf("Speed of strided_memcpy (blocksize = %zu, 20%% different) %g GiByte/s\n",
                   blocksize,
                   sizeof(struct stardata_t)*((float)i)/( GIGABYTE * (TIME_LEN * 1e-6) ));
            fflush(NULL);
         
            if(!Memory_equal(s1,s2,size))
            {
                Exit_binary_c(1,
                              "s1 and s2 differ :( memcpy failed\n");       
            }
           
            /*
             * 100% difference
             */
            Reset_buffers;
            j = 1;
            for(i=0; i<size; i++)
            {
                *( s1int + i/sizeof(int) ) = j;
            }
            i = 0;
            start_tick = getticks();
            while(Timer_microseconds(getticks()-start_tick) < TIME_LEN)
            {
                strided_memcpy(s2,s1,size,blocksize);
                i++;
            }
            Printf("Speed of strided_memcpy (blocksize = %zu, 100%% different) %g GiByte/s\n",
                   blocksize,
                   sizeof(struct stardata_t)*((float)i)/( GIGABYTE * (TIME_LEN * 1e-6) ));
            fflush(NULL);
            if(!Memory_equal(s1,s2,size))
            {
                Exit_binary_c(1,
                              "s1 and s2 differ :( memcpy failed\n");       
            }
           
            /*
             * Now the two areas with 10% difference, but different blocksizes
             */
            i = 0;
            Boolean last = FALSE;
            size_t blksize = sizeof(char);
            while(blksize <= size && last == FALSE)
            {
                Reset_buffers;
                j = 1;
                for(i=0; i<size; i+=size/10)
                {
                    *( s1int + i/sizeof(int) ) = j;
                }

                i = 0;
                if(size==blksize) last = TRUE;
                start_tick = getticks();
                while(Timer_microseconds(getticks()-start_tick) < TIME_LEN)
                {
                    strided_memcpy(s2,s1,size,blocksize);
                    i++;
                }
                Printf("Speed of strided_memcpy (blocksize = %zu, 10%% different) %g GiByte/s\n",
                       blksize,
                       sizeof(struct stardata_t)*((float)i)/( GIGABYTE * (TIME_LEN * 1e-6) ));
                fflush(NULL);
                if(!Memory_equal(s1,s2,size))
                {
                    Exit_binary_c(1,
                                  "s1 and s2 differ :( memcpy failed\n");       
                }
           
                blksize *= 2;
                blksize = Min(size,blksize);
            }

            
            /*
             * Free memory
             */
            Safe_free(s1);
            Safe_free(s2);
            Safe_free(p);
            Safe_free(q);
        }

        
        /*
         * Compare pow to fastPow
         */
        {
            int n=1000;
            double lnx,dlnx = (log(DBL_MAX)-log(DBL_MIN))/(n-1);
            for(lnx=log(DBL_MIN);
                lnx<log(DBL_MAX);
                lnx += dlnx)
            {
                double y = 10 * random_number(stardata,NULL);
                double x = exp(lnx);
                double _pow = pow(x,y);
                double _fastpow = fastPow(x,y);
                Printf("POWCHECK %g %g %g %g %g \n",
                       x,
                       y,
                       _pow,
                       _fastpow,
                       Is_really_not_zero(_fastpow) ? (_pow / _fastpow) : 1.0);
            }
        }

        /*
         * compare multiplication to division
         */
        {
            double x MAYBE_UNUSED;
            const double y=3.346182736,iy=1.0/y;
            long int mult = 0, div = 0;
            Timeme(mult,{x=random_number(stardata,NULL);x*=iy;});
            Timeme(div, {x=random_number(stardata,NULL);x/=y;});
            Printf("Speed of mult/div = %g\n",(double)mult/(double)div);
        }
        
        
        {
            /*
             * Compare log vs log10 operations
             */
            double x MAYBE_UNUSED;
            long int __log = 0, __log10 = 0;
            Timeme(__log,{x=log(random_number(stardata,NULL));});
            Timeme(__log10,{x=log10(random_number(stardata,NULL));});
            Printf("Speed of log/speed of log10 = %g\n",
                   (double)__log/(double)__log10);
        }

        
        {
            /*
             * Compare Pow2,Pow3,... to pow(x,2), pow(x,3), ...
             */



#define powtest(n,...)                                                  \
            {                                                           \
                nPOW=npow=0;                                            \
                Timeme(npow,                                            \
                       {                                                \
                           double __y = random_number(stardata,NULL);   \
                           double __x MAYBE_UNUSED =                    \
                               pow(__y,(double)n);                      \
                       }                                                \
                    );                                                  \
                                                                        \
                Timeme(nPOW,                                            \
                       {                                                \
                           double __y = random_number(stardata,NULL);   \
                           double __x MAYBE_UNUSED =                    \
                               Pow##n(__y);                             \
                       }                                                \
                    );                                                  \
                printf("Speed of POW%d %ld vs pow(x,%d) %ld : speed increase %5.2f %%\n", \
                       n,                                               \
                       nPOW,                                            \
                       n,                                               \
                       npow,                                            \
                       Speed_increase_pc(npow,nPOW)                     \
                    );                                                  \
            }
            
            long int nPOW=0,npow=0;
            powtest(2);
            powtest(3);
            powtest(4);
            powtest(5);
            powtest(6);
            powtest(7);
            powtest(8);
            powtest(9);
        }

    }
#endif //  TIMER && CPUFREQ && millisecond

    
    {
        struct rlimit limit;
        getrlimit (RLIMIT_STACK, &limit);
        Printf("Stack limit current = %ju, max = %ju (-1 means no limit)\n",
	       (uintmax_t)limit.rlim_cur,
	       (uintmax_t)limit.rlim_max);
    }
    Macrotest(SEGFAULTS);
    Macrotest(CATCH_SIGVTALRM);
    Show_string_macro(OPERATING_SYSTEM);
    Macrotest(LINUX);
    Macrotest(WINDOWS);
    Macrotest(DARWIN);
    Macrotest(UNKNOWN_OPERATING_SYSTEM);
    Macrotest(__APPLE__);
    Macrotest(__arm__);
    Macrotest(__i386__);
#ifdef ALIGNSIZE
    Printf("Memory is aligned to %d bytes (set in ALIGNSIZE)\n",ALIGNSIZE);
#else
    Printf("ALIGNSIZE is not defined: malloc calls are not aligned\n");
#endif 
    Show_string_macro(Builtin_aligned);
    Show_string_macro(Assume_aligned);
    Macrotest(__NATIVE_MALLOC__);
    Macrotest(ALLOC_CHECKS);
    Macrotest(HAVE_MALLOC_USABLE_SIZE);
    Macrotest(CHECK_MEMCPY_ALIGNMENT);
    Show_string_macro(Malloc_usable_size);
    Macrotest(USE_SKYWIND_FAST_MEMCPY);
    Macrotest(USE_APEX_MEMMOVE);
    Macrotest(USE_ASMLIB);
    Macrotest(TARGET_STACK_PROTECT_RUNTIME_ENABLED_P);
    Show_string_macro(STACK_SIZE);
    Macrotest(STACK_ACTION);
    Show_string_macro(STACK_ACTION);
    Macrotest(STACK_CHECK_BUILTIN);
    Macrotest(STACK_CHECK_STATIC_BUILTIN);
    Macrotest(STACK_CHECKS);
    Macrotest(BAN_UNSAFE_FUNCTIONS);
    Macrotest(MEMOIZE);
    Macrotest(__HAVE_LIBMEMOIZE__);
    Show_string_macro(MEMOIZE_VERSION);
    Show_string_macro(MEMOIZE_HEADER_FILE);
    if((int)sizeof(long int)==4)
    {
        Printf("Compiled for 32 bit architecture\n");
    }
    else if((int)sizeof(long int)==8)
    {
        Printf("Compiled for 64 bit architecture\n");
    }
    else
    {
        Printf("Compiled for an architecture with an unknown number of bits\n");
    }
    Macrotest(TIMER);
    Macrotest(TIMER_USE_ASM);
    Macrotest(TIMER_PER_SYSTEM);
    Show_int_macro(TIMEOUT_SECONDS);
    Show_int_macro(TIMEOUT_SECONDS_WITH_VALGRIND);
    Macrotest(CPUFREQ);
    Show_int_macro(CPUFREQ);
    Macrotest(CLOCKS_PER_SEC);
    Show_int_macro(CLOCKS_PER_SEC);
    Printf("_SC_CLK_TCK = %g\n",(double)sysconf (_SC_CLK_TCK));
    
    Macrotest(millisecond);
    Show_float_macro(millisecond);
    
    /* end of compiler/environment macros */
    /************************************************************/
    /* now come binary_c macros */

    Macrotest(BINARY_C_API);
    Macrotest(BINARY_C_API_FORTRAN);
    Show_string_macro(binary_c_API_function);
    Show_int_macro(STARDATA_DUMP_FORMAT);
    Macrotest(DEBUG);
#ifdef DEBUG
    Show_int_macro(DEBUG);
#endif
    Macrotest(DEBUG_CALLER);
    Macrotest(DEBUG_CALLER_LINE);
    Macrotest(DEBUG_REMOVE_NAN_FROM_CALLER);
    Macrotest(DEBUG_REMOVE_NAN_FROM_FILESNAMES);
    Show_string_macro(Debug_expression);
#ifdef Debug_stop_expression
    Show_string_macro(Debug_stop_expression);
#endif
    Macrotest(ALPHA_ARGS);
    Macrotest(USE_ANSI_COLOURS);
    Macrotest(USE_POINTER_LOOPS);

    Macrotest(__NATIVE_PRINTF__);
    Macrotest(__NATIVE_EXIT__);
    Macrotest(MAX_EXIT_STATEMENT_PRINT_SIZE);
#ifdef MAX_EXIT_STATEMENT_PRINT_SIZE
    Show_int_macro(MAX_EXIT_STATEMENT_PRINT_SIZE);
#endif
    Macrotest(FLUSH_LOG);
    Macrotest(__HAVE_VALGRIND__);
    Macrotest(__HAVE_LIBGSL__);
    Macrotest(__HAVE_LIBBFD__);
    Macrotest(__HAVE_LIBBSD__);
    Macrotest(__HAVE_LIBIBERTY__);
    Macrotest(__HAVE_LIBIBERTYH__);
    Macrotest(__HAVE_LIBIBERTY_LIBIBERTYH__);
    Macrotest(__HAVE_LIBBACKTRACE__);
    Macrotest(__HAVE_MALLOC_H__);
    Macrotest(__HAVE_DRAND48__);
    Macrotest(__HAVE_SETITIMER__);
    Macrotest(__HAVE_PKG_CONFIG__);
    Macrotest(BACKTRACE);
#ifdef BACKTRACE
    Show_int_macro(BACKTRACE_METHOD);
    Show_string_macro(Backtrace);
    Show_string_macro(BACKTRACE_GNU_BUFFER_STREAM);
#endif
    Macrotest(__STRLCPY_IS_MACRO__);
    Macrotest(__STRLCAT_IS_BSD_COPY__);
    Macrotest(DISABLE_DPRINT);
    Macrotest(DEBUG_LINENUMBERS);
    Macrotest(DEBUG_SHOW_FILENAMES);
    Macrotest(DEBUG_FAIL_ON_NAN);

    Printf("Compiled in parameters:\n");
    Show_int_macro(NUMBER_OF_STARS);
    Show_int_macro(TRUE);
    Show_int_macro(FALSE);
    Macrotestif(CFAMANDA,
                "We use values to compare to Amanda's models\n");
    Macrotest(BATCHMODE);
    Macrotest(NANCHECKS);
    Macrotest(USE_NATIVE_ISNAN);
    Macrotest(USE_2011_MASS_LOSS);
    Macrotest(USE_2012_MAIN_SEQUENCE_LIFETIMES_TABLE);
    Macrotest(MODULATE_TIMESTEP_TO_RESOLVE_ROTATION_MASS_LOSS);
    Show_float_macro(TIDAL_STRENGTH_FACTOR_DEFAULT);
    Macrotest(EXPONENTIAL_TIDES);
    Show_float_macro(MAGNETIC_BRAKING_GAMMA_DEFAULT);
    Show_float_macro(BETA);
    Show_float_macro(SPHERICAL_ANGMOM_ACCRETION_FACTOR);
    Show_float_macro(MINIMUM_STELLAR_ANGMOM);
    Show_float_macro(MAXIMUM_STELLAR_ANGMOM);
    Show_float_macro(MINIMUM_ORBITAL_ANGMOM);
    Show_float_macro(MAXIMUM_ORBITAL_ANGMOM);
    Show_float_macro(UNDEFINED_ORBITAL_ANGULAR_FREQUENCY);
    Show_string_macro(ANGULAR_MOMENTUM_CGS);
    Show_float_macro(ANGULAR_MOMENTUM_CGS);
    Show_int_macro(BURN_IN_TIMESTEPS);
    Macrotest(TRIPLE);
    Macrotest(REVERSE_TIME);
    
    Show_float_macro(NGTV);
    Show_float_macro(NGTV2);
    Show_string_macro(Total_mass);
    Show_float_macro(K2);
    Show_float_macro(K3);
    Show_float_macro(MINIMUM_SEPARATION_TO_BE_CALLED_BINARY);
    Show_float_macro(MAXIMUM_SEPARATION_TO_BE_CALLED_BINARY);
    Show_string_macro(System_is_single);
    Show_string_macro(System_is_binary);
    Show_float_macro(MIN_OBSERVATIONAL_STELLAR_MASS);
    Show_float_macro(MAX_OBSERVATIONAL_BINARY_ORBITAL_PERIOD);
    Show_float_macro(OBSERVABLE_RADIAL_VELOCITY_MIN_DEFAULT);
    Show_string_macro(Observable_binary);
    Show_string_macro(Angular_momentum_from_stardata);
    Show_float_macro(TINY_ECCENTRICITY);


    Printf("TIDES is deprecated! (always on)\n");
    Macrotest(FILE_LOG);
    Macrotest(FILE_LOG_REWIND);
    Macrotest(FILE_LOG_FLUSH);
    Macrotest(HRDIAG);
#ifdef HRDIAG
    Printf("HRDIAG logging : start time %g, time bin size=%g, bin sizes Teff=%g logL=%g logg=%g\n",
           HRDIAG_START_LOG,
           HRDIAG_BIN_SIZE_LOGTIME,
           HRDIAG_BIN_SIZE_TEFF,
           HRDIAG_BIN_SIZE_LOGL,
           HRDIAG_BIN_SIZE_LOGG
        );
#endif

    
    Show_float_macro(MIN_SEP_FOR_CLOSE_ANG_MOM_LOSS1);
    Show_float_macro(MIN_SEP_FOR_CLOSE_ANG_MOM_LOSS2);
    Show_float_macro(MIN_SEP_FOR_CLOSE_ANG_MOM_LOSS);
    Show_float_macro(NOVA_RETENTION_FRACTION_DEFAULT);
    Show_float_macro(ACCRETION_RATE_NOVAE_UPPER_LIMIT_HYDROGEN_DONOR_DEFAULT);
    Show_float_macro(ACCRETION_RATE_NEW_GIANT_ENVELOPE_LOWER_LIMIT_HYDROGEN_DONOR_DEFAULT);
    Show_float_macro(ACCRETION_RATE_NOVAE_UPPER_LIMIT_HELIUM_DONOR_DEFAULT);
    Show_float_macro(ACCRETION_RATE_NEW_GIANT_ENVELOPE_LOWER_LIMIT_HELIUM_DONOR_DEFAULT);
    Show_float_macro(ACCRETION_RATE_NOVAE_UPPER_LIMIT_OTHER_DONOR_DEFAULT);
    Show_float_macro(ACCRETION_RATE_NEW_GIANT_ENVELOPE_LOWER_LIMIT_OTHER_DONOR_DEFAULT);

    // RLOF-specific options
    Show_float_macro(NONCONSERVATIVE_ANGMOM_GAMMA_DEFAULT);
    Show_float_macro(RLOF_ENTRY_THRESHOLD);
    Show_float_macro(RLOF_OVERFLOW_THRESHOLD);
    Show_float_macro(RLOF_STABILITY_FACTOR);
    Show_float_macro(ADAPTIVE_RLOF_TARGET_FACTOR);
#ifdef ADAPTIVE_RLOF
    {
        int i=1;
        Printf("ADAPTIVE_RLOF ADAPTIVE_RLOF_IMIN=%d ADAPTIVE_RLOF_IMAX=%d ADAPTIVE_RLOF_ALPHA=%g ADAPTIVE_RLOF_MINIMUM_RADIUS_ERROR=%g ADAPTIVE_RLOF_MAX_ENV_FRAC=%g ",ADAPTIVE_RLOF_IMIN,ADAPTIVE_RLOF_IMAX,ADAPTIVE_RLOF_ALPHA,ADAPTIVE_RLOF_MINIMUM_RADIUS_ERROR,ADAPTIVE_RLOF_MAX_ENV_FRAC);
    }
    {
        double mass=10.0;
        Printf("[ADAPTIVE_RLOF_DM_MAX=%g for M=10 and ",ADAPTIVE_RLOF_DM_MAX);
        mass=20.0; 
        Printf("%g for M=20] ",ADAPTIVE_RLOF_DM_MAX);
    }
    Macrotest(ADAPTIVE_RLOF_CONVECTIVE_CHECK);
    Macrotest(ADAPTIVE_RLOF_LOG);
    Macrotest(RLOF_REDUCE_TIMESTEP);
    Macrotest(WTTS_LOG);
#ifdef RLOF_REDUCE_TIMESTEP
    Show_float_macro(RLOF_REDUCE_TIMESTEP_AGGRESSION);
    Show_float_macro(RLOF_REDUCE_TIMESTEP_MINIMUM_FACTOR);
    Show_float_macro(RLOF_REDUCE_TIMESTEP_THRESHOLD);
#endif
    Macrotest(SLOW_DOWN_PREROCHE_EAGB);
    Macrotest(SLOW_DOWN_PREROCHE_TPAGB);
#ifdef RLOF_STABILITY_FACTOR
    Show_float_macro(RLOF_STABILITY_FACTOR);
#endif
#ifdef RLOF_MINIMUM_TIMESTEP
    Show_float_macro(RLOF_MINIMUM_TIMESTEP);
#endif
#ifdef RLOF_MINIMUM_SPEEDUP_FACTOR
    Show_float_macro(RLOF_MINIMUM_SPEEDUP_FACTOR);
#endif
    Macrotest(ADAPTIVE_RLOF_CALC_TIMESCALES_EVERY_TIME);
    Macrotest(RLOF_MDOT_MODULATION);
    Macrotest(RLOF_NO_STELLAR_TYPE_CHANGE_IF_DTM_NEGATIVE);
    Macrotest(LIMIT_RLOF_ACCRETION_TO_ACCRETORS_THERMAL_RATE);
    Macrotest(RLOF_RADIATION_CORRECTION);
    Macrotest(STELLAR_TIMESCALES_SANITY);
    Macrotest(STELLAR_TIMESCALES_CACHE);
#ifdef RLOF_RADIATION_CORRECTION_F_DEFAULT
    Show_float_macro(RLOF_RADIATION_CORRECTION_F_DEFAULT);
#endif
#endif // ADAPTIVE_RLOF
    Show_float_macro(CHEN_HAN_FIRST_TIMESTEP_BETA);
    
    Macrotest(PATHOLOGICAL_RLOF_CHECK);
    Show_float_macro(MERGER_ANGULAR_MOMENTUM_FACTOR_DEFAULT);
    Show_int_macro(CEFLAG,"(spin-energy correction in common envelope)");
    Macrotest(DISCS);
#ifdef DISCS
    Show_int_macro(DISCS_MAX_N_ZONES);
    Show_int_macro(DISCS_NUMBER_OF_POWER_LAWS);
    Show_int_macro(NDISCS);
    Show_int_macro(DISC_DEBUG);
    Show_float_macro(DISC_MAX_LIFETIME);
    Show_float_macro(DISC_MIN_TIMESTEP);
    Show_float_macro(DISC_MAX_TIMESTEP);
    Show_float_macro(DISC_STRIP_FRAC_M_INNER);
    Show_float_macro(DISC_STRIP_FRAC_J_INNER);
    Show_float_macro(DISC_STRIP_FRAC_M_OUTER);
    Show_float_macro(DISC_STRIP_FRAC_J_OUTER);
    Show_int_macro(DISC_LOSS_ISM_ALGORITHM);
    Show_float_macro(DISC_MINIMUM_TORQUE);
    Show_float_macro(DISC_MAXIMUM_TORQUE);
    Show_int_macro(DISC_ADAM_RESOLUTION);
    Show_int_macro(DISC_ROOT_FINDER_MAX_ATTEMPTS);
    Show_float_macro(DISC_MINIMUM_DISC_F);
    Show_float_macro(DISC_MINIMUM_DISC_MASS_MSUN);
    Show_float_macro(DISC_RIN_MIN);
    Show_float_macro(DISC_ROUT_MAX);
    Show_float_macro(DISC_TVISC0_MIN);
    Show_float_macro(DISC_TVISC0_MAX);
    Show_float_macro(DISC_LARGE_RADIUS);
    Show_float_macro(DISC_LONG_TIME);
    Show_float_macro(DISC_LARGE_MASS);
    Show_float_macro(DISC_MINIMUM_WIDTH);
    Show_float_macro(DISC_BISECT_RM_TOLERANCE);
    Show_int_macro(DISC_BISECT_RM_ATTEMPTS);
    Show_float_macro(DISC_BISECT_RJ_TOLERANCE);
    Show_int_macro(DISC_BISECT_RJ_ATTEMPTS);
    Show_float_macro(DISC_PRESSURE_RADIUS_TOLERANCE);
    Show_float_macro(DISC_EDGE_EPS);
    Show_float_macro(DISC_TOLERANCE);
    Show_float_macro(DISC_MASS_TOLERANCE);
    Show_int_macro(DISC_BISECTION_MASS_MAX_ITERATIONS);
    Show_float_macro(DISC_ANGMOM_TOLERANCE);
    Show_int_macro(DISC_BISECTION_ANGMOM_MAX_ITERATIONS);
    Show_float_macro(DISC_ANGMOM_FLUX_TOLERANCE);
    Show_int_macro(DISC_BISECTION_ANGMOM_FLUX_MAX_ITERATIONS);
    Show_float_macro(DISC_TORQUEF_TOLERANCE);
    Show_int_macro(DISC_BISECTION_TORQUEF_MAX_ITERATIONS);
    Show_float_macro(DISC_TIMESTEP_LOW_FACTOR);
    Show_float_macro(DISC_TIMESTEP_HIGH_FACTOR);
    Show_float_macro(DISC_MINIMUM_EVAPORATION_TIMESCALE);
    Macrotest(DISC_MONTE_CARLO_GUESSES_USE_LOG);
    Show_int_macro(DISC_BISECT_USELOG);
    Show_int_macro(DISC_BISECT_OWEN_RADIUS_USELOG);
    Show_int_macro(DISC_BISECT_RJ_USELOG);
    Show_int_macro(DISC_BISECT_RM_USELOG);
    Show_int_macro(DISC_BISECT_PRESSURE_RADIUS_USELOG);
    Show_int_macro(DISC_BISECT_VISCOUS_USELOG);

#endif
    Macrotest(DISCS_CIRCUMBINARY_FROM_COMENV);
    Macrotest(DISCS_COMENV_APPEND);
    Macrotest(CBDISC_ECCENTRICITY_PUMPING);
    Macrotest(DISC_EVAPORATE_ON_FAILURE);
    
    Macrotest(RESOLVE_POSTAGB);
    Macrotest(DISC_LOG_POPSYN);
    Macrotest(DISC_LOG);
    Macrotest(DISC_LOG_2D);
    Macrotest(DISC_SAVE_EPSILONS);
    Macrotest(DISC_RESIDUAL_WARNING);
#ifdef DISC_RESIDUAL_WARNING
    Show_float_macro(DISC_RESIDUAL_WARNING_THRESHOLD);
#endif
    Show_float_macro(SN_SIGMA_DEFAULT);
    Macrotest(EVOLUTION_SPLITTING);
    Show_float_macro(WD_SIGMA_DEFAULT);
    Show_float_macro(REIMERS_ETA_DEFAULT);
    Macrotest(VW93_MIRA_SHIFT);
    Macrotest(VW93_MULTIPLIER);
    Show_float_macro(CRAP_PARAMETER_DEFAULT,"(Enhanced RLOF wind loss)");
    Show_float_macro(LUM0);
    Show_float_macro(KAP);
    Show_float_macro(AHE);
    Show_float_macro(ACO);
    Show_float_macro(TAUMIN);
    Show_float_macro(MLP);
    Show_float_macro(BONDI_HOYLE_CONVERSION_FACTOR);
    Show_float_macro(BONDI_HOYLE_ACCRETION_FACTOR_DEFAULT);
    Show_float_macro(DTFAC_DEFAULT);
    Show_float_macro(MINIMUM_TIMESTEP_DEFAULT);
    Show_float_macro(MAXIMUM_STELLAR_MASS);
    Macrotest(MINIMUM_STELLAR_MASS);
#ifdef MINIMUM_STELLAR_MASS
    Show_float_macro(MINIMUM_STELLAR_MASS);
#endif
    Macrotest(TRIM_SUPERMASSIVE_STARS);
    Show_float_macro(BINARY_C_MINIMUM_STELLAR_MASS);
    Show_float_macro(BINARY_C_MAXIMUM_STELLAR_MASS);

    Macrotest(OMEGA_CORE);
    
    Show_float_macro(HeWD_HeWD_IGNITION_MASS,
                     "more massive HeWDs with reignite helium on accretion and/or merger");
    Macrotestif(ALLOW_HeWD_SUPERNOVAE,
                "HeWDs exlpode as 'HeIa' supernovae (ALLOW_HeWD_SUPERNOVAE defined)");
    Macrotestif(ALLOW_HeWD_SUPERNOVAE,
                "HeWDs do not explode (ALLOW_HeWD_SUPERNOVAE not defined)\n");
    Show_float_macro(MASS_ACCRETION_FOR_ELD_DEFAULT);
    Macrotest(MILLISECOND_RANDOMNESS);
    Macrotest(NANOSECOND_RANDOMNESS);
    Macrotest(RANDOM_SYSTEMS);
    Macrotest(RANDOM_SYSTEMS_SHOW_ARGS);
    Macrotest(RANDOM_SYSTEMS_SHOW_ARGS_AND_START_TIME);
    Show_string_macro(RANDOM_SYSTEMS_SHOW_ARGS_STREAM);
    Macrotest(USE_DRAND48_R);
    Macrotest(USE_MERSENNE_TWISTER);

    /*
     * Simple random number test.
     *
     * Of course this is not a robust test, but it detects 
     * a basic failure of the routine.
     */
    {
        static const int maxn = 10000;
        int n = maxn;
        double sum = 0.0;
        struct stardata_t * s = New_clear_stardata;
        s->common.random_seed = random_seed();
        while(n-->0)
        {
            sum += random_number(s,&s->common.random_seed);
        }
        sum /= (double)(maxn+0.1);
        Printf("Random number mean %g\n",sum);
        Safe_free(s);
    }
    
    Macrotest(RANDOM_SEED_LOG);
#ifdef RANDOM_SEED_LOG
    Show_string_macro(RANDOM_SEED_LOG_STREAM);
#endif
    Macrotest(COMENV_LOG);
    Macrotest(GIANT_AGE_FATAL_ERRORS);
    Macrotest(DTR_FUDGES);
    Macrotest(ACQUIRE_FIX_NEGATIVE_AGES);
    Macrotest(ACQUIRE_TEST_FOR_NEGATIVE_AGES);
    Show_float_macro(ACCRETION_LIMIT_EDDINGTON_MULTIPLIER_DEFAULT);
    Show_float_macro(DEFAULT_LAMBDA_IONISATION);
    Show_float_macro(DEFAULT_LAMBDA_CE);
    Show_float_macro(DEFAULT_ALPHA_CE);
    Show_float_macro(DEFAULT_COMENV_PRESCRIPTION);
    Show_float_macro(DEFAULT_NELEMANS_GAMMA);
    Show_float_macro(NELEMANS_MIN_Q_DEFAULT);
    Show_float_macro(NELEMANS_MAX_FRAC_J_CHANGE_DEFAULT);
    Show_float_macro(NELEMANS_N_COMENVS_DEFAULT);
    Show_float_macro(MAX_CONVECTIVE_MASS);
    Show_float_macro(MIN_CONVECTIVE_MASS);
    
    Macrotest(ANG_MOM_CHECKS);
    Macrotest(NO_IMMEDIATE_MERGERS);

    /************************************************************/
    Macrotest(NUCSYN);
#ifdef NUCSYN
    Show_int_macro(ISOTOPE_ARRAY_SIZE);
    Show_float_macro(MR23YR);
    
    if(stardata!=NULL)
    {
        // list isotopes
        Printf("Isotopes are : \n");
        {

#ifndef ALL_ISOTOPES
            const Boolean all = FALSE;
#endif

            char s[][10]=NUCSYN_SHORT_ELEMENT_STRINGS;
            Isotope i;
            Ordered_isotope_loop(i)
            {
#ifndef ALL_ISOTOPES
                if(!all && i>=Xelse)
                {
                    Printf("Isotope %u is else\n",i);
                }
                else
                {
#endif
                    Printf("Isotope %u is %s%d (mass=%20.12e g (/amu=%20.12e, /MeV=%20.12e), Z=%d, A=%d)\n",
                           i,
                           stardata->store->nucleon_number[i]==0 ? "e" : 
                           s[stardata->store->atomic_number[i]],
                           stardata->store->nucleon_number[i],
                           stardata->store->mnuc[i],
                           stardata->store->mnuc[i]/AMU_GRAMS,
                           stardata->store->mnuc[i]*Pow2(SPEED_OF_LIGHT)/MEGA_ELECTRON_VOLT,
                           (int)stardata->store->atomic_number[i],
                           stardata->store->nucleon_number[i]
                        );
#ifndef ALL_ISOTOPES
                }
#endif
            }
        }
    }
    else
    {
        Printf("No isotope information because stardata is NULL\n");
    }
    Macrotestif(NUCSYN_STRIP_AND_MIX,
                "Main sequence stripping (i.e. CN in Algols, case A/B RLOF)");
    Macrotest(NUCSYN_FIRST_DREDGE_UP);
    Macrotest(NUCSYN_FIRST_DREDGE_UP_AMANDAS_TABLE);
    Macrotest(NUCSYN_FIRST_DREDGE_UP_RICHARDS_TABLE,
              "Richard Stancliffe's table in preference to Amanda's when available\n");
    Macrotest(NUCSYN_PREGUESS_FIRST_DREDGE_UP);
    Macrotest(NUCSYN_FIRST_DREDGE_UP_ACCRETION_CORRECTION);
    Macrotest(NUCSYN_FIRST_DREDGE_UP_ACCRETION_CORRECTION_FROM_TAMS);
    Macrotest(NUCSYN_FIRST_DREDGE_UP_PHASE_IN);
    Macrotest(NUCSYN_FORCE_DUP_IN_COMENV);
    Macrotest(NUCSYN_SECOND_DREDGE_UP);
    Macrotest(NUCSYN_TPAGB);
    Show_int_macro(TPAGB_LUMTYPE);
    Show_float_macro(PULSE_LUM_DROP_N_PULSES);
    Show_float_macro(THERMAL_PULSE_LUM_DROP_FACTOR);
    Show_float_macro(THERMAL_PULSE_LUM_DROP_TIMESCALE);
    Show_float_macro(KARAKAS2002_TPAGB_MIN_LUMINOSITY);
    Macrotest(PADOVA_MC1TP);
    Macrotest(NUCSYN_THIRD_DREDGE_UP);
    Macrotest(THIRD_DREDGE_UP_KARAKAS2002);
    Macrotest(NUCSYN_THIRD_DREDGE_UP_TABULAR_NCAL);
    Macrotest(THIRD_DREDGE_UP_STANCLIFFE);
    Show_float_macro(MINIMUM_ENVELOPE_MASS_FOR_THIRD_DREDGEUP_DEFAULT);
    Macrotest(NUCSYN_HUGE_PULSE_CRISTALLO);
    Macrotest(USE_TABULAR_INTERSHELL_ABUNDANCES_KARAKAS_2002);
    Macrotest(USE_TABULAR_INTERSHELL_ABUNDANCES_KARAKAS_2012);
    Macrotest(NUCSYN_HS_LS_LOG);
    Macrotest(NUCSYN_THIRD_DREDGE_UP_MULTIPLIERS);
    Macrotest(NUCSYN_BURN_TDUP_MATERIAL);
    Macrotest(NUCSYN_THIRD_DREDGE_UP_HYDROGEN_SHELL);
    Macrotest(KARAKAS2002_REFITTED_TPAGB_INTERPULSES);
    Show_float_macro(MAX_TPAGB_TIME);
    Show_float_macro(MINIMUM_INTERPULSE_PERIOD);
    Macrotest(NUCSYN_TPAGB_RUNTIME);
    Macrotest(NUCSYN_ANAL_BURN);
    Macrotest(NUCSYN_NUMERICAL_BURN);
    Show_float_macro(NUCSYN_NETWORK_BURN_MAX_FAILURE_COUNT);
    Macrotest(NUCSYN_TPAGB_HBB);
#ifdef NUCSYN_TPAGB_HBB
    Show_float_macro(HBBTFAC_DEFAULT);
    Printf("HBB on : cycles allowed ");
#ifdef NUCSYN_TPAGB_HBB_CN_CYCLE
    Printf("CN ");
#endif//NUCSYN_TPAGB_HBB_CN_CYCLE
#ifdef NUCSYN_TPAGB_HBB_ON_CYCLE
    Printf("ON ");
#endif//NUCSYN_TPAGB_HBB_ON_CYCLE
#ifdef NUCSYN_CNO_ASSUME_STABLE
    Printf("(assume CNO stable) ");
#endif
#ifdef NUCSYN_ALLOW_COOL_CNO_BURNING
    Printf("(allow cool CNO burning) ");
#endif
#ifdef NUCSYN_TPAGB_HBB_NeNa_CYCLE
    Printf("NeNa ");
#endif//NUCSYN_TPAGB_NeNa_CYCLE
#ifdef NUCSYN_NENA_LEAK
    Printf("(with Na23(pg)Mg24 leak) ");
#endif//NUCSYN_TPAGB_NeNa_LEAK
#ifdef NUCSYN_Na22_EQUILIBRIUM
    Printf("(with Na22 equilibrium) ");
#endif
#ifdef NUCSYN_TPAGB_HBB_MgAl_CYCLE
    Printf("MgAl ");
#endif//NUCSYN_TPAGB_MgAl_CYCLE
#ifdef NUCSYN_MGAL_LEAKBACK
    Printf("(with MgAl leakback) "); 
#endif
#ifdef NUCSYN_Al26m
    Printf("(with Al26m) ");
#endif
#ifdef NUCSYN_HBB_RENORMALIZE_MASS_FRACTIONS
    Printf("(with HBB mass fraction renormalization) ");
#endif
#ifdef NUCSYN_NORMALIZE_NUCLEONS
    Printf("(normalize nucleons in burn) ");
#endif
#ifdef NUCSYN_NETWORK_PP 
    Printf("pp=%d ",NUCSYN_NETWORK_PP);
#endif
#ifdef NUCSYN_NETWORK_COLDCNO
    Printf("cold CNO=%d ",NUCSYN_NETWORK_COLDCNO);
#endif
#ifdef NUCSYN_NETWORK_HOTCNO
    Printf("hot CNO=%d ",NUCSYN_NETWORK_HOTCNO);
#endif
#ifdef NUCSYN_NETWORK_NeNaMgAl 
    Printf("NeNaMgAl=%d ",NUCSYN_NETWORK_NeNaMgAl);
#endif
#ifdef NUCSYN_NETWORK_NeNaMgAlnoleak 
    Printf("NeNaMgAlnoleak=%d ",NUCSYN_NETWORK_NeNaMgAl);
#endif
#ifdef NUCSYN_NETWORK_NeNa
    Printf("NeNa=%d ",NUCSYN_NETWORK_NeNaMgAl);
#endif
#ifdef NUCSYN_NETWORK_PPfast
    Printf("PPfast=%d ",NUCSYN_NETWORK_NeNaMgAl);
#endif

    Printf("\n");
#else
    Printf("No HBB\n");    
#endif//NUCSYN_TPAGB_HBB

    Macrotest(RATES_OF_AMANDA,
              "Using Amanda's nuclear reaction rates");
    Macrotest(RATES_OF_AMANDA_NE22PG_FIX,"with Ne22(pg) fix");
    Macrotest(RATES_OF_MARIA,"Using Maria's nuclear reaction rates\n");
    Macrotest(RATES_OF_AREND_JAN,"Using Arend Jan's nuclear reaction rates\n");
    Macrotest(NUCLEAR_REACTION_RATE_MULTIPLIERS);
    Macrotest(NUCSYN_SIGMAV_PRE_INTERPOLATE);
#ifdef NUCSYN_SIGMAV_PRE_INTERPOLATE
    Show_int_macro(NUCSYN_SIGMAV_INTERPOLATION_RESOLUTION);
    Show_float_macro(NUCSYN_SIGMAV_INTERPOLATION_MIN);
    Show_float_macro(NUCSYN_SIGMAV_INTERPOLATION_MAX);
    Macrotestif(NUCSYN_SIGMAV_INTERPOLATE_LOGT9,"interpolate log T9");
    Macrotestifnot(NUCSYN_SIGMAV_INTERPOLATE_LOGT9,"interpolate linear T9");
    Macrotestif(NUCSYN_SIGMAV_INTERPOLATE_LOGSIGMAV,"interpolate log sigmav");
    Macrotestifnot(NUCSYN_SIGMAV_INTERPOLATE_LOGSIGMAV,"interpolate linear sigmav");
#endif
    Show_float_macro(SIGMAV_TINY);
    Macrotest(NUCSYN_HOT_SIGMAV);
    Macrotest(NUCSYN_THERMALIZED_CORRECTIONS);
    Macrotest(HELIUM_BURNING_REACTIONS);
    Macrotest(NUCSYN_S_PROCESS_LOG);

    Macrotest(NUCSYN_LITHIUM);
    Macrotest(NUCSYN_LITHIUM_TABLES);
    Macrotest(NUCSYN_ANGELOU_LITHIUM);
#ifdef NUCSYN_ANGELOU_LITHIUM
    Show_float_macro(NUCSYN_ANGELOU_LITHIUM_MINIMUM_Li7);
    Show_float_macro(NUCSYN_ANGELOU_LITHIUM_MAX_TIMESTEP_FACTOR);
#endif    
    Macrotest(NUCSYN_R_PROCESS);
    Macrotest(NUCSYN_R_PROCESS_ARLANDINI1999);
    Macrotest(NUCSYN_R_PROCESS_SIMMERER2004);
    Macrotest(NUCSYN_RADIOACTIVE_DECAY);
    Macrotest(NUCSYN_WR);
    Show_float_macro(NUCSYN_WR_MASS_BREAK);
    Macrotest(NUCSYN_WR_TABLES,
              "interpolate from Lynnette's tables");
    Macrotest(NUCSYN_WR_RS_TABLE
              "interpolate from Richard's tables ");
    Macrotest(NUCSYN_WR_METALLICITIY_CORRECTIONS,
              "with metallicity corrections (if non-tabular) ");
    Macrotest(NUCSYN_WR_ACCRETION,
              "with NUCSYN_WR_ACCRETION (if tabular) ");
    Macrotest(NUCSYN_WR_LOG);
    Macrotest(NUCSYN_LOWZ_SUPERNOVAE);
#ifdef NUCSYN_LOWZ_SNE_THRESHOLD
    Show_float_macro(NUCSYN_LOWZ_SNE_THRESHOLD);
#endif
    Macrotest(NUCSYN_SUPERNOVAE);
    Macrotest(NUCSYN_CCSNE_WOOSLEY_WEAVER_1995);
    Macrotest(NUCSYN_SUPERNOVAE_USE_TABULAR_WW95_MODELS);
    Macrotest(NUCSYN_SUPERNOVAE_CC_WW95_A);
    Macrotest(NUCSYN_SUPERNOVAE_CC_WW95_B);
    Macrotest(NUCSYN_SUPERNOVAE_CC_WW95_C);
    Macrotest(NUCSYN_CCSNE_CHIEFFI_LIMONGI_2004);
    Macrotest(NUCSYN_CCSNE_CHIEFFI_LIMONGI_2004_PORTINARI);
    Macrotest(NUCSYN_CCSNE_CHIEFFI_LIMONGI_2004_EXTRAPOLATE);
    Macrotest(NUCSYN_CCSNE_CHIEFFI_LIMONGI_ELEMENTAL_Na);
    Macrotest(NUCSYN_LIMONGI_CHIEFFI_2006_Al26);
    Macrotest(NUCSYN_SN_ELECTRON_CAPTURE_WANAJO_2008);
    Macrotest(NUCSYN_SN_ELECTRON_CAPTURE_WANAJO_2008_ST);
    Macrotest(NUCSYN_SN_ELECTRON_CAPTURE_WANAJO_2008_FP3);
    Macrotest(NUCSYN_HETEROGENEOUS_HE_STAR);
    Macrotest(NUCSYN_NOVAE);
    Macrotest(COMENV_NORMALIZE_ABUNDS);
    Macrotest(NUCSYN_LOGGING);
    Macrotest(NUCSYN_YIELDS);
    Macrotest(NUCSYN_LOG_YIELDS);
    Macrotest(NUCSYN_SPARSE_YIELDS);
    Macrotest(NUCSYN_YIELDS_COMPRESSION);
    Macrotest(NUCSYN_YIELDS_VS_TIME);
    Macrotest(NUCSYN_YIELD_COMPRESSION);
    Macrotest(NUCSYN_CHECK_LESS_THAN_ZERO_YIELDS);
    Macrotest(NUCSYN_COMRPESS_YIELD_AND_ENSEMBLE_OUTPUT);
    Macrotest(NUCSYN_LOG_BINARY_X_YIELDS);
    Macrotest(NUCSYN_LOG_BINARY_MPYIELDS);
    Macrotest(NUCSYN_LOG_BINARY_DX_YIELDS);
    Macrotest(NUCSYN_LOG_INDIVIDUAL_DX_YIELDS);
    Macrotest(NUCSYN_LOG_MPYIELDS);
    Macrotest(NUCSYN_LOG_SINGLE_X_YIELDS);
    Macrotest(NUCSYN_IGNORE_ZERO_BINARY_DX_YIELD);
    Macrotest(NUCSYN_LOG_INDIVIDUAL_DX_YIELDS);
    Macrotest(NUCSYN_LOG_YIELDS_EVERY_TIMESTEP);
    Macrotest(NUCSYN_LONG_YIELD_FORMAT);
    Macrotest(NUCSYN_ABUNDANCE_LOG);
    Macrotest(NUCSYN_GCE);
#ifdef GCE_ISOTOPES
    Show_int_macro(GCE_ISOTOPES);
#endif
    Macrotest(NUCSYN_GCE_ELSE);
    Macrotest(NUCSYN_GCE_OUTFLOW_CHECKS);
#ifdef NUCSYN_GCE_OUTFLOW_CHECKS
    Show_float_macro(NUCSYN_GCE_OUTFLOW_FACTOR);
    Show_float_macro(NUCSYN_GCE_SUPERNOVA_OUTFLOW_VELOCITY);
    Show_float_macro(NUCSYN_GCE_OUTFLOW_ESCAPE_FRACTION);
#endif
    
    Macrotest(NUCSYN_STELLAR_POPULATIONS_ENSEMBLE);
    Macrotest(NUCSYN_GCE_DISABLE_ENSEMBLE_OUTPUT);
#ifdef NUCSYN_STELLAR_POPULATIONS_ENSEMBLE
    Show_int_macro(NUCSYN_ENSEMBLE_N);
#endif
    Macrotest(NUCSYN_STELLAR_POPULATIONS_ENSEMBLE_SPARSE);

    {
        char * ensemble_strings[] = {NUCSYN_ENSEMBLE_STRINGS};
        int i;
        for(i=0;i<NUCSYN_ENSEMBLE_N;i++)
        {
            Printf("Ensemble %d is %s\n",
                   i,
                   ensemble_strings[i]);
        }
    }

    Macrotest(NUCSYN_ID_SOURCES);

#ifdef NUCSYN_ID_SOURCES
    Macrotest(NUCSYN_ID_SOURCES_OLD_METHOD);
    Macrotest(NUCSYN_ID_SOURCES_GCE);
    Show_int_macro(NUMBER_OF_SOURCES);

    {
        Source_number i;
        Sourceloop(i)
        {
            Printf("Nucleosynthesis source %u is %s\n",
                   i,
                   Nucsyn_source_string(i));
        }
    }
#endif//NUCSYN_ID_SOURCES

    Macrotest(NUCSYN_XTOT_CHECKS);
    Macrotest(NUCSYN_STRUCTURE_LOG);
    Macrotest(NUCSN_MU_FUZZ);
#ifdef NUCSYN_MU_FUZZ
    Show_float_macro(NUCSYN_MU_FUZZ);
#endif
    Macrotest(NUCSYN_LOG);
    Macrotest(NUCSYN_SHORT_LOG);
    Macrotest(NUCSYN_LONG_LOG);
    Macrotest(NUCSYN_GIANT_ABUNDS_LOG);
    Macrotest(NUCSYN_R_STAR_LOG);
    Macrotest(NUCSYN_MERGER_DREDGEUP);
#ifdef NUCSYN_MERGER_DREDGEUP
    Show_float_macro(NUCSYN_MERGER_MASS_DREDGEDUP_FRACTION);
#endif
    Macrotest(NUCSYN_PLANETARY_NEBULAE);
    Macrotest(LOG_BARIUM_STARS);
    Macrotest(NUCSYN_J_LOG);
    Macrotest(NUCSYN_Globular_Cluster_LOG);
    Macrotest(NUCSYN_CEMP_LOGGING);
#ifdef NUCSYN_CEMP_LOGGING
    double cemp_min_dt[]=CEMP_MIN_DT_ARRAY; 
    Printf("CEMP logging is enabled: min age default %g, min dt for logging %g, default [C/Fe] minimum for CEMPS=%g, timesteps (Myr): [MS=%g,%g, HG=%g, GB=%g, CHeB=%g, EAGB=%g, TPAGB=%g]\n",
           DEFAULT_CEMP_MINIMUM_AGE,
           CEMP_DT,
           DEFAULT_CEMP_CFE_MINIMUM,
           cemp_min_dt[LOW_MASS_MAIN_SEQUENCE],cemp_min_dt[MAIN_SEQUENCE],
           cemp_min_dt[HG],cemp_min_dt[GIANT_BRANCH],
           cemp_min_dt[CHeB],cemp_min_dt[EAGB],cemp_min_dt[TPAGB]);
    Macrotest(NUCSYN_CEMP_CRITERIA_O1);
#endif
    Macrotest(NUCSYN_LOG_JL);
    Macrotest(LOG_FINAL_TPAGB_ABUNDS);
    Macrotest(NO_MERGER_NUCLEOSYNTHESIS);
    Macrotest(TPAGB_SPEEDUP);
#ifdef TPAGB_SPEEDUP
    Show_float_macro(TPAGB_SPEEDUP_AFTER);
#endif  //TPAGB_SPEEDUP
    Macrotest(ZAMS_MENV_METALLICITY_CORRECTION);
    Macrotest(NUC_MASSES_DEBUG);
#ifdef CEMP_VERSION
    Printf("Equivalent CEMP version %s (code version %s)\n",
           CEMP_VERSION,BINARY_C_VERSION);
#endif
#endif //NUCSYN
    
    Show_long_int_macro(BUFFERED_PRINTF_MAX_BUFFER_SIZE);
    Show_long_int_macro(BUFFERED_PRINTF_ERROR_BUFFER_SIZE);
    Macrotest(BUFFER_MEMORY_DEBUGGING);
    Show_string_macro(BUFFER_MEMORY_DEBUGGING_STREAM);
    Macrotest(BUFFERED_STRING_OVERRUN_WARNINGS);
    
    Show_string_macro(BUFFERED_STRING_OVERRUN_WARNINGS_STREAM);
    Show_float_macro(BUFFERED_PRINTF_INCREASE_RATIO);
    
    Show_float_macro(EVOLUTION_DIFFLOG_INCREASE_RATIO);
    
    Macrotest(__SHOW_STARDATA__);
    Macrotest(SHOW_STARDATA);

    Show_float_macro(FUV_eV_MIN);
    Show_float_macro(FUV_eV_MAX);
    Show_float_macro(EUV_eV_MIN);
    Show_float_macro(EUV_eV_MAX);
    Show_float_macro(XRAY_eV_MIN);
    Show_float_macro(XRAY_eV_MAX);
        
    Macrotest(SHORT_SUPERNOVA_LOG);
    Macrotest(BINARY_STARS_ONLY);
    Macrotestif(LOG_SUPERNOVAE,
                "default filename %s",
                DEFAULT_SN_DAT_FILENAME);
    Macrotest(ZW2019_SNIA_LOG);
    Macrotest(CGI_EXTENSIONS);
    Macrotest(SDB_CHECKS);
    Macrotest(BI_LYN);
    Macrotest(XRAY_BINARIES);
    Macrotest(STELLAR_TYPE_LOG);
    Macrotest(STELLAR_COLOURS);
    Macrotest(LOG_JJE,
              "JJE (Eldridge) log activated");
    Macrotestif(SELMA,
              "Selma's logging activated");
    Macrotestif(FORM_STARS_AT_BREAKUP_ROTATION,
              "Form stars at breakup rotation rate");
    Macrotestif(AXEL_RGBF_FIX,
                "Use Axel's RGB radius function fix");
    Printf("Comenv accretion: ");
#ifdef COMENV_NS_ACCRETION
    Printf("NS : default fraction %g, default mass %g ... ",
           COMENV_NS_ACCRETION_FRACTION_DEFAULT,
           COMENV_NS_ACCRETION_MASS_DEFAULT);
#endif
#ifdef COMENV_MS_ACCRETION
    Printf("MS: default mass %g ",
           COMENV_MS_ACCRETION_MASS_DEFAULT);
#endif
    Printf("\n");
    Macrotestif(NS_BH_AIC_LOG,
                "NS/BH AIC log enabled");
    Macrotestif(NS_NS_BIRTH_LOG
                "NS/NS birth log enabled");
    Macrotest(RS_LOG_ACCREITON_LOG);
    Macrotest(DETMERS_LOG);
    Macrotest(LOG_HERBERT);
    Macrotestif(LOG_TO_STDOUT,
                "Log file will be sent to stdout");
    Macrotestif(LOG_TO_STDOUT
                "Log file will be sent to stderr");
    Macrotestif(LOG_COMENV_RSTARS,
                "Log comenv R-star formation enabled");
    Macrotestif(HALL_TOUT_2014_RADII,
              "Using core radii from Hall and Tout 2014");
#ifdef HRDIAG
    Printf("HRDIAG logging (for Peter Anders) is on: HRDIAG_BIN_SIZE_LOGTIME=%g HRDIAG_START_LOG=%g HRDIAG_BIN_SIZE_TEFF=%g HRDIAG_BIN_SIZE_LOGL=%g HRDIAG_BIN_SIZE_LOGG=%g\n",
           HRDIAG_BIN_SIZE_LOGTIME,
           HRDIAG_START_LOG,
           HRDIAG_BIN_SIZE_TEFF,
           HRDIAG_BIN_SIZE_LOGL,
           HRDIAG_BIN_SIZE_LOGG);
    Macrotest(HRDIAG_EXTRA_RESOLUTION_AT_END_AGB);
#endif//HRDIAG

    Macrotest(FINAL_MASSES_LOG);
    Macrotest(HeWD_HeCORE_MERGER_INITION);
    Macrotest(LOG_RSTARS);
    Macrotest(RSTARS,"RSTARS master switch is on");
    Macrotest(MESTEL_COOLING,"WD Mestel cooling");
    Macrotest(MODIFIED_MESTEL_COOLING,"WD Modified Mestel cooling");
    Macrotest(COMENV_CO_MERGER_GIVES_IIa);
    Macrotestif(COMENV_CO_MERGER_GIVES_IIa,
                "Comenv COWD-COWD mergers give SN IIa");
    Macrotestifnot(COMENV_CO_MERGER_GIVES_IIa,
                   "Comenv COWD-COWD mergers give electron-capture SN");
    Macrotest(NO_HeWD_NOVAE);
    Show_float_macro(He_REIGNITION_LAYER_THICKNESS); 
    Macrotest(STARDATA_STATUS);
    Macrotestif(MANUAL_VROT,"Rotational velocities can be specified on the command line"); 
    Show_float_macro(VROT_NON_ROTATING);
    Show_float_macro(VROT_BSE);
    Show_float_macro(VROT_BREAKUP);
    Show_float_macro(VROT_SYNC);
    
    Macrotest(FABIAN_IMF_LOG);
#ifdef FABIAN_IMF_LOG
    Show_float_macro(NEXT_FABIAN_IMF_LOG_TIME_DEFAULT);
    Show_float_macro(FABIAN_IMF_LOG_TIMESTEP_DEFAULT);
#endif
    Macrotest(TIMESTEP_MODULATION);
    Macrotest(RLOF_TIMESTEP_MODULATION);

    {
        int i;
        char *  dtstring[DT_NUMBER_OF_TIMESTEP_LIMITERS] = { DT_LIMIT_STRINGS };
        for(i=0;i<DT_NUMBER_OF_TIMESTEP_LIMITERS;i++)
        {
            Printf("DTlimit %d : %s : %g\n",
                   i,
                   dtstring[i],
                   stardata->preferences->timestep_multipliers[i]);
        }
    }

    Macrotestif(STOP_ACCRETION_OF_MATERIAL_BEYOND_BREAKUP,
                "Material is not accreted beyond breakup according to Hurley et al algorithm");
    Macrotest(SELMA_EXTRA_FEATURES);
    Macrotest(SELMA_FIX_ACCRETED_MASS_IN_REJUVENATION);
    Macrotest(SELMA_BETTER_TREATMENT_OF_MS_ACCRETORS);
    Macrotest(SELMA_BETTER_TREATMENT_OF_MS_MERGERS);
#ifdef SELMA_BETTER_TREATMENT_OF_MS_MERGERS
    Show_float_macro(F_MASSLOSS_DURING_MS_MERGERS_DEFAULT);
    Show_float_macro(MIXINGPAR_MS_MERGERS_DEFAULT);
#endif
    Macrotest(SELMA_NEW_QCRIT);
#ifdef SELMA_NEW_QCRIT
    Show_float_macro(QCRIT_ON_MAIN_SEQUENCE_DEFAULT);
    Show_float_macro(QCRIT_HERTZSPRUNG_GAP_DEFAULT);
    Show_float_macro(QCRIT_POST_MS_NAKED_HE_STAR_DEFAULT);
    Show_float_macro(QCRIT_ELSE_DEFAULT);
#endif

    Macrotest(HE_STAR_ACCRETING_H_FIX);
    Macrotest(K2FUDGE);
    Macrotest(NOVA_BACK_ACCRETION);
    Macrotest(PN_PROJECT);
#ifdef PN_PROJECT
    Show_float_macro(PN_MIN_L);
    Show_float_macro(PN_MIN_TEFF);
#endif
    Macrotest(PN_FAST_WIND);
#ifdef PN_FAST_WIND
    Show_float_macro((double)PN_FAST_WIND_MDOT_GB);
    Show_float_macro((double)PN_FAST_WIND_DM_GB);
    Show_float_macro((double)PN_FAST_WIND_MDOT_AGB);
    Show_float_macro((double)PN_FAST_WIND_DM_AGB);
    Macrotest(RESOLVE_PN_FAST_WIND);
#ifdef RESOLVE_PN_FAST_WIND
    Show_float_macro((double)RESOLVE_PN_FAST_WIND_MAX_MENV);
    Show_float_macro((double)RESOLVE_PN_FAST_WIND_MIN_WD_LUM);
#endif // RESOLVE_PN_FAST_WIND
#endif // PN_FAST_WIND
    Macrotest(POST_CE_OBJECTS_HAVE_ENVELOPES);
#ifdef POST_CE_OBJECTS_HAVE_ENVELOPES
    Macrotest(POST_CE_ADAPTIVE_MENV);
#ifndef POST_CE_ADAPTIVE_MENV
    Show_float_macro(POST_CE_ENVELOPE_DM_GB);
    Show_float_macro(POST_CE_ENVELOPE_DM_EAGB);
    Show_float_macro(POST_CE_ENVELOPE_DM_TPAGB);
#endif //POST_CE_ADAPTIVE_MENV
#endif // POST_CE_OBJECTS_HAVE_ENVELOPES

    Macrotest(AGB_ROTATIONAL_WIND_ENHANCEMENT_DS06,"(Experiental rotational wind enhancement of Dijkstra & Speck 2006)");
    Macrotest(AGB_ROTATIONAL_WIND_ENHANCEMENT_MOE,"(Experiental rotational wind enhancement of Moe)");
    Macrotest(WRLOF_MASS_TRANSFER,"(Carlo Abate's Wind-RLOF)");
    Macrotest(USE_WIND_VELOCITY_OF_VW93_ON_TPAGB);
    Macrotest(FORCE_SINGLE_STAR_EVOLUTION_WHEN_SECONDARY_IS_A_PLANET);
    Macrotest(CN_THICK_DISC);

#ifdef EVOLUTION_SPLITTING
    Printf("EVOLUTION_SPLITTING is on : EVOLUTION_SPLITTING_HARD_MAX_DEPTH=%d EVOLUTION_SPLITTING_MAX_SPLITDEPTH_DEFAULT=%d EVOLUTION_SPLITTING_SUPERNOVA_N=%d\n",
           EVOLUTION_SPLITTING_HARD_MAX_DEPTH,
           EVOLUTION_SPLITTING_MAX_SPLITDEPTH_DEFAULT,
           EVOLUTION_SPLITTING_SUPERNOVA_N);
#else
    Printf("EVOLUTION_SPLITTING is off\n");
#endif

    Macrotest(BSE);
    Macrotest(BINT);
    
    Macrotest(BSE_EMULATION);
    Macrotest(MAKE_BSE_TABLES);
    Macrotest(UNIT_TESTS);
    Macrotest(REJUVENATE_HERZTSPRUNG_GAP_STARS);
    Macrotest(UNIT_TESTS);

    Show_float_macro(MIN_MASS_He_FOR_HYBRID_COWD);
    Macrotest(LOG_HYBRID_WHITE_DWARFS);
    
    /* opacity tables */
    Macrotest(OPACITY_ALGORITHMS);
    Macrotest(OPACITY_ENABLE_ALGORITHM_PACZYNSKI);
    Macrotest(OPACITY_ENABLE_ALGORITHM_FERGUSON_OPAL);
    Macrotest(OPACITY_ENABLE_ALGORITHM_STARS);
    
    /* mathematics */
    Macrotest(BISECT_FAIL_EXIT);
    Macrotest(BISECT_DO_MONOCHECKS);
    Macrotest(MIN_MAX_ARE_FUNCTIONS);
    Macrotest(MIN_MAX_ARE_MACROS);
    Macrotest(USE_FABS);
    Macrotest(ZERO_CHECKS_ARE_FUNCTIONS);
    Macrotest(POWER_OPERATIONS_ARE_FUNCTIONS);
    
    /* constants */
    Show_float_macro(SPEED_OF_LIGHT);
    Show_float_macro(GRAVITATIONAL_CONSTANT);
    Show_float_macro(PLANCK_CONSTANT);
    Show_float_macro(BOLTZMANN_CONSTANT);
    Show_float_macro(ELECTRON_CHARGE);
    Show_float_macro(AVOGADRO_CONSTANT);
    Show_float_macro(R_SUN);
    Show_float_macro(ASTRONOMICAL_UNIT);
    Show_float_macro(DAY_LENGTH_IN_SECONDS);
    Show_float_macro(L_SUN);
    Show_float_macro(M_SUN);
    Show_float_macro(R_SUN);
    Show_float_macro(M_EARTH);
    Show_float_macro(M_JUPITER);
    Show_float_macro(M_PROTON);
    Show_float_macro(M_ELECTRON);
    Show_float_macro(M_NEUTRON);
    Show_float_macro(SOLAR_APPARENT_MAGNITUDE);
    Show_float_macro(CMB_TEMPERATURE);
    Show_float_macro(PARSEC);
    Show_float_macro(KILOPARSEC);
    Show_float_macro(MEGAPARSEC);
    Macrotest(LIKE_ASTROPY);
    
    /* derived constants */
    Show_float_macro(ELECTRON_VOLT);
    Show_float_macro(MEGA_ELECTRON_VOLT);
    Show_float_macro(AMU_GRAMS);
    Show_float_macro(AMU_MEV);
    Show_float_macro(RYDBERG_CONSTANT);
    Show_float_macro(YEAR_LENGTH_IN_SECONDS);
    Show_float_macro(YEAR_LENGTH_IN_DAYS);
    Show_float_macro(YEAR_LENGTH_IN_HOURS);
    Show_float_macro(R_SUN_KM);
    Show_float_macro(AU_IN_SOLAR_RADII);
    Show_float_macro(PLANCK_CONSTANT_BAR);
    Show_float_macro(GMRKM);
    Show_float_macro(OMEGA_FROM_VKM);
    Show_float_macro(G_CODE_UNITS);
    Show_float_macro(GAS_CONSTANT);
    Show_float_macro(FINE_STRUCTURE_CONSTANT);
    Show_float_macro(SIGMA_THOMPSON);
    Show_float_macro(STEFAN_BOLTZMANN_CONSTANT);

   
    /*
     * Symbols
     */
    Show_string_macro(SOLAR_SYMBOL);
    Show_string_macro(L_SOLAR);
    Show_string_macro(M_SOLAR);
    Show_string_macro(R_SOLAR);
    
    
    Exit_or_return_void(BINARY_C_SPECIAL_EXIT,"version exit");
}

            
static float __test_seconds(void)
{
    char * t = getenv("BINARY_C_SPEEDTEST_SECONDS");
    return t == NULL ? 1 : strtod(t,NULL);
}
