#include "../binary_c.h"

void set_default_preferences(struct preferences_t * RESTRICT const preferences)
{
    /* 
     * Reset the preferences struct to its default values
     *
     * This is the location where binary_c's defaults are set.
     * 
     * Other software, such as binary_grid2, may have a backup 
     * set of defaults, but this is the master list.
     *
     * You should be able to run (in a terminal):
     *
     * ./binary_c-config defaults
     *
     * to list the default parameters. 
     *
     * Beware that these defaults are not
     */
    timestep_set_default_multipliers(preferences);
#ifdef TIMESTEP_MODULATION
    preferences->timestep_modulator=1.0;
#endif
#ifdef RLOF_MDOT_MODULATION
    preferences->RLOF_mdot_factor=1.0;
#endif  
    preferences->RLOF_interpolation_method=RLOF_INTERPOLATION_BINARY_C;
    preferences->monte_carlo_sn_kicks=TRUE;
    {
        Supernova i;
        for(i=0;i<NUM_SN_TYPES;i++)
        {
            preferences->sn_kick_distribution[i] = KICK_VELOCITY_FIXED;
            preferences->sn_kick_dispersion[i] = 0.0;
        }
    }
    preferences->sn_kick_distribution[SN_II]=KICK_VELOCITY_MAXWELLIAN;
    preferences->sn_kick_dispersion[SN_II]=SN_SIGMA_DEFAULT;
    preferences->sn_kick_distribution[SN_IBC]=KICK_VELOCITY_MAXWELLIAN;
    preferences->sn_kick_dispersion[SN_IBC]=SN_SIGMA_DEFAULT;
    preferences->sn_kick_distribution[SN_GRB_COLLAPSAR]=KICK_VELOCITY_MAXWELLIAN;
    preferences->sn_kick_dispersion[SN_GRB_COLLAPSAR]=SN_SIGMA_DEFAULT;
#ifdef WD_KICKS
    preferences->sn_kick_dispersion[SN_WDKICK] = WD_SIGMA_DEFAULT;
    preferences->wd_kick_when=WD_KICK_END_AGB;
#endif
    preferences->dtfac=DTFAC_DEFAULT;
#ifdef NUCSYN
    preferences->third_dup=TRUE;
#ifdef LITHIUM_TABLES
    preferences->lithium_hbb_multiplier=1.0;
    preferences->lithium_GB_post_1DUP=0.0;
#endif
    strlcpy(preferences->Seitenzahl2013_model,"N100",5);
#ifdef PRE_MAIN_SEQUENCE
    preferences->pre_main_sequence=FALSE;
#endif

#ifdef NUCSYN_S_PROCESS
    /* C13 pocket s-process efficiency */
    preferences->c13_eff=DEFAULT_C13_EFFICIENCY;
    preferences->mc13_pocket_multiplier=1.0;
#endif

#ifdef USE_TABULAR_INTERSHELL_ABUNDANCES_KARAKAS_2012
    preferences->pmz_mass=0.0;//CAB let's put pmz_mass=0 as a default value?
#endif //USE_TABULAR_INTERSHELL_ABUNDANCES_KARAKAS_2012

#ifdef NUCSYN_SPARSE_YIELDS
    preferences->yields_dt=NUCSYN_YIELDS_DT_DEFAULT;
#endif
#ifdef NUCSYN_GCE_OUTFLOW_CHECKS
    preferences->escape_velocity=1e9; // galactic escape velocity
    preferences->escape_fraction=0.0; // galactic escape fraction
#endif
#endif//NUCSYN

    // initialize other free parameters
    preferences->nova_retention_fraction=NOVA_RETENTION_FRACTION_DEFAULT;
    preferences->accretion_limit_eddington_multiplier=ACCRETION_LIMIT_EDDINGTON_MULTIPLIER_DEFAULT;
    preferences->accretion_limit_thermal_multiplier=ACCRETION_LIMIT_THERMAL_MULTIPLIER_DEFAULT;
    preferences->accretion_limit_dynamical_multiplier=ACCRETION_LIMIT_DYNAMICAL_MULTIPLIER_DEFAULT;
    preferences->donor_limit_thermal_multiplier=DONOR_LIMIT_THERMAL_MULTIPLIER_DEFAULT;
    preferences->donor_limit_dynamical_multiplier=DONOR_LIMIT_DYNAMICAL_MULTIPLIER_DEFAULT;
    preferences->donor_limit_dynamical_multiplier=DONOR_LIMIT_ENVELOPE_MULTIPLIER_DEFAULT;

    /* nova timestep accelerators */
    preferences->nova_timestep_accelerator_num = 100;
    preferences->nova_timestep_accelerator_max = -1.0;
    preferences->nova_timestep_accelerator_index = 0.5;
    
    preferences->rlof_angmom_gamma=NONCONSERVATIVE_ANGMOM_GAMMA_DEFAULT;
    preferences->nonconservative_angmom_gamma=NONCONSERVATIVE_ANGMOM_GAMMA_DEFAULT;
    preferences->wind_djorb_fac=1.0;
    preferences->gb_reimers_eta=REIMERS_ETA_DEFAULT;
    preferences->magnetic_braking_factor=1.0;
    preferences->magnetic_braking_gamma=MAGNETIC_BRAKING_GAMMA_DEFAULT;
    preferences->magnetic_braking_factor=1.0;
    
    preferences->CRAP_parameter=CRAP_PARAMETER_DEFAULT;
    preferences->Bondi_Hoyle_accretion_factor = BONDI_HOYLE_ACCRETION_FACTOR_DEFAULT;
    preferences->vwind_multiplier=1.0/8.0;

    // Set the agb wind to 0, whatever that is (see wind loss function)
    preferences->tpagbwind=0;
    // mira switch on defaults at 500 days
    preferences->tpagb_superwind_mira_switchon=500.0;
    // remiers eta value on tpagb
    preferences->tpagb_reimers_eta=1.0; 
    // assume Reimers GB wind (Hurley etal 2002)
    preferences->gbwind=0; 
    // wr wind is standard by default (Hurley et al 2002)
    preferences->wr_wind=WR_WIND_BSE;
    /* common envelope parameters */
    preferences->alpha_ce=DEFAULT_ALPHA_CE;
    preferences->lambda_ce=DEFAULT_LAMBDA_CE;
    preferences->lambda_ionisation=DEFAULT_LAMBDA_IONISATION;
    /* WR wind loss factor 1.0 by default */
    preferences->wr_wind_fac=1.0;
    /* black hole prescription */
    preferences->BH_prescription=BH_HURLEY2002;
#ifdef  RLOF_RADIATION_CORRECTION
    preferences->RLOF_f= RLOF_RADIATION_CORRECTION_F_DEFAULT;
#endif // RLOF_RADIATION_CORRECTION

    preferences->tidal_strength_factor=TIDAL_STRENGTH_FACTOR_DEFAULT;
    preferences->merger_angular_momentum_factor=MERGER_ANGULAR_MOMENTUM_FACTOR_DEFAULT;
    preferences->minimum_timestep=MINIMUM_TIMESTEP_DEFAULT;
    preferences->maximum_timestep=MAXIMUM_TIMESTEP_DEFAULT;
    preferences->maximum_nuclear_burning_timestep=MAXIMUM_TIMESTEP_DEFAULT;
    preferences->maximum_timestep_factor = MAXIMUM_TIMESTEP_FACTOR_DEFAULT;
    
#ifdef SELMA_BETTER_TREATMENT_OF_MS_MERGERS
    preferences->f_massloss_during_MS_mergers=F_MASSLOSS_DURING_MS_MERGERS_DEFAULT;
    preferences->mixingpar_MS_mergers=MIXINGPAR_MS_MERGERS_DEFAULT;
#endif //SELMA_BETTER_TREATMENT_OF_MS_MERGERS

    /* critical q ratios for dynamical mass transfer
     *
     * The defaults are based on Claeys et al 2014 (A&A)
     */

    /* non-degenerate */
    preferences->qcrit[LOW_MASS_MS] = 0.6944;
    preferences->qcrit[MAIN_SEQUENCE] = 1.6;
    preferences->qcrit[HERTZSPRUNG_GAP] = 4.0;
    preferences->qcrit[GIANT_BRANCH] = -1.0;
    preferences->qcrit[CHeB] = 3.0;
    preferences->qcrit[EAGB] = -1.0;
    preferences->qcrit[TPAGB] = -1.0;
    preferences->qcrit[HeMS] = 3.0;
    preferences->qcrit[HeHG] = 4.0;
    preferences->qcrit[HeGB] = 0.78125;
    preferences->qcrit[HeWD] = 3.0;
    preferences->qcrit[COWD] = 3.0;
    preferences->qcrit[ONeWD] = 3.0;
    preferences->qcrit[NEUTRON_STAR] = 3.0;
    preferences->qcrit[BLACK_HOLE] = 3.0;
    preferences->qcrit[MASSLESS_REMNANT] = 3.0; // ignored

    /* degenerate */
    preferences->qcrit_degenerate[LOW_MASS_MS] = 1.0;
    preferences->qcrit_degenerate[MAIN_SEQUENCE] = 1.0;
    preferences->qcrit_degenerate[HERTZSPRUNG_GAP] = 4.7619;
    preferences->qcrit_degenerate[GIANT_BRANCH] = 1.15;
    preferences->qcrit_degenerate[CHeB] = 3.0;
    preferences->qcrit_degenerate[EAGB] = 1.15;
    preferences->qcrit_degenerate[TPAGB] = 1.15;
    preferences->qcrit_degenerate[HeMS] = 3.0;
    preferences->qcrit_degenerate[HeHG] =4.7619;
    preferences->qcrit_degenerate[HeGB] = 1.15;
    preferences->qcrit_degenerate[HeWD] = 0.625;
    preferences->qcrit_degenerate[COWD] = 0.625;
    preferences->qcrit_degenerate[ONeWD] = 0.625;
    preferences->qcrit_degenerate[NEUTRON_STAR] = 0.625;
    preferences->qcrit_degenerate[BLACK_HOLE] = 0.625;
    preferences->qcrit_degenerate[MASSLESS_REMNANT] = 3.0; // ignored

    preferences->E2_prescription=E2_HURLEY_2002;
#ifdef WRLOF_MASS_TRANSFER
    preferences->WRLOF_method = WRLOF_Q_DEPENDENT;
#endif//WRLOF_MASS_TRANSFER
    preferences->comenv_prescription=DEFAULT_COMENV_PRESCRIPTION;
    preferences->nelemans_gamma=DEFAULT_NELEMANS_GAMMA;
    preferences->nelemans_minq=NELEMANS_MIN_Q_DEFAULT;
    preferences->nelemans_max_frac_j_change=NELEMANS_MAX_FRAC_J_CHANGE_DEFAULT;
    preferences->nelemans_n_comenvs=NELEMANS_N_COMENVS_DEFAULT;
    preferences->mass_accretion_for_eld=MASS_ACCRETION_FOR_ELD_DEFAULT;
    preferences->WD_accretion_rate_novae_upper_limit_hydrogen_donor = ACCRETION_RATE_NOVAE_UPPER_LIMIT_HYDROGEN_DONOR_DEFAULT;
    preferences->WD_accretion_rate_new_giant_envelope_lower_limit_hydrogen_donor = ACCRETION_RATE_NEW_GIANT_ENVELOPE_LOWER_LIMIT_HYDROGEN_DONOR_DEFAULT;
preferences->WD_accretion_rate_novae_upper_limit_helium_donor = ACCRETION_RATE_NOVAE_UPPER_LIMIT_HELIUM_DONOR_DEFAULT;
    preferences->WD_accretion_rate_new_giant_envelope_lower_limit_helium_donor = ACCRETION_RATE_NEW_GIANT_ENVELOPE_LOWER_LIMIT_HELIUM_DONOR_DEFAULT;
preferences->WD_accretion_rate_novae_upper_limit_other_donor = ACCRETION_RATE_NOVAE_UPPER_LIMIT_OTHER_DONOR_DEFAULT;
    preferences->WD_accretion_rate_new_giant_envelope_lower_limit_other_donor = ACCRETION_RATE_NEW_GIANT_ENVELOPE_LOWER_LIMIT_OTHER_DONOR_DEFAULT;

    preferences->hachisu_disk_wind=FALSE;
    preferences->hachisu_qcrit=HACHISU_DISK_WIND_QCRIT_DEFAULT;

    preferences->mass_for_Hestar_Ia_upper=0.0;
    preferences->mass_for_Hestar_Ia_lower=0.0;

    preferences->max_HeWD_mass = 0.7;
    
    preferences->rotationally_enhanced_mass_loss=ROTATIONALLY_ENHNACED_MASSLOSS_NONE;
    preferences->rotationally_enhanced_exponent=1.0;

#ifdef VW93_MIRA_SHIFT
    preferences->vw93_mira_shift=0.0;
#endif
#ifdef VW93_MULTIPLIER
    preferences->vw93_multiplier=1.0;
#endif
#ifdef COMENV_NS_ACCRETION
    preferences->comenv_ns_accretion_fraction=COMENV_NS_ACCRETION_FRACTION_DEFAULT;
    preferences->comenv_ns_accretion_mass=COMENV_NS_ACCRETION_MASS_DEFAULT;
#endif
#ifdef COMENV_MS_ACCRETION
    preferences->comenv_ms_accretion_mass=COMENV_MS_ACCRETION_MASS_DEFAULT;
#endif
#ifdef ADAPTIVE_RLOF
    preferences->RLOF_method=RLOF_METHOD_BSE;
#endif
    preferences->qcrit_giant_branch_method=QCRIT_GB_DEFAULT;
  
    /* by default have no rotation effects, though this might change in the future */
    preferences->third_dup=TRUE;
    preferences->delta_mcmin=0.0;
    preferences->lambda_min=0.0;
    preferences->lambda_multiplier=1.0;
#ifdef NUCSYN
#ifdef NUCSYN_THIRD_DREDGE_UP
    preferences->minimum_envelope_mass_for_third_dredgeup= MINIMUM_ENVELOPE_MASS_FOR_THIRD_DREDGEUP_DEFAULT;
#endif
    preferences->hbbtfac=HBBTFAC_DEFAULT;
    preferences->no_thermohaline_mixing=FALSE;
#if defined NUCSYN_THIRD_DREDGE_UP && \
    defined NUCSYN_THIRD_DREDGE_UP_MULTIPLIERS
    {
        preferences->boost_third_dup=FALSE;
        Isotope i;
        Isotope_loop(i)
        {
            preferences->third_dup_multiplier[i]=1.0;
        }
    }
#endif
#ifdef NUCLEAR_REACTION_RATE_MULTIPLIERS
    nucsyn_reset_reaction_rates(preferences);
#endif

    /*
     * Set all initial override abundances to -1.0,
     * which means unset. 
     */
    nucsyn_set_abunds_array(preferences->the_initial_abundances,-1.0);
    nucsyn_set_abunds_array(preferences->initial_abundance_multiplier,-1.0);
    
#ifdef NUCSYN_STAR_BY_STAR_MATCHING
    /* similarly, set ERRORS for the observed star to -1 */
    nucsyn_set_abunds_array(preferences->observed_star_error,-1.0);
#endif //NUCSYN_STAR_BY_STAR_MATCHING

    preferences->initial_abunds_only=FALSE;

    /* default to AG89 abundances */
    preferences->initial_abundance_mix=NUCSYN_INIT_ABUND_MIX_AG89;

#ifdef NUCSYN_CEMP_LOGGING
    /* minimum [C/Fe] for CEMPs */
    preferences->CEMP_cfe_minimum=DEFAULT_CEMP_CFE_MINIMUM;
    /* minimum age for CEMPs */
    preferences->CEMP_minimum_age=DEFAULT_CEMP_MINIMUM_AGE;
    /* maximum logg */
    preferences->CEMP_logg_maximum=DEFAULT_CEMP_MAXIMUM_LOGG;
#endif

#ifdef NUCSYN_TPAGB_HBB
    /* allow NeNa and MgAl burning */
    preferences->NeNaMgAl=TRUE;
#endif

#endif //NUCSYN

    /* multiplier for GB wind loss rate */
    preferences->gbwindfac=1.0;
    preferences->agbwindfac=1.0;
    preferences->cmd_line_random_seed=0;
    
    preferences->repeat=1;
#ifdef RANDOM_SYSTEMS
    preferences->random_systems=FALSE;
#endif

    /* default to Hurley et al. (2002) mass loss prescription */
    preferences->wind_mass_loss=WIND_ALGORITHM_HURLEY2002;

#ifdef FILE_LOG
    /*
     * Set default filename for the log file
     */
    strlcpy(preferences->log_filename,
            DEFAULT_LOG_OUTPUT_FILENAME,
            STRING_LENGTH);
#endif

    preferences->wind_angular_momentum_loss = WIND_ANGMOM_LOSS_BSE;
    preferences->lw = 1.0;


    preferences->internal_buffering = 0;
    preferences->rlperi=FALSE;
    preferences->max_tpagb_core_mass = DEFAULT_MAX_TPAGB_CORE_MASS;
    preferences->chandrasekhar_mass = DEFAULT_CHANDRASEKHAR_MASS;
    preferences->max_neutron_star_mass = DEFAULT_MAX_NEUTRON_STAR_MASS;
    preferences->minimum_mass_for_carbon_ignition = DEFAULT_MINIMUM_MASS_FOR_CARBON_IGNITION;
    preferences->minimum_mass_for_neon_ignition = DEFAULT_MINIMUM_MASS_FOR_NEON_IGNITION;



#ifdef NUCSYN
#ifdef CN_THICK_DISC
    /* 
     * look back times for the start and end of 
     * thick disk star formation
     */
    preferences->thick_disc_start_age = CN_THICK_DISC_START_AGE_DEFAULT;
    preferences->thick_disc_end_age = CN_THICK_DISC_END_AGE_DEFAULT;
    preferences->thick_disc_logg_min = CN_THICK_DISC_LOGG_MIN_DEFAULT;
    preferences->thick_disc_logg_max = CN_THICK_DISC_LOGG_MAX_DEFAULT;
#endif

    preferences->AGB_3dup_algorithm=AGB_THIRD_DREDGE_UP_ALGORITHM_KARAKAS;
    preferences->AGB_core_algorithm=AGB_CORE_ALGORITHM_KARAKAS;
    preferences->AGB_radius_algorithm=AGB_RADIUS_ALGORITHM_KARAKAS;
    preferences->AGB_luminosity_algorithm=AGB_LUMINOSITY_ALGORITHM_KARAKAS;

#else
    preferences->AGB_3dup_algorithm=AGB_THIRD_DREDGE_UP_ALGORITHM_HURLEY;
    preferences->AGB_core_algorithm=AGB_CORE_ALGORITHM_HURLEY;
    preferences->AGB_radius_algorithm=AGB_RADIUS_ALGORITHM_HURLEY;
    preferences->AGB_luminosity_algorithm=AGB_LUMINOSITY_ALGORITHM_HURLEY;

#endif
    
    preferences->observable_radial_velocity_minimum=OBSERVABLE_RADIAL_VELOCITY_MIN_DEFAULT;

#ifdef EVOLUTION_SPLITTING
    preferences->evolution_splitting_sn_n = EVOLUTION_SPLITTING_SUPERNOVA_N;
    preferences->evolution_splitting_maxdepth = EVOLUTION_SPLITTING_MAX_SPLITDEPTH_DEFAULT;
#endif
#ifdef DISCS
#if defined DISC_LOG || defined DISC_LOG_2D
    strlcpy(preferences->disc_log_directory,"/tmp/",6);
#endif
    preferences->disc_timestep_factor = 0.01;
    preferences->cbdisc_gamma = 5.0/3.0;
    preferences->cbdisc_max_lifetime = 0.0;
    preferences->cbdisc_alpha = 0.01;
    preferences->cbdisc_kappa = 0.01;
    preferences->cbdisc_torqueF = 1.0;
    preferences->cbdisc_init_dM = 0.1;
    preferences->cbdisc_init_dJdM = 0.5;
    preferences->cbdisc_inner_edge_stripping = TRUE;
    preferences->cbdisc_outer_edge_stripping = TRUE;
    preferences->cbdisc_minimum_evaporation_timescale = YEAR_LENGTH_IN_SECONDS;
    preferences->cbdisc_mass_loss_constant_rate=0.0;
    preferences->cbdisc_mass_loss_inner_viscous_multiplier=0.0;
    preferences->cbdisc_mass_loss_inner_viscous_angular_momentum_multiplier=1.0;
    preferences->cbdisc_mass_loss_inner_L2_cross_multiplier=0.0;
    preferences->cbdisc_mass_loss_ISM_ram_pressure_multiplier=1.0;
    preferences->cbdisc_mass_loss_ISM_pressure=3000.0;
    preferences->cbdisc_minimum_luminosity=1e-10;
    preferences->cbdisc_minimum_mass=1e-10;
    preferences->cbdisc_resonance_multiplier=1.0;
#ifdef DISCS_CIRCUMBINARY_FROM_COMENV
    #endif // DISCS_CIRCUMBINARY_FROM_COMENV
#endif // DISCS
#ifdef COMENV_POLYTROPES
    preferences->comenv_splitmass = 1.0;
#endif
    preferences->accretion_start_time=-1.0;
    preferences->accretion_end_time=-1.0;
    preferences->gravitational_radiation_modulator_J=1.0;
    preferences->gravitational_radiation_modulator_e=1.0;
    preferences->solver = SOLVER_FORWARD_EULER;
    preferences->timestep_solver_factor = 1.0;
    preferences->zoomfac_multiplier_decrease = DEFAULT_ZOOMFAC_DECREASE;
    preferences->zoomfac_multiplier_increase = DEFAULT_ZOOMFAC_INCREASE;
    preferences->force_corotation_on_RLOF = TRUE;           
    preferences->max_stellar_angmom_change = MAX_STELLAR_ANGMOM_CHANGE_DEFAULT;
#ifdef BINT
    preferences->BINT_dir = getenv("BINT_DIR");
#endif//BINT

    preferences->initial_M1 = 1.0;
    preferences->initial_M2 = 0.5;
    preferences->initial_orbital_period = 0.0;
    preferences->initial_orbital_separation = 0.0;
    preferences->initial_orbital_eccentricity = 0.0;
    preferences->initial_probability = 1.0;
    preferences->metallicity = 0.02;
    preferences->effective_metallicity = DEFAULT_TO_METALLICITY;
#ifdef NUCSYN
    preferences->nucsyn_metallicity = DEFAULT_TO_METALLICITY;
#endif//NUCSYN
    preferences->max_evolution_time = 15000.0;
    preferences->initial_stellar_type1 = MAIN_SEQUENCE;
    preferences->initial_stellar_type2 = MAIN_SEQUENCE;
    preferences->overspin_algorithm = OVERSPIN_BSE;
    {
        unsigned int i;
        for(i=0;i<NUMBER_OF_FIXED_TIMESTEPS;i++)
        {
            preferences->use_fixed_timestep[i] = TRUE;
        }
    }
#ifdef GAIAHRD
    preferences->gaia_Teff_binwidth = 0.1;
    preferences->gaia_L_binwidth = 0.1;
#endif//GAIAHRD
}
