
#include "../binary_c.h"
#include "cmd_line_args.h"
#include "cmd_line_function_macros.h"
#include "cmd_line_macro_pairs.h"

#ifdef ALPHA_ARGS
#define Aprint(...) /* do nothing */
#endif//ALPHA_ARGS

/**************/
static Boolean match_scanf(const char * const arg,
                           const struct cmd_line_arg_t * const cmd_line_arg,
                           size_t * const offset_p);
static void derived_arguments(struct stardata_t * const stardata);
/**************/

void parse_arguments(const int start,
                     const int argc,
                     char ** argv,
                     struct stardata_t * RESTRICT const stardata)
{
    /*
     * Parse command line arguments and set options into appropriate
     * places inside stardata, especially in the zero_age structure
     *
     * See cmd_line_args.h for the master table : that is where
     * they are defined. If you want to add an argument, do it there
     * and rebuild. Do NOT change this function!
     *
     * If argv is non-NULL, we use it as a list of arguments
     * to be parsed.
     *
     * If argv is NULL, we use it to set up the cmd_line_args
     * array in tmpstore. This must be done for each system because 
     * it contains function pointers, and pointers inside
     * preferences, star[0,1], model etc. which may have changed
     * since the previous call.
     *
     * See also argument_setting_functions.c and batchmode.c
     */

    Dprint("PARSE sizeof stardata %ld, store %ld, store = %p, cmd_line_args = %p\n",
           (long int)sizeof(struct stardata_t),
           (long int)sizeof(struct store_t),
           stardata->tmpstore,
           stardata->tmpstore->cmd_line_args);
            
    if(argv == NULL)
    {        
        /* 
         * Make args table
         */
        struct tmpstore_t * tmpstore = stardata->tmpstore;

        Dprint("ARGS setup in tmpstore %p\n",tmpstore);        

        /*
         * see cmd_line_args.h for master argument table
         */
        {
            /*
             * Note that this array must be set up 
             * every time: it is not possible to defineit as 
             * const type because it contains pointers to the star
             * and model structs, as well as function pointers,
             * which can change from run to run.
             */
            const struct cmd_line_arg_t cmd_line_args2[] = 
                { CMD_LINE_ARGS };

            /* size of the cmd_line_args array */
            const size_t size = sizeof(cmd_line_args2);
            
            /* count the number of arguments (last's name NULL) */
            tmpstore->arg_count = size / sizeof(struct cmd_line_arg_t);

            /* allocate memory in the store for it */ 
            if(tmpstore->cmd_line_args == NULL)
            {
                tmpstore->cmd_line_args=Malloc(size);
                Dprint("Allocated %zu bytes for tmpstore->cmd_line_args table at %p\n",
                       size,
                       tmpstore->cmd_line_args);
            }
                        
            /*
             * Copy the data
             */
            memcpy(tmpstore->cmd_line_args,
                   cmd_line_args2,
                   size);

            unsigned int i;
            for(i=0;i<tmpstore->arg_count;i++)
            {
                /* 
                 * clear macro pair counters
                 */
                tmpstore->cmd_line_args[i].pairs = NULL;
                tmpstore->cmd_line_args[i].npairs = 0;
                tmpstore->cmd_line_args[i].argpairs = NULL;
                tmpstore->cmd_line_args[i].nargpairs = 0;
            }

            /* add macro pairs */
            set_cmd_line_macro_pairs(stardata,
                                     tmpstore->cmd_line_args,
                                     tmpstore->arg_count);
        }

#ifdef ALPHA_ARGS
        if(tmpstore->cmd_line_args_alpha == NULL)
        {
            /*
             * Allocate space for the alpha map: this is 
             * an array of pointers with index given by the
             * a map from the first character to an index 
             * between 0 and 51 (inclusive).
             */
            tmpstore->cmd_line_args_alpha = 
                Malloc(sizeof(struct cmd_line_arg_t *)*ALPHA_ARGS_SIZE);
        
            /*
             * Set all counts to zero
             */
            tmpstore->cmd_line_args_alpha_count =
                Calloc(ALPHA_ARGS_SIZE,sizeof(unsigned int));

            unsigned int u;
            for(u=0; u<NUMBER_OF_ALPHABETICAL_CASES; u++)
            {
                /*
                 * u = 0 -> lower case
                 * u = 1 -> upper case
                 */
                unsigned int a; 
                const unsigned int offset =
                    (const unsigned int) (u == 0 ? 'a' : 'A');
                for(a=0; a<NUMBER_OF_LETTERS_IN_THE_ALPHABET; a++)
                {
                    /* 
                     * Hence the char that is the first letter
                     */
                    const unsigned char c = offset + a;
                    const unsigned int index = a + u * NUMBER_OF_LETTERS_IN_THE_ALPHABET;
                    Aprint("allocate alpha structure %d (char %d = %c = %c)\n",
                           a,
                           (int)c,
                           offset+a,
                           c);
                    /*
                     * NULL the array (its index is zero, set by memset above)
                     */
                    tmpstore->cmd_line_args_alpha[index] = NULL;                
                
                    /*
                     * Loop over cmd_line_args2 to find the appropriate 
                     * mapping
                     */
                    unsigned int i;
                    for(i=0;i<tmpstore->arg_count;i++)
                    {
                        if(tmpstore->cmd_line_args[i].name[0] == c)
                        {
                            Aprint("mapped arg %s here\n",
                                   tmpstore->cmd_line_args[i].name);
                            /*
                             * Increase size
                             */
                            tmpstore->cmd_line_args_alpha[index] = Realloc(
                                tmpstore->cmd_line_args_alpha[index],
                                sizeof(struct cmd_line_arg_t) *
                                (1+tmpstore->cmd_line_args_alpha_count[index]));
                        
                            /*
                             * Copy the arg data into the sublist
                             */
                            memcpy(&tmpstore->cmd_line_args_alpha[index][tmpstore->cmd_line_args_alpha_count[index]],
                                   &tmpstore->cmd_line_args[i],
                                   sizeof(struct cmd_line_arg_t));

                            /*
                             * Expand count for the next item
                             */
                            tmpstore->cmd_line_args_alpha_count[index]++;

                            Aprint("new count[index=%d] %d\n",
                                   index,
                                   tmpstore->cmd_line_args_alpha_count[index]);
                        }
                    }
                }
            }
        }
#endif // ALPHA_ARGS
        
#undef ALPHA_ARGS
        Dprint("Arg table built : we have %u args (argc=%d)\n",
               tmpstore->arg_count,argc);
    }
    else
    {
	/*
	 * Parse command line arguments
	 */
        struct cmd_line_arg_t *cmd_line_args = stardata->tmpstore->cmd_line_args;
	
#ifdef DEBUG_FAIL_ON_NAN
        const Boolean allow_debug_nan_was = stardata->preferences->allow_debug_nan;
        stardata->preferences->allow_debug_nan = TRUE;
#endif
#ifdef DEBUG_FAIL_ON_INF
        const Boolean allow_debug_inf_was = stardata->preferences->allow_debug_inf;
        stardata->preferences->allow_debug_inf = TRUE;
#endif

        /* 
         * loop over the arguments, looking for
         * matches and acting accordingly
         */
	int c;
	for(c=start; c<argc; c++)
	{
	    Dprint("CMD LINE ARG %d/%d allow nan ? %d",c,argc,stardata->preferences->allow_debug_nan);
            Dprint(" is %s : process ",argv[c]);
	    char * arg = (char*) argv[c];

	    /* ignore leading -- */
	    if((arg[0] == '-') && (arg[1] == '-')) arg += 2;

#ifdef ALPHA_ARGS
            /*
             * Override cmd_line_args with the appropriate 
             * alpha-selected list
             */
            Aprint("Check arg %s first char %c (ASCII letter? %s : upper case? %s, lower case? %s)\n",
                   arg,
                   arg[0],
                   Yesno(ASCII_letter(arg[0])),
                   Yesno(ASCII_upper_case(arg[0])),
                   Yesno(ASCII_lower_case(arg[0]))
                );
            
            int arg_count;
            if(ASCII_letter(arg[0]))
            {
                /*
                 * First character is a letter: use the alpha map 
                 */
                index = ASCII_letter_to_index(arg[0]);
                
                /*
                 * Override the cmd_line_args and arg count with those
                 * for this starting letter
                 */
                arg_count = stardata->tmpstore->cmd_line_args_alpha_count[index];
                cmd_line_args = stardata->tmpstore->cmd_line_args_alpha[index];
                Aprint("Using alpha map for %c : index %d : arg_count = %d\n",
                       arg[0],index,arg_count);
            }
            else
            {
                /*
                 * Not ASCII : use the big table
                 * (this shouldn't happen)
                 */
                arg_count = stardata->tmpstore->arg_count;
                cmd_line_args = stardata->tmpstore->cmd_line_args;
            }
#else
            /*
             * No map: check the whole cmd_line_args in store
             */
            const int arg_count = stardata->tmpstore->arg_count;
            cmd_line_args = stardata->tmpstore->cmd_line_args;
#endif//ALPHA_ARGS
            
            /* 
             * check this argument against all those in the list
             */
	    unsigned int i;
            size_t offset = 0;
	    Boolean success = FALSE;

            for(i=0; i<arg_count; i++)
            {
                if(
                    Strings_equal(arg,
                                  cmd_line_args[i].name)
                    ||
                    (
                        Arg_is_scanf(cmd_line_args[i].type) && 
                        match_scanf(arg,
                                    &cmd_line_args[i],
                                    &offset) == TRUE)
                    )
                    
                {
                    Dprint("match \"%s\" (at arg table %u, type %d, pointer %p : cf log_filename at %p) repeat = %d\n",
                           arg,
                           i,
                           cmd_line_args[i].type,
                           cmd_line_args[i].pointer,
                           &(stardata->preferences->log_filename),
                           stardata->preferences->repeat
                        );
                    
                    success = TRUE;

                    if(cmd_line_args[i].pointer == NULL)
                    {	    
                        /*
                         * If the pointer is NULL, ignore this
                         * arg, but skip as many following args
                         * as required. This means calling
                         * the appropriate subroutine, or 
                         * simply incrementing the counter (i.e. c++).
                         */
                        if(Arg_is_subroutine(cmd_line_args[i].type))
                        {
                            void (*func)(ARG_SUBROUTINE_DECLARATION);
                            func = cmd_line_args[i].pointer;
                            func(ARG_SUBROUTINE_ARGS);
                        }
                        else
                        {
                            c++;
                        }
                    }
                    else
                    {
                        /* code to check if the next arg exists when it's required */
#define Next_arg_check                                                  \
                        if(c+1>argc || argv[c+1] == NULL)               \
                        {                                               \
                            Exit_binary_c(                              \
                                BINARY_C_SETUP_UNKNOWN_ARGUMENT,        \
                                "Exit because cmd line arg #%d \"%s\" requires a value which is not the next argument (c+1=%d argc=%d prev arg =\"%s\" next arg = \"%s\").", \
                                c,                                      \
                                arg,                                    \
                                c+1,                                    \
                                argc,                                   \
                                argv[c == 0?0:c-1],                     \
                                (c<argc ? argv[c+1] : "NULL")           \
                                );                                      \
                        }


                        /* Act based on argument type */
                        switch(cmd_line_args[i].type)
                        {
                        case ARG_NONE:
                            /* do nothing */
                            break;
                        case ARG_FLOAT:
                            Next_arg_check;
                            Arg_set_double;
                            break;
                        case ARG_INTEGER:
                            /* integer argument */
                            Next_arg_check;
                            Arg_set_int;
                            break;
                        case ARG_UNSIGNED_INTEGER:
                            /* unsigned integer argument */
                            Next_arg_check;
                            Arg_set_unsigned_int;
                            break;
                        case ARG_INTEGER_SCANF:
                            /* integer argument with a scanf for the index */
                            Next_arg_check;
                            Arg_scanf_set_int;
                            break;
                        case ARG_BOOLEAN_SCANF:
                            /* integer argument with a scanf for the index */
                            Next_arg_check;
                            Arg_scanf_set_Boolean;
                            break;
                        case ARG_FLOAT_SCANF:
                            /* integer argument with a scanf for the index */
                            Next_arg_check;
                            Arg_scanf_set_double;
                            break;
                        case ARG_LONG_INTEGER:
                            /* long integer argument */
                            Next_arg_check;
                            Arg_set_long_int;
                            break;
                        case ARG_BOOLEAN:
                        case BATCH_ARG_BOOLEAN:
                            /* Boolean type set to TRUE */
                            Arg_set_boolean_true;
                            break;
                        case ARG_NOT_BOOLEAN:
                        case BATCH_ARG_NOT_BOOLEAN:
                            /* Boolean type set to FALSE */
                            Arg_set_boolean_false;
                            break;
                        case ARG_STRING:
                            /* string argument : NB no bounds checking! */
                            Next_arg_check;
                            Arg_set_string;
                            break;
                        case ARG_SUBROUTINE:
                        case ARG_SUBROUTINE_RETURN_FLOAT:
                        case ARG_SUBROUTINE_RETURN_INT:
                        case BATCH_ARG_SUBROUTINE:
                            /* subroutine to call */
                            Arg_call_subroutine;
                            break;
                        default:
                            /* Matched argument but no idea what to do with it */
                            Dprint(
                                "Exit because argument type %d is unknown (check cmd_line_args.h at position %u)",
                                cmd_line_args[i].type,
                                i);
                            Exit_binary_c(
                                BINARY_C_SETUP_UNKNOWN_ARGUMENT,
                                "Exit because argument type %d is unknown (check cmd_line_args.h at position %u)",
                                cmd_line_args[i].type,
                                i);
                        }
                    }

                    /*
                     * Matched : break out of i loop 
                     */
                    break;
                }
            }

            /*
             * exit on error : this should never happen!
             */
            if(success == FALSE)
            {
                Dprint(
                    "Exit because given cmd line arg \"%s\" (number %d) failed to match any known argument.",
                    arg,c);
                Exit_binary_c(
                    BINARY_C_SETUP_UNKNOWN_ARGUMENT,
                    "Exit because given cmd line arg  \"%s\" (number %d) failed to match any known argument (prev args are \"%s\" and \"%s\").",
                    arg,
                    c,
                    argv[c<=1?0:c-2],
                    argv[c<=0?1:c-1]
                    );
            }
        } /* loop over cmd line args */
        Dprint("Finished processing cmd line arguments\n");

#ifdef DEBUG_FAIL_ON_NAN
        stardata->preferences->allow_debug_nan = allow_debug_nan_was;
#endif//DEBUG_FAIL_ON_NAN
#ifdef DEBUG_FAIL_ON_INF
        stardata->preferences->allow_debug_inf = allow_debug_inf_was;
#endif//DEBUG_FAIL_ON_INF

        /*
         * Derived variables cases
         */
        derived_arguments(stardata);
    }
}

/********************************************************************/

static Boolean match_scanf(const char * const arg,
                           const struct cmd_line_arg_t * const cmd_line_arg,
                           size_t * const offset_p)
{
    /*
     * Use sscanf to test the argument, return TRUE if it matches
     * and set *offset_p.
     *
     * Note: offset_p is an int pointer so it matches the %d
     *       pattern that is commonly used. The integer should
     *       always be positive, however,   
     */
    const int matching_int = 0;
    const int scantest = sscanf(arg,
                                cmd_line_arg->name,
                                &matching_int); 
    if(scantest > 0)
    {
        /*
         * Normal sscanf succeeded
         */
        *offset_p = (size_t) matching_int;
        return TRUE;
    }
    else if(cmd_line_arg->nargpairs > 0)
    {
        /*
         * Scan argpairs 
         */
        unsigned int i;
        for(i=0; i<cmd_line_arg->nargpairs; i++)
        {
            if(Strings_equal(arg,
                             cmd_line_arg->argpairs[i].string))
            {
                /*
                 * Matched : set the appropriate offset
                 * to be the offset
                 */
                *offset_p = (size_t) (cmd_line_arg->argpairs[i].value + TINY);
                return TRUE;
            }
        }
        return FALSE;
    }
    else
    {
        return FALSE;
    }
}

static void derived_arguments(struct stardata_t * const stardata)
{

    /*
     * Set stellar parameters 
     */
    stardata->star[0].mass = stardata->preferences->initial_M1;
    stardata->star[1].mass = stardata->preferences->initial_M2;
    stardata->common.orbit.period = stardata->preferences->initial_orbital_period;
    stardata->common.orbit.separation = stardata->preferences->initial_orbital_separation;
    stardata->common.orbit.eccentricity = stardata->preferences->initial_orbital_eccentricity;
    stardata->model.probability = stardata->preferences->initial_probability;
    stardata->common.metallicity = stardata->preferences->metallicity;
    Clamp(stardata->common.metallicity,1e-4,0.03);
    stardata->common.effective_metallicity = stardata->preferences->effective_metallicity;
#ifdef NUCSYN
    stardata->common.nucsyn_metallicity = stardata->preferences->nucsyn_metallicity;
#endif//NUCSYN
    stardata->model.max_evolution_time = stardata->preferences->max_evolution_time;
    stardata->star[0].stellar_type = stardata->preferences->initial_stellar_type1;
    stardata->star[1].stellar_type = stardata->preferences->initial_stellar_type2;
    
    stardata->common.orbit.angular_momentum = 0.0;
    Dprint("masses %g %g\n",
           stardata->star[0].mass,
           stardata->star[1].mass);

    if(Is_not_zero(stardata->star[0].mass) &&
       Is_not_zero(stardata->star[1].mass))
    {
        Dprint("preset : sep = %g per = %g\n",
               stardata->common.orbit.separation,
               stardata->common.orbit.period);
        if(Is_zero(stardata->common.orbit.period) &&
           Is_not_zero(stardata->common.orbit.separation))
        { 
            /*
             * Period is set to zero, but separation is given.
             * Hence calculate the initial period.
             */
            stardata->common.orbit.period = 
                calculate_orbital_period(stardata);
        }
        else if(Is_not_zero(stardata->common.orbit.period))
        {
            /*
             * Update the separation here so that the "INITIAL" log is 
             * correct. (This is reset later anyway.)
             */
            stardata->common.orbit.period /= YEAR_LENGTH_IN_DAYS;
            stardata->common.orbit.separation =
                calculate_orbital_separation(stardata);
            stardata->common.orbit.period *= YEAR_LENGTH_IN_DAYS;
        }
    }

    Dprint("postset: sep = %g per = %g\n",
           stardata->common.orbit.separation,
           stardata->common.orbit.period);
    
    /*
     * If we are using log times, choose the next
     * max_evolution_time that correponds to an 
     * exact number of log timesteps 
     */
#if defined NUCSYN && defined NUCSYN_YIELDS
    if(stardata->preferences->yields_logtimes == TRUE)
    {
        stardata->model.max_evolution_time =
            pow(10.0,
                stardata->preferences->yields_logdt *
                (1.0+(int)(log10(stardata->model.max_evolution_time)/
                           stardata->preferences->yields_logdt)));
    }
#endif // NUCSYN && NUCSYN_YIELDS

    Dprint("end parse arguments\n");
}
