#pragma once
#ifndef BINARY_C_CMD_LINE_ARGS_H
#define BINARY_C_CMD_LINE_ARGS_H

/*
 * Definitions for the command-line argument reading module
 *
 * To add an argument to binary_c, add a line to the table defined at
 * CMD_LINE_ARGS below and see the comment immediately before it.
 *
 * See also cmd_line_function_macros.h
 */

/*
 * Suck in macros that depend on code build options
 */ 
#include "cmd_line_macros.h"
#include <errno.h>

/* 
 * Argument types
 * 
 * Most arguments are called as
 * "--argument <value>"
 * where the value is a float, integer or string.
 * These are types ARG_FLOAT, ARG_INTEGER, ARG_STRING, 
 * ARG_BOOLEAN etc.
 *
 * ARG_NONE is a special case which is ignored.

 * Finally ARG_SUBROUTINE calls a subroutine which
 * handles more complex cases, e.g. multi-value 
 * settings. If you want to make one of these, just
 * copy-paste one of the existing functions and
 * hack it to your requirements.
 *
 * ARG_SUBROUTINE_WITH_RETURN_(INT|FLOAT) is a variant on
 * the above whereby the function returns a value.
 * This is useful for determining default values.
 *
 * The ARG*SCANF functions take an integer as a suffix, 
 * this is converted to an index to be added to the pointer
 * provided.
 */

#define ARG_FLOAT 0
#define ARG_STRING 1
#define ARG_NONE 2
#define ARG_INTEGER 3
#define ARG_SUBROUTINE 4
#define ARG_SUBROUTINE_RETURN_INT 5
#define ARG_SUBROUTINE_RETURN_FLOAT 6
#define ARG_BOOLEAN 7
#define ARG_LONG_INTEGER 8
#define ARG_NOT_BOOLEAN 9
#define BATCH_ARG_SUBROUTINE 10
#define BATCH_ARG_BOOLEAN 11
#define BATCH_ARG_NOT_BOOLEAN 12
#define ARG_INTEGER_SCANF 13
#define ARG_FLOAT_SCANF 14
#define ARG_UNSIGNED_INTEGER 15
#define ARG_BOOLEAN_SCANF 16

#include "../binary_c_code_options.h"
#include "../binary_c_debug.h"

#define Cprint(...)
//#define Cprint(...) printf(__VA_ARGS__)

/*
 * Macros for setting argument values from the command
 * line strings.
 *
 * Under/overflows are caught for (long) integer and double 
 * types.
 *
 * Strings are limited to STRING_LENGTH characters 
 * (defined in binary_c_macros.h).
 */


/* define a logical section for each argument */
#define ARG_SECTION_STARS 0
#define ARG_SECTION_BINARY (ARG_SECTION_STARS+1)
#define ARG_SECTION_NUCSYN (ARG_SECTION_BINARY+1)
#define ARG_SECTION_OUTPUT (ARG_SECTION_NUCSYN+1)
#define ARG_SECTION_INPUT (ARG_SECTION_OUTPUT+1)
#define ARG_SECTION_IO (ARG_SECTION_INPUT+1)
#define ARG_SECTION_ALGORITHMS (ARG_SECTION_IO+1)
#define ARG_SECTION_MISC  (ARG_SECTION_ALGORITHMS+1)
#define NUMBER_OF_ARG_SECTIONS (ARG_SECTION_MISC+1)

/* strings to describe the sections */
#define ARG_SECTION_STRINGS {                   \
        "Stars",                                \
            "Binary",                           \
            "Nucsyn",                           \
            "Output",                           \
            "Input",                            \
            "I/O",                              \
            "Algorithms",                       \
            "Misc"                              \
            }

/*
 * strings labelled with WTTS_USE_DEFAULT should
 * output the default value, when available, rather 
 * than a string specified here
 */
#define WTTS_USE_DEFAULT NULL

/*
 * Arguments to automated subroutine calls
 * and equivalent declarations are defined here.
 * Given that stardata is included, as is the 
 * array of arguments and the counter position (*c)
 * you shouldn't need to change these.
 */
#define ARG_SUBROUTINE_ARGS                     \
    cmd_line_args,                              \
        stardata,                               \
        &c,                                     \
        i,                                      \
        stardata->tmpstore->arg_count,          \
        argv,                                   \
        argc

#define ARG_SUBROUTINE_ARGS2                    \
    cmd_line_args,                              \
        stardata,                               \
        c,                                      \
        i,                                      \
        stardata->tmpstore->arg_count,          \
        argv,                                   \
        argc

#define ARG_SUBROUTINE_DECLARATION                      \
    struct cmd_line_arg_t * const cmd_line_args,        \
        struct stardata_t * RESTRICT const stardata,    \
        int * const c,                                  \
        int i,                                          \
        int arg_count,                                  \
        char ** argv,                                   \
        int argc

/*
 * Structure to hold a macro - value pair
 */
struct arg_pair_t
{
    char * string;
    double value; /* NB ints and Booleans can be cast in here, we rely on sizeof(double)>=Max(sizeof(Boolean),sizeof(int)) which is always true */
};

/* 
 * Structure to hold the information for each argument
 */
struct cmd_line_arg_t
{
    int section;
    const char *name;
    const char *help;
    int type;
    const char *wtts_string;
    void *pointer;
    double modulate;
    struct arg_pair_t * pairs; 
    unsigned int npairs;
    struct arg_pair_t * argpairs;
    unsigned int nargpairs;
};

/*
 * Each cmd_line_art structure should contain:
 *
 * section (int) : this is the section of the help the argument should be in 
 *                 (useful for WTTS)
 *
 * name (string) : this is the argument itself, e.g. "metallicity" for argument "--metallicity"
 *
 * help (string) : a string containing help text
 *
 * type (int) : one of ARG_FLOAT, ARG_STRING, ARG_INTEGER, ARG_BOOLEAN etc.
 *              (as defined at the top of this file) 
 *              depending on whether the following arguments is(are) floating-point 
 *              number(s), string(s), integers, booleans.
 *
 *              Alternatively, it is ARG_SUBROUTINE, which calls the subroutine instead.
 *
 *              ARG_NONE is a type that is ignored.
 *
 * wtts_string (string) [currently for WTTS only]
 *        : a string for WTTS
 *                 WTTS_USE_DEFAULT : uses the default,
 *                  "Ignore" to be ignored by WTTS,
 *                 "No_Show <default>" to not be shown by WTTS but have its value sent to the
 *                 binary_c command. If this is a macro you can use Stringof(MACRONAME).
 * NB binary_grid ignores this value and uses the code's actual default.
 *
 * pointer (void *) : this points to either the value to be set, or the 
 *                    subroutine to be called.
 *
 *                    NB the type of the variable to which the pointer points
 *                    MUST be the same as the type (ARG_...) as defined above,
 *                    otherwise there will be trouble.
 * 
 *                    NB If the type is ARG_STRING, there must be
 *                    sizeof(char)*STRING_LENGTH of memory allocated at the pointer's location.
 *
 * modulate (FLOAT_TYPE) : floating point number used to modulate passed in numbers (very useful for 
 *            converting units, Myr->years, etc) : ignored unless type ARG_FLOAT
 *
 * pairs : a set of macro-integer value pairs. Is not defined below, so is NULL
 *         by default. See set_cmd_line_macro_pairs.
 *
 * npairs : the number of macro-integer value pairs, 0 by default.
 * 
 * argpairs : arguments that are of the form xyz%d can have macros mapped
 *            to the integer that maps %d. Put them in here. NULL by default,
 *            and not set here. See set_cmd_line_macro_pairs.c)
 * 
 * nargpairs : the number of argpairs, 0 by default.
 */   


/*
 * These args are only defined with BATCHMODE defined
 */

#ifdef BATCHMODE
#define BATCHMODE_ARGS                                                  \
    {                                                                   \
        ARG_SECTION_IO,                                                 \
            "go",                                                       \
            "batchmode control command",                                \
            BATCH_ARG_SUBROUTINE,                                       \
            "Ignore",                                                   \
            &(batchmode_go) ,                                           \
            1.0                                                         \
            } ,                                                         \
    {                                                                   \
        ARG_SECTION_IO,                                                 \
            "gogo",                                                     \
            "batchmode control command",                                \
            BATCH_ARG_SUBROUTINE,                                       \
            "Ignore",                                                   \
            &(batchmode_gogo) ,                                         \
            1.0                                                         \
            } ,                                                         \
{                                                                       \
    ARG_SECTION_IO,                                                     \
        "reset_stars",                                                  \
        "Reset the star structures. Used in batchmode",                 \
        BATCH_ARG_SUBROUTINE,                                           \
        "Ignore",                                                       \
        &(batchmode_reset_stars) ,                                      \
        1.0                                                             \
        } ,                                                             \
{                                                                       \
    ARG_SECTION_IO,                                                     \
        "reset_stars_defaults",                                         \
        "Reset the star structures and set defaults. Used in batchmode", \
        BATCH_ARG_SUBROUTINE,                                           \
        "Ignore",                                                       \
        &(batchmode_reset_stars_defaults) ,                             \
        1.0                                                             \
        } ,                                                             \
{                                                                       \
    ARG_SECTION_IO,                                                     \
        "defaults",                                                     \
        "Reset all defaults. Used in batchmode",                        \
        BATCH_ARG_SUBROUTINE,                                           \
        "Ignore",                                                       \
        &(batchmode_defaults) ,                                         \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_IO,                                                     \
        "echo",                                                         \
        "Activate batchmode command echoing, i.e. when you enter a command, binary_c repeats the command then executes it.", \
        BATCH_ARG_SUBROUTINE,                                           \
        "Ignore",                                                       \
        &(batchmode_set_echo_on),                                       \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_IO,                                                     \
        "noecho",                                                       \
        "Deactivate batchmode command echoing. See 'echo'.",            \
        BATCH_ARG_SUBROUTINE,                                           \
        "Ignore",                                                       \
        &(batchmode_set_echo_off),                                      \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_IO,                                                     \
        "noechonow",                                                    \
        "Deactivate batchmode command echoing. See 'echo'.",            \
        BATCH_ARG_SUBROUTINE,                                           \
        "Ignore",                                                       \
        &(batchmode_set_echo_off),                                      \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_IO,                                                     \
        "bye",                                                          \
        "Quit binary_c. Used in batchmode.",                            \
        BATCH_ARG_SUBROUTINE,                                           \
        "Ignore",                                                       \
        &(batchmode_bye) ,                                              \
        1.0                                                             \
        } ,                                                             \
{                                                                       \
    ARG_SECTION_IO,                                                     \
        "fin",                                                          \
        "batchmode control command",                                    \
        BATCH_ARG_SUBROUTINE,                                           \
        "Ignore",                                                       \
        &(batchmode_fin) ,                                              \
        1.0                                                             \
        } ,                                                             \
{                                                                       \
    ARG_SECTION_IO,                                                     \
        "reset_prefs",                                                  \
        "Reset preferences struct. Used in batchmode",                  \
        BATCH_ARG_SUBROUTINE,                                           \
        "Ignore",                                                       \
        &(batchmode_reset_prefs) ,                                      \
        1.0                                                             \
        } ,                                                             \
{                                                                       \
    ARG_SECTION_IO,                                                     \
        "status",                                                       \
        "Output batchmode status information.",                         \
        BATCH_ARG_SUBROUTINE,                                           \
        "Ignore",                                                       \
        &(batchmode_status) ,                                           \
        1.0                                                             \
        },

#else
#define BATCHMODE_ARGS
#endif


/*
 * Now define the main CMD_LINE_ARGS array of structures.
 *
 * Put the most often called at the top...
 */
#define CMD_LINE_ARGS                                                   \
    BATCHMODE_ARGS                                                      \
    {                                                                   \
        ARG_SECTION_STARS,                                              \
            "metallicity",                                              \
            "This sets the metallicity of the stars, i.e. the amount (by mass) of matter which is not hydrogen or helium. If you are using the BSE algorithm, this must be 1e-4 <= metallicity <= 0.03. See also nucsyn_metallicity and effective_metallicity.", \
            ARG_FLOAT,                                                  \
            WTTS_USE_DEFAULT,                                           \
            &stardata->preferences->metallicity,                        \
            1.0                                                         \
            } ,                                                         \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "effective_metallicity",                                        \
        "This sets effective metallicity of stars as used in routines like the Schneider wind loss. If not set, or set to DEFAULT_TO_METALLICITY (==-1, the default), this is just the same as metallicity. The main difference between effective_metallicity and metallicity is the range of validity: 0 <= effective_metallicity <= 1, while metallicity's range of validity is limited by the stellar evolution algorithm (so, for BSE, is 1e-4 <= metallicity <= 0.03).", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->effective_metallicity),                \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "M_1",                                                          \
        "The mass of star one (in solar units).",                       \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        &stardata->preferences->initial_M1,                             \
        1.0                                                             \
        } ,                                                             \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "M_2",                                                          \
        "The mass of star two (in solar units).",                       \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        &stardata->preferences->initial_M2,                             \
        1.0                                                             \
        } ,                                                             \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "eccentricity",                                                 \
        "Set the eccentricity of the binary orbit (0-1).",              \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->initial_orbital_eccentricity)	,       \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "orbital_period",                                               \
        "Set the orbital period in days. See also separation.",         \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->initial_orbital_period)	,       \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "probability",                                                  \
        "The probability is a weighting applied to the star based on, say, the initial mass function. When running a grid of stars to simulate *all* stars, the summed probability of all the stars should be 1.0.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->initial_probability)	,               \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "phasevol",                                                     \
        "The system's phase volume, used by binary_grid.",              \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        PHASEVOL_VAR,                                                   \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "stellar_type_2",                                               \
        "Set the stellar type of star 2, usually 1 (main sequence)",    \
        ARG_INTEGER,                                                    \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->initial_stellar_type2)	,               \
        1                                                               \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "stellar_structure_algorithm",                                  \
        "Set the stellar structure algorithm. 0=modified BSE (default), 1=none, 2=external function (must be defined by the calling code), 3=binary_c (not yet implemented).", \
        ARG_INTEGER,                                                    \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->stellar_structure_algorithm),          \
        1                                                               \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "solver",                                                       \
        "The type of solver. Default is the Forward-Euler (0), but could be RK2 (1), RK4 (2) or a predictor-corretor (3).", \
        ARG_INTEGER,                                                    \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->solver),                               \
        1                                                               \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "max_evolution_time",                                           \
        "Set the maximum age for the stars (Myr).",                     \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->max_evolution_time),                   \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "max_model_number",                                             \
        "Set the maximum number of models, ignored if 0 (default is 0).", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->max_model_number),                     \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "monte_carlo_kicks",                                            \
        "Turn on Monte-Carlo SN kicks. On (True) by default, and indeed other algorithms are probably broken.", \
        ARG_BOOLEAN ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->monte_carlo_sn_kicks),                 \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "disc_timestep_factor",                                         \
        "Factor that multiplies the natural timestep of a disc.",       \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        DISC_TIMESTEP_FACTOR_VAR,                                       \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_mass_loss_inner_viscous_accretion_method",              \
        "Chooses where the mass that is accreted from the inner edge of a circumbinary disc goes, i.e. to which star. 0 = Young and Clarke 2015, 1 = Gerosa et al 2015, 2 = 50:50 (i.e. not dependence on mass).", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_MASS_LOSS_INNER_VISCOUS_ACCRETION_METHOD_VAR,            \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_inner_edge_stripping",                                  \
        "If True, allow inner edge mass stripping.",                    \
        ARG_BOOLEAN ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_INNER_EDGE_STRIPPING_VAR,                                \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_outer_edge_stripping",                                  \
        "If True, allow outer edge mass stripping.",                    \
        ARG_BOOLEAN ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_OUTER_EDGE_STRIPPING_VAR,                                \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "disc_n_monte_carlo_guesses",                                   \
        "Number of monte carlo guesses to try in the disc solver if the normal list of guesses fails (0).", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        DISC_N_MONTE_CARLO_GUESSES_VAR,                                 \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "disc_log",                                                     \
        "If 1, turn on the disc log. Requires DISC_LOG to be defined on build.", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        DISC_LOG_VAR,                                                   \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "disc_log2d",                                                   \
        "If 1, turn on the 2d disc log. Requires DISC_LOG to be defined on build.", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        DISC_LOG2D_VAR,                                                 \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "disc_log_dt",                                                  \
        "If non-zero, only allows disc log output every disc_log_dt Myr.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        DISC_LOG_DT_VAR,                                                \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "disc_log_directory",                                           \
        "Directory into which disc logging is sent (must exist!).",     \
        ARG_STRING ,                                                    \
        "/tmp/",                                                        \
        DISC_LOG_DIRECTORY_VAR,                                         \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_eccentricity_pumping_method",                           \
        "Select from various eccentricity-pumping methods when there is a circumbinary disc. Requires DISCS. 0 = off.", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_ECCENTRICITY_PUMPING_METHOD_VAR,                         \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_viscous_photoevaporation_coupling",                     \
        "Set to 1 to turn on viscous-photoevaporation coupling in circumbinary discs. Requires DISCS. 0 = off.", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_VISCOUS_PHOTOEVAPORATION_COUPLING_VAR,                   \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_inner_edge_stripping_timescale",                        \
        "Defines the timescale for mass loss from by inner edge stripping. 0 = instant, 1 = very long, 2 = viscous at Revap_in, 3 = orbital at Revap_in.", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_INNER_EDGE_TIMESCALE_VAR,                                \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_outer_edge_stripping_timescale",                        \
        "Defines the timescale for mass loss from by outer edge stripping. 0 = instant, 1 = very long, 2 = viscous at Revap_in, 3 = orbital at Revap_out.", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_OUTER_EDGE_TIMESCALE_VAR,                                \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_viscous_L2_coupling",                                   \
        "Set to 1 to turn on viscous-L2-loss coupling in circumbinary discs. Requires DISCS. 0 = off.", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_VISCOUS_L2_COUPLING_VAR,                                 \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "gravitational_radiation_model",                                \
        "Model for gravitational radiation from the system. 0=Hurley et al 2002 (Landau and Lifshitz 1951). 1 = as 0 but only when there is no RLOF. 2 = none.", \
        ARG_INTEGER,                                                    \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->gravitational_radiation_model),        \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "nova_irradiation_multiplier",                                  \
        "Multiplier for nova-radiative induced mass loss. (Shara+1986)", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->nova_irradiation_multiplier),          \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "gravitational_radiation_modulator_J",                          \
        "Modulator for gravitational wave radiation angular momentum loss rate (1.0).", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->gravitational_radiation_modulator_J),  \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "gravitational_radiation_modulator_e",                          \
        "Modulator for gravitational wave radiation eccentricity pumping rate (1.0).", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->gravitational_radiation_modulator_e),  \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "nova_faml_multiplier",                                         \
        "Nova friction-induced angular momentum loss multiplier. (Shara+1986)", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->nova_faml_multiplier),                 \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "RLOF_angular_momentum_transfer_model",                         \
        "Choose angular momentum transfer model in RLOF. 0=BSE (with discs), 1=conservative", \
        ARG_INTEGER,                                                    \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->RLOF_angular_momentum_transfer_model), \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "post_SN_orbit_method",                                         \
        "Method by which the post-SN orbit is calculated. 0=BSE, 1=Tauris&Taken 1998.", \
        ARG_INTEGER,                                                    \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->post_SN_orbit_method),                 \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "evolution_splitting",                                          \
        "If True, turn on splitting of an evolutionary run if splitpoint (e.g. supernova) occurs.", \
        ARG_BOOLEAN,                                                    \
        WTTS_USE_DEFAULT,                                               \
        EVOLUTION_SPLITTING_VAR,                                        \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "disable_events",                                               \
        "Whether to disable the new events code (defaults to False, so we use events by default)\n", \
        ARG_BOOLEAN,                                                    \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->disable_events),                       \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "evolution_splitting_sn_n",                                     \
        "Number of runs to split into when a SN occurs.",               \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        EVOLUTION_SPLITTING_SN_N_VAR,                                   \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "evolution_splitting_maxdepth",                                 \
        "Max number of splits in an evolutionary run.",                 \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        EVOLUTION_SPLITTING_MAXDEPTH_VAR,                               \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_MISC,                                                   \
        "random_seed",                                                  \
        "Random number seed, usually a (possibly negative) integer. Useful for exactly reproducing the evolution of a system which involves a kick (which is a Monte-Carlo, i.e. pseudorandom, process)." , \
        ARG_LONG_INTEGER,                                               \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->cmd_line_random_seed),                 \
        1.0                                                             \
        } ,                                                             \
{                                                                       \
    ARG_SECTION_MISC,                                                   \
        "random_skip",                                                  \
        "Skip the first <random_seed> random numbers that are generated. Usually this is 0 so they are all used.", \
        ARG_LONG_INTEGER,                                               \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->random_skip),                          \
        1.0                                                             \
        } ,                                                             \
{                                                                       \
    ARG_SECTION_MISC,                                                   \
        "idum",                                                         \
        "[NB: deprecated, please use 'random_seed' instead.] Random number seed, usually a (possibly negative) integer. Useful for exactly reproducing the evolution of a system which involves a kick (which is a Monte-Carlo, i.e. pseudorandom, process)." , \
        ARG_LONG_INTEGER,                                               \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->cmd_line_random_seed),                 \
        1.0                                                             \
        } ,                                                             \
{                                                                       \
    ARG_SECTION_ALGORITHMS,                                             \
        "repeat",                                                       \
        "If > 1, repeats the system as many times as required. Handy if you're using Monte-Carlo kicks and want to sample the parameter space well. Also, if you are running speed tests this is good to give a statistically more reasonable result. (See e.g. 'tbse pgo')." , \
        ARG_INTEGER,                                                    \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->repeat),                               \
        1.0                                                             \
        } ,                                                             \
{                                                                       \
    ARG_SECTION_ALGORITHMS,                                             \
        "random_systems",                                               \
        "Experimental. Use this to apply random initial system parameters (masses, separations, etc.). Useful for testing only." , \
        ARG_INTEGER,                                                    \
        WTTS_USE_DEFAULT,                                               \
        RANDOM_SYSTEMS_VAR,                                             \
        1.0                                                             \
        } ,                                                             \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "stellar_type_1",                                               \
        "Set the stellar type of star 1, usually 1 (main sequence)",    \
        ARG_INTEGER,                                                    \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->initial_stellar_type1)	,               \
        1                                                               \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "equation_of_state_algorithm",                                  \
        "Set the equation of state algorithm. 0 = Paczynski.",          \
        ARG_INTEGER,                                                    \
        WTTS_USE_DEFAULT,                                               \
        EQUATION_OF_STATE_VAR,                                          \
        1                                                               \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "opacity_algorithm",                                            \
        "Set the opacity algorithm. 0 = Paczynski, 1 = Ferguson/Opal.", \
        ARG_INTEGER,                                                    \
        WTTS_USE_DEFAULT,                                               \
        OPACITY_VAR,                                                    \
        1                                                               \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "wind_mass_loss",                                               \
        "Defines the algorithm used for stellar winds. 0 = none, 1 = Hurley et al. (2002), 2 = Schneider (2018).", \
        ARG_UNSIGNED_INTEGER ,                                          \
        "0",                                                            \
        &(stardata->preferences->wind_mass_loss),                       \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "gbwind",                                                       \
        "Wind prescription for first red giant branch stars.  0=Reimers (Hurley et al 2000/2002; choose gb_reimers_eta=0.5 for their mass loss rate), 1=Schroeder+Cuntz 2005 (set gb_reimers_eta=1.0 for their mass loss rate).", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->gbwind),                               \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "mattsson_Orich_tpagbwind",                                     \
        "Experimental : turns on Mattsson's TPAGB wind when the star is oxygen rich. Requires MATTSSON_MASS_LOSS.", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        MATTSSON_ORICH_TPAGBWIND_VAR,                                   \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "magnetic_braking_factor",                                      \
        "Multiplier for the magnetic braking angular momentum loss rate.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->magnetic_braking_factor),              \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "magnetic_braking_gamma",                                       \
        "gamma factor in Rappaport style magnetic braking expression.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->magnetic_braking_gamma),               \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "magnetic_braking_algorithm",                                   \
        "Algorithm for the magnetic braking angular momentum loss rate. 0 = Hurley et al. 2002, 1 = Andronov, Pinnsonneault and Sills 2003, 2 = Barnes and Kim 2010 ", \
        ARG_INTEGER,                                                    \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->magnetic_braking_algorithm),           \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "gb_reimers_eta",                                               \
        "First red giant branch wind multiplication factor, cf. eta in Reimers' mass loss formula. (This multiplies the 4e-13 in Reimers' formula, or the 8e-14 in Schroeder and Cuntz.)", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->gb_reimers_eta),                       \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "gbwindfac",                                                    \
        "Multiplier for the giant branch wind mass loss rate",          \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->gbwindfac),                            \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "agbwindfac",                                                   \
        "Multiplier for the AGB wind mass loss rate",                   \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->agbwindfac),                           \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "tpagbwind",                                                    \
        "Wind prescription during the TPAGB. 0=Karakas 2002 (a modified Vassiliadis and Wood 1993), 1=Hurley et al 2000/2002 (Vassiliadis and Wood 1993), 2=Reimers, 3=Bloecker, 4=Van Loon,  5=Rob's C-wind (broken?), 6,7=Vassiliadis and Wood 1993 (Karakas,Hurley variants respectively) when C/O>1, 8=Mattsson", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->tpagbwind),                            \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "vwind_multiplier",                                             \
        "Multiplier for the wind velocity as a function of the star's escape speed. In BSE (Hurley et al 2000/2002) this is 1/8=0.125. Does NOT apply on the AGB, when the Vassiliadis and Wood wind velocity is used instead.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->vwind_multiplier),                     \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "superwind_mira_switchon",                                      \
        "In the Vassiliadis and Wood (1993) AGB wind prescription, the superwind is turned on at a given Mira period, usually 500 days. You can vary when this switch-on happens with this parameter.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->tpagb_superwind_mira_switchon),        \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "vw93_mira_shift",                                              \
        "In the Vassiliadis and Wood (1993) AGB wind prescription, the wind loss rate depends on the Mira period plus this offset. Requires VW93_MIRA_SHIFT.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        VW93_MIRA_SHIFT_VAR,                                            \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "vw93_multiplier",                                              \
        "In the Vassiliadis and Wood (1993) AGB wind prescription, the wind loss rate is multiplied by this factor. Requires VW93_MULTIPLIER.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        VW93_MULTIPLIER_VAR,                                            \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "tpagb_reimers_eta",                                            \
        "TPAGB Reimers wind multiplication factor, cf. eta in Reimers' mass loss formula. (This multiplies the 4e-13 in Reimers' formula, or the 8e-14 in Schroeder and Cuntz.) Note that Reimers is not the default TPAGB wind prescription. See also tpagbwind.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->tpagb_reimers_eta),                    \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "mass_accretion_rate1",                                         \
        "Constant mass accretion rate on star 1.",                      \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->mass_accretion_rate1),                 \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "mass_accretion_rate2",                                         \
        "Constant mass accretion rate on star 2.",                      \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->mass_accretion_rate2),                 \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "angular_momentum_accretion_rate1",                             \
        "Constant angular momentum accretion rate on star 1.",          \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->angular_momentum_accretion_rate1),     \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "angular_momentum_accretion_rate2",                             \
        "Constant angular momentum accretion rate on star 2.",          \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->angular_momentum_accretion_rate2),     \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "angular_momentum_accretion_rate_orbit",                        \
        "Constant angular momentum accretion rate on the orbit.",       \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->angular_momentum_accretion_rate_orbit), \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "accretion_start_time",                                         \
        "Time at which artificial accretion stars. Ignored if <0 (default is -1).", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->accretion_start_time),                 \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "accretion_end_time",                                           \
        "Time at which artificial accretion ends. Ignored if <0 (default is -1).", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->accretion_end_time),                   \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "wr_wind",                                                      \
        "Massive-star (WR) wind prescription. 0 = Hurley et al 2000/2002, 1=Maeder and Meynet, 2=Nugis and Lamers, 3=John Eldridge's version of Vink's early-2000s wind (See Lynnette Dray's thesis, or John Eldridge's thesis)", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->wr_wind),                              \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "wr_wind_fac",                                                  \
        "Massive-star (WR) wind multiplication factor.",                \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->wr_wind_fac),                          \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "wrwindfac",                                                    \
        "Massive-star (WR) wind multiplication factor. Synonymous with wr_wind_fac (which you should use instead).", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->wr_wind_fac),                          \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "accretion_limit_eddington_multiplier",                         \
        "Accretion is limited by the Eddington instability, with limiting rate given by the accretion_limit_eddington_multiplier * the normal (spherically symmetric) Eddington rate. This is known in the trade as the Eddington factor, and anything greater than 1.0 potentially gives you super-Eddington accretion.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->accretion_limit_eddington_multiplier), \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "accretion_limit_thermal_multiplier",                           \
        "Mass transfer onto a MS, HG or CHeB star is limited by the accretor's thermal rate times this multiplier.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->accretion_limit_thermal_multiplier),   \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "accretion_limit_dynamical_multiplier",                         \
        "Mass transfer is limited by the accretor's dynamical rate times this multiplier.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->accretion_limit_dynamical_multiplier), \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "donor_limit_envelope_multiplier",                              \
        "Mass transfer by RLOF is limited by this fraction of the donor's envelope mass per timestep", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->donor_limit_envelope_multiplier),      \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "donor_limit_thermal_multiplier",                               \
        "Mass transfer by RLOF is limited by the accretor's thermal rate times this multiplier.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->donor_limit_thermal_multiplier),       \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "donor_limit_dynamical_multiplier",                             \
        "Mass transfer by RLOF is limited by the donor's dynamical rate times this multiplier.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->donor_limit_dynamical_multiplier),     \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "Bondi_Hoyle_accretion_factor",                                 \
        "Wind accretion rate, as calculated by the Bondi-Hoyle-Littleton formula, multiplcation factor. (Used to be called 'acc2' which is now deprecated.) Hurley et al 2002 use 1.5, which is the default.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->Bondi_Hoyle_accretion_factor),         \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "BH_prescription",                                              \
        "Black hole mass prescrition: relates the mass of a newly formed black hole to its progenitor's (CO) core mass. 0=Hurley et al 2000/2002, 1=Belczynski (early 2000s).", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->BH_prescription),                      \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_distribution_II",                                      \
        "Set the distribution of speeds applied to kick type II core collapse supernova systems. 0=fixed, 1=maxwellian (hurley/BSE), 2=custom function (see monte_carlo_kicks.c).", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_distribution[SN_II]),          \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_distribution_ECAP",                                    \
        "Set the distribution of speeds applied to the remnants of electron-capture supernovae. 0=fixed, 1=maxwellian (hurley/BSE), 2=custom function (see monte_carlo_kicks.c).", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_distribution[SN_ECAP]),        \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_distribution_NS_NS",                                   \
        "Set the distribution of speeds applied to kick neutron stars and black holes that survive a NS-NS merger. 0=fixed, 1=maxwellian (hurley/BSE), 2=custom function (see monte_carlo_kicks.c).", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_distribution[SN_NS_NS]),       \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_distribution_IBC",                                     \
        "Set the distribution of speeds applied to kick newly-born neutron stars and black holes after a type Ib/c core-collapse supernova. 0=fixed, 1=maxwellian (hurley/BSE), 2=custom function (see monte_carlo_kicks.c).", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_distribution[SN_IBC]),         \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_distribution_GRB_COLLAPSAR",                           \
        "Set the distribution of speeds applied to kick newly-born neutron stars and black holes after a type Ib/c core-collapse supernova which is also a collapsar. 0=fixed, 1=maxwellian (hurley/BSE), 2=custom function (see monte_carlo_kicks.c).", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_distribution[SN_GRB_COLLAPSAR]), \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_distribution_TZ",                                      \
        "Set the distribution of speeds applied to kick newly-born neutron stars and black holes at the death of a Thorne-Zytkow object. 0=fixed, 1=maxwellian (hurley/BSE), 2=custom function (see monte_carlo_kicks.c).", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_distribution[SN_TZ]),          \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_distribution_AIC_BH",                                  \
        "Set the distribution of speeds applied to kick newly-born neutron stars black holes after accretion induced collapse of a neutron star. 0=fixed, 1=maxwellian (hurley/BSE), 2=custom function (see monte_carlo_kicks.c).", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_distribution[SN_AIC_BH]),      \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_distribution_BH_BH",                                   \
        "Set the distribution of speeds applied to black holes formed by the merger of two black holes. 0=fixed, 1=maxwellian (hurley/BSE), 2=custom function (see monte_carlo_kicks.c).", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_distribution[SN_BH_BH]),       \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_distribution_BH_NS",                                   \
        "Set the distribution of speeds applied to black holes formed by the merger of a neutron star and a black holes. 0=fixed, 1=maxwellian (hurley/BSE), 2=custom function (see monte_carlo_kicks.c).", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_distribution[SN_BH_NS]),       \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_distribution_IA_Hybrid_HeCOWD",                        \
        "Set the distribution of speeds applied to any survivor of a hybrid He-COWD SNIa explosion. 0=fixed, 1=maxwellian (hurley/BSE), 2=custom function (see monte_carlo_kicks.c).", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_distribution[SN_IA_Hybrid_HeCOWD]), \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_distribution_IA_Hybrid_HeCOWD_subluminous",            \
        "Set the distribution of speeds applied to any survivor of a subluminous hybrid He-COWD SNIa explosion. 0=fixed, 1=maxwellian (hurley/BSE), 2=custom function (see monte_carlo_kicks.c).", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_distribution[SN_IA_Hybrid_HeCOWD_subluminous]), \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_dispersion_II",                                        \
        "Set the dispersion of speeds applied to kick type II core collapse supernova systems. 0=fixed, 1=maxwellian (hurley/BSE), 2=custom function (see monte_carlo_kicks.c).", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_dispersion[SN_II]),            \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_dispersion_ECAP",                                      \
        "Set the dispersion of speeds applied to the remnants of electron-capture supernovae. 0=fixed, 1=maxwellian (hurley/BSE), 2=custom function (see monte_carlo_kicks.c).", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_dispersion[SN_ECAP]),          \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_dispersion_NS_NS",                                     \
        "Set the dispersion of speeds applied to kick neutron stars and black holes that survive a NS-NS merger. 0=fixed, 1=maxwellian (hurley/BSE), 2=custom function (see monte_carlo_kicks.c).", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_dispersion[SN_NS_NS]),         \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_dispersion_IBC",                                       \
        "Set the dispersion of speeds applied to kick newly-born neutron stars and black holes after a type Ib/c core-collapse supernova. 0=fixed, 1=maxwellian (hurley/BSE), 2=custom function (see monte_carlo_kicks.c).", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_dispersion[SN_IBC]),           \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_dispersion_GRB_COLLAPSAR",                             \
        "Set the dispersion of speeds applied to kick newly-born neutron stars and black holes after a type Ib/c core-collapse supernova which is also a collapsar. 0=fixed, 1=maxwellian (hurley/BSE), 2=custom function (see monte_carlo_kicks.c).", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_dispersion[SN_GRB_COLLAPSAR]), \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_dispersion_TZ",                                        \
        "Set the dispersion of speeds applied to kick newly-born neutron stars and black holes at the death of a Thorne-Zytkow object. 0=fixed, 1=maxwellian (hurley/BSE), 2=custom function (see monte_carlo_kicks.c).", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_dispersion[SN_TZ]),            \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_dispersion_AIC_BH",                                    \
        "Set the dispersion of speeds applied to kick newly-born neutron stars black holes after accretion induced collapse of a neutron star. 0=fixed, 1=maxwellian (hurley/BSE), 2=custom function (see monte_carlo_kicks.c).", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_dispersion[SN_AIC_BH]),        \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_dispersion_BH_BH",                                     \
        "Set the dispersion of speeds applied to black holes formed by the merger of two black holes. 0=fixed, 1=maxwellian (hurley/BSE), 2=custom function (see monte_carlo_kicks.c).", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_dispersion[SN_BH_BH]),         \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_dispersion_BH_NS",                                     \
        "Set the dispersion of speeds applied to black holes formed by the merger of a neutron star and a black holes. 0=fixed, 1=maxwellian (hurley/BSE), 2=custom function (see monte_carlo_kicks.c).", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_dispersion[SN_BH_NS]),         \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_dispersion_IA_Hybrid_HeCOWD",                          \
        "Set the dispersion of speeds applied to the survivor of a SNIa explosion of a hybrid He-COWD. 0=fixed, 1=maxwellian (hurley/BSE), 2=custom function (see monte_carlo_kicks.c).", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_dispersion[SN_IA_Hybrid_HeCOWD]), \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_dispersion_IA_Hybrid_HeCOWD_subluminous",              \
        "Set the dispersion of speeds applied to the survivor of a subluminous SNIa explosion of a hybrid He-COWD. 0=fixed, 1=maxwellian (hurley/BSE), 2=custom function (see monte_carlo_kicks.c).", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_dispersion[SN_IA_Hybrid_HeCOWD_subluminous]), \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_companion_IA_He",                                      \
        "Set the speed (if >=0) of, or the algothim (if <0) used to calculate the, kick on the companion when a Ia He supernova occurs. 0 = none, 1 = Liu+2015, 2 = Wheeler+ 1975.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_companion[SN_IA_He]),          \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_companion_IA_ELD",                                     \
        "Set the speed (if >=0) of, or the algothim (if <0) used to calculate the, kick on the companion when a Ia ELD (sub-Mch) supernova occurs. 0 = none, 1 = Liu+2015, 2 = Wheeler+ 1975.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_companion[SN_IA_ELD]),         \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_companion_IA_CHAND",                                   \
        "Set the speed (if >=0) of, or the algothim (if <0) used to calculate the, kick on the companion when a Ia Mch supernova occurs. 0 = none, 1 = Liu+2015, 2 = Wheeler+ 1975.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_companion[SN_IA_CHAND]),       \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_companion_AIC",                                        \
        "Set the speed (if >=0) of, or the algothim (if <0) used to calculate the, kick on the companion when an accretion induced collapse (supernova) occurs. 0 = none, 1 = Liu+2015, 2 = Wheeler+ 1975.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_companion[SN_AIC]),            \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_companion_ECAP",                                       \
        "Set the speed (if >=0) of, or the algothim (if <0) used to calculate the, kick on the companion when an electron capture supernova occurs. 0 = none, 1 = Liu+2015, 2 = Wheeler+ 1975.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_companion[SN_ECAP]),           \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_companion_IA_He_Coal",                                 \
        "Set the speed (if >=0) of, or the algothim (if <0) used to calculate the, kick on the companion when a Ia helium merger supernova occurs. 0 = none, 1 = Liu+2015, 2 = Wheeler+ 1975.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_companion[SN_IA_He_Coal]),     \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_companion_IA_CHAND_Coal",                              \
        "Set the speed (if >=0) of, or the algothim (if <0) used to calculate the, kick on the companion when a Ia Mch merger supernova occurs. 0 = none, 1 = Liu+2015, 2 = Wheeler+ 1975.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_companion[SN_IA_CHAND_Coal]),  \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_companion_NS_NS",                                      \
        "Set the speed (if >=0) of, or the algothim (if <0) used to calculate the, kick on the companion when a neutron-star neutron-star merger. 0 = none, 1 = Liu+2015, 2 = Wheeler+ 1975.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_companion[SN_NS_NS]),          \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_companion_GRB_COLLAPSAR",                              \
        "Set the speed (if >=0) of, or the algothim (if <0) used to calculate the, kick on the companion when a GRB Collapsar (rapidly rotating SN Ibc) supernova occurs. 0 = none, 1 = Liu+2015, 2 = Wheeler+ 1975.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_companion[SN_GRB_COLLAPSAR]),  \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_companion_HeStarIa",                                   \
        "Set the speed (if >=0) of, or the algothim (if <0) used to calculate the, kick on the companion when a He-star Ia supernova occurs. 0 = none, 1 = Liu+2015, 2 = Wheeler+ 1975.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_companion[SN_HeStarIa]),       \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_companion_IBC",                                        \
        "Set the speed (if >=0) of, or the algothim (if <0) used to calculate the, kick on the companion when a type Ib/c supernova occurs. 0 = none, 1 = Liu+2015, 2 = Wheeler+ 1975.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_companion[SN_IBC]),            \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_companion_II",                                         \
        "Set the speed (if >=0) of, or the algothim (if <0) used to calculate the, kick on the companion when a type II supernova occurs. 0 = none, 1 = Liu+2015, 2 = Wheeler+ 1975.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_companion[SN_II]),             \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_companion_IIa",                                        \
        "Set the speed (if >=0) of, or the algothim (if <0) used to calculate the, kick on the companion when a type IIa supernova occurs. 0 = none, 1 = Liu+2015, 2 = Wheeler+ 1975.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_companion[SN_IIa]),            \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_companion_WDKICK",                                     \
        "Set the speed (if >=0) of, or the algothim (if <0) used to calculate the, kick on the companion when a WD is kicked. 0 = none, 1 = Liu+2015, 2 = Wheeler+ 1975.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_companion[SN_WDKICK]),         \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_companion_TZ",                                         \
        "Set the speed (if >=0) of, or the algothim (if <0) used to calculate the, kick on the companion when a Thorne-Zytkow object is formed. 0 = none, 1 = Liu+2015, 2 = Wheeler+ 1975.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_companion[SN_TZ]),             \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_companion_AIC_BH",                                     \
        "Set the speed (if >=0) of, or the algothim (if <0) used to calculate the, kick on the companion when a neutron star collapses to a black hole. 0 = none, 1 = Liu+2015, 2 = Wheeler+ 1975.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_companion[SN_AIC_BH]),         \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_companion_BH_BH",                                      \
        "Set the speed (if >=0) of, or the algothim (if <0) used to calculate the, kick on the companion when two black holes merge. 0 = none, 1 = Liu+2015, 2 = Wheeler+ 1975.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_companion[SN_BH_BH]),          \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_companion_BH_NS",                                      \
        "Set the speed (if >=0) of, or the algothim (if <0) used to calculate the, kick on the companion when a black hole merges with a neutron star. 0 = none, 1 = Liu+2015, 2 = Wheeler+ 1975.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_companion[SN_BH_NS]),          \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_companion_IA_Hybrid_HeCOWD",                           \
        "Set the speed (if >=0) of, or the algothim (if <0) used to calculate the kick on the companion, if it survives, in a hybrid He-COWD type Ia explosion. 0 = none, 1 = Liu+2015, 2 = Wheeler+ 1975.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_companion[SN_IA_Hybrid_HeCOWD]), \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "sn_kick_companion_IA_Hybrid_HeCOWD_subluminous",               \
        "Set the speed (if >=0) of, or the algothim (if <0) used to calculate the kick on the companion, if it survives, in a subluminous hybrid He-COWD type Ia explosion. 0 = none, 1 = Liu+2015, 2 = Wheeler+ 1975.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->sn_kick_companion[SN_IA_Hybrid_HeCOWD_subluminous]), \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "wd_sigma",                                                     \
        "Set the speed at which white dwarfs are kicked when they form, in km/s. Default is zero (i.e. no kick). Requires WD_KICKS.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        WD_SIGMA_VAR,                                                   \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "wd_kick_direction",                                            \
        "Set the direction of white dwarf kicks. 0 = random, 1 = up, 2 = forward, 3 = backward, 4 = inward, 5 = outward. Requires WD_KICKS.", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        WD_KICK_DIRECTION_VAR,                                          \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "wd_kick_when",                                                 \
        "Decide when to kick a white dwarf. 0=at birth, 1=at first RLOF, 2=at given pulse number (see wd_kick_pulse_number), 3 at every pulse Requires WD_KICKS.", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        WD_KICK_WHEN_VAR,                                               \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "wd_kick_pulse_number",                                         \
        "Apply a kick to a star at a desired pulse number on the TPAGB (i.e. pre-WD). Requires WD_KICKS.", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        WD_KICK_PULSE_NUMBER_VAR,                                       \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "max_tpagb_core_mass",                                          \
        "Maximum mass of the core during the TPAGB. Typically around 1.38Msun.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->max_tpagb_core_mass),                  \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "minimum_helium_ignition_core_mass",                            \
        "Minimum helium core mass required to ignite helium in the case that the hydrogen envelope is stripped on the giant branch, e.g. to make an sdB or sdO star. Typically 0.4, if 0.0 then the BSE algorithm (based on the total mass) is used.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->minimum_helium_ignition_core_mass),    \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "minimum_mass_for_carbon_ignition",                             \
        "Minimum core mass for carbon ignition. Typically around 1.6Msun.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->minimum_mass_for_carbon_ignition),     \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "minimum_mass_for_neon_ignition",                               \
        "Minimum core mass for neon ignition. Typically around 2.85Msun.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->minimum_mass_for_neon_ignition),       \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "max_neutron_star_mass",                                        \
        "Maximum mass of a neutron star before it collapses to a black hole. Typically around 1.8Msun.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->max_neutron_star_mass),                \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "chandrasekhar_mass",                                           \
        "The Chandrasekhar mass, usually 1.44Msun",                     \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->chandrasekhar_mass),                   \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "delta_mcmin",                                                  \
        "A parameter to reduce the minimum core mass for third dredge up to occur on the TPAGB. As used by Izzard and Tout (2004) to increase the amount of dredge up, hence carbon, in Magellanic cloud stars.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        NUCSYN_DELTA_MCMIN_VAR,                                         \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "lambda_min",                                                   \
        "A parameter to increase the efficiency of third dredge up on the TPAGB. The efficiency is lambda * lambda_mult, and setting lambda_min>0 implies that, once Mc>Mcmin (see delta_mcmin) lambda=Max(lambda(fit to Karakas), lambda_min). As used by Izzard and Tout (2004) to increase the amount of dredge up, hence carbon, in Magellanic cloud stars. See also lambda_multiplier.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        NUCSYN_LAMBDA_MIN_VAR,                                          \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "lambda_multiplier",                                            \
        "A parameter to increase the efficiency of third dredge up on the TPAGB. The efficiency is lambda * lambda_mult, and setting lambda_min>0 implies that, once Mc>Mcmin (see delta_mcmin) lambda=Max(lambda(fit to Karakas), lambda_min). As used by Izzard and Tout (2004) to increase the amount of dredge up, hence carbon, in Magellanic cloud stars.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        NUCSYN_LAMBDA_MULTIPLIER_VAR,                                   \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "minimum_envelope_mass_for_third_dredgeup",                     \
        "The minimum envelope mass for third dredge up on the TPAGB. Early, solar metallicity models by Straniero et al suggested 0.5Msun is typical. However, circumstantial evidence (Izzard et al 2009) as well as newer models by Stancliffe and Karakas suggest that at low metallicity a value nearer zero is more appropriate.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        NUCSYN_MINIMUM_ENVELOPE_MASS_FOR_THIRD_DREDGE_UP_VAR,           \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "mass_of_pmz",                                                  \
        "The mass in the partial mixing zone of a TPAGB star, using the Karakas 2012 tables. Ask Carlo Abate for more details, or see the series of papers Abate et al 2012, 2013, 2014. Requires NUCSYN and USE_TABULAR_INTERSHELL_ABUNDANCES_KARAKAS_2012.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        NUCSYN_MASS_OF_PMZ_VAR,                                         \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "c13_eff",                                                      \
        "The \"efficiency\" of partial mixing in a TPAGB star intershell region, when using the s-process tables of Gallino, Busso, Lugaro et al. as provided by Maria Lugaro for the Izzard et al. 2009 paper. Requires NUCSYN and NUCSYN_S_PROCESS.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        NUCSYN_C13_EFF_VAR,                                             \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "mc13_pocket_multiplier",                                       \
        "Multiplies the mass in the partial mixing zone of a TPAGB star, when using the s-process tables of Gallino, Busso, Lugaro et al. as provided by Maria Lugaro for the Izzard et al. 2009 paper. Requires NUCSYN and NUCSYN_S_PROCESS.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        NUCSYN_MC13_POCKET_MULTIPLIER_VAR,                              \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "E2_prescription",                                              \
        "Choose how to calculate the E2 structural parameter (used in tidal timescale calculations). 0=Hurley 1=Izzard (see Siess et al 2013)", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->E2_prescription),                      \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "dtfac",                                                        \
        "A parameter to decrease the timestep ONLY during the TPAGB phase.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        NUCSYN_DTFAC_VAR,                                               \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "hbbtfac",                                                      \
        "A parameter to modulate the temperature at the base of the hot-bottom burning zone in TPAGB stars. (Works only if NUCSYN is defined)", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        NUCSYN_HBBTFAC_VAR,                                             \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "tidal_strength_factor",                                        \
        "A modulator for the tidal strength. If this factor > 1 then tides are stronger, i.e. tidal timescales are reduced." , \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->tidal_strength_factor),                \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "hachisu_qcrit",                                                \
        "Critical q=Maccretor/Mdonor above which Hachisu's disk wind turns on.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->hachisu_qcrit),                        \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "hachisu_disk_wind",                                            \
        "Set to True to turn on Hachisu's disk wind when material accretes too fast onto a white dwarf. This helps to make more SNeIa. See also hachisu_qcrit.", \
        ARG_BOOLEAN ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->hachisu_disk_wind),                    \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "mass_accretion_for_eld",                                       \
        "The mass that must be accreted onto a COWD for it to ignite as an edge-lit detonation SNIa.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->mass_accretion_for_eld),               \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "WDWD_merger_algorithm",                                        \
        "Algorithm to be used when merging two white dwarfs. 0 = Hurley et al. (2002), 1 = Perets+ (2019), 2 = Chen+ (2016, todo) ", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->WDWD_merger_algorithm),                \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "type_Ia_MCh_supernova_algorithm",                              \
        "Algorithm to be used when calculating type Ia yields from Chandrasekhar-mass exploders. 0 = DD7 (Iwamoto 1999), 1 = Seitenzahl 2013 3D hydro yields (you must also set Seitenzahl2013_model) ", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        NUCSYN_TYPE_IA_MCH_SUPERNOVA_ALGORITHM_VAR,                     \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "Seitenzahl2013_model",                                         \
        "Which of Seitenzahl et al. 2013's models to use? One of N1,N3,N5,N10,N20,N40,N100L,N100,N100H,N150,N200,N300C,N1600,N1600C,N100_Z0.5,N100_Z0.1,N100_Z0.01 (defaults to N100).", \
        ARG_STRING ,                                                    \
        "N1",                                                           \
        NUCSYN_SEITENZAHL2013_VAR,                                      \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "type_Ia_sub_MCh_supernova_algorithm",                          \
        "Algorithm to be used when calculating type Ia yields from sub-Chandrasekhar-mass exploders. (Currently unused.)", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        NUCSYN_TYPE_IA_SUB_MCH_SUPERNOVA_ALGORITHM_VAR,                 \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "max_HeWD_mass",                                                \
        "The maximum mass a HeWD can have before it ignites helium (0.7).", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->max_HeWD_mass),                        \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "merger_angular_momentum_factor",                               \
        "When two stars merge the resulting single star retains a fraction of the total system angular momentum (or the critical spin angular momentum, if it is smaller) multiplied by this factor.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->merger_angular_momentum_factor),       \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "wind_angular_momentum_loss",                                   \
        "Prescription for losing angular momentum in a stellar wind. 0=Hurley et al 2002 ('Tout') prescription, 1=lw i.e. a factor multiplying the specific orbital angular momentum, 2=lw hybrid for fast winds. Set wind_djorb_fac to the desired factor..", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->wind_angular_momentum_loss),           \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "wind_djorb_fac",                                               \
        "Factor multiplying angular momentum loss in a stellar wind when wind_angular_momentum_loss=0 (the Tout/Hurley et al 2002 prescription). See wind_angular_momentum_loss.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->wind_djorb_fac),                       \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "lw",                                                           \
        "Factor multiplying angular momentum loss in a stellar wind when wind_angular_momentum_loss=1,2 (the 'lw' prescription). See wind_angular_momentum_loss.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->lw),                                   \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_OUTPUT,                                                 \
        "yields_dt",                                                    \
        "When doing yield calculations, yields are output every yields_dt Myr. Requires NUCSYN and NUCSYN_SPARSE_YIELDS.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        NUCSYN_YIELDS_DT_VAR,                                           \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_OUTPUT,                                                 \
        "yields_logdt",                                                 \
        "When doing yield calculations, and when logyieldtimes is set, yields are output every yields_logdt Myr. Requires NUCSYN and NUCSYN_SPARSE_YIELDS.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        NUCSYN_YIELDS_LOGDT_VAR,                                        \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_OUTPUT,                                                 \
        "yields_startlogtime",                                          \
        "Start log yields at yields_startlogtime.",                     \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        NUCSYN_YIELDS_STARTLOGTIME_VAR,                                 \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_OUTPUT,                                                 \
        "yields_logtimes",                                              \
        "When doing yield calculations, set this to output at log times rather than linear times.", \
        ARG_BOOLEAN ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        NUCSYN_YIELDS_LOGTIMES_VAR,                                     \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_MISC,                                                   \
        "reverse_time",                                                 \
        "Make time go backwards. To be considered very experimental!",  \
        ARG_BOOLEAN ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        REVERSE_TIME_VAR,                                               \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_MISC,                                                   \
        "start_time",                                                   \
        "Start time for the simulation.",                               \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->start_time),                           \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_OUTPUT,                                                 \
        "CEMP_logg_maximum",                                            \
        "Maximum logg that CEMP stars are allowed to have. See Izzard et al 2009. See also CEMP_cfe_minimum, CEMP_minimum_age.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        NUCSYN_CEMP_LOGG_MAXIMUM_VAR,                                   \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_OUTPUT,                                                 \
        "CEMP_cfe_minimum",                                             \
        "Minimum [C/Fe] that CEMP stars are required to have. See Izzard et al 2009. See also CEMP_logg_maximum, CEMP_minimum_age.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        NUCSYN_CEMP_CFE_MINIMUM_VAR,                                    \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_OUTPUT,                                                 \
        "thick_disc_start_age",                                         \
        "Lookback time for the start of the thick disc star formation, e.g. 13e3 Myr. Units = Myr.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        THICK_DISC_START_AGE,                                           \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_OUTPUT,                                                 \
        "thick_disc_end_age",                                           \
        "Lookback time for the end of the thick disc star formation, e.g. 4e3 Myr. Units = Myr.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        THICK_DISC_END_AGE,                                             \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_OUTPUT,                                                 \
        "thick_disc_logg_min",                                          \
        "Minimum logg for thick disc giants to be logged.",             \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        THICK_DISC_LOGG_MIN,                                            \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_OUTPUT,                                                 \
        "thick_disc_logg_max",                                          \
        "Maximum logg for thick disc giants to be logged.",             \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        THICK_DISC_LOGG_MAX,                                            \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_OUTPUT,                                                 \
        "CEMP_minimum_age",                                             \
        "Minimum age that CEMP stars are required to have. See Izzard et al 2009. See also CEMP_cfe_minimum, CEMP_logg_maximum.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        NUCSYN_CEMP_MINIMUM_AGE_VAR,                                    \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_OUTPUT,                                                 \
        "escape_velocity",                                              \
        "A parameter used in constructing galactic chemical evolution (GCE) models. If the stellar wind velocity exceeds this value, any chemical yield from the wind is ignored, i.e. assumed lost. (km/s) Requires NUCSYN_GCE_OUTFLOW_CHECKS. Default 1e9 km/s. See also escape_fraction.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        NUCSYN_ESCAPE_VELOCITY_VAR,                                     \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_OUTPUT,                                                 \
        "escape_fraction",                                              \
        "A parameter used in constructing galactic chemical evolution (GCE) models. If the stellar wind velocity exceeds this value, any chemical yield from the wind is ignored, i.e. assumed lost. (km/s) Requires NUCSYN_GCE_OUTFLOW_CHECKS. Default 0.0. See also escape_velocity.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        NUCSYN_ESCAPE_FRACTION_VAR,                                     \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "pre_main_sequence",                                            \
        "Set to True to turn on pre-main sequence evolution. Currently this is not a special stellar type, rather the first (small) fraction of the main sequence has increased radii to match the Railton et al 2014 fits to Tout's pre-main sequence stars. Requires PRE_MAIN_SEQUENCE. See also pre_main_sequence_fit_lobes.", \
        ARG_BOOLEAN ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        PRE_MAIN_SEQUENCE_VAR,                                          \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "pre_main_sequence_fit_lobes",                                  \
        "Set to True force a pre-main sequence star into its Roche lobe. This is done by artificially aging it. Requires PRE_MAIN_SEQUENCE", \
        ARG_BOOLEAN ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        PRE_MAIN_SEQUENCE_FIT_LOBES_VAR,                                \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "rlperi",                                                       \
        "Set this to True to use the Roche lobe radius at periastron, rather than (the default to) assume a circular orbit. This will be useful one day when we treat RLOF in eccentric orbits properly, hopefully.", \
        ARG_BOOLEAN ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->rlperi),                               \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_LMMS",                                                   \
        "Apply critical q=Mdonor/Maccretor value for low-mass main sequence stars to determine the stability of Roche-lobe overflow for non-degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit[LOW_MASS_MAIN_SEQUENCE]),        \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_MS",                                                     \
        "Apply critical q=Mdonor/Maccretor value for (non-low mass) main sequence stars to determine the stability of Roche-lobe overflow for non-degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit[MAIN_SEQUENCE]),                 \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_HG",                                                     \
        "Apply critical q=Mdonor/Maccretor value for Hertzsprung gap stars to determine the stability of Roche-lobe overflow for non-degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit[HERTZSPRUNG_GAP]),               \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_GB",                                                     \
        "Apply critical q=Mdonor/Maccretor value for first red giant branch stars to determine the stability of Roche-lobe overflow for non-degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit[GIANT_BRANCH]),                  \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_CHeB",                                                   \
        "Apply critical q=Mdonor/Maccretor value for core helium burning stars to determine the stability of Roche-lobe overflow for non-degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit[CHeB]),                          \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_EAGB",                                                   \
        "Apply critical q=Mdonor/Maccretor value for early-AGB stars to determine the stability of Roche-lobe overflow for non-degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit[EAGB]),                          \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_TPAGB",                                                  \
        "Apply critical q=Mdonor/Maccretor value for TP-AGB stars to determine the stability of Roche-lobe overflow for non-degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit[TPAGB]),                         \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_HeMS",                                                   \
        "Apply critical q=Mdonor/Maccretor value for helium main sequence stars to determine the stability of Roche-lobe overflow for non-degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit[HeMS]),                          \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_HeHG",                                                   \
        "Apply critical q=Mdonor/Maccretor value for helium Hertzsprung gap stars to determine the stability of Roche-lobe overflow for non-degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit[HeHG]),                          \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_HeGB",                                                   \
        "Apply critical q=Mdonor/Maccretor value for helium red giant stars to determine the stability of Roche-lobe overflow for non-degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit[HeGB]),                          \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_HeWD",                                                   \
        "Apply critical q=Mdonor/Maccretor value for helium white dwarf stars to determine the stability of Roche-lobe overflow for non-degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit[HeWD]),                          \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_COWD",                                                   \
        "Apply critical q=Mdonor/Maccretor value for carbon-oxygen white dwarf stars to determine the stability of Roche-lobe overflow for non-degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit[COWD]),                          \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_ONeWD",                                                  \
        "Apply critical q=Mdonor/Maccretor value for oxygen-neon white dwarf stars to determine the stability of Roche-lobe overflow for non-degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit[ONeWD]),                         \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_NS",                                                     \
        "Apply critical q=Mdonor/Maccretor value for neutron stars to determine the stability of Roche-lobe overflow for non-degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit[NEUTRON_STAR]),                  \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_BH",                                                     \
        "Apply critical q=Mdonor/Maccretor value for black holes to determine the stability of Roche-lobe overflow for non-degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit[BLACK_HOLE]),                    \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_degenerate_LMMS",                                        \
        "Apply critical q=Mdonor/Maccretor value for (low mass) main sequence stars to determine the stability of Roche-lobe overflow for degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit_degenerate[MAIN_SEQUENCE]),      \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_degenerate_MS",                                          \
        "Apply critical q=Mdonor/Maccretor value for (non-low mass) main sequence stars to determine the stability of Roche-lobe overflow for degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit_degenerate[MAIN_SEQUENCE]),      \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_degenerate_HG",                                          \
        "Apply critical q=Mdonor/Maccretor value for Hertzsprung gap stars to determine the stability of Roche-lobe overflow for degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit_degenerate[HERTZSPRUNG_GAP]),    \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_degenerate_GB",                                          \
        "Apply critical q=Mdonor/Maccretor value for first red giant branch stars to determine the stability of Roche-lobe overflow for degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit_degenerate[GIANT_BRANCH]),       \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_degenerate_CHeB",                                        \
        "Apply critical q=Mdonor/Maccretor value for core helium burning stars to determine the stability of Roche-lobe overflow for degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit_degenerate[CHeB]),               \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_degenerate_EAGB",                                        \
        "Apply critical q=Mdonor/Maccretor value for early-AGB stars to determine the stability of Roche-lobe overflow for degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit_degenerate[EAGB]),               \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_degenerate_TPAGB",                                       \
        "Apply critical q=Mdonor/Maccretor value for TP-AGB stars to determine the stability of Roche-lobe overflow for degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit_degenerate[TPAGB]),              \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_degenerate_HeMS",                                        \
        "Apply critical q=Mdonor/Maccretor value for helium main sequence stars to determine the stability of Roche-lobe overflow for degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit_degenerate[HeMS]),               \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_degenerate_HeHG",                                        \
        "Apply critical q=Mdonor/Maccretor value for helium Hertzsprung gap stars to determine the stability of Roche-lobe overflow for degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit_degenerate[HeHG]),               \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_degenerate_HeGB",                                        \
        "Apply critical q=Mdonor/Maccretor value for helium red giant stars to determine the stability of Roche-lobe overflow for degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit_degenerate[HeGB]),               \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_degenerate_HeWD",                                        \
        "Apply critical q=Mdonor/Maccretor value for helium white dwarf stars to determine the stability of Roche-lobe overflow for degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit_degenerate[HeWD]),               \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_degenerate_COWD",                                        \
        "Apply critical q=Mdonor/Maccretor value for carbon-oxygen white dwarf stars to determine the stability of Roche-lobe overflow for degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit_degenerate[COWD]),               \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_degenerate_ONeWD",                                       \
        "Apply critical q=Mdonor/Maccretor value for oxygen-neon white dwarf stars to determine the stability of Roche-lobe overflow for degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit_degenerate[ONeWD]),              \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_degenerate_NS",                                          \
        "Apply critical q=Mdonor/Maccretor value for neutron stars to determine the stability of Roche-lobe overflow for degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit_degenerate[NEUTRON_STAR]),       \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "qcrit_degenerate_BH",                                          \
        "Apply critical q=Mdonor/Maccretor value for black holes to determine the stability of Roche-lobe overflow for degenerate accretors. See also qcrits_*, qcrits_degenerate_*.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->qcrit_degenerate[BLACK_HOLE]),         \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "mass_for_Hestar_Ia_upper",                                     \
        "Only helium stars below this mass can explode as SNIa. Default is zero, i.e. it never happens. See also mass_for_Hestar_Ia_lower.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->mass_for_Hestar_Ia_upper),             \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "mass_for_Hestar_Ia_lower",                                     \
        "Only helium stars above this mass can explode as SNIa. Default is zero, i.e. it never happens. See also mass_for_Hestar_Ia_upper.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->mass_for_Hestar_Ia_lower),             \
        1.0,                                                            \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "alphaCB",                                                      \
        "Circumbinary disk viscosity parameter, alpha." ,               \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        ALPHACB_VAR,                                                    \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "minimum_donor_menv_for_comenv",                                \
        "Minimum donor envelope mass for common envelope evolution to be triggered (Msun). Default 0." , \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->minimum_donor_menv_for_comenv),        \
        1.0,                                                            \
        },                                                              \
{         /* common envelope evolution */                               \
    ARG_SECTION_BINARY,                                                 \
        "comenv_prescription",                                          \
        "Use this to choose which common envelope prescription you should use. 0=Hurley et al 2002 (based on the Paczyński energy model) or 1=Nelemans and Tout (angular momentum model). See also alpha_ce, comenv_ms_accretion_mass, comenv_ms_accretion_fraction, comenv_ns_accretion_fraction, comenv_ns_accretion_mass, nelemans_gamma, nelemans_minq, nelemans_max_frac_j_change, nelemans_n_comenvs, lambda_ce, lambda_ionisation.", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->comenv_prescription),                  \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "comenv_ejection_spin_method",                                  \
        "When a common envelope is ejected, we need to decide how fast the stars are left spinning. COMENV_EJECTION_SPIN_METHOD_DO_NOTHING (0) is the default, this just leaves the stars/stellar cores spinning with the same spin rate (omega = angular velocity) with which they entered the common envelope phase. COMENV_EJECTION_SPIN_METHOD_SYCHRONIZE instead tidally synchronizes the stars with their new orbital angular velocity.", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->comenv_ejection_spin_method),          \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "comenv_merger_spin_method",                                    \
        "When a common envelope binary merges, we need to decide how fast the resulting single star is left spinning. COMENV_MERGER_SPIN_METHOD_SPECIFIC (0) is the default, this preserves angular momentum but limits the specific angular momentum of the merged star to the specific angular momentum of the system at the onset of common envelope evolution. COMENV_MERGER_SPIN_METHOD_CONSERVE_ANGMOM (1) sets the merger's angular momentum to be that of the system at the onset of common envelope evolution (which means the star may be rotating supercritically). COMENV_MERGER_SPIN_METHOD_CONSERVE_OMEGA (2) sets the spin rate (angular frequency = omega) of the merged star to be that of the orbit just at the onset of common envelope evolution.", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->comenv_merger_spin_method),            \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "comenv_ms_accretion_mass",                                     \
        "Experimental. During common envelope evolution, a main sequence star may accrete some of the envelope's mass. Requires COMENV_MS_ACCRETION. See also comenv_ms_accretion_fraction.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        CE_MS_ACCRETION_MASS_VAR,                                       \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "comenv_ms_accretion_fraction",                                 \
        "Experimental. During common envelope evolution, a main sequence may accrete a fraction of the envelope's mass. Requires COMENV_MS_ACCRETION. See also comenv_ms_accretion_mass.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        CE_MS_ACCRETION_FRACTION_VAR,                                   \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "comenv_ns_accretion_mass",                                     \
        "Experimental. During common envelope evolution, a neutron star may accrete some of the envelope's mass. Requires COMENV_NS_ACCRETION. See also comenv_ns_accretion_fraction.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        CE_NS_ACCRETION_MASS_VAR,                                       \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "comenv_ns_accretion_fraction",                                 \
        "Experimental. During common envelope evolution, a neutron star may accrete a fraction of the envelope's mass. Requires COMENV_NS_ACCRETION. See also comenv_ns_accretion_mass.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        CE_NS_ACCRETION_FRACTION_VAR,                                   \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "alpha_ce",                                                     \
        "Common envelope energy formalism parameter. A fraction alpha of the orbital energy is used to eject the envelope. See Hurley et al 2002 for details." , \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->alpha_ce),                             \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "lambda_ce",                                                    \
        "Common envelope parameter. The binding energy of the common envelope is G*M*Menv/(lambda*R). Typically this is taken to be 0.5, but if set to -1 binary_c uses the Dewi and Tauris fits instead, -2 uses the formalism of Wang, Jia and Li (2016) and if -3 then a polytropic formalism is used (see also comenv_splitmass).", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->lambda_ce),                            \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "comenv_splitmass",                                             \
        "When lambda_ce=-2, the envelope binding energy, lambda, is calculated using a polytropic formalism. The comenv_splitmass defines the point, in the units of the core mass, above which material is ejected.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        COMENV_SPLITMASS_VAR,                                           \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "nelemans_recalc_eccentricity",                                 \
        "If True, recalculate the eccentricity after angular momentum is removed. " , \
        ARG_BOOLEAN ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->nelemans_recalc_eccentricity),         \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "comenv_post_eccentricity",                                     \
        "Eccentricity remaining after common envelope ejection." ,      \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->comenv_post_eccentricity),             \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "nelemans_gamma",                                               \
        "Set the fraction of the orbital specific angular momentum that is used to eject the common envelope according to the Nelemans and Tout prescription. See also nelemans_minq, nelemans_max_frac_j_change, nelemans_n_comenvs." , \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->nelemans_gamma),                       \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "nelemans_minq",                                                \
        "Only activate the Nelemans and Tout common envelope prescription for q>nelemans_minq. See also nelemans_gamma, nelemans_max_frac_j_change, nelemans_n_comenvs." , \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->nelemans_minq),                        \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "nelemans_max_frac_j_change",                                   \
        "Maximum fractional angular momentum change in the Nelemans and Tout common envelope prescription. See also nelemans_gamma, nelemans_minq, nelemans_n_comenvs." , \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->nelemans_max_frac_j_change),           \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "nelemans_n_comenvs",                                           \
        "Set the maximum number of common envelope ejections allowed to follow the Nelemans and Tout prescription, after which the standard alpha prescription is used." , \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->nelemans_n_comenvs),                   \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "lambda_ionisation",                                            \
        "A fraction lambda_ionisation of the recombination energy in the common envelope goes into ejecting the envelope. This is usually 0.0, but a positive value can make a big difference to the outcome of common envelope evolution.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->lambda_ionisation),                    \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "lambda_enthalpy",                                              \
        "A fraction of the enthalpy to be included in the common envelope evolution binding energy. Only used for the Wang 2016 prescription (so far).", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->lambda_enthalpy),                      \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_gamma",                                                 \
        "Circumbinary disc gamma (equation of state) parameter. Requires DISCS.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_GAMMA_VAR,                                               \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_alpha",                                                 \
        "Circumbinary disc alpha (viscosity) parameter. Requires DISCS.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_ALPHA_VAR,                                               \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_kappa",                                                 \
        "Circumbinary disc kappa (opacity) parameter. Requires DISCS.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_KAPPA_VAR,                                               \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_minimum_evaporation_timescale",                         \
        "Circumbinary disc minimum evaporation timescale (years). If (slow, not edge stripped) mass loss would evaporate the disc on a timescale less than this, simply evaporate the disc immediated. Usually set to 1y, ignore if zero. Requires DISCS.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_MINIMUM_EVAPORATION_TIMESCALE_VAR,                       \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_torquef",                                               \
        "Circumbinary disc binary torque multiplier. Requires DISCS.",  \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_TORQUEF_VAR,                                             \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_max_lifetime",                                          \
        "Circumbinary disc maximum lifetime (years, ignored if 0). Requires DISCS.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_MAX_LIFETIME_VAR,                                        \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_init_dM",                                               \
        "On cbdisc start, reduce mass by a fraction dM if it won't converge. Requires DISCS.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_START_DM_VAR,                                            \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_init_dJdM",                                             \
        "On cbdisc start, reduce angular momentum by a fraction dJ/dM*dM if it won't converge. Requires DISCS.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_START_DJDM_VAR,                                          \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_mass_loss_constant_rate",                               \
        "Circumbinary disc constant mass loss rate (Msun/year). Requires DISCS.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_MASS_LOSS_RATE_VAR,                                      \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_mass_loss_FUV_multiplier",                              \
        "Circumbinary disc FUV mass loss rate multiplier (no units). Requires DISCS.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_MASS_LOSS_FUV_MULTIPLIER_VAR,                            \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_mass_loss_Xray_multiplier",                             \
        "Circumbinary disc X-ray mass loss rate multiplier (no units). Requires DISCS.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_MASS_LOSS_XRAY_MULTIPLIER_VAR,                           \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_mass_loss_ISM_ram_pressure_multiplier",                 \
        "Circumbinary disc interstellar medium ram pressure stripping mass loss rate multiplier (no units). Requires DISCS.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_MASS_LOSS_ISM_RAM_PRESSURE_MULTIPLIER_VAR,               \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_mass_loss_ISM_pressure",                                \
        "Circumbinary disc interstellar medium ram pressure in units of Boltzmann constant per Kelvin (I think...). Requires DISCS. Typically 3000.0. See e.g. http://www.astronomy.ohio-state.edu/~pogge/Ast871/Notes/Intro.pdf page 15 or https://arxiv.org/pdf/0902.0820.pdf Fig. 1 (left panel).", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_MASS_LOSS_ISM_PRESSURE_VAR,                              \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_mass_loss_inner_viscous_multiplier",                    \
        "Circumbinary disc inner edge viscous mass loss rate multiplier (no units). Requires DISCS.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_MASS_LOSS_INNER_VISCOUS_MULTIPLIER_VAR,                  \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_mass_loss_inner_viscous_angular_momentum_multiplier",   \
        "Circumbinary disc inner edge viscous angular momentum multiplier (no units). The inner edge angular momentum Requires DISCS.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_MASS_LOSS_INNER_VISCOUS_ANGULAR_MOMENTUM_MULTIPLIER_VAR, \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_resonance_multiplier",                                  \
        "Circumbinary disc resonant interaction multiplier, affects eccentricity pumping and angular momentum rates. Requires DISCS.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_RESONANCE_MULTIPLIER_VAR,                                \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_resonance_damping",                                     \
        "Circumbinary disc resonant interaction damping: should be on (True) to damp the l=1, m=2 resonance when the disc inner edge lies outside the resonance location. Requires DISCS.", \
        ARG_BOOLEAN,                                                    \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_RESONANCE_DAMPING_VAR,                                   \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_fail_ring_inside_separation",                           \
        "If True, while converging on a structure, circumbinary discs with Rring < the binary separation are immediately failed.", \
        ARG_BOOLEAN,                                                    \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_FAIL_RING_INSIDE_SEPARATION,                             \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_mass_loss_inner_L2_cross_multiplier",                   \
        "Circumbinary disc inner edge L2-crossing mass loss rate multiplier (no units). Requires DISCS.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_MASS_LOSS_INNER_L2_CROSS_MULTIPLIER_VAR,                 \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_minimum_luminosity",                                    \
        "Circumbinary disc minimum luminosity. If the disc becomes dimmer than this, the disc is evaporated instantly. Requires DISCS.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_MINIMUM_LUMINOSITY_VAR,                                  \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_minimum_mass",                                          \
        "Circumbinary disc minimum mass. If the disc becomes less massive than this, the disc is evaporated instantly. Requires DISCS.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_MINIMUM_MASS_VAR,                                        \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "cbdisc_minimum_fRing",                                         \
        "Circumbinary disc minimum fRing. If the disc becomes a ring, and fRing = |Rout/Rin-1| < this value (and this value is non-zero), the disc is evaporated instantly. Requires DISCS.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        CBDISC_MINIMUM_FRING_VAR,                                       \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "comenv_disc_angmom_fraction",                                  \
        "If >0 Fraction of the common envelope's angular momentum that goes into the circumbinary disc. If -1 then uses the moments of inertia to calculate (deprecated), if -2 use the common envelope's specific angular momentum, if -3 uses the L2 point at the end of the common envelope to set the angular momentum. Requires DISCS and DISCS_CIRCUMBINARY_FROM_COMENV.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        COMENV_DISC_ANGMOM_FRACTION_VAR,                                \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "comenv_disc_mass_fraction",                                    \
        "Fraction of the common envelope's mass that goes into the circumbinary disc. Requires DISCS and DISCS_CIRCUMBINARY_FROM_COMENV.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        COMENV_DISC_MASS_FRACTION_VAR,                                  \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "wind_disc_angmom_fraction",                                    \
        "If >0 Fraction of the wind envelope's angular momentum that goes into the circumbinary disc. If -1 then uses the L2 point's specific angular momentum. Requires DISCS and DISCS_CIRCUMBINARY_FROM_WIND.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        WIND_DISC_ANGMOM_FRACTION_VAR,                                  \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "wind_disc_mass_fraction",                                      \
        "Fraction of the stellar wind's mass that goes into the circumbinary disc. Requires DISCS and DISCS_CIRCUMBINARY_FROM_WIND.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        WIND_DISC_MASS_FRACTION_VAR,                                    \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "WRLOF_method",                                                 \
        "Choose whether and how to apply wind-Roche-lobe-overflow. 0=none, 1=q-dependent, 2=quadratic See Abate et al 2013/14 for details. Requires WRLOF_MASS_TRANSFER.", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        WRLOF_METHOD_VAR,                                               \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "minimum_timestep",                                             \
        "The minimum timestep (Myr).",                                  \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->minimum_timestep),                     \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "timestep_solver_factor",                                       \
        "Factor applied in timestep_limits, e.g. to prevent X changing too fast by comparing to X/dX/dt, which is usually 1 but can be higher to lengthen timesteps when using an alternative solver.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->timestep_solver_factor),               \
        1.0                                                             \
        },                                                              \
{         /* Roche lobe overflow */                                     \
    ARG_SECTION_BINARY,                                                 \
        "RLOF_mdot_factor",                                             \
        "Multiplier applied to the mass transfer rate during Roche-lobe overflow. Requires RLOF_MDOT_MODULATION.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        RLOF_MDOT_MODULATOR_VAR,                                        \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "RLOF_f",                                                       \
        "Factor to enlarge a Roche lobe, nominally because of radiation pressure (see Dermine et al paper). Requires RLOF_RADIATION_CORRECTION.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        RLOF_F_VAR,                                                     \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "minimum_separation_for_instant_RLOF",                          \
        "If True, instead of evolving the system just report the minimum separation (on the zero-age main sequence) that would lead to instant RLOF. Used by binary_grid. See also minimum_orbital_period_for_instant_RLOF.", \
        ARG_BOOLEAN ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->show_minimum_separation_for_instant_RLOF), \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "separation",                                                   \
        "Set the orbital separation in solar radii. Note that if the orbital period is given, it is used to calculate the separation. So if you want to set the separation instead, either do not set the orbital period or set the orbital period to zero (0.0).", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->initial_orbital_separation),           \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "minimum_orbital_period_for_instant_RLOF",                      \
        "If True, instead of evolving the system just report the minimum orbital period (on the zero-age main sequence) that would lead to instant RLOF. Used by binary_grid. See also minimum_separeation_for_instant_RLOF.", \
        ARG_BOOLEAN ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->show_minimum_orbital_period_for_instant_RLOF), \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "RLOF_method",                                                  \
        "Use RLOF_method to choose the algorithm you use for Roche-lobe overflow mass loss rate calculations. 0=Hurley et al 2002, 1=Adaptive (for radiative stars) R=RL method, 2=Ritter (probably broken), 3=Claeys etal 2014 variant on Hurley et al 2002.", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->RLOF_method),                          \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "small_envelope_method",                                        \
        "Choose the method used to determine the stellar radius when the envelope mass is very thin. 0 = Hurley et al. (2002), 1 = Miller Bertolami et al. (2016+) for GB and AGB stars only.", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->small_envelope_method),                \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "RLOF_interpolation_method",                                    \
        "When a star overflows its Roche lobe, it always has R>RL because of the limited time resolution of the simulation. Binary_c then uses an algorithm to get back to when R~RL (within a desired tolerance, set in RLOF_ENTRY_THRESHOLD which is usually 1.02, i.e. overflow of 2%). You can choose algorithm 0, the Hurley et al 2002 method which reverses time (i.e. uses a Newton-like scheme), or 1 to use the binary_c method which rejects a timestep (and hence does no logging on that timestep) and repeats with half the timestep until R~RL. The latter is now the default, because this means there are no negative timesteps which break various other algorithms (e.g. nucleosynthesis).", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->RLOF_interpolation_method),            \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "vrot1",                                                        \
        "Equatorial rotational speed of star 1 (km/s). If 0.0, the Hurley et al 2000/2002 prescription is used to set the main-sequence velocity, so for a truly non-rotating star, set to something small (e.g. 0.001). Requires MANUAL_VROT. See also vrot2.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        VROT_1_VAR,                                                     \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "vrot2",                                                        \
        "Equatorial rotational speed of star 2 (km/s). If 0.0, the Hurley et al 2000/2002 prescription is used to set the main-sequence velocity, so for a truly non-rotating star, set to something small (e.g. 0.001). Requires MANUAL_VROT. See also vrot1.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        VROT_2_VAR,                                                     \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "timestep_modulator",                                           \
        "Multiplier applied to the global timestep. Requires TIMESTEP_MODULATION.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        TIMESTEP_MODULATOR_VAR,                                         \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "timestep_multiplier%d",                                        \
        "Multiplier applied to timestep limit <n>.",                    \
        ARG_FLOAT_SCANF,                                                \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->timestep_multipliers),                 \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "maximum_timestep",                                             \
        "The maximum timestep (MYr).",                                  \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->maximum_timestep),                     \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "zoomfac_multiplier_decrease",                                  \
        "When a timestep is rejected, decrease the timestep by this factor (0.5).", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->zoomfac_multiplier_decrease),          \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "zoomfac_multiplier_increase",                                  \
        "When a timestep is rejected, zooms, then succeeds, increase the timestep by this factor (1.2).", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->zoomfac_multiplier_increase),          \
        1.0,                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "maximum_timestep_factor",                                      \
        "The maximum factor between two subsequent timesteps (1.2).",   \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->maximum_timestep_factor),              \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "maximum_nuclear_burning_timestep",                             \
        "The maximum timestep (MYr) in any nuclear burning phase.",     \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->maximum_nuclear_burning_timestep),     \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "nova_retention_method",                                        \
        "Algorithm used to calculate the amount of mass retained during a nova explosion. 0=use nova_retention_fraction. (other methods pending)", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->nova_retention_method),                \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "nova_retention_fraction",                                      \
        "The mass accreted during a nova as fraction of mass transferred", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->nova_retention_fraction),              \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "beta_reverse_nova",                                            \
        "The fraction of mass ejected in a nova explosion which is accreted back onto the companion star. Set to -1 to automatically calculate based on a geometric argument, or 0 or positive to set the value.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->beta_reverse_nova),                    \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "WD_accretion_rate_novae_upper_limit_hydrogen_donor",           \
        "Upper limit of the stable mass transfer rate onto a white dwarf that leads to novae when the donor is hydrogen rich : above this rate the mass transfer leads to stable burning.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->WD_accretion_rate_novae_upper_limit_hydrogen_donor), \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "WD_accretion_rate_novae_upper_limit_helium_donor",             \
        "Upper limit of the stable mass transfer rate onto a white dwarf that leads to novae when the donor is helium rich : above this rate the mass transfer leads to stable burning.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->WD_accretion_rate_novae_upper_limit_helium_donor), \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "WD_accretion_rate_novae_upper_limit_other_donor",              \
        "Upper limit of the stable mass transfer rate onto a white dwarf that leads to novae when the donor is neither hydrogen nor helium rich : above this rate the mass transfer leads to stable burning.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->WD_accretion_rate_novae_upper_limit_other_donor), \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "WD_accretion_rate_new_giant_envelope_lower_limit_hydrogen_donor", \
        "Lower limit of the mass transfer rate onto a white dwarf that leads to a the formation of a new giant envelope with a hydrogen-rich donor. Below this mass transfer leads to stable burning.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->WD_accretion_rate_new_giant_envelope_lower_limit_hydrogen_donor), \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "WD_accretion_rate_new_giant_envelope_lower_limit_helium_donor", \
        "Lower limit of the mass transfer rate onto a white dwarf that leads to a the formation of a new giant envelope with a helium-rich donor. Below this mass transfer leads to stable burning.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->WD_accretion_rate_new_giant_envelope_lower_limit_helium_donor), \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "WD_accretion_rate_new_giant_envelope_lower_limit_other_donor", \
        "Lower limit of the mass transfer rate onto a white dwarf that leads to a the formation of a new giant envelope when the donor is neither hydrogen nor helium rich. Below this mass transfer leads to stable burning.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->WD_accretion_rate_new_giant_envelope_lower_limit_other_donor), \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "CRAP_parameter",                                               \
        "Tidally enhanced mass loss parameter. See Tout and Eggleton's paper on the subject. (Was the parameter bb).", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->CRAP_parameter),                       \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_OUTPUT,                                                 \
        "log_filename",                                                 \
        "Location of the output logging filename. If set to \"/dev/null\" then there is no logging.", \
        ARG_STRING ,                                                    \
        "",                                                             \
        LOG_FILENAME_VAR,                                               \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_OUTPUT,                                                 \
        "stardata_dump_filename",                                       \
        "Location of the stardata dump file.",                          \
        ARG_STRING ,                                                    \
        "",                                                             \
        &(stardata->preferences->stardata_dump_filename),               \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_OUTPUT,                                                 \
        "stardata_load_filename",                                       \
        "Location of the stardata file to load.",                       \
        ARG_STRING ,                                                    \
        "",                                                             \
        &(stardata->preferences->stardata_load_filename),               \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_OUTPUT,                                                 \
        "api_log_filename_prefix",                                      \
        "Location of the output logging filename prefix for the API. If set to \"/dev/null\" then there is no logging.", \
        ARG_STRING ,                                                    \
        "0",                                                            \
        API_LOG_FILENAME_PREFIX_VAR,                                    \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_OUTPUT,                                                 \
        "hrdiag_output",                                                \
        "Set to True to output high time-resolution Hertzstrpung-Russell diagram information. Requires HRDIAG.", \
        ARG_BOOLEAN ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        HRDIAG_OUTPUT_VAR,                                              \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_OUTPUT,                                                 \
        "internal_buffering",                                           \
        "Experimental. Set to non-zero values to implement internal buffering prior to output. For use with binary_grid, you shouldn't really be playing with this.", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->internal_buffering),                   \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "individual_novae",                                             \
        "If individual_novae is True, novae are resolved such that each explosion is performed separtaely.", \
        ARG_BOOLEAN,                                                    \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->individual_novae),                     \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "nova_timestep_accelerator_num",                                \
        "The nova timestep is accelerated if the nova number exceeds nova_timestep_accelerator_num. If zero or negative, acceleration is off. See also nova_timestep_accelerator_index and nova_timestep_accelerator_max. Only used if individual_novae is on.\n", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->nova_timestep_accelerator_num),        \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "nova_timestep_accelerator_index",                              \
        "The index at which the nova timestep is accelerated. A larger value gives longer timesteps. See also nova_timestep_accelerator_num and nova_timestep_accelerator_max. Only used if individual_novae is on.\n", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->nova_timestep_accelerator_index),      \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "nova_timestep_accelerator_max",                                \
        "The nova timestep is accelerated by a factor that is capped at nova_timestep_accelerator_max. This parameter is ignored if it is zero or negative. See also nova_timestep_accelerator_num and nova_timestep_accelerator_index. Only used if individual_novae is on.\n", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->nova_timestep_accelerator_max),        \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_OUTPUT,                                                 \
        "force_corotation_on_RLOF",                                     \
        "If True forces co-rotation of stars and orbit when RLOF starts, this is as in the BSE algorithm. (True)", \
        ARG_BOOLEAN ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->force_corotation_on_RLOF),             \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "nonconservative_angmom_gamma",                                 \
        "Mass lost from the system (but NOT from a stellar wind) takes a fraction gamma of the orbital angular momentum with it. Set to -1 to take the specific angular momentum of the donor star. Set to -2 to take super-Eddington, nova and disk-wind angular momenta as if a wind from the accretor.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->nonconservative_angmom_gamma),         \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "max_stellar_angmom_change",                                    \
        "Maxmimum fractional change in stellar angular momentum allowed before a timestep is rejected (0.05).", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->max_stellar_angmom_change),            \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_BINARY,                                                 \
        "comenv_ms_accretion_mass",                                     \
        "Experimental. During common envelope evolution, a main sequence star may accrete some of the envelope's mass. Requires COMENV_MS_ACCRETION. See also comenv_ms_accretion_fraction.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        CE_MS_ACCRETION_MASS_VAR,                                       \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "third_dup",                                                    \
        "If True, enables third dredge up. Requires NUCSYN and NUCSYN_THIRD_DREDGE_UP.", \
        ARG_BOOLEAN,                                                    \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->third_dup),                            \
        1.0                                                             \
        } ,                                                             \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "third_dup_multiplier",                                         \
        "Usage: --third_dup_multiplier <i> <f>. Multiplies the abundance of element <i> by <f> during third dredge up.", \
        ARG_SUBROUTINE,                                                 \
        "1.0",                                                          \
        &(set_third_dup_multiplier),                                    \
        1.0                                                             \
        } ,                                                             \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "NeNaMgAl",                                                     \
        "Enables NeNaMgAl reaction network. Requires NUCSYN and NUCSYN_HBB.", \
        ARG_BOOLEAN,                                                    \
        "Ignore",                                                       \
        NeNaMgAl_VAR,                                                   \
        1.0                                                             \
        } ,                                                             \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "nucreacmult%d",                                                \
        "Usage: --nucreacmult%d <f>. Multiply nuclear reaction given by the integer %d (integer) by f (float). ", \
        ARG_FLOAT_SCANF,                                                \
        WTTS_USE_DEFAULT,                                               \
        NUCSYN_REACTION_RATE_MULTIPLIERS_VAR,                           \
        1.0                                                             \
        } ,                                                             \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "nucsyn_metallicity",                                           \
        "This sets the metallicity of the nucleosynthesis algorithms, i.e. the amount (by mass) of matter which is not hydrogen or helium. Usually you'd just set this with the metallicity parameter, but if you want the nucleosynthesis to be outside the range of the stellar evolution algorithm (e.g. Z=0 or Z=0.04) then you need to use nucsyn_metallicity. That said, it's also outside the range of some of the nucleosynthesis algorithms as well, so you have been warned!", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        NUCSYN_METALLICITY_VAR,                                         \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "initial_abundance_mix",                                        \
        "initial abundance mixture: 0=AG89, 1=Karakas 2002, 2=Lodders 2003, 3=Asplund 2005 (not available?), 4=Garcia Berro, 5=Grevesse Noels 1993", \
        ARG_UNSIGNED_INTEGER,                                           \
        "0",                                                            \
        INITIAL_ABUNDANCE_MIX_VAR,                                      \
        1.0                                                             \
        } ,                                                             \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "init_abund",                                                   \
        "Usage: --init_abund <i> <X>. Sets the initial abundance of isotope number <i> to mass fraction <X>.", \
        ARG_SUBROUTINE,                                                 \
        "0.02",                                                         \
        &set_init_abund,                                                \
        1.0                                                             \
        } ,                                                             \
                                                                        \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "init_abund_mult",                                              \
        "Usage: --init_abund_mult <i> <f>. Multiplies the initial abundance of isotope number <i> by <f>.", \
        ARG_SUBROUTINE,                                                 \
        "1.0",                                                          \
        &set_init_abund_mult,                                           \
        1.0                                                             \
        } ,                                                             \
                                                                        \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "init_abund_dex",                                               \
        "Usage: --init_abund_dex <i> <f>. Changes the initial abundance of isotope number <i> by <f> dex.", \
        ARG_SUBROUTINE,                                                 \
        "0.0",                                                          \
        &set_init_abund_dex,                                            \
        0.0                                                             \
        } ,                                                             \
                                                                        \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "init_abunds_only",                                             \
        "If True, outputs only the initial abundances, then exits.",    \
        ARG_BOOLEAN,                                                    \
        WTTS_USE_DEFAULT,                                               \
        INIT_ABUNDS_VAR,                                                \
        1.0                                                             \
        } ,                                                             \
                                                                        \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "initial_abunds_only",                                          \
        "If True, outputs only the initial abundances, then exits.",    \
        ARG_BOOLEAN,                                                    \
        WTTS_USE_DEFAULT,                                               \
        INIT_ABUNDS_VAR,                                                \
        1.0                                                             \
        } ,                                                             \
                                                                        \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "no_thermohaline_mixing",                                       \
        "If True, disables thermohaline mixing.",                       \
        ARG_BOOLEAN,                                                    \
        WTTS_USE_DEFAULT,                                               \
        NO_THERMOHALINE_MIXING_VAR,                                     \
        1.0                                                             \
        } ,                                                             \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "lithium_GB_post_Heflash",                                      \
        "Sets the lithium abundances after the helium flash. Requires NUCSYN and LITHIUM_TABLES.", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        LITHIUM_GB_POST_HEFLASH_VAR,                                    \
        1.0                                                             \
        } ,                                                             \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "lithium_GB_post_1DUP",                                         \
        "Sets the lithium abundance after first dredge up. Requires NUCSYN and LITHIUM_TABLES.", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        LITHIUM_GB_POST_1DUP_VAR,                                       \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "lithium_hbb_multiplier",                                       \
        "Multiplies the lithium abundances on the AGB during HBB (based on Karakas/Fishlock et al models).Requires NUCSYN and LITHIUM_TABLES.", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        LITHIUM_GB_HBB_MULTIPLIER_VAR,                                  \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "angelou_lithium_decay_function",                               \
        "Functional form which describes Li7 decay. Requires NUCSYN and NUCSYN_ANGELOU_LITHIUM. Choices are : 0 expoential (see angelou_lithium_decay_time).", \
        ARG_INTEGER,                                                    \
        WTTS_USE_DEFAULT,                                               \
        ANGELOU_LITHIUM_DECAY_FUNCTION_VAR,                             \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "angelou_lithium_LMMS_time",                                    \
        "Time at which lithium manufacture is triggered in a low-mass (convective) main sequence (Myr). Requires NUCSYN and NUCSYN_ANGELOU_LITHIUM. Ignored if 0 (for the start, use 1e-6).\n", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        ANGELOU_LITHIUM_LMMS_TIME_VAR,                                  \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "angelou_lithium_MS_time",                                      \
        "Time at which lithium manufacture is triggered on the main sequence (Myr). Requires NUCSYN and NUCSYN_ANGELOU_LITHIUM. Ignored if 0 (for the start, use 1e-6).\n", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        ANGELOU_LITHIUM_MS_TIME_VAR,                                    \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "angelou_lithium_HG_time",                                      \
        "Time at which lithium manufacture is triggered on the Hertzsprung gap (Myr). Requires NUCSYN and NUCSYN_ANGELOU_LITHIUM. Ignored if 0 (for the start, use 1e-6).\n", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        ANGELOU_LITHIUM_HG_TIME_VAR,                                    \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "angelou_lithium_GB_time",                                      \
        "Time at which lithium manufacture is triggered on the giant branch (Myr). Requires NUCSYN and NUCSYN_ANGELOU_LITHIUM. Ignored if 0 (for the start, use 1e-6).\n", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        ANGELOU_LITHIUM_GB_TIME_VAR,                                    \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "angelou_lithium_CHeB_time",                                    \
        "Time at which lithium manufacture is triggered during core helium burning (Myr). Requires NUCSYN and NUCSYN_ANGELOU_LITHIUM. Ignored if 0 (for the start, use 1e-6).\n", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        ANGELOU_LITHIUM_CHeB_TIME_VAR,                                  \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "angelou_lithium_EAGB_time",                                    \
        "Time at which lithium manufacture is triggered on the early AGB (Myr). Requires NUCSYN and NUCSYN_ANGELOU_LITHIUM. Ignored if 0 (for the start, use 1e-6).\n", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        ANGELOU_LITHIUM_EAGB_TIME_VAR,                                  \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "angelou_lithium_TPAGB_time",                                   \
        "Time at which lithium manufacture is triggered on the thermally pulsing AGB (Myr). Requires NUCSYN and NUCSYN_ANGELOU_LITHIUM. Ignored if 0 (for the start, use 1e-6).\n", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        ANGELOU_LITHIUM_TPAGB_TIME_VAR,                                 \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "angelou_lithium_LMMS_decay_time",                              \
        "Decay time for surface lithium abundance during the low-mass (convective) main sequence (Myr). Requires NUCSYN and NUCSYN_ANGELOU_LITHIUM. Ignored if 0 (for the start, use 1e-6).\n", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        ANGELOU_LITHIUM_LMMS_DECAY_TIME_VAR,                            \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "angelou_lithium_MS_decay_time",                                \
        "Decay time for surface lithium abundance on the main sequence (Myr). Requires NUCSYN and NUCSYN_ANGELOU_LITHIUM. Ignored if 0 (for the start, use 1e-6).\n", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        ANGELOU_LITHIUM_MS_DECAY_TIME_VAR,                              \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "angelou_lithium_HG_decay_time",                                \
        "Decay time for surface lithium abundance on the Hertzsprung gap (Myr). Requires NUCSYN and NUCSYN_ANGELOU_LITHIUM. Ignored if 0 (for the start, use 1e-6).\n", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        ANGELOU_LITHIUM_HG_DECAY_TIME_VAR,                              \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "angelou_lithium_GB_decay_time",                                \
        "Decay time for surface lithium abundance on the giant branch (Myr). Requires NUCSYN and NUCSYN_ANGELOU_LITHIUM. Ignored if 0 (for the start, use 1e-6).\n", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        ANGELOU_LITHIUM_GB_DECAY_TIME_VAR,                              \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "angelou_lithium_CHeB_decay_time",                              \
        "Decay time for surface lithium abundance during core helium burning (Myr). Requires NUCSYN and NUCSYN_ANGELOU_LITHIUM. Ignored if 0 (for the start, use 1e-6).\n", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        ANGELOU_LITHIUM_CHeB_DECAY_TIME_VAR,                            \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "angelou_lithium_EAGB_decay_time",                              \
        "Decay time for surface lithium abundance on the early AGB (Myr). Requires NUCSYN and NUCSYN_ANGELOU_LITHIUM. Ignored if 0 (for the start, use 1e-6).\n", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        ANGELOU_LITHIUM_EAGB_DECAY_TIME_VAR,                            \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "angelou_lithium_TPAGB_decay_time",                             \
        "Decay time for surface lithium abundance on the thermally pulsing AGB (Myr). Requires NUCSYN and NUCSYN_ANGELOU_LITHIUM. Ignored if 0 (for the start, use 1e-6).\n", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        ANGELOU_LITHIUM_TPAGB_DECAY_TIME_VAR,                           \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "angelou_lithium_LMMS_massfrac",                                \
        "Lithium mass fraction when its manufacture is triggered during the low-mass (convective) main sequence (Myr). Requires NUCSYN and NUCSYN_ANGELOU_LITHIUM. Ignored if 0 (for the start, use 1e-6).\n", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        ANGELOU_LITHIUM_LMMS_MASSFRAC_VAR,                              \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "angelou_lithium_MS_massfrac",                                  \
        "Lithium mass fraction when its manufacture is triggered on the main sequence (Myr). Requires NUCSYN and NUCSYN_ANGELOU_LITHIUM. Ignored if 0 (for the start, use 1e-6).\n", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        ANGELOU_LITHIUM_MS_MASSFRAC_VAR,                                \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "angelou_lithium_HG_massfrac",                                  \
        "Lithium mass fraction when its manufacture is triggered on the Hertzsprung gap (Myr). Requires NUCSYN and NUCSYN_ANGELOU_LITHIUM. Ignored if 0 (for the start, use 1e-6).\n", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        ANGELOU_LITHIUM_HG_MASSFRAC_VAR,                                \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "angelou_lithium_GB_massfrac",                                  \
        "Lithium mass fraction when its manufacture is triggered on the giant branch (Myr). Requires NUCSYN and NUCSYN_ANGELOU_LITHIUM. Ignored if 0 (for the start, use 1e-6).\n", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        ANGELOU_LITHIUM_GB_MASSFRAC_VAR,                                \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "angelou_lithium_CHeB_massfrac",                                \
        "Lithium mass fraction when its manufacture is triggered during core helium burning (Myr). Requires NUCSYN and NUCSYN_ANGELOU_LITHIUM. Ignored if 0 (for the start, use 1e-6).\n", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        ANGELOU_LITHIUM_CHeB_MASSFRAC_VAR,                              \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "angelou_lithium_EAGB_massfrac",                                \
        "Lithium mass fraction when its manufacture is triggered on the early AGB (Myr). Requires NUCSYN and NUCSYN_ANGELOU_LITHIUM. Ignored if 0 (for the start, use 1e-6).\n", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        ANGELOU_LITHIUM_EAGB_MASSFRAC_VAR,                              \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "angelou_lithium_TPAGB_massfrac",                               \
        "Lithium mass fraction when its manufacture is triggered on the thermally pulsing AGB (Myr). Requires NUCSYN and NUCSYN_ANGELOU_LITHIUM. Ignored if 0 (for the start, use 1e-6).\n", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        ANGELOU_LITHIUM_TPAGB_MASSFRAC_VAR,                             \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "angelou_lithium_vrot_trigger",                                 \
        "Equatorial rotational velocity at which lithium manufacture is triggered (km/s). Requires NUCSYN and NUCSYN_ANGELOU_LITHIUM. Ignored if 0.\n", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        ANGELOU_LITHIUM_VROT_TRIGGER_VAR,                               \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_NUCSYN,                                                 \
        "angelou_lithium_vrotfrac_trigger",                             \
        "Fraction of Keplerian (breakup) equatorial rotational velocity at which lithium manufacture is triggered (must be <1, ignored if 0). Requires NUCSYN and NUCSYN_ANGELOU_LITHIUM. Ignored if 0.\n", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        ANGELOU_LITHIUM_VROTFRAC_TRIGGER_VAR,                           \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "gaia_Teff_binwidth",                                           \
        "log10(Effective temperature) bin width used to make Gaia-like HRDs\n", \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        GAIA_TEFF_BINWIDTH,                                             \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "gaia_L_binwidth",                                              \
        "log10(luminosity) bin width used to make Gaia-like HRDs\n",    \
        ARG_FLOAT,                                                      \
        WTTS_USE_DEFAULT,                                               \
        GAIA_L_BINWIDTH,                                                \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_OUTPUT,                                                 \
        "fabian_imf_log_time",                                          \
        "Time at which to output for Fabian Schneider's IMF project. Requires FABIAN_IMF_LOG", \
        ARG_FLOAT,                                                      \
        "Ignore",                                                       \
        NEXT_FABIAN_IMF_LOG_TIME_VAR,                                   \
        1.0                                                             \
        } ,                                                             \
{                                                                       \
    ARG_SECTION_OUTPUT,                                                 \
        "fabian_imf_log_timestep",                                      \
        "Timestep for Fabian Schneider's IMF project logging. Requires FABIAN_IMF_LOG", \
        ARG_FLOAT,                                                      \
        "Ignore",                                                       \
        NEXT_FABIAN_IMF_LOG_TIMESTEP_VAR,                               \
        1.0                                                             \
        }  ,                                                            \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "rotationally_enhanced_mass_loss",                              \
        "Set to 1 to enable rotationally enhanced mass loss rate algorithms: 0= none, 1=formula cf. Langer models (=ROTATIONALLY_ENHNACED_MASSLOSS_LANGER_FORMULA), 2=limit accretion rate before wind loss is applied, 3 = both 1 and 2. See also rotationally_enhanced_exponent", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        ROTATIONALLY_ENHANCED_MASS_LOSS_VAR,                            \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "AGB_core_algorithm",                                           \
        "Algorithm to use for calculating AGB core masses. 0=Hurley et al. 2002 if no NUCSYN, Karakas 2002 if NUCSYN is defined; 1=Hurley et al. 2002 (overshooting models); 1=Karakas 2002 (non-overshooting models).", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->AGB_core_algorithm),                   \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "AGB_radius_algorithm",                                         \
        "Algorithm to use for calculating radii on the TPAGB.",         \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->AGB_radius_algorithm),                 \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "AGB_luminosity_algorithm",                                     \
        "Algorithm to use for calculating luminosities on the TPAGB.",  \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->AGB_luminosity_algorithm),             \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "AGB_3dup_algorithm",                                           \
        "Algorithm to use for calculating third dredge up efficiency on the TPAGB.", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->AGB_3dup_algorithm),                   \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "overspin_algorithm",                                           \
        "Algorithm to determine what we do when a star is rotating at its breakup velocity. OVERSPIN_BSE (0) conservatively transfers the angular momentum back to the orbit. OVERSPIN_MASSLOSS uses the rotationally_enhanced_massloss parameter to lose mass which carries away the angular momentum.", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->overspin_algorithm),                   \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "rotationally_enhanced_exponent",                               \
        "The exponent (power) by which rotationally enhanced mass loss is raised. Requires ROTATIONALLY_ENHANCED_MASS_LOSS. See also rotationally_enhanced_mass_loss.", \
        ARG_FLOAT ,                                                     \
        WTTS_USE_DEFAULT,                                               \
        ROTATIONALLY_ENHANCED_EXPONENT_VAR,                             \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_MISC,                                                   \
        "warmup_cpu",                                                   \
        "Uses the CPU at maximum power the given number of seconds, prior to running normal stellar evolution.", \
        ARG_SUBROUTINE,                                                 \
        "Ignore",                                                       \
        &(binary_c_warmup_cpu),                                         \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_MISC,                                                   \
        "help",                                                         \
        "Display help pages. Usage: --help <help topic>.",              \
        ARG_SUBROUTINE,                                                 \
        "Ignore",                                                       \
        &(binary_c_help_from_arg)                                                \
        ,1.0                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_MISC,                                                   \
        "help_all",                                                     \
        "Display all help pages.",                                      \
        ARG_SUBROUTINE,                                                 \
        "Ignore",                                                       \
        &(binary_c_help_all)                                            \
        ,1.0                                                            \
        },                                                              \
{                                                                       \
    ARG_SECTION_MISC,                                                   \
        "list_args",                                                    \
        "Display list of arguments with their default values. Useful for batchmode.", \
        ARG_SUBROUTINE,                                                 \
        "Ignore",                                                       \
        &(list_available_args_and_exit) ,                               \
        1.0                                                             \
        } ,                                                             \
{                                                                       \
    ARG_SECTION_OUTPUT,                                                 \
        "version",                                                      \
        "Display binary_c version and build information. Also performs timing tests.", \
        ARG_SUBROUTINE,                                                 \
        "Ignore",                                                       \
        &binary_c_version_internal,                                     \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_OUTPUT,                                                 \
        "dumpversion",                                                  \
        "Display binary_c version number (short format).",              \
        ARG_SUBROUTINE,                                                 \
        "Ignore",                                                       \
        &dumpversion,                                                   \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_OUTPUT,                                                 \
        "version_only",                                                 \
        "Display binary_c version number and build information, but do not perform timing tests or anything that requires stardata to be non-NULL.", \
        ARG_SUBROUTINE,                                                 \
        "Ignore",                                                       \
        &version_only,                                                  \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "batchmode",                                                    \
        "Set the batchmode control variable. Use only if you know what you are doing!", \
        ARG_INTEGER ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->batchmode),                            \
        1.0                                                             \
        },                                                              \
                                                                        \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "speedtests",                                                   \
        "If True, turns on speedtests during version information (off by default).", \
        ARG_BOOLEAN ,                                                   \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->speedtests),                           \
        1.0                                                             \
        },                                                              \
{                                                                       \
    ARG_SECTION_STARS,                                                  \
        "use_fixed_timestep_%d",                                        \
        "Set to True to use fixed timestep <n>, False to turn off. Fixed timesteps are on (this is True) by default.", \
        ARG_BOOLEAN_SCANF,                                              \
        WTTS_USE_DEFAULT,                                               \
        &(stardata->preferences->use_fixed_timestep),                   \
        1.0                                                             \
        }                                                               \
                                                                        \
    
    




#endif // BINARY_C_CMD_LINE_ARGS_H
