#!/usr/bin/env perl
use strict;
use Sort::Versions;
use rob_misc;
use subs 'printx';
$|=1;

#
# Script to build binary_c with each available gcc and clang,
# checking for errors and warnings
#
my $logfile = '/tmp/check_compilers.log';
open(my $logfp,'>',$logfile)||die("cannot open $logfile");
my $args = "@ARGV";
my $vb = ($args=~/-v/) ? 1 : 0;
my @compilers = ('gcc','clang');
my $compiler_regexp = join('|',@compilers);
my $meson_version = (`meson --version`=~/^(\d+\.\d+\.\d+)/)[0]; # set to 1 to use meson
my $meson = defined $meson_version ? 1 : 0;
my $meson_builddir = "builddir_check_compilers";
# exclude these buggy compilers
my $exclude_compilers =
{
    'gcc' => {
        '4.7.4' => 'Crashes with internal errors on build'
        },
        'clang' => {
        },
};

printx "Check binary_c compilers (vb = $vb), logging to $logfile, building with ",($meson ? "meson $meson_version" : 'make')."\n";

foreach my $compiler (@compilers)
{
    my @executables = find_executables($compiler);

    printx "Compiler    : $compiler\n";
    printx "Executables : @executables\n";
 
    foreach my $executable (@executables)
    {
        build($executable);
    }
}

close $logfile;
exit;


sub find_executables
{
    # find available executables of the compiler
    my ($compiler) = @_;

    # find the standard executables
    my @executables = grep{
        /$compiler-?\d*(\.\d+)*$/
    }split(/\n/,`bash -c "compgen -ca $compiler"`);
    chomp @executables;
    
    # sort by string length
    @executables = sort {
        length($a) <=> length($b)
    } @executables;
    
    # get versions : we want a unique list by version
    my %versions;
    foreach my $executable (@executables)
    {
        my $v = version_of($executable);
        
        if($exclude_compilers->{$compiler}->{$v})
        {
            printx "Skipping $compiler $v because it is excluded ($exclude_compilers->{$compiler}->{$v})\n";
        }
        elsif(!defined $v)
        {
            printx "Warning: $executable gave no version\n";
        }
        else
        {
            $versions{$v} = $executable;
        }
    }
    
    # order by version
    @executables = sort {versioncmp($a,$b)} values %versions;
    
    return @executables;    
}

sub version_of
{
    # return version of the compiler, or undef
    my ($executable) = @_;
    return (`$executable -v  2>\&1`=~/version (\S+)/)[0] //
        (`$executable --version 2>\&1`=~/($compiler_regexp) \S+ (\d.*)/)[0] //
        undef;
}

sub build
{
    my ($compiler) = @_; 
    my $v = version_of($compiler)//'';
    
    # environment for commnads 
    my $env = "unset CC; export CC=$compiler; ";

    # run configure or meson
    printx "Configure with $compiler (version $v)\n";
    my $cmd = 
        $meson 
        ?
        "$env rm -rf $meson_builddir; $env meson $meson_builddir --buildtype release"
        :
        "$env ./configure 2>/dev/stdout";

    printx "CMD $cmd\n"if($vb);
    my $r = `$cmd`;
    printx $r if($vb);
    
    if($meson ? 
       $r=~/Found ninja/ :
       $r=~/Now run .\/make/)
    {
        # configure successful : do the build
        printx "Building\n";
        $cmd = 
            $meson ?
            "cd $meson_builddir; ninja 2>/dev/stdout" :
            "$env ./make clean 2>/dev/stdout; ./make 2>/dev/stdout";
        printx "CMD $cmd\n" if($vb);
        $r = `$cmd`;
        
        # clean the results of successful statements
        if($meson)
        {
            $r =~ s/^\[\d+\/\d+].*//mg;
        }
        else
        {    
            $r =~ s/(?:rm -f|Make on|Done|BUILD|LINK|DONE).*//g;
            
        }
        $r =~ s/\n\s*\n//g;
        $r =~ s/^\s+//;
        $r =~ s/\s+$//;
        $r = rob_misc::remove_ANSI($r);
        
        # check for remaining warnings or errors
        if($r)
        {
            # error or warning found
            printx "Found error or warning\n\n\"$r\"\n";
            close $logfile;
            print "\n\n\nSee $logfile\n";
            exit;
        }
        else
        {
            printx "Build was clean\n";
        }
    }
    else
    {
        printx "Configure with compiler \"$compiler\" (version $v) failed \n";
        exit;
    }
}


sub printx(@)
{
    my $s = join('',@_);
    print {$logfp} $s if(defined $logfp);
    print $s;
}
