
/*
 * Macros for stellar yield and population ensemble output.
 */

/*
 * Define when a yield is zero. It's best not to use
 * the Is_zero macro, because the yield output could be
 * <1e-14 for rare elements.
 */
#define Yield_is_zero(Y) (fabs(Y)<1e-100)

#define TOTAL_PMS_MASS (stardata->star[0].pms_mass+stardata->star[1].pms_mass)
#define TOTAL_MPYIELD (stardata->star[0].mpyield[i]+stardata->star[1].mpyield[i])

/*
 * Compress yields, e.g. "0 0 0 0" becomes "4x0".
 */
#define COMPRESS_YIELD_AND_ENSEMBLE_OUTPUT

/*
 * Perhaps number ensemble items?  Useful for debugging.
 */
//#define NUMBER_ENSEMBLE_ITEMS


#if defined NUCSYN_YIELD_COMPRESSION || defined NUCSYN_LOG_BINARY_DX_YIELDS
#define SAVE_YIELDS
#endif

/*
 * Enable this to check if yields are < 0 (note sometimes numerical errors give
 * <0 yields... so this will cause the code to fail!
 */
//#define NUCSYN_CHECK_LESS_THAN_ZERO_YIELDS

/* MAximum error in mass conservation */
#define MAXPCERR 10 

#define Star_nucsyn_output(K)                                                   \
    fprintf(stardata->model.log_fp,                                     \
            "Star %d : type %d : mass %g : core_mass %g\n",             \
            (K),                                                        \
            stardata->star[(K)].stellar_type,stardata->star[(K)].mass,  \
            stardata->star[(K)].core_mass);                             \
                                                                        \
    _printf("Star %d : type %d : mass %g : core_mass %g\n",             \
            (K),                                                        \
            stardata->star[(K)].stellar_type,                           \
            stardata->star[(K)].mass,                                   \
            stardata->star[(K)].core_mass);


#define Dump_zeros(N)                                \
    {                                           \
        if(zerocount<3)                         \
        {                                       \
            if(zerocount==1)                    \
            {                                   \
                Printf("0 ");                   \
            }                                   \
            else if(zerocount==2)               \
            {                                   \
                Printf("0 0 ");                 \
            }                                   \
        }                                       \
        else                                    \
        {                                       \
            Printf("%dx0 ",zerocount);          \
        }                                       \
    }

#define ENSEMBLE_Zeros(N)                       \
    {                                           \
        if(zerocount<3)                         \
        {                                       \
            if(zerocount==1)                    \
            {                                   \
                Ensemble_printf("0 ");          \
            }                                   \
            else if(zerocount==2)               \
            {                                   \
                Ensemble_printf("0 0 ");        \
            }                                   \
        }                                       \
        else                                    \
        {                                       \
            Ensemble_printf("%dx0 ",zerocount); \
        }                                       \
    }

#ifdef NUCSYN_LONG_YIELD_FORMAT
#define YIELD_FORMAT "%0+.12e "
#else
#define YIELD_FORMAT "%0+.3e "
#endif

/* 
 * Given a linear time T, calculate the equvialent 
 * yields time (which can be log or linear)
 */
#define Yields_time(T)                                  \
    (stardata->preferences->yields_logtimes == TRUE ?   \
     log10(Max(1e-50,(T))) :                            \
     (T))

/*
 * Given a yields time Y calculate a linear time 
 */
#define Time_from_yields_time(Y)                        \
    (stardata->preferences->yields_logtimes == TRUE ?   \
     pow(10.0,(Y)) :                                    \
     (Y))

/*
 * What is the yields timestep? If we're using log times,
 * return the yields_logdt, if we're using linear times, return 
 * the yields_dt
 */
#define Yields_timestep                                 \
    (stardata->preferences->yields_logtimes == TRUE ?   \
     stardata->preferences->yields_logdt :              \
     stardata->preferences->yields_dt)

/*
 * Return the yields timestep in linear time
 */
#define Yields_timestep_linear                                          \
    (stardata->preferences->yields_logtimes == TRUE ?                   \
     (log(10.0)*stardata->model.time*                                   \
      stardata->preferences->yields_logdt) :                            \
     stardata->preferences->yields_dt)

#define Yields_start_time_linear                                        \
    (stardata->preferences->yields_logtimes == TRUE ?                   \
     Time_from_yields_time(stardata->preferences->                      \
                           yields_startlogtime) :                       \
     0.0)

/*
 * Start of yielding in yield time units
 */
#define Yields_start_time                                               \
    (stardata->preferences->yields_logtimes == TRUE ?                   \
     log10(stardata->preferences->yields_startlogtime) :                \
     0.0)

/*
 * Round the number X to calculate the next yield time.
 * NB we have to take care with negative numbers, hence the TINY.
 */
#define Round_yields_t(X) ((X)>0.0 ? ((long int)(X)) : (-(long int)(-(X)+TINY)))

/*
 * Given a linear time T, what is the next yield time?
 *
 * To calculate this, we need the Yield_timestep_number, then add one
 * timestep to it. The timestep number starts at the yields start time.
 */

#define Yield_timestep_number(T)                                        \
    Round_yields_t((Yields_time(T)-Yields_start_time)/                  \
                   Yields_timestep)
        
/* hence the next yield time (in yield time units) */
#define Next_yields_t(T)                        \
    (                                           \
        (                                       \
            1.0 +                               \
            1.0 * Yield_timestep_number(T)      \
            ) *                                 \
        Yields_timestep                         \
        +                                       \
        Yields_start_time                       \
        )

/*
 * And a generic version for stardata->model.time
 */
#define Next_yields_time Next_yields_t(stardata->model.time)
            

/*
 * If TESTYIELDS is defined, output a constant rate 1Msun/Myr of 
 * TESTYIELDISOTOPE (which is usually neutrons) in each source.
 *
 * This allows you to test the various incarnations of yields_vs_time.pl
 * and its log/linear time algorithms. 
 */
//#define TESTYIELDS

#ifdef TESTYIELDS
#define TESTYIELDX (stardata->model.probability)
#define TESTYIELDSTAR 0
#define TESTYIELDISOTOPE (Xn)
#endif // TESTYIELDS


#define ENSEMBLE_START_TIME 1e-14

#define ENSEMBLE_TEST_STAR 0

#define ENSEMBLE_MINIMUM_PMS_MASS 0.02

/*
 * Output macros for the ensemble
 */
#ifdef NUCSYN_GCE_DISABLE_ENSEMBLE_OUTPUT
#define Ensemble_print(K,X) /* */
#define Ensemble_printf(K,X) /* */
#else

#ifdef NUMBER_ENSEMBLE_ITEMS
/* prepend ensemble item number */
#define __FORMAT__ "%d=%g "
#define Ensemble_print(K,X)                                     \
    Printf(                                                     \
        __FORMAT__,                                             \
        (K),(                                                   \
            ensemble_type[(K)]==ENSEMBLE_TYPE_MAGNITUDE ?       \
            magnitude_from_luminosity(X) :                      \
            (X)                                                 \
            )                                                   \
        );
#else
/* no prepend */
#define __FORMAT__ "%g "
#define Ensemble_print(K,X)                                     \
    Printf(                                                     \
        __FORMAT__,                                             \
        (                                                       \
            ensemble_type[(K)]==ENSEMBLE_TYPE_MAGNITUDE ?       \
            magnitude_from_luminosity(X) :                      \
            (X)                                                 \
            )                                                   \
        );
#endif
    
#define Ensemble_printf(...) Printf(__VA_ARGS__)    
#endif // ?NUCSYN_GCE_DISABLE_ENSEMBLE_OUTPUT

/*
 * Output macros for the yields
 */
#define Yield_print(Y)                                                  \
    Printf(YIELD_FORMAT,                                                \
           (                                                            \
               stardata->preferences->yields_logtimes == TRUE ?         \
               ((Y)/(log(10.0)*stardata->model.time)) :                 \
               (Y)                                                      \
               )                                                        \
        )



