#include "../binary_c.h"

/*
 * Subroutine to update the abundance of each star due to mass
 * mass transfer, gain and loss.
 *
 * This subroutine 
 */
#ifdef NUCSYN

#include "nucsyn_update_abundances.h"

//#undef MDEBUG
//#define MDEBUG(...) if(k==1){printf("star%d: ",k);printf(__VA_ARGS__);}
#undef MDEBUG
#define MDEBUG(...) Dprint(__VA_ARGS__);

void Hot_function nucsyn_update_abundances(struct stardata_t * RESTRICT const stardata)
{
    
    /*
     * Abundance arrays are
     * 
     * Xacc,Xenv [k], k=0,1 
     * The material already in star k's accretion layer and the
     * material in star k's envelope.
     * 
     * nXacc, nXenv [k], k=0,1 The abundances in the accretion layer and
     * envelope after accretion has taken place. At the end of the routine Xacc
     * and Xenv will be set to these values but NOT BEFORE because then things
     * will be all wrong!
     */
    Star_number k;    
    Boolean allocated_Xdonor;
    Boolean allocated_Xaccretor;
    Boolean allocated_XRLOF_gain;
    Boolean allocated_Xwind_loss;
    Boolean allocated_XRLOF_and_decretion_loss;
    
    /* accretion layer thicknesses */
    const double dmacc[NUMBER_OF_STARS] = {
        stardata->star[0].dmacc,
        stardata->star[1].dmacc
    };
    double ndmacc[NUMBER_OF_STARS];

    /* Abundance arrays for pre-mix values */
    const Abundance * const Xacc[NUMBER_OF_STARS] = {
        stardata->star[0].Xacc,
        stardata->star[1].Xacc
    };
    const Abundance *Xenv[NUMBER_OF_STARS] = {
        stardata->star[0].Xenv,
        stardata->star[1].Xenv
    };
    
    /*
     * Use pointers in tmpstore rather than 
     * local pointers when we can, this
     * saves the memory being allocated and freed repeatedly.
     */
#define __tmp stardata->tmpstore
#define Xwind_gain __tmp->Xwind_gain
#ifdef NUCSYN_NOVAE
#define Xnovae __tmp->Xnovae
#endif //NUCSYN_NOVAE
#define nXacc __tmp->nXacc
#define nXenv __tmp->nXenv

    /* timestep in years */
    const double dt = stardata->model.dtm * 1e6;

    if(unlikely(nXacc[0] == NULL))
    {
        Xwind_gain = New_isotope_array;
#ifdef NUCSYN_NOVAE
        Xnovae = New_isotope_array;
#endif//NUCSYN_NOVAE
        Starloop(k)
        {
            nXacc[k] = New_isotope_array;
            nXenv[k] = New_isotope_array;
        }
    }

    /* pointers that may be allocated */
    Abundance *Xdonor = NULL;
    Abundance *Xaccretor = NULL;
    Abundance *Xwind_loss = NULL;
    Abundance *XRLOF_gain = NULL;
    Abundance *XRLOF_and_decretion_loss = NULL;
            
    Evolving_STARLOOP(k)
    {
        SETstar(k);

        /* set the abundance arrays that may be altered */
        Copy_abundances(Xacc[k], nXacc[k]);
        Copy_abundances(Xenv[k], nXenv[k]);

        /* Set initial accretion layer thickness */
        ndmacc[k] = star->dmacc;
        
        MDEBUG("\nMIXING star %d type %d : dmacc=%g : env=%g : Mdot_wind=%g : Mdot_acc=%g (C12 Env %g Acc %g)\n",
               k,
               star->stellar_type,
               dmacc[k],
               Envelope_mass(k),
               Dm(DERIVATIVE_STELLAR_MASS_WIND_LOSS),
               Dm(DERIVATIVE_STELLAR_MASS_WIND_GAIN),
               Xenv[k][XC12],
               Xacc[k][XC12]
            );

        MDEBUG("XC12 in %d ACC %g ENV %g\n",
               k,
               Xacc[k][XC12],
               Xenv[k][XC12]);
    }
   
    MIXDEBUG_START;

        
    Evolving_STARLOOP(k)
    {
        SETstars(k);
        
        /* 
         * Stellar winds
         * 
         * Choose f, the fraction of the accreting wind which gets mixed
         * with the star's wind prior to accretion.
         * 
         * f=0 is like having all the mass from the companion's wind
         *      hit the star.
         *
         * f=1 is like having a shock so all the star's wind
         *     hit the star. 
         *
         * f is also limited by the accretion rate
         */
        const double dm_wind_loss = -Dm(DERIVATIVE_STELLAR_MASS_WIND_LOSS);
        const double dm_wind_gain = +Dm(DERIVATIVE_STELLAR_MASS_WIND_GAIN);

        const double f = Min(nucsyn_choose_wind_mixing_factor(stardata,k),
                             dm_wind_loss/(dm_wind_gain+VERY_TINY));

        /* mass from the companion */
        const double Dm_companion_wind = dm_wind_gain * (1.0 - f);

        /* mass from the star */
        const double Dm_star_wind = dm_wind_gain * f;
        
        Dprint("dm_wind loss=%g gain=%g f=%g Dm companion=%g star=%g\n",
               dm_wind_loss,
               dm_wind_gain,
               f,
               Dm_companion_wind,
               Dm_star_wind);

        struct star_t * prev_star =
            stardata->previous_stardata != NULL ? 
            &stardata->previous_stardata->star[star->starnum] :
            star;
        struct star_t * prev_companion =
            stardata->previous_stardata != NULL ? 
            &stardata->previous_stardata->star[companion->starnum] :
            companion;
        
        /* hence the abundances */
        nucsyn_remove_dm_from_surface(
#if defined DEBUG && DEBUG > 0
            stardata,
#endif
            prev_star,
            Dm_star_wind,
            &Xaccretor,
            &allocated_Xaccretor);

        nucsyn_remove_dm_from_surface(
#if defined DEBUG && DEBUG > 0
            stardata,
#endif
            prev_companion,
            Dm_companion_wind,
            &Xdonor,
            &allocated_Xdonor);

        /*
         * Set Xwind_gain to be the mixture of
         * companion and star wind
         */
        nucsyn_dilute_shell_to(Dm_companion_wind,
                               Xdonor,
                               Dm_star_wind,
                               Xaccretor,
                               Xwind_gain);
        
        /*
         * other mass gain:
         * dm_RLOF_gain is from the companion star
         * dm_cbdisc_gain is accretion from the innermost circumbinary disc
         */
        const double dm_RLOF_gain = Dm(DERIVATIVE_STELLAR_MASS_RLOF_GAIN);
#ifdef DISCS
        const double dm_cbdisc_gain = Dm(DERIVATIVE_STELLAR_MASS_CBDISC_GAIN);

        /* CBdisc loss : only do this for star 0 */
        const double dm_cbdisc_loss = k==1 ? 0.0 :
            -Dm(DERIVATIVE_SYSTEM_CBDISC_MASS_LOSS);
#endif
        
        /*
         * other mass loss:
         * dm_RLOF_loss, dm_excretion_loss are from this star
         * dm_disc_loss is from the circumstellar disc (not yet implemented)
         * dm_nonconservative_loss is from the other star 
         */
        const double dm_RLOF_loss = -Dm(DERIVATIVE_STELLAR_MASS_RLOF_LOSS);
        const double dm_excretion_loss = -Dm(DERIVATIVE_STELLAR_MASS_DECRETION_DISC);
        const double MAYBE_UNUSED dm_disc_loss = -Dm(DERIVATIVE_STELLAR_MASS_DISC_LOSS);
        const double dm_nonconservative_loss = -Dm(DERIVATIVE_STELLAR_MASS_DISC_LOSS);

        Dprint("NUA Gain RLOF %g, wind %g; Lose RLOF %g, wind %g; Decretion %g, nonconservative %g\n",
               dm_RLOF_gain,dm_wind_gain,dm_RLOF_loss,dm_wind_loss,
               dm_excretion_loss,dm_nonconservative_loss
            );

        /*
         * Wind accretion, abundance calculated above
         */
        nucsyn_add_mass_to_surface(k,
                                   dm_wind_gain,
                                   Xwind_gain,
                                   ndmacc,
                                   Xacc,
                                   nXacc);
        
        /*
         * RLOF accretion and non-conservative loss: 
         * take abundances from the companion
         */

        nucsyn_remove_dm_from_surface(
#if defined DEBUG && DEBUG > 0
            stardata,
#endif
            prev_companion,
            dm_RLOF_gain + dm_nonconservative_loss,
            &XRLOF_gain,
            &allocated_XRLOF_gain);

        /* only the RLOF gain is accreted */
        nucsyn_add_mass_to_surface(k,
                                   dm_RLOF_gain,
                                   XRLOF_gain,
                                   ndmacc,
                                   Xacc,
                                   nXacc);
        
#ifdef DISCS
        {
            /*
             * Get abundance from the first (assumed innermost)
             * circumbinary disc
             */
            if(stardata->common.ndiscs>=1 &&
               stardata->common.discs[0].type==DISC_CIRCUMBINARY)
            {
                nucsyn_add_mass_to_surface(k,
                                           dm_cbdisc_gain,
                                           stardata->common.discs[0].X,
                                           ndmacc,
                                           Xacc,
                                           nXacc);
            }
        }
#endif // DISCS

        /*
         * Wind loss
         */
        nucsyn_remove_mass_from_surface(
#if defined DEBUG && DEBUG > 0
            stardata,
#endif
            prev_star,
            dm_wind_loss,
            &Xwind_loss,
            ndmacc,
            &allocated_Xwind_loss);
        Dprint("post-add wind loss\n");
                
        /*
         * RLOF and excretion disc loss
         */
        nucsyn_remove_mass_from_surface(
#if defined DEBUG && DEBUG > 0
            stardata,
#endif
            prev_star,
            dm_RLOF_loss+dm_excretion_loss,
            &XRLOF_and_decretion_loss,
            ndmacc,
            &allocated_XRLOF_and_decretion_loss);
        Dprint("post-add RLOF loss\n");

        /* Yields : positive and negative */

        /* RLOF */
        Add_yield(dm_RLOF_loss, XRLOF_and_decretion_loss, // lost
                  dm_RLOF_gain, XRLOF_gain, // accreted
                  NUCSYN_SOURCE_RLOF);
        
        Add_yield(dm_nonconservative_loss,XRLOF_gain, //lost
                  0.0,NULL,
                  NUCSYN_SOURCE_RLOF);
        
        /* excretion disc */
        Add_yield(dm_excretion_loss, XRLOF_and_decretion_loss, // lost
                  0.0, NULL, // accreted
                  NUCSYN_SOURCE_DECRETION);
        
        /* wind loss / accretion */
        Add_yield(dm_wind_loss,Xwind_loss, //lost
                  dm_wind_gain,Xwind_gain, //accreted
                  ID_wind_source(prev_star,stardata));

#ifdef DISCS
        /* discs */
        Add_yield(dm_cbdisc_loss,stardata->common.discs[0].X,
                  dm_cbdisc_gain,stardata->common.discs[0].X,
                  NUCSYN_SOURCE_CBDISC);
#endif
        // to be implemented  : dm_disc_loss, NUCSYN_SOURCE_DISC
        
#ifdef NUCSYN_NOVAE
        /*
         * Novae : handles accretion also by making sure
         * the nova abundances are set by the star that is losing 
         * mass.
         *
         * Note: if dm_novae < 0 then we're dealing with
         * the white dwarf, otherwise the donor
         *
         * Really, we should call set_nova_abunds once!
         */
        const double dm_novae = Dm(DERIVATIVE_STELLAR_MASS_NOVA);
        
        if(Is_not_zero(dm_novae))
        {
            struct star_t * const white_dwarf_accretor = dm_novae < 0.0 ? star : companion;
            nucsyn_set_nova_abunds(stardata,
                                   white_dwarf_accretor,
                                   Xdonor,
                                   Xnovae);
            if(dm_novae < 0.0)
            {
                /*
                 * Nova loss from the white dwarf
                 */
                nucsyn_remove_mass_from_surface(
#if defined DEBUG && DEBUG > 0
                    stardata,
#endif
                    white_dwarf_accretor,
                    -dm_novae,
                    NULL,
                    ndmacc,
                    NULL);

                Add_yield(-dm_novae,Xnovae,
                          0.0,NULL,
                          NUCSYN_SOURCE_NOVAE);
            }
            else
            {
                /*
                 * Nova "reaccretion"
                 */
                nucsyn_add_mass_to_surface(k,
                                           dm_novae,
                                           Xnovae,
                                           ndmacc,
                                           Xacc,
                                           nXacc);
                Add_yield(0.0,NULL,
                          dm_novae,Xnovae,
                          NUCSYN_SOURCE_NOVAE);
            }
        }
#endif // NUCSYN_NOVAE

        /*
         * Free possibly allocated memory
         */
        Free_if_allocated(wind_loss);
        Free_if_allocated(RLOF_gain);
        Free_if_allocated(RLOF_and_decretion_loss);
        Free_if_allocated(donor);
        Free_if_allocated(accretor);
    }

    Evolving_STARLOOP(k)
    {
        SETstars(k);
  
        /************************************************************/
        
        /*
         * Now do interior mixing
         */
#ifdef NUCSYN_STRIP_AND_MIX
        Boolean use_strip_and_mix = can_use_strip_and_mix(stardata,star);
        if(use_strip_and_mix==TRUE)
        {
            /*
             * In pre-CHeB stars, use a new "strip and mix" code
             * which requires that we put accreted material into
             * an accretion layer and deal with it in detail in
             * nucsyn_strip_and_mix
             */
        }
        else
        {
#endif//NUCSYN_STRIP_AND_MIX
            Dprint("post-strip and mix\n");

            if(ndmacc[k]>0.0)
            {
                /*
                 * Accreted material is in the new accretion layer.
                 *
                 * If the accreting star is convective, just mix accreted material
                 * into its envelope.
                 */
                if(CONVECTIVE_ENVELOPE(star->stellar_type))
                {
                    MDEBUG("MIXING accreting star is convective\n");
                    nucsyn_dilute_shell(Max(0.0,Envelope_mass(k) - ndmacc[k]),
                                        nXenv[k],
                                        ndmacc[k],
                                        nXacc[k]);
                    ndmacc[k]=0.0;
                    
                    MDEBUG("MIXING accreted material mixed straight into the envelope (Xenv XC12 %g)\n",
                           nXenv[k][XC12]);
                }
                else 
                {
                    /*******************************/
                    /* Accreting star is radiative */
                    /*******************************/
              
                    /* Now, does the new accretion layer sink or swim? */

                    /*
                     * Calculate the molecular weight of the new accretion layer
                     * and the "new" envelope (which is probably the same as the
                     * old envelope). We assume the material that has accreted 
                     * is fully ionized. 
                     */
                    const double mu_acc = nucsyn_molecular_weight(nXacc[k],
                                                                  stardata,
                                                                  1.0);
                    const double mu_env = nucsyn_molecular_weight(nXenv[k],
                                                                  stardata,
                                                                  1.0);
                
                    MDEBUG("MIXING Seeing if the accretion layer will sink or swim : mu_acc=%g c.f. mu_env=%g, DELTAmu=%g\n",
                           mu_acc,mu_env,mu_acc-mu_env);
                
                    /*
                     * Compare the molecular weight of the new accretion layer
                     * against the new envelope : if the new accretion layer is
                     * *heavier* it will sink and mix entirely with the envelope.
                     * If it is lighter it will just sit on the surface and hang
                     * about like the kids on the street corner with the baseball
                     * caps i.e. very annoying and might be best to burn it! For
                     * the case where mu_acc==mu_env assume the accreting material
                     * floats : if anything it'll be hotter (due to the impact)
                     * than the stellar envelope, so will float due to "normal"
                     * convection.
                     */

                    if(mu_acc - mu_env > -VERY_TINY) 
                        /*
                         * NB use > -VERY_TINY so that if they're very similar we do
                         *  not have an accretion layer (this saves time)
                         */
                    {
                        MDEBUG("Mu_acc=%g mu_env=%g : call thermohaline mix of (acc: dm=%g XC12=%g) with (env: dm=%g XC12=%g)  ",
                               mu_acc,
                               mu_env,
                               ndmacc[k],
                               Xacc[k][XC12],
                               Envelope_mass(k),
                               nXenv[k][XC12]
                            );
                        nucsyn_thermohaline_mix(stardata,
                                                k,
                                                nXacc[k],
                                                ndmacc[k],
                                                Xenv[k],
                                                nXenv[k]);
                                               
                        ndmacc[k] = 0.0;

                        MDEBUG("\nafter mix C12 : acc %g env %g ",
                               Xacc[k][XC12],
                               nXenv[k][XC12]);
                        
                        MDEBUG("MIXING        : Difference : %8e %8e %8e %8e %8e\nMIXING set ndmacc[%d] to %g\n",
                               nXenv[k][XH1]-Xenv[k][XH1],
                               nXenv[k][XHe4]-Xenv[k][XHe4],
                               nXenv[k][XC12]-Xenv[k][XC12],
                               nXenv[k][XN14]-Xenv[k][XN14],
                               nXenv[k][XO16]-Xenv[k][XO16],k,ndmacc[k]);
                        MDEBUG("MIX onto star %d (type %d, mu=%3g onto type %d, mu=%3g) : sink\n",
                               k,
                               companion->stellar_type,
                               mu_acc,
                               star->stellar_type,
                               mu_env);

                    }
                    else
                    {  
                        /*
                         * Accretion layer floats : just leave it as it is
                         */
                        MDEBUG("MIXING Accretion layer floats on the radiative envelope\n MIXING sink or swim? SWIM!\nMIX onto star %d (type %d, mu=%3g onto type %d, mu=%3g) swim\n",
                               k,
                               companion->stellar_type,
                               mu_acc,
                               star->stellar_type,
                               mu_env);
                    }
                }
            }
#ifdef NUCSYN_STRIP_AND_MIX
        }
#endif//NUCSYN_STRIP_AND_MIX
    }
    
    /* 
     * Update abundances for the next timestep
     */
    Evolving_STARLOOP(k)
    {
        MDEBUG("MIXING memcpy new abundances for next timestep + wind losses\n");
        
        MDEBUG("MIXING update accretion layer for star %d\n",k);
        SETstar(k);


        Copy_abundances(nXacc[k],star->Xacc);
        Copy_abundances(nXenv[k],star->Xenv);

        MDEBUG("New Xacc %g Xenv %g\n",nXacc[k][XC12],nXenv[k][XC12]);
        
        star->dmacc = star->stellar_type==MASSLESS_REMNANT ? 0.0 : ndmacc[k];
        
#ifndef NUCSYN_STRIP_AND_MIX
        if(ON_MAIN_SEQUENCE(star->stellar_type))
        {
            /* by memcpying we only set Xinit, but we also 
             * want to set Xenv: for low and intermediate 
             * mass stars this is fine, because the main sequence
             * abundances can only change due to mass accretion
             * ...
             * massive stars will then have their surface abundances 
             * further changed (due to evolutionary effects)
             * in nucsyn_WR
             */
            if(stardata->preferences->no_thermohaline_mixing==FALSE)
            {
                /*
                 * In the case where thermohaline mixing is disabled,
                 * Xenv and Xinit cannot have changed because there 
                 * is no mixing into the envelope
                 */
                Copy_abundances(nXenv[k],star->Xenv);
                Copy_abundances(star->Xenv,star->Xinit);
            }
        }
#endif
        MDEBUG("MIXING star %d now has an accretion layer of size %g\n",k,star->dmacc);
        MDEBUG("XC12 out %d ACC %g ENV %g\n",
               k,
               star->Xacc[XC12],
               star->Xenv[XC12]);
        
    }

    MIXDEBUG_END;
}

#ifdef NUCSYN_ID_SOURCES
static int id_wind_source(const struct star_t * const star,
                          const struct stardata_t * const stardata)
{
    Yield_source source;
    /* 
     * Yield source ID function for wind-type sources
     */
    if(AGB(star->stellar_type))
    {
        const double mcbagb = mcagbf(star->phase_start_mass,
                                     stardata->common.giant_branch_parameters);
        source =
            mcbagb > stardata->preferences->minimum_mass_for_carbon_ignition ?
            NUCSYN_SOURCE_SAGB :
            NUCSYN_SOURCE_AGB;
    }
    else if(star->stellar_type==GIANT_BRANCH)
    {
        source = NUCSYN_SOURCE_GB;
    }
    else if(star->stellar_type < HeWD && WR_mu(star)<1.0)
    {
        source = NUCSYN_SOURCE_WR;
    }
    else
    {
        source = NUCSYN_SOURCE_WIND;
    }
    return source;
}
#endif //NUCSYN_ID_SOURCES



#endif /* NUCSYN */
