#include "nucsyn.h"
#ifdef NUCSYN

/*
 * Function to evolve a star during the Thermally Pulsing Asymptotic Giant
 * Branch stage of evolution and do nucleosynthesis in that star.
 */ 

void nucsyn_tpagb(double * RESTRICT const  mass,
                  double * RESTRICT const  age,
                  double * RESTRICT const  mc,
                  double * RESTRICT const  mt,
                  double * RESTRICT const  lum,
                  double * RESTRICT const  r,
                  double * RESTRICT const  rc,
                  struct star_t * const newstar,
                  struct stardata_t * RESTRICT const  stardata,
                  double * const GB)
{
    const double menv = *mt - *mc; /* Envelope mass */
    double Thbbmax=0.0;
    double tagb=0.0; /* Age along TPAGB in years */
    const double m0 = newstar->mc_1tp + newstar->menv_1tp; /* initial TPAGB mass */
    Boolean logged = FALSE; /* logging boolean : avoid excessive log files */    
#ifdef NUCSYN_TPAGB_HBB
    Boolean hbb = FALSE;
#endif

    /************************************************************/
    Dprint("TPAGB debug for star %d, pulse number %g\n",
           newstar->starnum,
           newstar->num_thermal_pulses);
    /*
     * Calculate time on the TPAGB in years
     */
    tagb = 1e6 * (newstar->age - newstar->time_first_pulse);
    Dprint("tagb = %g from aj=%30.20g start=%30.20g\n", 
           tagb,
           *age,
           TPAGB_start_time(newstar));

    /* Initialise stuff that needs to be done on the first pulse */
    if(First_pulse(newstar) || Is_zero(tagb))
    { 
        Dprint("First pulse at %g (t=%g) kw=%d\n",
               newstar->time_first_pulse,
               stardata->model.time,
               newstar->stellar_type); 
        nucsyn_init_first_pulse(newstar,
                                stardata,
                                TPAGB_MIXING_LENGTH_PARAMETER);
    }
    else
    {
        Dprint("Not first pulse\n");
    }
    Dprint("TIPF %g\n",newstar->interpulse_period);

    
    /* save the time we were last called */
    newstar->prev_tagb = tagb;

    /*
     * Does not work if we're interpolating for RLOF
     * (but you should use Rob's algorithm to do this
     * and hence AVOID such interpolation)
     */

    if(stardata->model.intpol==0)
    {
        Dprint("tagb : tagb=%g from age=%g macro=%g (dt=%g)\n",
               tagb,
               *age,
               newstar->time_first_pulse,
               stardata->model.dt*1e6);

        /*
         * If the envelope mass is zero, just return - we can't do any burning
         * or third dredge-up anyway
         */
        if(menv > TINY)
        {
            Dprint("Every step : tagb=%g m=%g menv=%g mc=%g l=%g r=%g \n",
                   tagb,*mt,menv,*mc,*lum,*r);

#ifdef NUCSYN_TPAGB_HBB
            /**********************/
            /* HOT BOTTOM BURNING */
            /**********************/
            /* Set maximum temperature this star can achieve */
            Thbbmax = nucsyn_set_hbb_conditions(newstar,stardata,m0,menv);
            hbb = HBB_Conditions(newstar->temp,newstar->rho,menv,m0);

            Dprint("Next pulse at %g, NTP=%g, tmax %g T=%g rho=%g risefac %g\n",
                   newstar->time_next_pulse,
                   newstar->num_thermal_pulses,
                   Thbbmax,
                   newstar->temp,
                   newstar->rho,
                   newstar->temp_rise_fac);
      
            if(hbb == TRUE)
            {
                Dprint("Confirm Burn!\n");

                /*
                 * As time goes on, burn the shell stored in Xhbb for the time given 
                 * by the timestep. Each timestep, mix the Xhbb shell with the 
                 * envelope shell abundances and save this in "tempshell" which we 
                 * only use for logging. At the end of each pulse finally mix the HBB 
                 * shell with the envelope.
                 */

                /*
                 * calculate burn time:
                 * This is approximate in the sense that we assume time_prev_pulse
                 * is constant from pulse to pulse (which is a very good approximation
                 * because the core mass never changes very quickly).
                 */
                
                /*
                double burn_time_this_pulse = 1e-6 * newstar->f_burn *
                    (Min(tagb,newstar->time_next_pulse) - newstar->time_prev_pulse);
                */
                
                double burn_time_this_pulse = newstar->interpulse_period *
                    newstar->f_burn * newstar->dntp;
                
                Dprint("Pulse NTP=%g at t=%g burntime %g yrs c.f. tipf=%g yrs * f_burn=%g c.f. dt=%g from tagb=%g time_prev_pulse=%g diff=%g\n",
                       newstar->num_thermal_pulses,
                       newstar->time_next_pulse,
                       burn_time_this_pulse,
                       1e6*newstar->interpulse_period,
                       newstar->f_burn,
                       stardata->model.dt,
                       tagb,
                       newstar->time_prev_pulse,
                       tagb - newstar->time_prev_pulse
                    );

                /*
                 * copy envelope abundance, Xenv, into Xhbb
                 * because we burn Xhbb
                 */
                Copy_abundances(newstar->Xenv,
                                newstar->Xhbb);

                /* 
                 * Do the nuclear burning with mixture in Xhbb.
                 */
                Dprint("HBB for time %g was C12=%g, N14=%g \n",
                       burn_time_this_pulse,
                       newstar->Xhbb[XC12],
                       newstar->Xhbb[XN14]);

                nucsyn_hbb(newstar->temp,
                           newstar->rho,
                           burn_time_this_pulse, // NB this is already in years
                           newstar->Xhbb,
                           newstar,
                           stardata);

                Dprint("Done burn (Env: C12=%g N14=%g)\n",
                       newstar->Xhbb[XC12],
                       newstar->Xhbb[XN14]);

#ifdef NUCSYN_LOG2
                /*
                 * For logging, copy the surface abundances into tempshell, pretend it's the
                 * size of the envelope (minus the small hbb layer) and dilute(/enhance) with
                 * the stuff in the hbb layer. Then output. If you set the timestep to be small
                 * (or dtfac to be small) then this logging will be activated.
                 */
                Abundance * tempshell = New_isotope_array_from(newstar->Xenv);
        
                /* Dilute into tempshell */
                nucsyn_dilute_shell((1.0-newstar->f_hbb)*menv,
                                    tempshell,
                                    newstar->f_hbb*menv,
                                    newstar->Xhbb);
                Log_nucsyn("HBB intermediate:",tagb,tempshell,0.0);       
                logged = TRUE;
                Safe_free(tempshell);
#endif // NUCSYN_LOG2
                Dprint("Done log\n");

            }
#if (DEBUG==1)
            else
            {
                Dprint("not hot/dense enough or m0 is not high enough\n");
            }
#endif
            Dprint("Done HBB\n");

#endif /* NUCSYN_TPAGB_HBB */
        }
        /*
         * check if we've gone through some pulses
         */
        Dprint("Check for pulses : dntp = %g\n",
               newstar->dntp);
        
        if(newstar->dntp > 0.5)
        {
            Dprint("NTP=%g tipf=%g\n",
                   newstar->num_thermal_pulses,
                   newstar->interpulse_period);

#ifdef NUCSYN_TPAGB_HBB            
            if(hbb==TRUE)
            {
                /*
                 * Mix the HBB layer into the envelope since we've had enough
                 * of burning it
                 */
                Dprint("Mix HBB: was H=%g He=%g (shell H=%g He=%g) -> ",
                       newstar->Xenv[XH1],
                       newstar->Xenv[XHe4],
                       newstar->Xhbb[XH1],
                       newstar->Xhbb[XHe4]);
                nucsyn_dilute_shell(1.0 - newstar->f_hbb,
                                    newstar->Xenv,
                                    newstar->f_hbb,
                                    newstar->Xhbb);
                Dprint("now H=%g He=%g\n", 
                       newstar->Xenv[XH1],
                       newstar->Xenv[XHe4]);
            }
#endif //NUCSYN_TPAGB_HBB
            Dprint("< 3DUP mc was %g \n",*mc);

            /* Do third dredge up but only if we've gone through a pulse! */
#ifdef NUCSYN_THIRD_DREDGE_UP

            if(newstar->dntp > TINY)
            {
                Dprint("Pre 3DUP abundances: H=%g He=%g\n",
                       newstar->Xenv[XH1], 
                       newstar->Xenv[XHe4]);

                /*********************/
                /** Third Dredge Up **/
                /*********************/
                nucsyn_third_dredge_up(newstar,
                                       stardata,
                                       tagb,
                                       r,rc,mc,mt,age,lum,mass,
                                       newstar->dntp,
                                       Thbbmax, /* NB not Thbb */
                                       newstar->rho,
                                       newstar->fmdupburn,
                                       newstar->menv_1tp,
                                       0.0
                    );
            }
#endif
            Dprint("Post 3DUP abundances: H=%g He=%g\n",
                   newstar->Xenv[XH1],
                   newstar->Xenv[XHe4]);
        }

#ifdef LOG_FINAL_TPAGB_ABUNDS
        Copy_abundances(newstar->Xenv, newstar->final_tpagbX);
#endif //LOG_FINAL_TPAGB_ABUNDS

        Dprint("prev %g\n",newstar->prev_tagb);
        if(logged==FALSE)
        {
#ifdef NUCSYN_LOG
            Log_nucsyn("Exit\n",tagb,newstar->Xenv,0.0);
#endif//NUCSYN_LOG
        }

        Dprint("end of tpagb function loop n=%g, ntp=%g\n\n",
               newstar->Xenv[Xn],newstar->num_thermal_pulses);


#ifdef NUCSYN_Globular_Cluster_LOG
        if(stardata->star[0].stellar_type==TPAGB)
        {
            printf("AGBNA %g %g %g %g %g %g %g %g %g %g %g %g %g %g %g %g %g %g %g %g\n",
                   tagb,
                   newstar->temp,
                   newstar->rho,
                   stardata->star[0].Xenv[XNa23],
                   stardata->star[0].Xenv[XC12],
                   stardata->star[0].Xenv[XN14],
                   stardata->star[0].Xenv[XO16],
                   stardata->star[0].Xenv[XMg24],
                   stardata->star[0].Xenv[XMg25],
                   stardata->star[0].Xenv[XMg26],
                   stardata->star[0].Xenv[XFe56],
                   newstar->luminosity,
                   *r,
                   newstar->mass,
                   newstar->core_mass,
                   pow(10,-2.07 - 0.90*log10(newstar->mass) + 1.940*log10(newstar->radius)),
                   stardata->star[0].Xenv[XC12]/12.0+
                   stardata->star[0].Xenv[XN14]/14.0+
                   stardata->star[0].Xenv[XO16]/16.0,
                   nucsyn_square_bracket(stardata->star[0].Xenv,
                                         stardata->common.Xsolar,
                                         XC12,
                                         XFe56),
                   nucsyn_square_bracket(stardata->star[0].Xenv,
                                         stardata->common.Xsolar,
                                         XN14,
                                         XFe56),
                   nucsyn_square_bracket(stardata->star[0].Xenv,
                                         stardata->common.Xsolar,
                                         XO16,
                                         XFe56) 

                );
        }
#endif

        Dprint("ABUNDS %g %g %g %g %g\n",
               tagb,
               newstar->Xenv[XH1],
               newstar->Xenv[XHe4],
               newstar->Xenv[XC12],
               newstar->Xenv[XN14]);

        Dprint("endTPAGB Mc=%g Menv=%g L=%g R=%g\n",*mc,*mt-*mc,*lum,*r);
    }
}

#endif /* NUCSYN */
