#include "../binary_c.h"

#if defined NUCSYN && defined NUCSYN_SUPERNOVAE
#include "nucsyn_sn_yield.h"
//#undef DEBUG
//#define DEBUG 1
/*
 * Function to calculate and add the yield due to a supernova
 *
 * Input parameters: 
 *
 * exploder contains the stellar structure *after* the explosion
 *
 * pre_explosion_star contains the stellar structure *before* the explosion
 *
 * dm is the mass ejected
 *
 * menv and mcore are the envelope and core mass or, if zero,
 * these are calculated automatically from the pre_explosion_star 
 */

void nucsyn_sn_yield(struct stardata_t *stardata,
                     struct star_t * exploder,
                     struct star_t * pre_explosion_star,
                     double dm, /* mass ejected */
                     double menv, /* envelope mass of exploder */
                     double mcore, /* core mass of exploder */
                     Stellar_type pre_explosion_companion_stellar_type
    )
{
    struct star_t * companion = Other_star_struct(exploder);
    if(Is_zero(menv))
    {
        menv = pre_explosion_star->mass - pre_explosion_star->core_mass;
    }
    if(Is_zero(mcore))
    {
        mcore = pre_explosion_star->core_mass;
    }

    /*
     * array to store the yields before calling calc_yields
     */
    Abundance * X = New_clear_isotope_array;

    /*
     * the metallicity for the explosion (might be <1e-4)
     */
    Abundance metallicity = stardata->common.nucsyn_metallicity>-TINY ? 
        stardata->common.nucsyn_metallicity : stardata->common.metallicity;
  
    Dprint("SN yield star %d, exploder->SN_type = %d, mass ejected dm = %g (menv = %g, mcore = %g), pre_explosion stellar type %d, nucsyn metallicity %g\n",
           exploder->starnum,
           exploder->SN_type,
           dm,
           menv,
           mcore,
           pre_explosion_star->stellar_type,
           metallicity);

    
#ifdef NUCSYN_GCE_OUTFLOW_CHECKS
    // set velocity for the SN wind
    exploder->vwind = NUCSYN_GCE_SUPERNOVA_OUTFLOW_VELOCITY;
#endif // NUCSYN_GCE_OUTFLOW_CHECKS
    exploder->sn_last_time = exploder->SN_type;

    /* 
     * Save the stellar remnant's abundances
     */
    Abundance * Xremnant = New_isotope_array_from(exploder->Xenv);

    /*
     * Use saved surface abundance prior to remnant formation
     */
    Copy_abundances(pre_explosion_star->Xenv,
                    exploder->Xenv);

#ifdef NUCSYN_STRIP_AND_MIX
    /* mix down to the core */
    mix_down_to_core(exploder);
#endif // NUCSYN_STRIP_AND_MIX
#ifdef SN_TEST
    // call the test function
    nucsyn_sn_test(stardata);
#endif // SN_TEST

    /*
     * Select yields according to explosion type
     */

    Yield_source src_id = -1;
    
    switch (exploder->SN_type)
    {    
    case SN_IBC:
    case SN_II:
    case SN_GRB_COLLAPSAR:

        Dprint("CORE_COLLAPSE_SNIBC or II mc=%g dm=%g\n",exploder->core_mass,dm);
        /*
         * SNIbc or SNII
         * NB SN_GRB_COLLAPSAR is really a IBC
         */
        dm = nucsyn_sn_core_collapse(X,
                                     pre_explosion_star->Xenv,
                                     pre_explosion_star->core_mass,
                                     menv,
                                     dm,
                                     metallicity,
                                     &src_id,
                                     stardata, 
                                     pre_explosion_star,
                                     exploder
            );
        break;
    case SN_IA_He:

        Dprint("SNIa of HeWD. Companion is stellar type %d\n",
               companion->stellar_type);

        if(pre_explosion_star->stellar_type == HeWD)
        {
            nucsyn_sn_woosley_taam_weaver_1986(X);
            src_id = NUCSYN_SOURCE_SNIa_He;
        }
        else
        {
            Dprint("The other star is exploding?");
            Other_exploder(1);
        }
        break;

    case SN_IA_He_Coal:

        Dprint("HeWD explosion by merger, companion's stellar type is %d",
               companion->stellar_type);

        if(pre_explosion_star->stellar_type == HeWD)
        {
            if(companion->stellar_type == HeWD)
            {
                nucsyn_sn_woosley_taam_weaver_1986(X);
                src_id = NUCSYN_SOURCE_SNIa_He;
            }
            else
            {
                Exit_binary_c(
                    BINARY_C_ALGORITHM_BRANCH_FAILURE,
                    "We have a HeWD-HeWD merger (SNIa?) but the other star is not a HeWD?!\n"
                    );
            }
        }
        else
        {
            Other_exploder(2);
        }
        break;

    case SN_IA_ELD:
        /*
         * CO and ONeWDs accrete helium-rich material until the accumulated
         * material exceeds a mass of 0.15 when it ignites. For a COWD with
         * mass less than 0.95 the system will be destroyed in a possible
         * Type 1a SN. COWDs with mass greater than 0.95 and ONeWDs will 
         * survive with all the material converted to ONe.
         *
         * V1.10 of BSE
         * Now changed to an ELD for all COWDs when 0.15 
         * accreted (JH 11/01/00).  
         */
        Dprint("ELD : CO core size %g : envelope size %g, pre-explosion stellar type %d\n",
               mcore,
               menv,
               pre_explosion_star->stellar_type);

        if(pre_explosion_star->stellar_type == COWD)
        {
            if(companion->stellar_type == HeWD ||
               NAKED_HELIUM_STAR(companion->stellar_type))
            {
                /*
                 * Fits to Livne and Arnett 1995 
                 * (as function of core and env mass)
                 */
                nucsyn_sn_livne_arnett_1995(X,
                                            mcore,
                                            menv);
            }
            else
            {
                /*
                 * H-rich : currently use Livne
                 */
                nucsyn_sn_livne_arnett_1995(X,
                                            mcore,
                                            menv);
            }
            dm = mcore + menv;
            src_id = NUCSYN_SOURCE_SNIa_ELD;
        }
        else
        {
            Other_exploder(3);
        }
        break;

    case SN_IA_CHAND:
    
        Dprint("Chandrasekhar mass SN Ia by accretion: accretor type %d mass %g, donor type %d mass %g\n",
               pre_explosion_star->stellar_type,
               pre_explosion_star->mass,
               companion->stellar_type,
               companion->mass
            );
        /*
         * NB allow HeWDs to explode this way also. 
         * Presumably they would convert to COWDs 
         * the process.
         */

        if(pre_explosion_star->stellar_type == COWD ||
           pre_explosion_star->stellar_type == HeWD)
        {
            if(stardata->preferences->type_Ia_MCh_supernova_algorithm ==
               TYPE_IA_MCH_SUPERNOVA_ALGORITHM_DD2)
            {
                nucsyn_sn_iwamoto_1999_DD2(X);
            }
            else if(stardata->preferences->type_Ia_MCh_supernova_algorithm ==
                    TYPE_IA_MCH_SUPERNOVA_ALGORITHM_SEITENZAHL2013)
            {
                nucsyn_sn_Seitenzahl2013(stardata,X);
            }
            else if(stardata->preferences->type_Ia_MCh_supernova_algorithm ==
                    TYPE_IA_MCH_SUPERNOVA_ALGORITHM_SEITENZAHL2013_AUTOMATIC)
            {
                nucsyn_sn_Seitenzahl2013_automatic(stardata,X);
            }
            src_id = NUCSYN_SOURCE_SNIa_CHAND;
        }
        else
        {
            show_cmd_line_args(stardata,stderr);
            Other_exploder(4);
        }
        break;

    case SN_IA_CHAND_Coal:
    
        // NB this function cannot be called by ONeWD+ONeWD merger case
        Dprint("M > Chandrasekhar mass by merger, stellar types are pre-explosion accretor %d, donor %d",
               pre_explosion_star->stellar_type,
               pre_explosion_companion_stellar_type);
        
        if(pre_explosion_star->stellar_type == HeWD &&
           pre_explosion_companion_stellar_type == HeWD)
        { 
            nucsyn_sn_woosley_taam_weaver_1986(X);
        }
        else if(pre_explosion_star->stellar_type == COWD &&
                pre_explosion_companion_stellar_type == COWD)
        {
            if(stardata->preferences->type_Ia_MCh_supernova_algorithm ==
               TYPE_IA_MCH_SUPERNOVA_ALGORITHM_DD2)
            {
                /*
                 * Use Iwamoto results for single COWD (M=MCh)
                 * for dm ~= MCh of the material
                 */
                nucsyn_sn_iwamoto_1999_DD2(X);
            }
            else if(stardata->preferences->type_Ia_MCh_supernova_algorithm ==
                    TYPE_IA_MCH_SUPERNOVA_ALGORITHM_SEITENZAHL2013)
            {
                nucsyn_sn_Seitenzahl2013(stardata,X);
            }
            else if(stardata->preferences->type_Ia_MCh_supernova_algorithm ==
                    TYPE_IA_MCH_SUPERNOVA_ALGORITHM_SEITENZAHL2013_AUTOMATIC)
            {
                nucsyn_sn_Seitenzahl2013_automatic(stardata,X);
            }
        }
        else
        {
            Other_exploder(5);
        }
        src_id = NUCSYN_SOURCE_SNIa_CHAND_Coal;
        
        break;
   
    case SN_IIa:
        /* 
         * SNIIa : the core explodes, no remnant is left.
         */
    {
        double dmCO=Max(0.0,dm - menv);
        
        Dprint("SNIIa menv=%g mco=%g core_mass=%g -> dmCO = %g\n",
               menv,dmCO,stardata->star[1].core_mass,dmCO);
        
        Abundance * XSN = New_clear_isotope_array;
        Abundance * Xenv = New_isotope_array_from(exploder->Xenv);
        if(dmCO > TINY)
        {
            if(stardata->preferences->type_Ia_MCh_supernova_algorithm ==
               TYPE_IA_MCH_SUPERNOVA_ALGORITHM_DD2)
            {
                nucsyn_sn_iwamoto_1999_DD2(X);
            }
            else if(stardata->preferences->type_Ia_MCh_supernova_algorithm ==
                    TYPE_IA_MCH_SUPERNOVA_ALGORITHM_SEITENZAHL2013)
            {
                nucsyn_sn_Seitenzahl2013(stardata,X);
            }
            else if(stardata->preferences->type_Ia_MCh_supernova_algorithm ==
                    TYPE_IA_MCH_SUPERNOVA_ALGORITHM_SEITENZAHL2013_AUTOMATIC)
            {
                nucsyn_sn_Seitenzahl2013_automatic(stardata,X);
            }      
        }   
        src_id = NUCSYN_SOURCE_SNIIa;
        nucsyn_mix_shells(dmCO,XSN, 
                          menv,exploder->Xenv);
        Copy_abundances(XSN,X);
        Safe_free(XSN);
        Safe_free(Xenv);
    }
    case SN_ECAP:
        /* 
         * Electron Capture supernova of an ONeMg core,
         * probably in a TPAGB star, maybe a CO core too (although
         * these are probably thermonuclear)
         */
    {
        src_id = NUCSYN_SOURCE_SNeCAP;

        double mcbagb=mcagbf(exploder->phase_start_mass,
                             stardata->common.giant_branch_parameters);
        
        Boolean ONe_core;

        /* determine if the core was CO or ONe */
        ONe_core = Boolean_(mcbagb > stardata->preferences->minimum_mass_for_carbon_ignition);

#ifdef NUCSYN_SN_ELECTRON_CAPTURE_WANAJO_2008
        if(ONe_core == TRUE)
        {
            /*
             * Assume burnt material comes from the envelope:
             * required to conserve mass!
             */
            menv -= ELECTRON_CAPTURE_EJECTED_MASS;
            menv = Max(0.0,menv);
        }
#endif// NUCSYN_SN_ELECTRON_CAPTURE_WANAJO_2008

        Dprint("Electron capture supernova with core mass %g made of %s, eject menv = %g\n",
               mcbagb,
               (ONe_core ? "ONe" : "CO"),
               menv
            );

        /* eject the envelope */ 
        Abundance * Xenv = New_isotope_array_from(exploder->Xenv);
        Abundance * Xcore = New_isotope_array;

        double dmCO = Max(0.0, dm - menv);
        double dmcore = 0.0;

        Dprint("Electron capture SN: dmCO=%g mcbagb=%g mc=%g\n",dmCO,
               mcbagb,stardata->preferences->minimum_mass_for_carbon_ignition);

#ifdef NUCSYN_SN_ELECTRON_CAPTURE_WANAJO_2008
        /* In the case of ONe cores, eject a burnt layer, leave a NS.
         * In the case of CO cores, eject the entire core.
         */
        if(ONe_core == TRUE)
        {
            Dprint("Electron capture SN : ONe burnt layer ejected, NS remains\n");
            nucsyn_sn_electron_capture(Xcore);
            dmcore = ELECTRON_CAPTURE_EJECTED_MASS;
        }
        else
        {
            /* should be like a Ia? */
            Clear_isotope_array(Xcore);
            Xcore[XC12]=0.2; 
            Xcore[XO16]=0.8;
            Dprint("Electron capture SN : CO core ejected\n");
            dmcore = dmCO;
        }
#else
        if(dmCO > 0.0)
        {
            /* is the core CO or ONe ? */
            Clear_isotope_array(X);
            if(ONe_core == TRUE)
            {
                Dprint("Electron capture SN : eject %g of ONe core\n",dmCO);
                Xcore[XO16]=0.95; 
                Xcore[XNe22]=0.02; 
                Xcore[XMg24]=0.03; 
            }
            else
            {
                Dprint("Electron capture SN : eject %g of CO core\n",dmCO);

                /* should be like a Ia? */
                Xcore[XC12]=0.2; 
                Xcore[XO16]=0.8;
            }
            dmcore = dmCO;
        }
#endif
        nucsyn_mix_shells(dmcore,Xcore,
                          menv,Xenv);
        Copy_abundances(Xcore,X);
        Safe_free(Xcore);
        Safe_free(Xenv);
    }
    break;
    case SN_AIC:
      
        /* 
         * Accretion Induced Collapse - assume direct 
         * change from WD to NS i.e. zero yield although there are suspicions
         * that there might be some r-process yield... ???? 
         *
         * what should we eject? try the surface abundance
         * for lack of anything else...
         */
        Dprint("AIC supernova eject dm = %g \n",dm);
        src_id = NUCSYN_SOURCE_SNAIC;
        Copy_abundances(exploder->Xenv,X);
        break;
    default:

        /*
         * Probably a NS/BH merger or Thorne-Zytkow
         * object. These either have no yields, or
         * the yields are dealt with elsewhere.
         */
        Dprint("SN of type %d : yields should be dealt with elsewhere\n",
               exploder->SN_type);
        break;
    }

    nancheck("SN X",X,ISOTOPE_ARRAY_SIZE);
    nancheck("companion Xenv",companion->Xenv,ISOTOPE_ARRAY_SIZE);

    if(src_id != -1)
    {
        nucsyn_calc_yields(stardata,
                           exploder,
                           dm,X,
                           0.0,NULL,
                           pre_explosion_star->starnum,
                           YIELD_NOT_FINAL,
                           src_id);

        if(companion->dm_companion_SN < TINY)
        {
            /*
             * mass ejected from companion
             */
            Abundance * Xcompanion;
            Boolean allocated;
            nucsyn_remove_dm_from_surface(
#if defined DEBUG && DEBUG > 0
                stardata,
#endif
                companion,
                companion->dm_companion_SN,
                &Xcompanion,
                &allocated);
            nucsyn_calc_yields(stardata,
                               exploder,
                               companion->dm_companion_SN,companion->Xenv,
                               0.0,NULL,
                               pre_explosion_star->starnum,
                               YIELD_NOT_FINAL,
                               NUCSYN_SOURCE_SNSTRIP);
            if(allocated == TRUE)
            {
                Safe_free(Xcompanion);
            }
        }
        else
        {
            /*
             * Mass accreted into companion.
             *
             * Assume
             * 
             * 1) mixing of any pre-existing accretion layer
             *
             * 2) that the impacting ejecta are a mix of the 
             *    ejecta from the exploding star. Probably the 
             *    slowest ejecta are likely to accrete most, but
             *    then again the first to impact will be the
             *    outer envelope. What to do...???
             */
            nucsyn_mix_accretion_layer_and_envelope(stardata,companion,-1.0);
            nucsyn_calc_yields(stardata,
                               companion,
                               0.0,NULL,
                               companion->dm_companion_SN,X,
                               companion->starnum,
                               YIELD_NOT_FINAL,
                               NUCSYN_SOURCE_OTHER);
            nucsyn_dilute_shell(companion->dm_companion_SN,X,
                                companion->mass,companion->Xenv);
        }
        companion->dm_companion_SN = 0.0;
    }

    /* restore remnant's surface abundances */
    Copy_abundances(Xremnant,exploder->Xenv);

    Safe_free(Xremnant);
    Safe_free(X);
}


/*
 * test functions : not maintained actively,
 * use at your own peril!
 */

#ifdef SN_TEST
void nucsyn_sn_test(struct stardata_t *stardata)
{
    /* this is a test function to enable you to obtain yields for various masses at the current metallicity */

#define SN_TEST_ISOTOPE_NUMBER 14

    // the isotopes array contains the isotopes you wish to output
    int isotopes[SN_TEST_ISOTOPE_NUMBER]={XH1,XHe4,XC12,XO16,XNe22,XMg24,XSi28,XCa44,XTi48,XCr52,XMn55,XFe56,XCo59,XV51};
    int i,j; // counters
    int isotope;
    double m;
    double *X; // used to store results!
    const char *c[]=NUCSYN_SHORT_ELEMENT_STRINGS;

    // header
    printf("\nIsotope : \n");
    for(i=0;i<SN_TEST_ISOTOPE_NUMBER;i++)
    {
        isotope=isotopes[i];
        //printf("%12s ",c[stardata->store->atomic_number[isotope]]);
        printf("i=%d isotope=%d z=%d mass=%g name %s%d\n",i,isotope,stardata->store->atomic_number[isotope],
               stardata->store->mnuc[isotope]/AMU_GRAMS,
               c[stardata->store->atomic_number[isotope]],
               (int)(0.5+stardata->store->mnuc[isotope]/AMU_GRAMS));
    }
    printf ("\n");

    // now data
  
    // first, type 1a SNe

#define CLEARX for(j=0;ISOTOPE_ARRAY_SIZE;j++){X[j]=0.0;}
    X=New_clear_isotope_array;

    printf("SN1a ELD and exploding HeWD models\n");
         
    for(m=0.4;m<=stardata->preferences->chandrasekhar_mass;m+=0.1)
    {
        printf("\nProgenitor mass %g\n\n",m);

        for(i=0;i<SN_TEST_ISOTOPE_NUMBER;i++)
        {
            isotope=isotopes[i];
            printf("%8.12s%2.d ",c[stardata->store->atomic_number[isotope]],
                   (int)(0.5+stardata->store->mnuc[isotope]/AMU_GRAMS)
                );

            // woosley taam weaver 1986
            CLEARX;
            nucsyn_sn_woosley_taam_weaver_1986(X);
            printf("%12.3e",X[isotope]);

            // Livne+Arnett 1995 with envelope of 0.1 Msun of He
            CLEARX;
            nucsyn_sn_livne_arnett_1995(X,m,0.1);
            printf("%12.3e",X[isotope]);
          
            // Livne+Arnett 1995 with envelope of 0.2 Msun of He
            CLEARX;
            nucsyn_sn_livne_arnett_1995(X,m,0.2);
            printf("%12.3e",X[isotope]);

            // no more models!
            printf("\n");       
        }
        printf("\n");
    }

    printf("Chandrasekhar mass models\n");

    m=stardata->preferences->chandrasekhar_mass;

    printf("\nProgenitor mass %g\n\n",m);

    for(i=0;i<SN_TEST_ISOTOPE_NUMBER;i++)
    {
        isotope=isotopes[i];
        printf("%8.12s%2.d ",c[stardata->store->atomic_number[isotope]],
               (int)(0.5+stardata->store->mnuc[isotope]/AMU_GRAMS)
            );

        // iwamoto 1999
        CLEARX;
        nucsyn_sn_iwamoto_1999_DD2(X);
        printf("%12.3e %12.3e",X[isotope],X[XV51]/X[isotope]);

        // no more models!
        printf("\n");   
    }
    printf("\n");
  
    printf("\nTypical type II (1.4Msun NS remnant)\n");
    
    double * XH = New_isotope_array_from(stardata->preferences->Xsolar);
    double * XII = New_isotope_array;
    nucsyn_sn_core_collapse(XII,
                            stardata->preferences->Xsolar,
                            4.0,
                            12.0,
                            1.4,
                            0.02,
                            EAGB,
                            stardata,
                            i
            );

    printf("\nTypical type Ibc 2.0Msun NS remnant)\n");

    double * XIbc = New_isotope_array;
    double * XHe = New_isotope_array_from(stardata->preferences->Xsolar);
    XHe[XHe4]+=XHe[XH1];
    XHe[XH1] = 0.0;
    
    nucsyn_sn_core_collapse(XIbc,
                            XHe,
                            8.0,
                            6.0,
                            2.0,
                            0.02,
                            HeGB,
                            stardata,
                            i
            );


    for(i=0;i<SN_TEST_ISOTOPE_NUMBER;i++)
    {
        isotope=isotopes[i];
        printf("%8.12s%2.d ",c[stardata->store->atomic_number[isotope]],
               (int)(0.5+stardata->store->mnuc[isotope]/AMU_GRAMS)
            );

        printf("%12.3e %12.3e",XII[i],XIbc[i]);

        // no more models!
    }

    Safe_free(XHe);
    Safe_free(XIbc);
    Safe_free(X);
    Safe_free(XII);
    Safe_free(XH);
    Exit_binary_c(BINARY_C_NORMAL_EXIT,"");
}
#endif// SN_TEST

#endif// NUCSYN_SUPERNOVAE && NUCSYN
