#include "nucsyn.h"
#ifdef NUCSYN_SUPERNOVAE
#ifdef NUCSYN_CCSNE_WOOSLEY_WEAVER_1995


void nucsyn_sn_woosley_weaver_1995(struct stardata_t * const stardata,
                                   Abundance * const X,
                                   double mco,
                                   Abundance z)
{
    /*
     * Woosley & Weaver 1995 SN yields, corrected by Portinari's method
     */

#ifdef NUCSYN_SUPERNOVAE_USE_TABULAR_WW95_MODELS
    // use table interpolation to calculate yields
#ifdef NUCSYN_SUPERNOVAE_CC_WW95_A
#include "WW95_Amodels.h"
#ifdef SN_DEBUG
    printf("Use WW95 table A\n");
#endif
    mco=Max(1.35,Min(mco,16.37));
#endif
#ifdef NUCSYN_SUPERNOVAE_CC_WW95_B
#include "WW95_Bmodels.h"
#ifdef SN_DEBUG
    printf("Use WW95 table B\n");
#endif
    mco=Max(1.68,Min(mco,15.4082));
#endif
#ifdef NUCSYN_SUPERNOVAE_CC_WW95_C
#include "WW95_Cmodels.h"      
#ifdef SN_DEBUG
    printf("Use WW95 table C\n");
#endif
    mco=Max(1.68,Min(mco,15.4082));
#endif
    Const_data_table table[]={WW95_TABLE};
    // table is of length WW95_TABLE_LINES, with 2 free parameters (Z,MCO)
    z=Max(0.0,Min(0.02,z));
      
#ifdef SN_DEBUG
    printf("Table interpolation : Z=%g MCO=%g\n",z,mco);
#endif


    double x[2];
    x[0]=z; x[1]=mco;
    double y[147]; // 147 isotopes from WW95! (+MFe) we have to set them manually :(
    // NB subtract 1 to get the number to put into interpolate
#ifdef SN_DEBUG
    printf("Interpolate...\n");fflush(stdout);
#endif
    interpolate(table,2,146,WW95_TABLE_LINES,x,y,TRUE);
#ifdef SN_DEBUG
    printf("Interpolate complete\n");fflush(stdout);
#endif
    X[XH1]+=y[1]; X[XH2]+=y[2]; X[XHe3]+=y[3]; X[XHe4]+=y[4]; X[XBe7]+=y[5]; X[XLi7]+=y[6]; X[XBe9]+=y[7]; X[XB10]+=y[8]; X[XB11]+=y[9]; X[XC11]+=y[10]; X[XC12]+=y[11]; X[XC13]+=y[12]; X[XN13]+=y[13]; X[XC14]+=y[14]; X[XN14]+=y[15]; X[XN15]+=y[16]; X[XO16]+=y[17]; X[XO17]+=y[18]; X[XO18]+=y[19]; X[XF19]+=y[20]; X[XNe20]+=y[21]; X[XNe21]+=y[22]; X[XNa22]+=y[23]; X[XNe22]+=y[24]; X[XNa23]+=y[25]; X[XMg24]+=y[26]; X[XNa24]+=y[27]; X[XMg25]+=y[28]; X[XAl26]+=y[29]; X[XMg26]+=y[30]; X[XAl27]+=y[31]; X[XAl28]+=y[32]; X[XSi28]+=y[33]; X[XSi29]+=y[34]; X[XSi30]+=y[35]; X[XP31]+=y[36]; X[XS32]+=y[37]; X[XS33]+=y[38]; X[XS34]+=y[39]; X[XCl35]+=y[40]; X[XS35]+=y[41]; X[XAr36]+=y[42]; X[XCl36]+=y[43]; X[XS36]+=y[44]; X[XAr37]+=y[45]; X[XCl37]+=y[46]; X[XAr38]+=y[47]; X[XK39]+=y[48]; X[XAr40]+=y[49]; X[XCa40]+=y[50]; X[XK40]+=y[51]; X[XCa41]+=y[52]; X[XK41]+=y[53]; X[XCa42]+=y[54]; X[XCa43]+=y[55]; X[XSc43]+=y[56]; X[XCa44]+=y[57]; X[XTi44]+=y[58]; X[XCa45]+=y[59]; X[XSc45]+=y[60]; X[XTi45]+=y[61]; X[XCa46]+=y[62]; X[XTi46]+=y[63]; X[XCa47]+=y[64]; X[XTi47]+=y[65]; X[XV47]+=y[66]; X[XCa48]+=y[67]; X[XCr48]+=y[68]; X[XTi48]+=y[69]; X[XV48]+=y[70]; X[XCr49]+=y[71]; X[XTi49]+=y[72]; X[XV49]+=y[73]; X[XCr50]+=y[74]; X[XTi50]+=y[75]; X[XV50]+=y[76]; X[XCr51]+=y[77]; X[XMn51]+=y[78]; X[XV51]+=y[79]; X[XCr52]+=y[80]; X[XFe52]+=y[81]; X[XMn52]+=y[82]; X[XCr53]+=y[83]; X[XFe53]+=y[84]; X[XMn53]+=y[85]; X[XCr54]+=y[86]; X[XFe54]+=y[87]; X[XMn54]+=y[88]; X[XCo55]+=y[89]; X[XFe55]+=y[90]; X[XMn55]+=y[91]; X[XCo56]+=y[92]; X[XFe56]+=y[93]; X[XNi56]+=y[94]; X[XCo57]+=y[95]; X[XFe57]+=y[96]; X[XNi57]+=y[97]; X[XCo58]+=y[98]; X[XFe58]+=y[99]; X[XNi58]+=y[100]; X[XCo59]+=y[101]; X[XCu59]+=y[102]; X[XFe59]+=y[103]; X[XNi59]+=y[104]; X[XCo60]+=y[105]; X[XCu60]+=y[106]; X[XFe60]+=y[107]; X[XNi60]+=y[108]; X[XZn60]+=y[109]; X[XCo61]+=y[110]; X[XCu61]+=y[111]; X[XNi61]+=y[112]; X[XZn61]+=y[113]; X[XCu62]+=y[114]; X[XNi62]+=y[115]; X[XZn62]+=y[116]; X[XCu63]+=y[117]; X[XNi63]+=y[118]; X[XZn63]+=y[119]; X[XCu64]+=y[120]; X[XGa64]+=y[121]; X[XGe64]+=y[122]; X[XNi64]+=y[123]; X[XZn64]+=y[124]; X[XCu65]+=y[125]; X[XGa65]+=y[126]; X[XGe65]+=y[127]; X[XNi65]+=y[128]; X[XZn65]+=y[129]; X[XCu66]+=y[130]; X[XGa66]+=y[131]; X[XGe66]+=y[132]; X[XZn66]+=y[133]; X[XGa67]+=y[134]; X[XGe67]+=y[135]; X[XZn67]+=y[136]; X[XGa68]+=y[137]; X[XGe68]+=y[138]; X[XZn68]+=y[139]; X[XGa69]+=y[140]; X[XGe69]+=y[141]; X[XZn69]+=y[142]; X[XGa70]+=y[143]; X[XGe70]+=y[144]; X[XGe71]+=y[145];

#ifdef SN_DEBUG
    printf("Masses ejected: H1=%g C12=%g N14=%g O16=%g Fe56=%g \n",
           X[XH1],X[XC12],X[XN14],X[XO16],X[XFe56]);
    printf("Total mass in yields : %g, WW95 MFe=%g hence m_ej from CO core is %g in WW95, c.f. BSE %g\n",XXsum(X),y[0],XXsum(stardata,X)-y[0],dm_core);
#endif
      
#else
    // use fits instead of a tabular interpolation
    //printf("Use WW95 fits to table A\n");
    /* CORE */
    // Use fits to Woosley and Weaver 1995 corrected as in Portinari et al. 1998
    // for removal of the envelope. The fits are f(M_CO,Z)

    // these fits are accurate over the range 1.9 < M_CO/Msun < 16 - outside that range
    // use the limiting values (you're unlikely to get outside that range!)
    mco=Max(mco,1.9);
    mco=Min(mco,16.0);

    // remember that some of them might go negative so include a Max(0, ... 0);

    /* z=metallicity, mco=CO core mass */
    /* m is the amount of the CO core ejected */

    double mc2,mc3,mc4,mc5,mc6,mc13;
    double lz,ilz;

    /* set the X array with the appropriate mass fractions, normalize then multiply by dm_core - the amount ejected from the CO core */
    // copy and paste the C funcs bits to here

    X[XK40] += Max(0.0,(4.00470e-08+3.05790e-05*z) + (2.16070e-07+3.79180e-04*z)*exp(-Pow2(mco-5.36190e+00))+-1.86260e-09*mco);

    mc2=Pow2(mco);
    mc3=Pow3(mco);
    mc4=pow(mco,4.000000e+00);
    mc5=pow(mco,5.000000e+00);
    mc6=mco-6;
    mc13=mco-13;
    lz=log10(z);
    ilz=1.0/lz;

    /* Ni65 */
    X[XNi65] += Max(0.0, (-3.93020e-07+(z)*((5.52090e-03-3.57270e-03*mco+9.37860e-04*mc2-1.16420e-04*mc3+6.71390e-06*mc4-1.44350e-07*mc5))));
    /* Ar37 */
    X[XAr37] += Max(0.0, (3.30390e-01+((1.0+2.49370e-04*z))*(-3.30370e-01+4.68620e-05*exp(-Pow2(5.62320e-01*(mc6)))-2.11830e-05*exp(-Pow2(-7.29280e-02*(mc13))))));
    /* Ga68 */
    X[XGa68] += Max(0.0, 1e-8*(-1.58730e-03+8.43230e+00*pow(z,7.90010e-01))*(-1.52880e+00+4.03540e+00*mco+-2.28470e-01*mc2)   );

    /* Fe52 */
    X[XFe52] += Max(0.0, (6.54810e-04+((1.0-2.61130e+01*z))*((3.85000e-03-2.46420e-03*mco+7.97710e-04*mc2-1.27330e-04*mc3+8.97420e-06*mc4-2.25380e-07*mc5))));
    /* Mn51 */
    X[XMn51] += Max(0.0, (1.89990e-06+((ilz))*((2.95970e-06-2.50300e-05*mco+6.70460e-06*mc2-5.74630e-07*mc3+1.58320e-08*mc4))));
    /* Ga66 */
    X[XGa66] += Max(0.0,1.46290e-04*pow(z,1.42100e-01)*pow(mco,-5.18750e+00)+2.18300e-06*exp(-Pow2(mco-5.89650e+00)/5.78970e-01)+4.30480e-04*z*exp(-Pow2(mco-1.33480e+01)/6.82310e-01));

    /* N13 */

    // interpolate between result for Z=0.0002 and zero for 0.002<Z<0.0002
    X[XN13] += Max(0.0, Min(1.0,Max(0.0,(z-0.0002)/1.8e-3)) * (-3.98610e-07+8.95840e-06/mco));

    /* Zn60 */
    X[XZn60] += Max(0.0, (1.52250e-04+(lz)*(6.60080e-04-3.05850e-05*exp(-Pow2(-4.18700e+00*(mc6)))-5.11800e-04*exp(-Pow2(9.70360e-02*(mc13)))-1.60100e-03/mco)));
    /* Si30 */
    X[XSi30] += Max(0.0, (1.81350e-04+(z)*((2.13050e-01-4.47200e-02*mco+4.04170e-03*mc2+3.75840e-04*mc3-3.02110e-05*mc4))));
    /* V49 */
    X[XV49] += Max(0.0, 
                   4.67990e-06*pow(mco,-1.02290e+00)+(2.72880e-07*-lz)*sin(8.74750e-01*mco+2.49440e+00)   );

    /* Zn64 */
    X[XZn64] += Max(0.0, 2.57320e-06+9.83410e-05*pow(z,-3.82120e-01)*pow(mco,-3.68530e+00) +7.20310e-04*z   );

    /* Ca44 */
    X[XCa44] += Max(0.0, (-6.48200e-07+(z)*(-3.57700e-03+5.23750e-04*exp(-Pow2(-4.25450e-01*(mc6)))+3.54460e-03*exp(-Pow2(8.38930e-02*(mc13)))+7.62110e-03/mco)));
    /* B10 */
    X[XB10]+=Max(0.0,(-8.38190e-09+4.28690e-06*z)*pow(mco,-3.02020e+00));
    /* Fe57 */
    X[XFe57] += Max(0.0, (-9.51900e-06+(z)*(1.19650e-01/mco-2.12580e-02+1.61330e-03*mco)));
    /* Mg24 */
    X[XMg24] += Max(0.0,2.62770e-02*exp(-Pow2(mco-2.85380e+00)/2.15090e+00)+6.83620e-02*exp(-Pow2(mco-1.08230e+01)/1.68750e+01)   );
    /* Cl35 */
    X[XCl35]+=Max(0.0,(-5.97420e-05+5.42130e-03*z+1.83660e-04*pow(mco,-4.35020e-01)) + (3.87970e-04+2.03060e-02*z)*exp(-Pow2(mco-5.65870e+00+3.51800e+01*z)));

    /* C12 */
    X[XC12] += Max(0.0, (2.14860e-01+-2.99670e-02*lz)*pow(mco,-7.70670e-01)   );
    /* Fe54 */
    X[XFe54]+=Max(0.0,
                  (4.12930e-03+5.99750e-01*z)*exp(-Pow2(mco+4.78310e+00)/7.07190e+01)+(8.21810e-05+9.56260e-02*z)*exp(-Pow2(mco-1.39650e+01)/2.10230e+00)-1.39050e-06*pow(mco,-2)/z);

    /* Ni63 */
    X[XNi63] += Max(0.0,4.49690e-02*pow(z,1.73140e+00)   );

    /* Co60 */
    X[XCo60] += Max(0.0,4.62520e-03*pow(z,1.5));

    /* Cr53 */
    X[XCr53] += Max(0.0,(-2.62630e-07+4.40460e-04*z+4.02490e-04*pow(mco,-1.14900e+01+3.83410e+02*z)));

    /* Al28 */
    X[XAl28] += Max(0.0, (2.49990e-01+(1.0)*(-2.49990e-01)));
    /* O16 */
    
    X[XO16] += Max(0.0, (6.17900e-01+((1.0-6.59240e+00*z))*((-8.85190e-01+5.01790e-01*mco-1.07990e-01*mc2+1.22490e-02*mc3-6.85300e-04*mc4+1.44180e-05*mc5))));

    /* Ge65 */
    X[XGe65] += Max(0.0, 2.79400e-08*pow(z,-2.92180e-01)*pow(mco,-5.31940e+00)   
        );

    /* Be7 */
    X[XBe7] += Max(0.0, (-1.65780e-07+((ilz))*(1.98280e-06-2.20160e-06*exp(-Pow2(9.13260e-02*(mc13)))-6.03030e-06/mco)));
    /* Al26 */
    X[XAl26] += Max(0.0, 5.71290e-03*z/mco+-6.69340e-06*lz*exp(-Pow2(mco-3.86090e+00+-3.94760e-01*lz)/1.76660e+00)+(6.60460e-05+1.24250e-05*lz)*exp(-Pow2(mco-1.14070e+01)/1.70180e+01)   );


    /* Ni57 */
    X[XNi57] += Max(0.0, (3.15200e-01+((1.0-6.47870e-02*z))*(-3.45950e-01+2.60030e-03*exp(-Pow2(5.86570e-01*(mc6)))+2.74540e-02*exp(-Pow2(8.62680e-02*(mc13)))+6.13260e-02/mco)));
    /* Cu66 */
    X[XCu66] += Max(0.0, (-3.07340e-08+(z)*((5.60630e-05+1.49100e-05*mco-5.47990e-06*mc2+4.81490e-07*mc3-1.30390e-08*mc4))));
    /* Be9 */
    X[XBe9]+=Max(0.0,(-9.31320e-10+4.61000e-07*z)*pow(mco,-3.02200e+00));

    /* Cu61 */
    X[XCu61] += Max(0.0, (2.48840e-04+((1.0+1.40750e+00*z))*((2.75860e-03-2.18340e-03*mco+5.90520e-04*mc2-7.36330e-05*mc3+4.24310e-06*mc4-9.12700e-08*mc5))));
    /* Fe60 */
    X[XFe60] += Max(0.0, (-7.85100e-07+(z)*(9.84260e-03/mco-1.40940e-03+5.79540e-05*mco)));
    /* Ca42 */
    X[XCa42] += Max(0.0, (8.69860e-07+3.37010e-03*z)*exp(-Pow2(mco-2.32830e+00)/7.55420e-01)+(2.42130e-05+4.95440e-03*z)*exp(-Pow2(mco-4.88620e+00)/1.27880e+00)+(7.29030e-04*z)*exp(-Pow2(mc13)/2.0)   );

    /* Cl37 */
    X[XCl37] += Max(0.0,2.25370e-03*z*(1+-1.41430e-02*mco)+1.14380e-05*exp(-Pow2(mco-5.49050e+00)/1.33290e-01)+6.44140e-02*z*pow(mco,-9.14470e+00));

    /* Ga64 */
    X[XGa64] += Max(0.0, 1.05810e-05*pow(z,-4.62210e-01)*pow(mco,-3.32910e+00));

    /* S32 */
    X[XS32] += Max(0.0, 3.75490e-02*exp(-Pow2(mco-2.07710e+00)/8.85200e-01) + 6.03790e-02*exp(-Pow2(mco-5.02810e+00)/3.49050e+00) + 1.46500e-02*exp(-Pow2(mco-1.24240e+01)/30.43200e-01));

    /* Cu62 */
    X[XCu62] += Max(0.0, (-3.35090e-06+((ilz))*(1.97450e-04-3.26780e-05*exp(-Pow2(4.23440e-01*(mc6)))-1.86030e-04*exp(-Pow2(1.04840e-01*(mc13)))-4.24850e-04/mco)));
    /* Mn53 */
    X[XMn53] += Max(0.0, (5.02630e-06+((ilz))*((-1.51650e-05-9.98440e-05*mco+2.63310e-05*mc2-2.28170e-06*mc3+6.42610e-08*mc4))));
    /* Co58 */
    X[XCo58] += Max(0.0,4.93600e-07+1.22750e-03*pow(z,1.80990e-01)*pow(mco,-5.67690e+00));

    /* Ca48 */
    X[XCa48] += Max(0.0, 1.68860e-03*Pow2(z)+7.28200e-02*Pow2(z)*pow(mco,-3.52350e+00)   );

    /* Na23 */
    X[XNa23] += Max(0.0, (3.11710e-04+7.46210e-02*z)/(1+pow(0.1,mc6)) + (5.20080e-04+1.50190e-01*z)*exp(-Pow2(mco-3)/1) + 5.19760e-01*z*pow(mco,-2)   );
    /* Zn62 */
    X[XZn62] += Max(0.0, (7.25280e-05+((1.0+8.89110e+01*z))*((1.29060e-02-9.76660e-03*mco+2.67750e-03*mc2-3.34160e-04*mc3+1.91550e-05*mc4-4.08850e-07*mc5))));
    /* Ge71 */
    X[XGe71] += Max(0.0, 4.61680e-02*pow(z,1.38190e+00)*exp(-Pow2(mco-1.14680e+01)/3.18550e+01)   );

    /* Ca45 */
    X[XCa45] += Max(0.0, (3.18300e-04+-7.40080e-05*mco+1.16010e-05*mc2+-4.62870e-07*mc3)*pow(z,1.58500e+00)   );

    /* Cr52 */


    X[XCr52]=Max(0.0,(-3.80910e-07+3.95040e-03*z+3.29600e-03*pow(mco,-9.78370e+00+3.06160e+02*z)));



    /* Ti47 */
    X[XTi47] += Max(0.0, (1.02450e-08+(z)*(-2.17050e-04+2.15690e-04*exp(-Pow2(4.81170e-01*(mc6)))+2.52420e-04*exp(-Pow2(1.93820e-01*(mc13)))+9.97580e-04/mco)));
    /* Ni61 */
    X[XNi61] += Max(0.0, (3.30210e-01+((1.0-8.33450e-03*z))*(-3.30220e-01+(1.18400e-05+0.5e-8/z)*exp(-Pow2(3.34690e-01*(mc6)))+3.43560e-05*(1+z/0.02)*exp(-Pow2(5.94480e-01*(mc13)))))    ) ;
    /* Mg25 */
    X[XMg25] += Max(0.0, 5.35320e-05+3.96690e-01*z*pow(mco,-1.96530e-01) - (5.52160e-05+1.79760e-01*z)*exp(-Pow2(mco-5.08750e+00)/8.73570e-01)   );

    /* Co56 */
    X[XCo56] += Max(0.0, 9.40780e-03*exp(-Pow2(mco+4.38210e+00)/1.41740e+01)+9.66750e-04*exp(-Pow2(mco-6.25990e+00)/1.08770e+00)+(6.61540e-04+9.63610e-02*z)*exp(-Pow2(mco-1.32780e+01)/5.24830e-01)  );

    /* Fe56 */
    X[XFe56] += Max(0.0, (5.85880e-02+((1.0-1.92560e+01*z))*((1.39410e+00-9.70700e-01*mco+2.53400e-01*mc2-3.12950e-02*mc3+1.81280e-03*mc4-3.95170e-05*mc5))));
    /* K39 */
    X[XK39]+=Max(0.0,(2.63970e-05+5.67050e-03*z+1.05780e-04/mco) + (2.56390e-04+3.50110e-03*z)*exp(-Pow2(mco-5.35820e+00))+-4.62680e-06*mco);if(mco>7){X[XK39]/=2;}


    /* Cr49 */
    X[XCr49] += Max(0.0, (1.41560e-06+((ilz))*((-4.07690e-05+1.43440e-05*mco-1.78530e-06*mc2+9.59260e-08*mc3-1.86260e-09*mc4))));
    /* H1 */
    //X[XH1] += Max(0.0, (2.49990e-01+(1.0)*(-2.49990e-01)));
    /* Cr48 */
    X[XCr48] += Max(0.0, (3.29650e-01+((1.0-2.61970e-04*z))*(-3.29940e-01+9.83210e-05*exp(-Pow2(5.45060e-01*(mc6)))+2.49450e-04*exp(-Pow2(1.58620e-01*(mc13)))+1.15200e-03/mco)));
    /* H2 */
    //X[XH2] += Max(0.0, (2.49990e-01+(1.0)*(-2.49990e-01)));
    /* Ga69 */
    X[XGa69] += Max(0.0, (-2.03960e-07+(z)*((6.24910e-04-5.07020e-04*mco+2.20360e-04*mc2-3.26520e-05*mc3+2.02000e-06*mc4-4.47030e-08*mc5))));
    /* C13 */
    X[XC13] += Max(0.0, (2.49990e-01+(1.0)*(-2.49990e-01)));
    /* Mg26 */
    X[XMg26] += Max(0.0, z*(8.20900e-01+-3.36160e-01*mco+5.95710e-02*mc2+-2.94390e-03*mc3+-6.35090e-05*mc4+5.57770e-06*mc5)   );

    /* Cu64 */
    X[XCu64] += Max(0.0, (-2.04890e-08+(z)*(3.78030e-05+7.05520e-05*exp(-Pow2(3.01490e-01*(mc6)))+5.39090e-05*exp(-Pow2(6.70620e-01*(mc13)))+6.35720e-06/mco)));
    /* Ti45 */
    X[XTi45] += Max(0.0, (2.09550e-07+((1.0-1.76480e+01*z))*((9.26110e-06-4.30830e-06*mco+6.83590e-07*mc2-4.37720e-08*mc3+9.31320e-10*mc4))));
    /* V51 */
    X[XV51] += Max(0.0, 1.11010e-04*z+1.74990e+00*pow(z,2.54170e+00)*pow(mco,-3.69570e+00));

    /* Ca40 */
    X[XCa40] += Max(0.0, (3.21960e-03+(lz)*((2.80270e-03-3.02560e-03*mco+7.16000e-04*mc2-5.95060e-05*mc3+1.63910e-06*mc4))));
    /* K41 */
    X[XK41] += Max(0.0, (-1.00580e-07+(z)*(-6.33500e-04+9.11810e-05*exp(-Pow2(4.68680e-01*(mc6)))+6.53100e-04*exp(-Pow2(9.59550e-02*(mc13)))+1.60030e-03/mco)));
    /* V47 */
    X[XV47] += Max(0.0, (2.21650e-07+((ilz))*(6.52670e-06-5.82080e-07*exp(-Pow2(7.01090e-01*(mc6)))-5.02080e-06*exp(-Pow2(1.06390e-01*(mc13)))-1.82380e-05/mco)));
    /* Fe58 */
    X[XFe58] += Max(0.0, (-7.09390e-06+(z)*(7.14290e-03+1.52130e-03*exp(-Pow2(2.01790e-01*(mc13)))+1.49080e-02/mco)));
    /* S34 */
    X[XS34] += Max(0.0, 1.32750e-01*pow(z,7.49700e-01)*exp(-Pow2(mco-4.80270e+00)/5.68360e-01)+ 3.63670e-02*z*exp(-Pow2(mc13)/5)+2.72960e-01*z*pow(mco,-9.68930e-01)   );

    /* Zn65 */
    X[XZn65] += Max(0.0,
                    1.37400e-05*pow(z,5.26020e-01)   );

    /* F19 */
    X[XF19]+=Max(0.0,(2.99420e-05+8.15090e-06*lz+6.14950e-01*z*pow(mco,-1.34840e+01)) + (1.78990e-05+4.19490e-04*z)*exp(-Pow2(mco-3.03890e+00))+3.67870e-07*mco   );

    /* Si28 */
    X[XSi28] += Max(0.0, 7.21770e-02*exp(-Pow2(mco-1.72000e+00)/1.32400e+00)+ 1.11990e-01*exp(-Pow2(mco-4.99980e+00)/4.96350e+00)+ 4.76490e-02*exp(-Pow2(mco-1.21160e+01)/6.26360e+00)   );

    /* Mn55 */
    X[XMn55] += Max(0.0, (-6.48200e-06+(z)*(5.30550e-02/mco-1.09860e-02+7.34270e-04*mco)));
    /* Cr51 */
    X[XCr51] += Max(0.0,
                    (1.71460e-06+1.70170e-04*z)*exp(-Pow2(mco-4.42010e+00)/2.30810e+01)+(4.03170e-06+-8.06970e-05*z)*exp(-Pow2(mco-1.35570e+01)/1.96040e-01));

    /* Co61 */
    X[XCo61] += Max(0.0, (-6.09080e-07+(z)*((4.50140e-03-1.72870e-03*mco+2.37320e-04*mc2-1.30140e-05*mc3+2.43080e-07*mc4))));
    /* Ge70 */
    X[XGe70] += Max(0.0, (-3.49250e-07+(z)*(3.95950e-04+2.41710e-03*exp(-Pow2(5.28550e-01*(mc6)))+1.77990e-03*exp(-Pow2(2.79600e-01*(mc13)))-6.83490e-05/mco)));
    /* _C11 */
    X[XC11] += Max(0.0, (1.40630e-06+2.40380e-05*z)*pow(mco,-1.73760e+00)+(9.31320e-10+1.74160e-07*z)*mco   );


    /* Ga67 */
    X[XGa67] += Max(0.0, (1.21070e-08+(z)*(-9.41190e-06+7.27640e-06*exp(-Pow2(3.56110e-01*(mc6)))+1.15310e-05*exp(-Pow2(1.88870e-01*(mc13)))+2.26280e-05/mco)));
    /* Zn63 */
    X[XZn63] += Max(0.0, 
                    5.76240e-04*pow(mco,-4.56730e+00)*pow(z,3.87540e-02)   );

    /* Cu60 */
    X[XCu60] += Max(0.0, (-3.98370e-05+((1.0-8.91270e+00*z))*((3.11260e-02-2.03750e-02*mco+5.04370e-03*mc2-5.85600e-04*mc3+3.19330e-05*mc4-6.58450e-07*mc5))));
    /* Ga70 */
    X[XGa70] += Max(0.0,8.61580e-04*pow(z,1.59990e+00) );

    /* S33 */
    X[XS33] += Max(0.0, 3.68150e-04*exp(-Pow2(mco-5.18330e+00)/3.80480e+00)+ 4.54820e-03*z*exp(-Pow2(mc13)/5)+3.97420e-02*z*pow(mco,-1.70120e+00)   );

    /* Cu65 */
    X[XCu65] += Max(0.0, (-9.21080e-07+(z)*(2.20310e-04+7.75150e-04*exp(-Pow2(2.22870e-01*(mc6)))+1.79710e-03*exp(-Pow2(1.08530e-01*(mc13)))+6.04810e-04/mco)));
    /* Ca43 */
    X[XCa43]=Max(0.0,(-5.21540e-08+6.05340e-05*z+1.01540e-03*pow(mco,-1.37280e+01)) + (1.47150e-07+3.39480e-04*z)*exp(-Pow2(mco-5.72890e+00+3.44540e+01*z)));

    /* Ni59 */
    X[XNi59] += Max(0.0, (3.28850e-01+((1.0-1.51710e-02*z))*(-3.30670e-01+2.44910e-04*exp(-Pow2(-4.80580e-01*(mc6)))+1.58520e-03*exp(-Pow2(1.03870e-01*(mc13)))+3.99690e-03/mco)));
    /* Na22 */
    X[XNa22] += Max(0.0, 1.69970e-06*exp(-Pow2(mco-3.22110e+00)/7.42940e-01)+(8.64270e-07+-1.82830e-05*z)/(1+pow(0.1,mco-6.25390e+00))   );

    /* Cu59 */
    X[XCu59] += Max(0.0, (6.31340e-06+((ilz))*(1.86520e-04-2.00080e-05*exp(-Pow2(4.84340e-01*(mc6)))-1.49740e-04*exp(-Pow2(1.12870e-01*(mc13)))-4.47870e-04/mco)));
    /* N14 */
    X[XN14] += Max(0.0, (2.49990e-01+(1.0)*(-2.49990e-01)));
    /* Ge68 */
    X[XGe68] += Max(0.0,1.25950e-05*pow(z,1.82880e-01)*pow(mco,-6.00960e+00));

    /* Ni64 */
    X[XNi64] += Max(0.0, 2.38880e-03*pow(z,1.39590e+00)*(1.51700e+01+-3.30350e+00*mco+8.21230e-01*mc2+-3.82540e-02*mc2*mco)   );
    //(-4.96670e-06+(z)*((5.21180e-03+3.81640e-04*mco))));
    /* Ar36 */
    X[XAr36]+=Max(0.0,-2.54570e-03+1.07990e-02*exp(-Pow2(5.34270e-01*(mco-5.23810e+00)))+3.38810e-03*exp(-Pow2(-4.34530e-01*(mco-1.34150e+01)))+1.97310e-02/mco);

    /* Al27 */
    X[XAl27] += Max(0.0, 4.21960e-01*z/mco+(2.59010e-03+1.73010e-01*z)*exp(-Pow2(mco-3.76160e+00+3.00450e+01*z)/1.80500e-01)+(2.44040e-03+4.18770e-01*z)*exp(-Pow2(mco-1.13080e+01)/1.95710e+01)   );

    /* O18 */
    X[XO18] += Max(0.0, (2.49990e-01+(1.0)*(-2.49990e-01)));
    /* Mn52 */
    X[XMn52] += Max(0.0,8.14070e-05*(Pow2(sin(5.15740e-01*Max(mco,4.5)+1.40960e+00))+1e-5));

    /* Co57 */
    X[XCo57] += Max(0.0,8.65790e-03*exp(-Pow2(mco+1.33510e+00)/2.36310e+00)+1.12530e-04*exp(-Pow2(mco-6.11260e+00)/8.99750e-01)+(4.19850e-05+1.73760e-02*z)*exp(-Pow2(mco-1.31020e+01)/1.03720e+00));

    /* Ni62 */
    X[XNi62] += Max(0.0, (-1.17630e-06+(z)*(-7.33650e-03+1.50370e-02*exp(-Pow2(3.31640e-01*(mc6)))+2.61940e-02*exp(-Pow2(2.91560e-01*(mc13)))+2.61710e-02/mco)));
    /* Zn68 */
    X[XZn68] += Max(0.0, (-3.60980e-06+(z)*((2.93370e-03-9.89710e-04*mco+2.36530e-04*mc2+2.59660e-05*mc3-4.40610e-06*mc4+1.35040e-07*mc5))));
    /* B11 */
    X[XB11] += Max(0.0, (-9.77890e-08+((ilz))*(-4.75630e-06/mco+9.31320e-09-6.51930e-09*mco)));
    /* Sc45 */
    X[XSc45] += Max(0.0, (8.47500e-08+(z)*(-4.15240e-05+1.43000e-04*exp(-Pow2(5.01950e-01*(mc6)))+7.39040e-05*exp(-Pow2(-3.00760e-01*(mc13)))+1.86360e-04/mco)));
    /* Ti49 */
    X[XTi49] += Max(0.0, -1.00580e-07+1.41460e-03*pow(z,8.46490e-01)*pow(mco,-4.58700e+00) +9.83780e-05*z   );

    /* S36 */
    X[XS36] += Max(0.0, (-4.74040e-07+(z)*((3.68230e-04+7.03500e-05*mco-1.32620e-06*mc2-1.71360e-07*mc3))));
    /* Cr54 */
    X[XCr54] += Max(0.0, (-3.61350e-07+(z)*((2.06700e-03-9.17900e-04*mco+1.65290e-04*mc2-1.18050e-05*mc3+2.89640e-07*mc4))));
    /* N15 */
    X[XN15] += Max(0.0,-7.57720e-06+(-3.56980e-04+1.74550e-02*z)/mco+(3.39970e-04+-7.33500e-03*z)/pow(mco,7.00290e-01));

    /* Ti46 */
    X[XTi46] += Max(0.0, (1.78810e-06+(z)*((2.77720e-03-1.56440e-03*Min(mco,10)+5.53810e-04*Pow2(Min(mco,10))-8.93420e-05*pow(Min(mco,10),3.000000e+00)+6.13180e-06*pow(Min(mco,10),4.000000e+00)-1.48080e-07*pow(Min(mco,10),5.000000e+00)))));

    /* Ge64 */
    X[XGe64] += Max(0.0,7.73930e-07*pow(z,-3.91080e-01)*pow(mco,-3.20870e+00));

    /* Ge66 */
    X[XGe66] += Max(0.0, 3.09630e-03*pow(z,1.32550e-01)*pow(mco,-4.84380e+00)   
        );

    /* Ar38 */

    X[XAr38]+=Max(0.0,-5.77100e-03+(1.22100e-03+1.16420e-01*z)*exp(-Pow2(4.36820e-01*(mco-4.69350e+00)))+(5.45360e-03+1.66110e-02*z)*exp(-Pow2(7.11430e-02*(mco-1.32310e+01)))+5.56810e-03/mco);

    /* Ca41 */
    X[XCa41] += Max(0.0, (2.50030e-01+(1.0)*(-2.50030e-01+1.30910e-05*exp(-Pow2(5.23490e-01*(mc6)))+1.68940e-06*exp(-Pow2(-3.57000e-01*(mc13)))+1.24620e-05/mco)));
    /* Cl36 */
    X[XCl36] += Max(0.0, 3.07220e-05*z+2.51180e-06*pow(-lz,-2.82280e+00)*(1+2.26910e+01*exp(-Pow2(mco-5)/1.06140e+00))   );

    /* Ni60 */
    X[XNi60] += Max(0.0,
                    (-3.31040e-03+2.63340e-02*z+8.44550e-03/mco) + (2.34760e-03+-3.49320e-02*z)*exp(-Pow2(mc6)/5.73330e+00)+(3.26500e-03+2.83800e-02*z)*exp(-Pow2(mc13)/2.25220e+01)   );

    /* He4 */
    //X[XHe4] += Max(0.0, (2.49990e-01+(1.0)*(-2.49990e-01)));
    /* Cu63 */
    X[XCu63] += Max(0.0, (7.01290e-07+(z)*(-2.09030e-03+6.23560e-04*exp(-Pow2(4.10430e-01*(mc6)))+2.45400e-03*exp(-Pow2(9.71550e-02*(mc13)))+4.52470e-03/mco)));
    /* Zn61 */
    X[XZn61] += Max(0.0, (7.34350e-06+(lz)*(3.71240e-05-2.44100e-06*exp(-Pow2(4.44370e-01*(mc6)))-3.09770e-05*exp(-Pow2(8.11190e-02*(mc13)))-6.02450e-05/mco)));
    /* Ga65 */
    X[XGa65] += Max(0.0,2.68410e-05*pow(z,-2.22150e-02)*pow(mco,-4.39770e+00)  );

    /* Ca47 */
    X[XCa47] += Max(0.0,1.00020e-06*pow(z,1.00000e+00)*pow(mco,-2.50000e+00)   );

    /* Ca46 */
    X[XCa46] += Max(0.0, 2.79400e-09+2.30510e-03*Pow2(z)*(1+-3.30820e-01*mco+4.35340e-02*mc2+-1.62470e-03*mc2*mco));
    /* He3 */
    //X[XHe3] += Max(0.0, (2.49990e-01+(1.0)*(-2.49990e-01)));
    /* Fe55 */
    X[XFe55] += Max(0.0, (-1.17100e-05+2.52730e-03*z)*Pow2(sin(3.41110e-01*mco))+1.10960e-04/mco);

    /* Ti48 */
    X[XTi48] += Max(0.0, (-5.57860e-07+(z)*(-4.33900e-03-8.94940e-04*exp(-Pow2(-2.59760e-02*(mc6)))+5.12820e-03*exp(-Pow2(7.06500e-02*(mc13)))+1.01060e-02/mco)));
    /* Ni58 */
    X[XNi58] += Max(0.0, 6.20910e-04+(3.69440e-01+2.02810e+01*z)/pow(mco,5.29340e+00));

    /* Zn67 */
    X[XZn67] += Max(0.0, (-3.39000e-07+(z)*((2.60720e-04-1.70720e-05*mco+2.08570e-05*mc2-1.06730e-06*mc3))));
    /* Ti44 */
    X[XTi44] += Max(0.0, (2.49880e-01+(1.0)*(-2.50180e-01+4.31980e-05*exp(-Pow2(5.51510e-01*(mc6)))+2.67020e-04*exp(-Pow2(1.00780e-01*(mc13)))+6.92000e-04/mco)));
    /* Ti50 */
    X[XTi50] += Max(0.0, (-1.49010e-07+(z)*(-5.03970e-04+5.98800e-05*exp(-Pow2(1.81370e-01*(mc6)))+6.37480e-04*exp(-Pow2(8.43200e-02*(mc13)))+9.49910e-04/mco)));
    /* Si29 */
    X[XSi29] += Max(0.0, (2.03260e-04+(z)*((1.97500e-01-5.17980e-02*mco+3.21810e-03*mc2+1.27600e-03*mc3-1.32270e-04*mc4+3.20000e-06*mc5))));
    /* Ne21 */
    X[XNe21] += Max(0.0, 4.02700e-02*pow(z,7.59360e-01)*pow(mco,-1.36730e+00)   );

    /* Fe59 */
    X[XFe59] += Max(0.0, (-8.69860e-07+(z)*((5.86290e-03-1.31750e-03*mco+1.01590e-04*mc2-2.51180e-06*mc3))));
    /* Na24 */
    if((mco>6)&&(z<0.001))
    { 
        X[XNa24] += 1e-6;
    }

    /* Ne20 */
    X[XNe20] += Max(0.0, (8.35900e-02+-2.89750e-01*z)/(1+pow(0.1,mc6)) + (1.08720e-01+4.76280e-01*z)*exp(-Pow2(mco-3)/1) + 6.54910e+00*z*pow(mco,-2)   );

    /* Ge67 */
    X[XGe67] += Max(0.0,1e-8*6.51930e-09*pow(z,-2.61110e+00)*pow(mco,-2.84340e+00));

    /* Ni56 */
    X[XNi56] += Max(0.0, (2.49990e-01+(1.0)*(-2.49990e-01)));
    /* Co55 */
    X[XCo55] += Max(0.0,-3.73610e-04+1.72390e-03*pow(mco,-6.86370e-01+9.03510e+00*z)+1.64950e-04*exp(-Pow2(mc13)/2)   );

    /* Li7 */
    X[XLi7] += Max(0.0,9.40640e-08+-4.65660e-09*mco+6.11320e-06*pow(mco,-4.07420e+00));
    /* Cr50 */
    X[XCr50] += Max(0.0,z*(8.63580e-03+-2.73970e-03*mco+7.35540e-04*mc2+-1.21690e-04*mc3+9.04500e-06*mc4+-2.35620e-07*mc5)   );

    /* C14 */
    X[XC14] += Max(0.0,2.06840e-02*z*pow(mco,-3.55780e+00)+-3.95930e-03*exp(-Pow2(mco-2.89750e+00+3.68740e+01*z)/8.96780e-03)   );
    /* Mn54 */
    X[XMn54] += Max(0.0,(4.35860e-07+1.17090e-04*z)*exp(-Pow2(mco-3.47790e+00)/1.30240e+01)+(5.58790e-08+3.65400e-05*z)*exp(-Pow2(mco-1.36140e+01)/8.56400e-01)   );

    /* Co59 */
    X[XCo59] += Max(0.0, (-2.94480e-06+(z)*(-2.55930e-03+4.52530e-04*exp(-Pow2(2.46210e-01*(mc6)))+4.96440e-03*exp(-Pow2(9.52590e-02*(mc13)))+1.48610e-02/mco)));
    /* Zn66 */
    X[XZn66] += Max(0.0, (-9.59260e-07+(z)*(3.60810e-04+2.66640e-03*exp(-Pow2(3.89980e-01*(mc6)))+3.16590e-03*exp(-Pow2(1.61040e-01*(mc13)))+2.23570e-03/mco)));
    /* P31 */
    X[XP31] += Max(0.0,(1.32110e-03+3.96150e-01*z)*(8.58900e-02+4.51030e-03*mco+-4.57340e-04*mc2)+4.57110e-04*exp(-Pow2(mco-5.37960e+00)/6.35880e-01)+0.5*1.69130e-01*pow(z,1.26990e+00)*exp(-Pow2(mco-1.29150e+01)/6));
    /* V50 */
    X[XV50] += Max(0.0, 2.47480e-04*pow(z,1.47610e+00)*exp(-Pow2(mco-4.98030e+00)/1.65290e-01) +3.53340e-06*z);
    /* Ne22 */
    X[XNe22] += Max(0.0, 1.88850e-01*z*(1+4.12510e-01*mco+-2.04950e-02*mc2)+4.87020e-01*pow(z,1.51510e-01)*pow(mco,-8.61990e+00)   );
    //(8.15190e-05+(z)*(3.54460e-01+2.28180e-01*exp(-pow(1.62260e+00*(mc6),2))+1.34030e-01*exp(-pow(1.00160e-01*(mc13),2)))));
    /* Sc43 */
    X[XSc43] += Max(0.0, (-3.44590e-08+((ilz))*(5.99210e-06-7.30160e-07*exp(-Pow2(6.18220e-01*(mc6)))-5.33740e-06*exp(-Pow2(8.64050e-02*(mc13)))-1.36110e-05/mco)));
    /* Ar40 */
    X[XAr40] += Max(0.0, (-1.01510e-07+(z)*((3.85610e-04-1.40720e-04*mco+2.78060e-05*mc2-2.05170e-06*mc3+5.02910e-08*mc4))));
    /* O17 */
    X[XO17] += Max(0.0, (2.49990e-01+(1.0)*(-2.49990e-01)));
    /* Ge69 */
    X[XGe69] += Max(0.0, (9.31320e-10+(z)*(-5.50410e-07+9.05430e-06*exp(-Pow2(4.89280e-01*(mc6)))+4.14530e-06*exp(-Pow2(3.50900e-01*(mc13)))+2.07960e-06/mco)));
    /* Zn69 */
    X[XZn69] += Max(0.0, 1.04190e-01*pow(z,2.27540e+00)*pow(mco,-8.18770e-01));

    /* S35 */
    X[XS35] += Max(0.0, (-1.31320e-07+(z)*((3.71110e-04-3.52110e-04*mco+2.18990e-04*mc2-4.05790e-05*mc3+2.94200e-06*mc4-7.35740e-08*mc5))));
    /* V48 */
    X[XV48] += Max(0.0, (5.20610e-07+((ilz))*(2.04700e-06-1.78690e-05*exp(-Pow2(1.49530e+00*(mc6)))-1.35990e-05*exp(-Pow2(5.20250e-01*(mc13)))-8.51790e-06/mco)));
    /* Fe53 */
    X[XFe53] += Max(0.0, (5.57670e-05+1.07950e-03*z)*exp(-Pow2(mco-1.99670e+00)/1.16020e+01)+(3.01280e-06+6.35250e-04*z)*exp(-Pow2(mco-1.32430e+01)/3.80680e-01));

#endif
}


#endif
#endif
