#include "../binary_c.h"
#if defined NUCSYN && defined NUCSYN_SUPERNOVAE

double nucsyn_sn_core_collapse(Abundance * const X, //space calloc'd previously
                               Abundance * const Xenv,
                               const double mco,// co core mass
                               const double menv,//env mass
                               double dm,// mass to be ejected
                               Abundance z,
                               Yield_source * const src_id,
                               struct stardata_t * const stardata,
                               const struct star_t * const pre_explosion_star,
                               struct star_t * const exploder)
{
    /* 
     * Calculate yields from core collapse supernovae.
     *
     * The mass ejected in the supernova is dm. However, this is made up of
     * material both from the envelope and the core so treat them separately.
     *
     * pre_explosion_star->stellar_type is the stellar type of the star just prior to explosion 
     *
     * exploder is the structure after explosion
     * 
     * The abundance array X stores the ejected mass fractions,
     * this can be used to accrete onto the companion star, 
     * if required.
     * 
     * Returns total ejected mass.
     */
    if(X==NULL)
    {
        Exit_binary_c(
            BINARY_C_POINTER_FAILURE,
            "nucsyn_sn_core_collapse has X null on input - oops\n");
    }
    Clear_isotope_array(X);
    
    double ejecta_mass = 0.0;
#ifdef NUCSYN_CCSNE_CHIEFFI_LIMONGI_2004
    double mcut = menv + mco - dm; // location of the mass cut (NS/BH mass)
#endif
    double dm_env; // envelope mass to be ejected
    double dm_core; // core mass to be ejected
    double m1,m2;
    
    /*
     * if there is no expelled material, just return
     */
    if(dm < TINY)
    {
        Dprint("no material expelled : nothing to do");
    }
    else
    {
#ifdef NUCSYN_R_PROCESS_FROM_SNE
        /*
         * Set some ejecta mass to be from r-process
         */
        dm -= NUCSYN_R_PROCESS_FROM_SNE_MASS;
#endif//NUCSYN_R_PROCESS_FROM_SNE

        /*
         * If nucsyn_metallicity is set, use it
         */
        if(stardata->common.nucsyn_metallicity > -TINY) 
            z = stardata->common.nucsyn_metallicity;

        /*
         * decide whether we're SNIb/c or SNII 
         * (SN_type can be SN_GRB_COLLAPSAR,
         * but there is no equivalent nucsyn source).
         */
        *src_id = 
            (exploder->SN_type == SN_GRB_COLLAPSAR ||
             exploder->SN_type == SN_IBC) ? 
            NUCSYN_SOURCE_SNIbc : NUCSYN_SOURCE_SNII;

        Dprint("Nucsyn source %d from SN_type %d\n",
               *src_id,
               exploder->SN_type);

        /*
         * decide where expelled material is from
         */
        if(dm < menv)
        {
            dm_env = dm;
            dm_core = 0.0;
        }
        else
        {
            dm_env = menv;
            dm_core = dm-menv;
        }

        /*
         * make sure both dm_env and dm_core are at least TINY to prevent
         * numerical problems
         */
        dm_env = Max(TINY, dm_env);
        dm_core = Max(TINY, dm_core);

        if(dm_env > TINY)
        {
            /* 
             * ENVELOPE
             *
             * for stellar types EAGB and TPAGB (AGB) the envelope is uniform and
             * the helium core is (almost) coincident with the CO core
             * so we can treat the envelope as homogeneous with composition
             * the same as the surface
             *
             * Deal with helium stars a bit differently, however.
             */
            if(NAKED_HELIUM_STAR(pre_explosion_star->stellar_type))
            {
#ifdef NUCSYN_HETEROGENEOUS_HE_STAR
                /* This is based on a 56Msun star just prior to explosion...
                 * Between the surface and 1.5Msun outside the CO core the C,O and He
                 * mass fractions change ~ linearly to C=0.3 O=0.6 He=0.06 Ne20=0.03
                 * The core is dealt with by the SN fits so we 
                 * just need to worry about the section with 
                 * any helium in it
                 * Xenv is the surface abundance 
                 */

                /***** DEPRECATED *****/
                /*
                  m1=Max(0.0,dm_env-1.5); // region of changing abunds
                  m2=dm_env-m1; // homogeneous region outside the CO core
                  Xenv[XC12]=(m1*0.5*(Xenv[XC12]+0.3)+m2*0.6)/dm_env;
                  Xenv[XO16]=(m1*0.5*(Xenv[XO16]+0.6)+m2*0.3)/dm_env;
                  Xenv[XHe4]=(m1*0.5*(Xenv[XHe4]+0.06)+m2*0.06)/dm_env;
                  Xenv[XNe20]=(m1*0.5*(Xenv[XNe20]+0.03)+m2*0.03)/dm_env;
                */

                /*
                 * Improved fit based on many Z=0.02 models from Lynnette, where 
                 * the region outside the CO core is defined by Y>0.05. 
                 */
          
                /*
                 * At Z=0.02:
                 * Combining the low-core-mass (MM) and high-core-mass(NL)
                 * data, we have the homogeneous helium abundance given as
                 * a function of core mass
                 *
                 * he = Max(0.01,Min(0.1,0.0135760+4.9056*exp(-mco)))
                 *
                 * (note the 0.01 lower limit is somewhat arbitrary)
                 *
                 * with
                 * c=2.43910e-01-7.48800e-03*mco+1.92790e+00*mco*exp(-mco)
                 * o=6.78700e-01+8.04440e-03*mco-2.59100e+00*mco*exp(-mco)
                 *
                 * The size of the region is 0.5Msun for Mcore < 10,
                 * the whole envelope mass for Mcore>10. This is a sudden
                 * change (it would seem...)
                 *
                 * At other metallicities...
                 * It seems the core mass is always > 10 for Z<=0.004 
                 * (EXCEPT IN BINARIES!)
                 * so the heterogeneous zone is always the entire envelope.
                 * The fit for the internal he, c and o is reasonable for all Z 
                 * down to 1e-4.
                 *
                 * Note that for MM mass-loss it's very difficult because
                 * the star is very evolved just prior to the end of the evolution
                 * and indeed is mostly oxygen! It is unlikely that the nucsyn_wr function 
                 * actually follows this behaviour... indeed I have tried to fit the high
                 * oxygen abundances to M, Mc etc. and cannot. Ooops...
                 */

                // mass of heterogenous region
                m1 = mco<10 ? Min(dm_env,0.5) : dm_env;
            
                m2=dm_env-m1; // mass of homogeneous region

                // limit the core mass to the range given by 
                preheco=Xenv[XHe4]+Xenv[XC12]+Xenv[XO16];
        
                /*
                 * This needs some work!
                 *
                 mcofit=Max(1.7,Min(25.0,mco));
                 he=Max(0.01,Min(0.1,0.0135760+4.9056*exp(-mcofit)));
                 c=Max(0,Min(0.9,2.43910e-01-7.48800e-03*mcofit+1.92790e+00*mcofit*exp(-mcofit)));
                 o=preheco-he-c; // calculate oxygen from the rest
                */

                /*
                 * Everything else broke, so just assume 80% O, 20% C,
                 * no helium at the internal boundary (or less C if the 
                 * surface has less)
                 */
                he=0.0;
                c=Min(0.2,Xenv[XC12]);
                o=1.0-he-c;

                /*
                 * In m1: triangle area + rectangle below
                 * In m2: just the rectangle
                 */
                Xenv[XHe4]=(m1*0.5*(Xenv[XHe4]-he)+m1*he+m2*Xenv[XHe4])/dm_env;
                Xenv[XC12]=(m1*0.5*(Xenv[XC12]-c)+m1*c+m2*Xenv[XC12])/dm_env;
                Xenv[XO16]=(m1*0.5*(Xenv[XO16]-o)+m1*o+m2*Xenv[XO16])/dm_env;


#endif // NUCSYN_HETEROGENEOUS_HE_STAR
            }
            else if(pre_explosion_star->stellar_type==CHeB ||
                    pre_explosion_star->stellar_type==EAGB ||
                    pre_explosion_star->stellar_type==TPAGB)
            {
                /*
                 * Star explodes on EAGB : some of the star is helium, not hydrogen!
                 * NB This also applies on the TPAGB, for which we should set mcx,
                 * the CO core mass
                 */
                if(pre_explosion_star->stellar_type==TPAGB)
                {
                    exploder->max_EAGB_He_core_mass = exploder->core_mass;
                }

                /* this mass of envelope is hydrogen-burnt */
                m1 = exploder->max_EAGB_He_core_mass-
                    exploder->core_mass;

                /* this mass is in the convective *unburnt* envelope */
                m2 = dm_env-m1;

                /* convert to mass fractions */
                m1/=dm_env;
                m2/=dm_env;

                
                /*
                 * Convert all H to He in the burnt region
                 * and convert all CNO to N in the burnt region
                 */
                Xenv[XHe4]=(m1 * (Xenv[XHe4]+Xenv[XH1]) + 
                            m2 * (Xenv[XHe4]));
                Xenv[XH1]=(m1 * 0.0+
                           m2 * Xenv[XH1]);
                Xenv[XN14]=(m1 * (Xenv[XN14]+Xenv[XC12]+Xenv[XC13]+Xenv[XN15]+Xenv[XO16]+Xenv[XO17]+Xenv[XO18]) + 
                            m2 * Xenv[XN14]);
                Xenv[XC12]=(m1 * 0.0+
                            m2 * Xenv[XC12]);
                Xenv[XC13]=(m1 * 0.0+
                            m2 * Xenv[XC13]);
                Xenv[XN15]=(m1 * 0.0+
                            m2 * Xenv[XN15]);
                Xenv[XO16]=(m1 * 0.0+
                            m2 * Xenv[XO16]);
                Xenv[XO17]=(m1 * 0.0+
                            m2 * Xenv[XO17]);
                Xenv[XO18]=(m1 * 0.0+
                            m2 * Xenv[XO18]);
            }

            /* 
             * normalize the abundance array to 1.0 : this prevents small errors,
             * or in the chieffi/limongi case it normalizes the abundances array
             * correctly
             */
            nucsyn_renormalize_abundance(stardata,Xenv);
            
            /*
             * Put the ejecta in X
             */
            nucsyn_mix_shells(dm_env,Xenv,
                              ejecta_mass,X);
            ejecta_mass += dm_env;
        }

        /* 
         * now yield from the core: first calculate the abundances from the
         * explosion of the CO core according to the chosen prescription
         */
        if(dm_core > 2.0*TINY)
        {
            Abundance * Xcore = New_clear_isotope_array;

#if defined(NUCSYN_CCSNE_WOOSLEY_WEAVER_1995)
            nucsyn_sn_woosley_weaver_1995(stardata,Xcore,mco,z);
#elif defined(NUCSYN_CCSNE_CHIEFFI_LIMONGI_2004)
            nucsyn_sn_chieffi_limongi_2004(stardata,Xcore,mco,z,mcut,exploder);
#endif
            /* 
             * normalize the abundance array to 1.0 : this prevents small errors,
             * or in the chieffi/limongi case it normalizes the abundances array
             * correctly
             */
            nucsyn_renormalize_abundance(stardata,Xcore);
            
            /*
             * Now add yield lost from the core, dm_core. remember Xtot normalization factor.
             */
            Dprint("exploder is star %d, using yields of %s with mco=%g menv=%g : explosive yields (M=%g) : C=%g O=%g Fe56=%g) : Yields H1=%g He4=%g C12=%g O16=%g\n",
                   exploder->starnum,
#ifdef NUCSYN_CCSNE_WOOSLEY_WEAVER_1995
                   "Woosley and Weaver 1995",
#elif defined NUCSYN_CCSNE_CHIEFFI_LIMONGI_2004
                   "Chieffi and Limongi 2004",
#else
                   "???",
#endif
                   mco,
                   menv,
                   mco + menv,
                   dm_core*X[XC12],
                   dm_core*X[XO16],
                   dm_core*X[XFe56],
                   exploder->Xyield[XH1],
                   exploder->Xyield[XHe4],
                   exploder->Xyield[XC12],
                   exploder->Xyield[XO16]);
            
            /*
             * Put the ejecta in X
             */
            nucsyn_mix_shells(dm_core,Xcore,
                              ejecta_mass,X);
            ejecta_mass += dm_core;
            Safe_free(Xcore);
        }
        
        
#if defined NUCSYN_R_PROCESS && defined NUCSYN_R_PROCESS_FROM_SNE
        /*
         * We still have a mass NUCSYN_R_PROCESS_FROM_SNE_MASS to 
         * be ejected as r-process material, so do it.
         */
        Abundance * Xr = New_clear_isotope_array;
        nucsyn_r_process(Xr,stardata); // set r-process abundances
        nucsyn_mix_shells(NUCSYN_R_PROCESS_FROM_SNE_MASS,Xr,
                          ejecta_mass,X);
        ejecta_mass += NUCSYN_R_PROCESS_FROM_SNE_MASS;
        Safe_free(Xr);
#endif
        nancheck("SN X",X,ISOTOPE_ARRAY_SIZE);
    }
    
    Dprint("Returning ejecta_mass %g\n",ejecta_mass);
    return ejecta_mass;
}

#endif // NUCSYN && NUCSYN_SUPERNOVAE
