#pragma once
/*
 * The binary_c stellar population nucleosynthesis framework.
 *
 * Contact: r.izzard@surrey.ac.uk or rob.izzard@gmail.com
 *
 * http://personal.ph.surrey.ac.uk/~ri0005/binary_c.html
 * https://gitlab.eps.surrey.ac.uk/ri0005/binary_c
 * https://groups.google.com/forum/#!forum/binary_c-nucsyn-announce
 * https://groups.google.com/forum/#!forum/binary_c-nucsyn-devel
 * https://twitter.com/binary_c_code
 * https://www.facebook.com/groups/149489915089142/
 *
 * Please see the files README, LICENCE and CHANGES
 *
 * This file contains prototypes for the nucsyn library 
 */

#ifndef NUCSYN_PROTOTYPES_H
#define NUCSYN_PROTOTYPES_H
#include "nucsyn.h"
#include "../binary_c_structures.h"

/* 
 * nucsyn_sn_yield is called even if NUCSYN is not defined, it is
 * an exceptional function :)
 */
void nucsyn_sn_yield(struct stardata_t * RESTRICT const stardata,
                     struct star_t * exploder,
                     struct star_t * pre_explosion_star,
                     double dm,                    
		     double menv,
		     double mcore,
                     Stellar_type pre_explosion_stellar_type
    );

/* all other functions require NUCSYN to be defined */
#ifdef NUCSYN
void nucsyn_log(const double t,
                const double m,
                const double mc,
		double r,
                const double rc,
                const double lum,
                struct star_t *star,
                struct stardata_t *stardata,
		double lambda,
                Abundance * RESTRICT Xabund,
                const double tpagb_runtime);
void nucsyn_short_log(struct stardata_t * RESTRICT const stardata);
void nucsyn_long_log(struct stardata_t * RESTRICT const stardata);

void nucsyn_hbb(const double temp,
                const double rho,
                const double thisdt,
                Abundance * RESTRICT Xhbb,
		struct star_t *star,
		struct stardata_t *  stardata);
void nucsyn_set_nuc_masses(Nuclear_mass * RESTRICT mnuc,
			   Nuclear_mass * RESTRICT imnuc,
			   Nuclear_mass * RESTRICT mnuc_anu,
                           Nuclear_mass * RESTRICT imnuc_anu,
			   Molecular_weight * RESTRICT molweight,
                           double * RESTRICT ZonA,
                           Atomic_number * RESTRICT atomic_number,
			   Nucleon_number * RESTRICT nucleon_number);

double Pure_function nucsyn_molecular_weight(const Abundance * RESTRICT X,
                                             const struct stardata_t * RESTRICT const stardata,
                                             const double ionisation_fraction);
double Pure_function nucsyn_effective_molecular_weight(const Abundance * const RESTRICT X,
                                                       const Stellar_type stellar_type,
                                                       const double * RESTRICT const molweights);

void Xmult(Abundance * RESTRICT X,
           const double f);
void X_to_N(const Nuclear_mass * RESTRICT imnuc,
            const double dens,
            Number_density * RESTRICT N,
            const Abundance * RESTRICT X,
            const Isotope num_species);
void N_to_X(const Nuclear_mass * RESTRICT mnuc,
            const double dens,
	    const Number_density * RESTRICT N,
            Abundance * RESTRICT X,
            const Isotope num_species);

double nucsyn_third_dredge_up(struct star_t *const newstar,
                              struct stardata_t *const stardata,
                              const double tagb,
                              double * RESTRICT const  r,
                              double * RESTRICT const  rc,
                              double * RESTRICT const  mc,
                              double * RESTRICT const  mt,
                              double * RESTRICT const  age,
                              double * RESTRICT const  lum,
                              double * RESTRICT const  mass,
                              const int dntp,
                              const double Thbbmax,
                              const double rhohbb,
                              const double fmdupburn,
                              const double menv_1tp,
                              const double tpagb_runtime);
			
#ifndef NUCSYN_CALIBRATION_LOG
Constant_function
#endif
double nucsyn_hbbtmax( double menv_1tp,
                       double mc_1tp,
                       const Abundance Z);

void nucsyn_set_sigmav(struct stardata_t * const stardata,
                       struct store_t * const store,
                       const double temp, /* temperature */
                       Reaction_rate * const sigmav /* reaction rates */
#ifdef NUCLEAR_REACTION_RATE_MULTIPLIERS
		       ,const double * RESTRICT const multipliers /* reaction rate multipliers */
#endif
#ifdef RATES_OF_AMANDA_NE22PG_FIX
		       , const Boolean ne22pg_cf88
#endif
    );

void nucsyn_mix_shells(const double m1,
                       Abundance * const RESTRICT X1,
                       const double m2,
                       Abundance * const RESTRICT X2);

Constant_function double nucsyn_lamaxf(const double m,
				const Abundance z,
				const struct star_t *star,
				const struct stardata_t *stardata
    );


void nucsyn_tpagb(double * RESTRICT const  mass,
                  double * RESTRICT const  age,
                  double * RESTRICT const  mc,
                  double * RESTRICT const  mt,
                  double * RESTRICT const  lum,
                  double * RESTRICT const  r,
                  double * RESTRICT const  rc,
                  struct star_t * const newstar,
                  struct stardata_t * const stardata ,
                  double * const GB);

void nucsyn_set_1st_dup_abunds(Abundance * RESTRICT const X,
			       double m, 
			       Abundance z,
			       const Stellar_type stellar_type,
			       struct star_t *const star,
			       struct stardata_t *const stardata,
                               Boolean asevent);

void nucsyn_set_post_2nd_dup_abunds(Abundance * RESTRICT const X,
                                    const double m, 
                                    const Abundance z,
                                    const double mcbagb,
				    struct star_t *const star,
				    struct stardata_t *RESTRICT const stardata);




Constant_function double nucsyn_mhbbmin(const Abundance  z);
void nucsyn_dilute_shell(const double m1,
                         Abundance * RESTRICT const X1,
                         const double m2,
                         const Abundance * const RESTRICT X2);

void nucsyn_dilute_shell_to(const double m1,
                            const Abundance * const RESTRICT X1,
                            const double m2,
                            const Abundance * const RESTRICT X2,
                            Abundance * RESTRICT X3_to_be);

Constant_function double nucsyn_wgcoremasslum(const double mc,
                                       const double mass,
                                       const double mc_1tp,
                                       const Abundance z,
                                       const double alpha,
                                       Boolean hbbB);

#ifdef NUCSYN_ANAL_BURN
void nucsyn_anal_CNO_burn(double * RESTRICT t,
                          Number_density * RESTRICT Nin,
                          double dtseconds,
                          Boolean eqonly); 
int nucsyn_anal_NeNa_burn(double * RESTRICT t,
			  Number_density * RESTRICT Nin,
			  const double dtseconds,
			  const Boolean eqonly);
int nucsyn_anal_NeNa_burn1(struct stardata_t * stardata,
                           double * RESTRICT t,
                           Number_density * RESTRICT Nin,
			   const double dtseconds);

void nucsyn_anal_MgAl_burn(struct stardata_t * stardata,
                           double * RESTRICT t,
                           Number_density * RESTRICT Nin,
                           const double dtseconds);
#endif // NUCSYN_ANAL_BURN

                               
void nucsyn_set_third_dredgeup_abunds(Abundance * RESTRICT const dup_material,
                                      struct star_t * const newstar,
                                      const Abundance z,
				      struct stardata_t * const stardata,
				      const double mc);



double nucsyn_set_hbb_conditions(struct star_t * const newstar,
                                 struct stardata_t * RESTRICT const stardata,
                                 const double m0,
                                 const double menv);
Constant_function double nucsyn_tpagb_rhomax(const double m0,
                                      const Abundance z);

void nucsyn_init_first_pulse(struct star_t * const star,
                             struct stardata_t * const stardata,
                             const double alpha);

void Hot_function nucsyn_calc_yields(struct stardata_t * RESTRICT const stardata,
                                     struct star_t * const star,
                                     const double dmlose, /* mass lost as wind/otherwise */
                                     const Abundance * const Xlose, /* abundance of this "wind" */
                                     const double dmacc, /* mass gained by accretion */
                                     const Abundance * const Xacc, /* abundance of this accretion */
                                     const Star_number starnum,
                                     const int final,
                                     const Yield_source source
    );




void nucsyn_set_tpagb_free_parameters(double m,
                                      Abundance z,
				      struct star_t * const star,
				      struct stardata_t * const stardata);

void Hot_function nucsyn_update_abundances(struct stardata_t * RESTRICT const stardata);

double Pure_function nucsyn_totalX(struct stardata_t * const stardata,
                                   const Abundance * RESTRICT X,
                                   const Isotope Xmin,
                                   const Isotope Xmax) Nonnull_some_arguments(1,2);

void nucsyn_binary_yield(struct stardata_t * RESTRICT const stardata,
			 const Boolean final);


void nucsyn_burn(const double T,
		 const double dt,
		 Number_density * RESTRICT const N,
		 struct star_t * const star,
		 struct stardata_t * const stardata);

void nucsyn_set_WD_abunds(struct stardata_t * const stardata,
                          Abundance * RESTRICT const Xenv,
                          const Stellar_type stellar_type);

void Nonnull_some_arguments(1) nucsyn_set_remnant_abunds(Abundance * RESTRICT const Xenv);

double Pure_function Nonnull_some_arguments(1,2) XXsum(const struct stardata_t * const stardata,
                                                       const Abundance * RESTRICT const X);

#ifdef NUCSYN_THERMOHALINE_MIXING_TIME
Constant_function double nucsyn_thermohaline_mixing_time(const double mu0,
                                                         const double mu1,
                                                         const double dmacc,
                                                         const double m,
                                                         const double l,
                                                         const double r);
#endif

void nucsyn_set_nova_abunds(struct stardata_t * const stardata,
                            const struct star_t * const star,
                            Abundance * RESTRICT const Xdonor,
                            Abundance * RESTRICT const Xnova);
			
void nucsyn_remove_dm_from_surface(
#if defined DEBUG && DEBUG > 0
    struct stardata_t * RESTRICT const stardata,
#endif
    struct star_t * RESTRICT const star, 
    const double dm,
    Abundance ** RESTRICT const X,
    Boolean * RESTRICT const allocated);
				   


double Pure_function nucsyn_choose_wind_mixing_factor(const struct stardata_t * RESTRICT const stardata,
                                                      const Star_number k /* Number of ACCRETOR star */);


void nucsyn_s_process(Abundance * RESTRICT const dup_material, 
                      struct star_t * const newstar,
		      Abundance z,
		      struct stardata_t * const stardata);


void nucsyn_WR(struct star_t * const star,
	       struct stardata_t * RESTRICT const stardata,
	       const double stellar_mass);

void nucsyn_WR_table(struct star_t * const star,
                     struct stardata_t * RESTRICT const stardata,
                     const double stellar_mass);

void nucsyn_WR_table_lookup(struct stardata_t * RESTRICT const stardata,
                            double * RESTRICT const x,
			    Abundance * RESTRICT const r,
			    const int wr_wind);
void nucsyn_WR_RS_table_lookup(struct stardata_t * RESTRICT const stardata,
                               double * RESTRICT const x,
			       Abundance * RESTRICT const r);

WR_type Pure_function nucsyn_WR_type(const struct star_t * RESTRICT const star);

#ifdef NUCSYN_J_LOG
void nucsyn_j_log(struct stardata_t * RESTRICT const stardata);
#endif

void nucsyn_radioactive_decay(struct stardata_t * RESTRICT const stardata);

Abundance Pure_function * nucsyn_observed_surface_abundances(struct star_t * RESTRICT star);

void nucsyn_mix_accretion_layer_and_envelope(struct stardata_t * const stardata,
                                             struct star_t * RESTRICT const star,
                                             double menv);

void nucsyn_merge_NSs(const struct stardata_t * RESTRICT const stardata);

Moles Pure_function nucsyn_mole_fraction(const Isotope i, /* the species */
                                         const Abundance * RESTRICT const X, /* mass fractions */
                                         const Nuclear_mass * RESTRICT const mnuc /* nuclear masses */);
Abundance_ratio Pure_function nucsyn_square_bracket(const Abundance * RESTRICT const X,
                                                    const Abundance * RESTRICT const Xsolar,
                                                    const Isotope top_isotope,
                                                    const Isotope bot_isotope);

Abundance_ratio Pure_function nucsyn_square_multibracket(const Abundance * RESTRICT const X,
                                                         const Abundance * RESTRICT const Xsolar,
                                                         const Isotope top_isotopes[],
                                                         const int n_top,
                                                         const Isotope bottom_isotope[],
                                                         const int n_bottom);

void nucsyn_initial_abundances(Abundance * X,
			       Abundance z,
                               struct stardata_t * RESTRICT const stardata,
			       Abundance_mix mix);

void nucsyn_planetary_nebulae(const struct stardata_t * RESTRICT const stardata,
			      const Boolean PNetype,
			      const Star_number starnum,
			      const double dmass,
			      const Abundance *  X,
			      const double vexp);

void nucsyn_set_abunds_array(Abundance * RESTRICT const array,
			     const Abundance c);

void nucsyn_ensemble_log(struct stardata_t * RESTRICT const stardata);
		   

#ifdef NUCSYN_CEMP_LOGGING
void nucsyn_cemp_log(struct stardata_t * RESTRICT const stardata);
#endif

void nucsyn_lowz_yields(Abundance * RESTRICT const X,
			double mass);

/* Functions used to add or calculate supernova yields */
void nucsyn_sn_iwamoto_1999_DD2(Abundance * RESTRICT const X);
void nucsyn_sn_livne_arnett_1995(Abundance * RESTRICT const X,
                                 const double mcore,
                                 const double menv);
void nucsyn_sn_woosley_taam_weaver_1986(Abundance * RESTRICT const X );

double nucsyn_sn_core_collapse(Abundance * const X,
                               Abundance * const Xenv,
                               const double mco,
                               const double menv,
                               double dm,
                               Abundance z,
                               Yield_source * const src_id,
                               struct stardata_t * const stardata,
                               const struct star_t * const pre_exploder_star,
                               struct star_t * const exploder);

void nucsyn_sn_woosley_weaver_1995(struct stardata_t * const  stardata,
                                   Abundance * const X,
                                   double mco,
                                   Abundance z);

void nucsyn_sn_chieffi_limongi_2004(struct stardata_t * const stardata,
                                    Abundance * const X,
				    const double mco,
				    const Abundance z,
				    const double mcut,
				    struct star_t * const star);

void nucsyn_r_process(Abundance * RESTRICT const Xr,
		      struct stardata_t * RESTRICT const stardata);
void nucsyn_r_process_Simmerer2004(Abundance * RESTRICT const Xr,
				   struct stardata_t * RESTRICT const stardata);
void nucsyn_r_process_Arlandini1999(Abundance * RESTRICT const Xr,
				    struct stardata_t * RESTRICT const stardata);

#if defined NUCSYN_NETOWRK_HOTCNO &&            \
    defined NUCSYN_NETWORK_COLDCNO
void nucsyn_network_test(struct stardata_t * RESTRICT const stardata);
#endif

double nucsyn_network_burn(double * RESTRICT const Nin,
                           const double * RESTRICT const sigmav,
			   const double maxt,
			   double (*burnfunc)(double * RESTRICT const Nin,
                                              const double * RESTRICT const sigmav,
                                              const double h),
			   void (*logfunc)(struct stardata_t * const stardata,
                                           const double * RESTRICT const Nin,
                                           const double t),
			   const double dtguess,
			   struct stardata_t * RESTRICT const stardata);

/* nuclear burning functions */

#ifdef NUCSYN_NETWORK_PP
double nucsyn_burn_pp (double * RESTRICT const Nin,
                       const double * RESTRICT const sigmav,
                       const double h);
#endif
#ifdef NUCSYN_NETWORK_PPfast
double nucsyn_burn_ppfast (double * RESTRICT const Nin,
                           const double * RESTRICT const sigmav,
                           const double h);
#endif

#ifdef NUCSYN_NETWORK_COLDCNO
double nucsyn_burn_coldCNO (double * RESTRICT const Nin,
                            const double * RESTRICT const sigmav,
                            const double h);
#endif
#ifdef NUCSYN_NETWORK_HOTCNO
double nucsyn_burn_hotCNO (double * RESTRICT const Nin,
                           const double * RESTRICT const sigmav,
                           const double h);
#endif

#ifdef NUCSYN_NETWORK_NeNa
double nucsyn_burn_NeNa (double * RESTRICT Nin,
                         double * RESTRICT sigmav,
                         const double h);
#endif
#ifdef NUCSYN_NETWORK_NeNaMgAlnoleak
double nucsyn_burn_NeNaMgAlnoleak (double * RESTRICT Nin,
                                   double * RESTRICT sigmav,
                                   const double h);
#endif
#ifdef NUCSYN_NETWORK_NeNaMgAl
double nucsyn_burn_NeNaMgAl (double * RESTRICT const Nin,
                             const double * RESTRICT const sigmav,
                             const double h);
#endif



double Constant_function nucsyn_CN_timescale(const double t9,
                                             const double NH1,
                                             const double f);

void nucsyn_thermohaline_mix(struct stardata_t * const stardata,
			     const Star_number k, // accreting star number
                             Abundance * const Xacc, // accretion abundances
			     const double dmacc, // mass accreted
			     const Abundance * const Xenv, // envelope abundances: pre mix
			     Abundance * const nXenv // envelope abundances: post mix (written)			    
    );

void nucsyn_set_sigmav_from_table(struct stardata_t * const stardata,
                                  struct store_t * const store,
				  const double t9, // T(K)/1e9
				  double * RESTRICT const sigmav // target for the interpolation result
    );

void print_sigmav_table(struct store_t * RESTRICT store);

double Pure_function nucsyn_free_electron_density(const double * RESTRICT const N,// number density
                                                  const int * RESTRICT const Z); // atomic numbers

void nucsyn_burning_cycles(Abundance * RESTRICT const N,
			   const Reaction_rate * RESTRICT const sigmav,
			   const double dt,
			   struct stardata_t * RESTRICT const stardata);

void nucsyn_reset_reaction_rates(struct preferences_t * RESTRICT const preferences);

void nucsyn_setup_default_mass_shells(struct stardata_t * RESTRICT const stardata);

void nucsyn_set_nuclear_burning_timestep(struct stardata_t * RESTRICT const stardata,
					 struct star_t * RESTRICT star);

void nucsyn_main_sequence_convective_envelope_size_table(struct stardata_t * stardata);


double Constant_function nucsyn_main_sequence_convective_envelope_size(double x,
                                                                       double y,
                                                                       double z);



double Pure_function nucsyn_envelope_mass(const struct star_t * RESTRICT const star);

			  
void nucsyn_first_dup_interpolation(struct stardata_t * const stardata,
				    double m,
				    const double z,
				    double * RESTRICT r);

void nucsyn_first_dup_apply(const double m,
                            struct stardata_t * const stardata,
			    Abundance * RESTRICT const X,
			    double * RESTRICT const r);
double nucsyn_match_chi_squared(const struct stardata_t * RESTRICT const stardata,
                                const struct star_t * RESTRICT const star);
void nucsyn_match_set_abundance(const Isotope i,
				const Abundance abundance,
				const Abundance error,
				struct stardata_t * RESTRICT const stardata
    );

void nucsyn_sn_electron_capture(Abundance * RESTRICT const X);
void nucsyn_huge_pulse(struct star_t * RESTRICT const star,
		       struct stardata_t * RESTRICT const stardata);

void nucsyn_check_abundance_array(struct stardata_t * RESTRICT const stardata,
                                  const Abundance * RESTRICT const X,
                                  const char * RESTRICT const s);
#ifdef LITHIUM_TABLES
void nucsyn_lithium(struct star_t * RESTRICT const star,
		    struct stardata_t * RESTRICT const stardata);

#endif


Abundance nucsyn_elemental_abundance(const char * RESTRICT const element,
				     const Abundance * RESTRICT const X,
				     struct stardata_t * const stardata,
                                     struct store_t * const store);

Abundance_ratio nucsyn_elemental_square_bracket(const char * RESTRICT const top,
						const char * RESTRICT const bottom,
						const Abundance * RESTRICT const X,
						const Abundance * RESTRICT const Xsolar,
						struct stardata_t * RESTRICT const stardata);

Atomic_number nucsyn_element_to_atomic_number(struct store_t * RESTRICT const store,
                                              const char * RESTRICT const element);

void nucsyn_TZ_surface_abundances(struct star_t * RESTRICT const star);


void nucsyn_strip_and_mix(struct stardata_t * const stardata,
                          struct star_t * const star);
void mix_down_to_core(struct star_t * RESTRICT const star);
double mshells(struct star_t * RESTRICT const star);
void sort_by_molecular_weight(struct star_t * RESTRICT const star,
                              struct stardata_t * RESTRICT const stardata);
void nucsyn_mix_stars(struct stardata_t * RESTRICT const stardata,
                      const double mlost,
                      const double m3);
Boolean can_use_strip_and_mix(struct stardata_t * const stardata,
                              struct star_t * RESTRICT const star);
void strip_and_mix_remove_mass(struct star_t * RESTRICT const star,
                               double dmlose);
void halve_shells(struct star_t * RESTRICT const star);

void nucsyn_build_store_contents(struct store_t * RESTRICT const store);
void nucsyn_free_store_contents(struct store_t * RESTRICT const store);

double Pure_function nucsyn_recombination_energy_per_gram(struct stardata_t * RESTRICT const stardata,
                                                          Abundance * RESTRICT const X);

void nucsyn_check_for_second_dredge_up(struct stardata_t * const stardata,
                                       struct star_t * const newstar,
                                       const double mcbagb);


void nucsyn_renormalize_abundance(struct stardata_t * const stardata,
                                  Abundance * RESTRICT const X);



void nucsyn_sn_Seitenzahl2013(struct stardata_t * RESTRICT const stardata,
                              Abundance * RESTRICT const X);
void nucsyn_sn_Seitenzahl2013_automatic(struct stardata_t * RESTRICT const stardata,
                                        Abundance * RESTRICT const X);

void nucsyn_angelou_lithium(struct stardata_t * const stardata,
                            struct star_t * const star);

Yield_source Pure_function nucsyn_sn_source(struct star_t * RESTRICT const exploder);

Abundance nucsyn_elemental_abundance_by_number(const char * RESTRICT const element, 
                                               const Abundance * RESTRICT const X,
                                               struct stardata_t * const stardata,
                                               struct store_t * const store,
                                               const double density);



void nucsyn_make_icache(struct store_t * const store);
void nucsyn_free_icache(Isotope ** icache);

void nucsyn_add_mass_to_surface(const Star_number k,
                                const double dm,
                                const Abundance * RESTRICT const X,
                                double * RESTRICT const ndmacc,
                                const Abundance * const * RESTRICT const Xacc,
                                Abundance ** RESTRICT nXacc);

void nucsyn_remove_mass_from_surface(
#if defined DEBUG && DEBUG > 0
    struct stardata_t * stardata,
#endif
    struct star_t * RESTRICT star,
    const double dm,
    Abundance ** RESTRICT X,
    double * RESTRICT ndmacc,
    Boolean *RESTRICT allocated);



Boolean nucsyn_thermohaline_unstable(struct stardata_t * const stardata,
                                     struct star_t * const star);

void nucsyn_thermohaline_mix_star(struct stardata_t * const stardata,
                                  struct star_t * const star);
#endif /* NUCSYN */
#endif /* NUCSYN_PROTOYPES */


