#pragma once
#ifndef NUCSYN_ISOTOPES_H
#define NUCSYN_ISOTOPES_H

#include "../binary_c_parameters.h"

#ifdef NUCSYN


/*
 * The isotopes used in binary_c are stored in an array
 *
 * This array is an array of double precision floating point
 * numbers ('doubles') and contains ISOTOPE_ARRAY_SIZE elements
 *
 * The elements are individually referred to using the macros defined below as
 * XH1, XHe4, XC12 etc. but you should use the macros Set_isotopic_property
 * and Increment_isotopic_property to set e.g. abundances.
 *
 * NB all functions should access the isotopes via the X... macros defined
 * below, unless looping over all the isotopes (e.g. to mix things) in which
 * case you don't care which isotope is which, just that it's there. In this
 * way direct isotope access is removed and the system is totally flexible :
 * assuming you keep this file up to date. It is also as fast as you'd like it,
 * because all the macros are compiled in. Just be careful if you change
 * something! You'll have to recompile all objects/libraries which use these
 * macro definitions.
 *
 * If you change the size of arrays, you'll need to do a full rebuild.
 */


/*
 * If ALL_ISOTOPES is defined, then binary_c uses > 400 isotopes.
 * If it is not, a reduced set of isotopes is used instead, which
 * makes the code a lot faster.
 *
 * NB Changing this will require a full rebuild. 
 */

#define ALL_ISOTOPES
//#undef ALL_ISOTOPES

/* load in isotope list */
#include "nucsyn_isotope_list.h"

/*******************************************************************/
/* REMEMBER to set a nuclear mass for each of the isotopes YOU ADD */
/*******************************************************************/

/* Useful macros */
#define CNO(A) (A[XC12]+A[XC13]+A[XN13]+A[XN14]+A[XN15]+A[XO15]+A[XO16]+A[XO17])
#define Clear_CNO_abundances(A) A[XC12]=0.0;A[XC13]=0.0;A[XN13]=0.0;A[XN14]=0.0;A[XN15]=0.0;A[XO15]=0.0;A[XO16]=0.0;A[XO17]=0.0;
#define CNO_abundance_by_number(A) (A[XC12]/12.0+A[XC13]/13.0+A[XN14]/14.0+A[XN15]/15.0+A[XO16]/16.0+A[XO17]/17.0+A[XO18]/18.0)
#define All_CNO_to_N14(A) A[XN14]=CNO_abundance_by_number(A); A[XC12]=A[XC13]=A[XN15]=A[XO16]=A[XO17]=A[XO18]=0.0;
#define All_hydrogen_to_helium(A) A[XHe4]+=A[XH1]; A[XH1]=0.0; 
#define Total_carbon_abundance(A) (A[XC12]/12.0+A[XC13]/13.0)
#define Total_oxygen_abundance(A) (A[XO16]/16.0+A[XO17]/17.0+A[XO18]/18.0)
#define Carbon_oxygen_ratio(A) (Is_not_zero(Total_oxygen_abundance(A)) ? (Total_carbon_abundance(A)/Total_oxygen_abundance(A)) : 0.0)

#define Burn_CNO_completely(A) All_hydrogen_to_helium(A);All_CNO_to_N14(A)

#define ISOTOPE_MEMSIZE (sizeof(double)*(ISOTOPE_ARRAY_SIZE))

// S-process macros
#define S_process_hs_abundance(X) (0.5*(nucsyn_square_bracket((X),XY,XFe) +          \
                      nucsyn_square_bracket((X),XZr,XFe)))
#define S_process_ls_abundance(X) (0.2*(nucsyn_square_bracket((X),XBa,XFe) +         \
                      nucsyn_square_bracket((X),XLa,XFe) +         \
                      nucsyn_square_bracket((X),XCe,XFe) +         \
                      nucsyn_square_bracket((X),XNd,XFe) +         \
                      nucsyn_square_bracket((X),XSm,XFe)))
#define S_process_hs_minus_ls(X) (S_process_hs_abundance(X) - S_process_ls_abundance(X))
#define S_process_overabundance(X) ((X[XNd] + X[XY])/\
                            (0.02*1.61434282262822e-07 +0.02*5.58660876791233e-07))

// below this abundance, an isotope may be considered to have zero abundance
#define ABUND_ZERO 1e-30

/*
 * Macro to check if an isotope number A is valid
 */
#define Isotope_is_valid(A) (likely((A)<(ISOTOPE_ARRAY_SIZE)))

/*
 * Safe macros to set/increment abundances, nuclear masses etc.
 * The property of the isotope to be set (or incremented) is X.
 * The isotope is A and the value (or change of value) is B.
 */
#define Set_isotopic_property(X,A,B) if(Isotope_is_valid(A)) *((X)+(A))=(B);
#define Increment_isotopic_property(X,A,B) if(Isotope_is_valid(A)) *((X)+(A))+=(B);

/*
 * Unordered isotope loop
 */
#define Isotope_loop_sign -1
#define Isotope_loop(I) for((I)=(ISOTOPE_ARRAY_SIZE);likely((I)=(I)+Isotope_loop_sign);)


/*
 * Ordered isotope loop (probably slower)
 */
#define Ordered_isotope_loop(I) for((I)=0; likely((I)<(ISOTOPE_ARRAY_SIZE)); (I)++)

/*
 * Make a new isotope array
 */
#define New_isotope_array Malloc((ISOTOPE_MEMSIZE))
#define New_clear_isotope_array Calloc(ISOTOPE_ARRAY_SIZE,      \
                                       sizeof(Abundance))


/*
 * Clear an isotope array
 */
#define Clear_isotope_array(X) memset((Abundance*)(X),          \
                                      0,                        \
                                      (ISOTOPE_MEMSIZE))

/*
 * Copy abundances : the use of memcpy, which should be fastest,
 * relies on the regions NOT overlapping.
 */
#define Copy_abundances(FROM,TO) memcpy((Abundance*)(TO),       \
                                        (Abundance*)(FROM),     \
                                        (ISOTOPE_MEMSIZE))

#define New_isotope_array_from(X) Copy_abundances((X),                  \
                                                  (New_isotope_array)) 

#endif /* NUCSYN */
#endif /* NUCSYN_ISOTOPES_H */
