#include "../binary_c.h"

#if defined NUCSYN && defined NUCSYN_GCE

#include "nucsyn_yield_macros.h"

void nucsyn_ensemble_log(struct stardata_t * RESTRICT const stardata)
{
    /* 
     * "Ensemble" stellar population data output.
     * activated by NUCSYN_STELLAR_POPULATIONS_ENSEMBLE
     * for yield_vs_time.pl runs.
     *
     * If NUCSYN_GCE_DISABLE_ENSEMBLE_OUTPUT is defined, 
     * everything is performed *except* the output step,
     * so integrals are correct, etc., but yields_vs_time can
     * only work correctly for chemical yields.
     */
#ifdef NUCSYN_STELLAR_POPULATIONS_ENSEMBLE
    Dprint("Enter GCE log\n");

    double ensemble_weight[NUCSYN_ENSEMBLE_N];
    double teff,l,mu;

#define ensemble_store (stardata->model.ensemble_store)
#define lastensembleouttime (stardata->model.lastensembleouttime)

    Ensemble_index k;
    Star_number i;
    const static Ensemble_index merger_matrix[NUMBER_OF_STELLAR_TYPES-1][NUMBER_OF_STELLAR_TYPES-1] = 
        NUCSYN_ENSEMBLE_MERGER_MATRIX;
    Boolean reset_time=FALSE;

#endif //  NUCSYN_STELLAR_POPULATIONS_ENSEMBLE

    /*
     * Linear timestep 
     */
    double lineardt;
#ifdef NUCSYN_STELLAR_POPULATIONS_ENSEMBLE_SPARSE
    double dt_since_last; 
#endif
    double linear_dt_since_last;    
    struct stardata_t * previous_stardata = stardata->previous_stardata;

    /* 
     * dt_since_last is the time between this output and the
     * previous : it is usually the yield_dt but may be less
     * if we have reached the maximum evolution time (in which 
     * case final==TRUE and we're being called to flush all the 
     * arrays).
     *
     * NB this is the difference between log times if logs are set 
     * to be on.
     *
     * If you want the linear time difference, use linear_dt_since_last.
     */

    if(stardata->preferences->yields_logtimes == TRUE)
    {
        dt_since_last = log10(stardata->model.time) - stardata->model.fixed_timesteps[FIXED_TIMESTEP_YIELDS].previous_trigger;
        linear_dt_since_last = stardata->model.time - pow(10.0,stardata->model.fixed_timesteps[FIXED_TIMESTEP_YIELDS].previous_trigger);
    }
    else
    {
        dt_since_last = stardata->model.time - stardata->model.fixed_timesteps[FIXED_TIMESTEP_YIELDS].previous_trigger;
        linear_dt_since_last = dt_since_last;
    }
    
    /*
     * Make the ensemble type array
     */
    Ensemble_type * ensemble_type;
    if(stardata->tmpstore->ensemble_type==NULL)
    {
        ensemble_type = stardata->tmpstore->ensemble_type
            = Calloc(NUCSYN_ENSEMBLE_N,sizeof(Ensemble_type));
               
        /* default all to type "normal" */
        for(i=0;i<NUCSYN_ENSEMBLE_N;i++)
        {
            ensemble_type[i] = ENSEMBLE_TYPE_NORMAL;
        }
        
        /* some ensemble variables are rates : treat these differently */

        /* supernovae */
        ensemble_type[NUCSYN_ENSEMBLE_SN_IA_He] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_SN_IA_ELD] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_SN_IA_CHAND] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_SN_AIC] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_SN_IA_He_Coal] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_SN_IA_CHAND_Coal] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_SN_IA_Hybrid_HeCOWD] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_SN_IA_Hybrid_HeCOWD_subluminous] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_SN_IBC] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_SN_II] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_SN_IIa] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_GRB_NS_NS] = ENSEMBLE_TYPE_RATE;    
        ensemble_type[NUCSYN_ENSEMBLE_GRB_COLLAPSAR] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_SN_TZ] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_SN_AIC_BH] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_SN_BH_NS] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_SN_BH_BH] = ENSEMBLE_TYPE_RATE;

        /* test rate */
        ensemble_type[NUCSYN_ENSEMBLE_TEST_RATE] = ENSEMBLE_TYPE_RATE;

        /* common envelope rates */
        for(i=NUCSYN_ENSEMBLE_COMENV_HG_MERGER;
            i<NUCSYN_ENSEMBLE_COMENV_DETACHED+1;
            i++)
        {
            ensemble_type[i] = ENSEMBLE_TYPE_RATE;
        }

        /* merger rates */
        for(i=NUCSYN_ENSEMBLE_MERGER_INTO_LMMS;
            i<NUCSYN_ENSEMBLE_MERGER_BH_BH+1;
            i++)
        {
            ensemble_type[i] = ENSEMBLE_TYPE_RATE;
        }

        /* special case : hybrid COWD mergers */
        for(i=NUCSYN_ENSEMBLE_MERGER_LMMS_Hybrid_HeCOWD;
            i<NUCSYN_ENSEMBLE_MERGER_Hybrid_HeCOWD_Hybrid_HeCOWD+1;
            i++)
        {
            ensemble_type[i] = ENSEMBLE_TYPE_RATE;
        }
        
        for(i=NUCSYN_ENSEMBLE_PERETS_HeWD_COWD_LT0p8;
            i<=NUCSYN_ENSEMBLE_PERETS_HeWD_LT0p25_ONeWD;
            i++)
        {
            ensemble_type[i] = ENSEMBLE_TYPE_RATE;         
        }
        ensemble_type[NUCSYN_ENSEMBLE_MERGER_HeWD_gt_COWD] = ENSEMBLE_TYPE_RATE;
        
        /*
         * Various luminosities and momentum fluxes should 
         * be considered as rates
         */
        ensemble_type[NUCSYN_ENSEMBLE_WIND_KINETIC_POWER] = ENSEMBLE_TYPE_RATE; 
        ensemble_type[NUCSYN_ENSEMBLE_WIND_RSG_KINETIC_POWER] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_WIND_ISG_KINETIC_POWER] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_WIND_BSG_KINETIC_POWER] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_WIND_WOLF_RAYET_KINETIC_POWER] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_WIND_AGB_KINETIC_POWER] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_SUPERNOVA_KINETIC_POWER] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_SUPERNOVA_CORE_COLLAPSE_KINETIC_POWER] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_SUPERNOVA_IA_KINETIC_POWER] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_NOVA_KINETIC_POWER] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_SUPERNOVA_LUMINOSITY] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_SUPERNOVA_CORE_COLLAPSE_LUMINOSITY] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_SUPERNOVA_IA_LUMINOSITY] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_NOVA_LUMINOSITY] = ENSEMBLE_TYPE_RATE; 
        ensemble_type[NUCSYN_ENSEMBLE_WIND_MOMENTUM] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_WIND_RSG_MOMENTUM] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_WIND_ISG_MOMENTUM] = ENSEMBLE_TYPE_RATE; 
        ensemble_type[NUCSYN_ENSEMBLE_WIND_BSG_MOMENTUM] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_WIND_WOLF_RAYET_MOMENTUM] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_WIND_AGB_MOMENTUM] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_SUPERNOVA_MOMENTUM] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_SUPERNOVA_CORE_COLLAPSE_MOMENTUM] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_SUPERNOVA_IA_MOMENTUM] = ENSEMBLE_TYPE_RATE;
        ensemble_type[NUCSYN_ENSEMBLE_NOVA_MOMENTUM] = ENSEMBLE_TYPE_RATE;

        /* Thorne-Zytkow formation rates as a function of binary mass */
        for(i=NUCSYN_ENSEMBLE_TZ_M1_LT8;
            i<NUCSYN_ENSEMBLE_TZ_M1_GT30+1;
            i++)
        {
            ensemble_type[i] = ENSEMBLE_TYPE_RATE;
        }

        Stellar_magnitude_index m;
        for(m=0; m<NUMBER_OF_STELLAR_MAGNITUDES; m++)
        {
            Ensemble_index n = NUCSYN_ENSEMBLE_MAGNITUDE_U + (Ensemble_index)m;      
            ensemble_type[n] = ENSEMBLE_TYPE_MAGNITUDE;
        }
    }
    else
    {
        ensemble_type = stardata->tmpstore->ensemble_type;
    }

    /*
     * Calculate time since last call that is after the start time
     */
    double prevtime =
        stardata->preferences->yields_logtimes == TRUE ?
        Max(stardata->preferences->yields_startlogtime,
            pow(10.0,stardata->model.fixed_timesteps[FIXED_TIMESTEP_YIELDS].previous_test)) :
        stardata->model.fixed_timesteps[FIXED_TIMESTEP_YIELDS].previous_test;

    lineardt = Max(0.0,stardata->model.time - prevtime);
        
    /* 
     * dt<0 can happen with the old BSE algorithm
     * but shouldn't with modern binary_c. If it happens,
     * hope that it's not for long and ignore it. If you
     * really care, use the new binary_c algorithm instead
     * (which is now the default).
     *
     * Note that dt == 0 is perfectly fine. This happens when there
     * is an event (e.g. a common envelope merging).
     */
    if(lineardt < -TINY) return;
    
#ifdef NUCSYN_STELLAR_POPULATIONS_ENSEMBLE
    if((stardata->preferences->yields_logtimes == TRUE &&
        stardata->model.time  < stardata->preferences->yields_startlogtime + TINY))
    {
        /*
         * When log time is used, we cannot log before a certain time.
         */
        lastensembleouttime =
            stardata->preferences->yields_logtimes==TRUE ?
            log10(stardata->preferences->yields_startlogtime) :
            0.0;
    }
    else
    {
        Dprint("Grand ensemble\n");
        Boolean born_binary = WAS_A_STAR(0) && WAS_A_STAR(1);    
    
        /*
         * A grand ensemble of things we can observe, all nicely pre-binned.
         */
        Starloop(i)
        {
            Dprint("Star %d",i);
            SETstar(i);
            int other = Other_star(i);
            struct star_t * other_star = &stardata->star[other];
            struct star_t * previous_star = &previous_stardata->star[i];
            struct star_t * previous_other_star = &previous_stardata->star[other];
        
            /*
             * First set all detections to false.
             */
            memset(ensemble_weight,0,sizeof(double)*NUCSYN_ENSEMBLE_N);

            /*
             * Test variables
             */
            if(star->starnum == ENSEMBLE_TEST_STAR)
            {
                /*
                 * There is one "star" at any time
                 */
                ensemble_weight[NUCSYN_ENSEMBLE_TEST_COUNT] = 1.0;             

                /*
                 * The test rate is 1/Myr, so in one timestep 
                 * we expect 1.0 * (linear timestep/Myr) events,
                 * i.e. dt * 1e-6.
                 */
                ensemble_weight[NUCSYN_ENSEMBLE_TEST_RATE] =
                    lineardt * 1e-6;
            }
        
            /*
             * Stars which form with M>0.02 are logged: everything
             * with a lower mass is a planet and we don't know what to do
             */
            if(star->pms_mass > ENSEMBLE_MINIMUM_PMS_MASS)
            {
                Dprint("HRD");

                /*
                 * calculate effective temperature, log L
                 */
                teff = 1000.0*pow((1130.0*star->luminosity/
                                   Pow2(star->radius)),
                                  0.25);
                teff=log10(teff);
                l=log10(star->luminosity);

                /*
                 * Save the total number of stars and the total luminosity in stars
                 */
                if(IS_A_STAR(i))
                {
                    ensemble_weight[NUCSYN_ENSEMBLE_NUMBER_OF_STARS]=1.0; 
                    ensemble_weight[NUCSYN_ENSEMBLE_LUMINOSITY_OF_STARS]=star->luminosity;
                    ensemble_weight[NUCSYN_ENSEMBLE_MASS_IN_STARS]=star->mass;
                }

                /*
                 * Binary/single statistics : only do this for index==0
                 * to avoid double-counting. We assume that for single 
                 * stars, the star is in star 0 and that star 1 is a 'planet'
                 * that can be ignored.
                 */
                if(i==0)
                { 
                    Dprint("Binary statistics");

                    if(NEITHER_STAR_MASSLESS)
                    {
                        ensemble_weight[Observable_binary ?
                                        NUCSYN_ENSEMBLE_NUMBER_OF_OBSERVABLE_BINARY_STARS :
                                        NUCSYN_ENSEMBLE_NUMBER_OF_OBSERVABLE_SINGLE_STARS]=1.0;
                    }
                    
                    if(born_binary==TRUE)
                    {
                        /*
                         * Only stars born binary can still be binary stars,
                         * mergers or disrupted binaries. 
                         */

                        if(IS_A_STAR(0) && IS_A_STAR(1))
                        {
                            /* both stars are observable as stars : 
                             * check the orbit to see if they are bound.
                             * NB this cannot be a merged star because
                             * were it, the companion would be a MASSLESS_REMNANT
                             * and IS_A_STAR would fail.
                             */
                            if(stardata->common.orbit.separation > TINY)
                            {
                                /* binary system : still bound */
                                ensemble_weight[NUCSYN_ENSEMBLE_NUMBER_OF_BINARY_STARS]=1.0;
                                ensemble_weight[NUCSYN_ENSEMBLE_LUMINOSITY_OF_BINARY_STARS]=stardata->star[0].luminosity+stardata->star[1].luminosity;
                                ensemble_weight[NUCSYN_ENSEMBLE_MASS_IN_BINARY_STARS]=stardata->star[0].mass+stardata->star[1].mass;
                            }
                            else
                            {
                                /* binary system but unbound : each star would look single */
                                ensemble_weight[NUCSYN_ENSEMBLE_NUMBER_OF_DISRUPTED_BINARIES]=1.0;
                                ensemble_weight[NUCSYN_ENSEMBLE_LUMINOSITY_OF_DISRUPTED_BINARIES]=stardata->star[0].luminosity+stardata->star[1].mass;
                                ensemble_weight[NUCSYN_ENSEMBLE_MASS_IN_DISRUPTED_BINARIES]=stardata->star[0].mass+stardata->star[1].mass;

                                ensemble_weight[NUCSYN_ENSEMBLE_NUMBER_OF_SINGLE_STARS]=2;
                                ensemble_weight[NUCSYN_ENSEMBLE_LUMINOSITY_OF_SINGLE_STARS]=stardata->star[0].luminosity+stardata->star[1].luminosity;
                                ensemble_weight[NUCSYN_ENSEMBLE_MASS_IN_SINGLE_STARS]=stardata->star[0].mass+stardata->star[1].mass;
                
                                /* both stars would be runaways */
                                ensemble_weight[NUCSYN_ENSEMBLE_NUMBER_OF_RUNAWAY_STARS]=2;
                                ensemble_weight[NUCSYN_ENSEMBLE_LUMINOSITY_OF_RUNAWAY_STARS]=stardata->star[0].luminosity+stardata->star[1].luminosity;
                                ensemble_weight[NUCSYN_ENSEMBLE_MASS_IN_RUNAWAY_STARS]=stardata->star[0].mass+stardata->star[1].mass;
                            }
                        }
                        /* one of the stars is a planet or massless remnant: find out which */
                        else if(IS_A_STAR(0))
                        {
                            /* star 1 is observable as a single star */
                            ensemble_weight[NUCSYN_ENSEMBLE_NUMBER_OF_SINGLE_STARS]=1.0;
                            ensemble_weight[NUCSYN_ENSEMBLE_LUMINOSITY_OF_SINGLE_STARS]=stardata->star[0].luminosity;
                            ensemble_weight[NUCSYN_ENSEMBLE_MASS_IN_SINGLE_STARS]=stardata->star[0].mass;

                            if(stardata->model.merged == TRUE || stardata->model.coalesce == TRUE)
                            {
                                /* it is a merged star */
                                ensemble_weight[NUCSYN_ENSEMBLE_NUMBER_OF_MERGED_BINARIES]=1.0;
                                ensemble_weight[NUCSYN_ENSEMBLE_LUMINOSITY_OF_MERGED_BINARIES]=stardata->star[0].luminosity;
                                ensemble_weight[NUCSYN_ENSEMBLE_MASS_IN_MERGED_BINARIES]=stardata->star[0].mass;
                            }
                        }
                        else if(IS_A_STAR(1))
                        {
                            /* star 2 is observable as a single star */
                            ensemble_weight[NUCSYN_ENSEMBLE_NUMBER_OF_SINGLE_STARS]=1.0;
                            ensemble_weight[NUCSYN_ENSEMBLE_LUMINOSITY_OF_SINGLE_STARS]=stardata->star[1].luminosity;
                            ensemble_weight[NUCSYN_ENSEMBLE_MASS_IN_SINGLE_STARS]=stardata->star[1].mass;

                            if(stardata->model.merged == TRUE)
                            {
                                /* it is a merged star */
                                ensemble_weight[NUCSYN_ENSEMBLE_NUMBER_OF_MERGED_BINARIES]=1.0;
                                ensemble_weight[NUCSYN_ENSEMBLE_LUMINOSITY_OF_MERGED_BINARIES]=stardata->star[1].luminosity;
                                ensemble_weight[NUCSYN_ENSEMBLE_MASS_IN_MERGED_BINARIES]=stardata->star[1].mass;
                            }
                        }
                    }
                    else
                    {
                        /* system was born single in star 1 : check if it's still a star */
                        if(IS_A_STAR(0))
                        {
                            ensemble_weight[NUCSYN_ENSEMBLE_NUMBER_OF_SINGLE_STARS]=1.0;
                            ensemble_weight[NUCSYN_ENSEMBLE_LUMINOSITY_OF_SINGLE_STARS]=stardata->star[0].luminosity;
                            ensemble_weight[NUCSYN_ENSEMBLE_MASS_IN_SINGLE_STARS]=stardata->star[0].mass;
                        }
                    }
                }

                /*
                 * Save the number of each stellar type and the luminosity in each stellar type 
                 */
                ensemble_weight[NUCSYN_ENSEMBLE_LMMS+2*star->stellar_type]=1.0;
                ensemble_weight[NUCSYN_ENSEMBLE_LMMS_LUM+2*star->stellar_type]=star->luminosity;

                /*
                 * Hybrid He-COWDs are a special case
                 */
                if(star->stellar_type == COWD && star->hybrid_HeCOWD == TRUE)
                {
                    ensemble_weight[NUCSYN_ENSEMBLE_Hybrid_HeCOWD] = 1.0;
                    ensemble_weight[NUCSYN_ENSEMBLE_Hybrid_HeCOWD_LUM] = star->luminosity;
                }

                Dprint("WR checks");
                /*
                 * Check for WR stars
                 */
                if(star->stellar_type<10)
                {
                    // avoid WDs, which would qualify as WO stars!

                    switch(nucsyn_WR_type(&(stardata->star[i])))
                    {
                    case WR_WNL:
                        // WNL star
                        ensemble_weight[NUCSYN_ENSEMBLE_WNL]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_WNL_LUM]=star->luminosity;
                        break;
                    case WR_WNE:
                        // WNE star
                        ensemble_weight[NUCSYN_ENSEMBLE_WNE]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_WNE_LUM]=star->luminosity;
                        break;
                    case WR_WC:
                        // WC star
                        ensemble_weight[NUCSYN_ENSEMBLE_WC]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_WC_LUM]=star->luminosity;
                        break;
                    case WR_WO:
                        // WO star
                        ensemble_weight[NUCSYN_ENSEMBLE_WO]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_WO_LUM]=star->luminosity;
                        break;
                    }
                }

                Dprint("B/RSG checks");
                /*
                 * Check for BSG or RSG
                 */
                if(star->Xenv[XH1]>0.4)
                {
                    if(l>4.9)
                    {
                        if(teff<3.66)
                        {
                            ensemble_weight[NUCSYN_ENSEMBLE_RSG]=1.0;
                            ensemble_weight[NUCSYN_ENSEMBLE_RSG_LUM]=star->luminosity;
                        }
                        else if(teff>3.9)
                        {
                            ensemble_weight[NUCSYN_ENSEMBLE_BSG]=1.0;
                            ensemble_weight[NUCSYN_ENSEMBLE_BSG_LUM]=star->luminosity;
                        }
                        else
                        {
                            ensemble_weight[NUCSYN_ENSEMBLE_ISG]=1.0;
                            ensemble_weight[NUCSYN_ENSEMBLE_ISG_LUM]=star->luminosity;
                        }
                    }

                    /* 
                     * RSGs for Ben Davies (Liverpool, 2016)
                     * L>10^3 and Teff<5000 (log10Teff < 3.69897)
                     */
                    if(l>3.0 && teff<3.69897)
                    {
                        ensemble_weight[NUCSYN_ENSEMBLE_DAVIES_RSG] = 1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_DAVIES_RSG_LUM] = star->luminosity;
                    }
                }

                Dprint("SN checks");
                /*
                 * Check for SNe: NB set to -1 so we do NOT multiply by dt
                 * when calculating populations
                 */ 

           
                if(star->SN_type!=-1 &&
                   star->SN_type!=SN_NONE)
                {
                    switch(star->SN_type)
                    {
                    case SN_IA_He:
                        ensemble_weight[NUCSYN_ENSEMBLE_SN_IA_He]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_KINETIC_POWER]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_IA_KINETIC_POWER]=
                            supernova_kinetic_energy(star);
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_MOMENTUM]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_IA_MOMENTUM]=
                            supernova_momentum(star);
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_LUMINOSITY]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_IA_LUMINOSITY]=
                            supernova_luminous_energy(star);
                        break;
                    case SN_IA_ELD:
                        ensemble_weight[NUCSYN_ENSEMBLE_SN_IA_ELD]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_KINETIC_POWER]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_IA_KINETIC_POWER]=
                            supernova_kinetic_energy(star);
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_MOMENTUM]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_IA_MOMENTUM]=
                            supernova_momentum(star);
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_LUMINOSITY]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_IA_LUMINOSITY]=
                            supernova_luminous_energy(star);
                        break;
                    case SN_IA_CHAND:
                        ensemble_weight[NUCSYN_ENSEMBLE_SN_IA_CHAND]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_KINETIC_POWER]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_IA_KINETIC_POWER]=
                            supernova_kinetic_energy(star);
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_MOMENTUM]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_IA_MOMENTUM]=
                            supernova_momentum(star);
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_LUMINOSITY]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_IA_LUMINOSITY]=
                            supernova_luminous_energy(star);
                        break;
                    case SN_IA_Hybrid_HeCOWD:
                        ensemble_weight[NUCSYN_ENSEMBLE_SN_IA_Hybrid_HeCOWD]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_KINETIC_POWER]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_IA_KINETIC_POWER]=
                            supernova_kinetic_energy(star);
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_MOMENTUM]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_IA_MOMENTUM]=
                            supernova_momentum(star);
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_LUMINOSITY]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_IA_LUMINOSITY]=
                            supernova_luminous_energy(star);
                        break;
                    case SN_IA_Hybrid_HeCOWD_subluminous:
                        ensemble_weight[NUCSYN_ENSEMBLE_SN_IA_Hybrid_HeCOWD_subluminous]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_KINETIC_POWER]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_IA_KINETIC_POWER]=
                            supernova_kinetic_energy(star);
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_MOMENTUM]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_IA_MOMENTUM]=
                            supernova_momentum(star);
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_LUMINOSITY]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_IA_LUMINOSITY]=
                            supernova_luminous_energy(star);
                        break;
                    case SN_AIC:
                        ensemble_weight[NUCSYN_ENSEMBLE_SN_AIC]=1.0;
                        break;
                    case SN_IA_He_Coal:
                        ensemble_weight[NUCSYN_ENSEMBLE_SN_IA_He_Coal]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_KINETIC_POWER]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_IA_KINETIC_POWER]=
                            supernova_kinetic_energy(star);
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_MOMENTUM]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_IA_MOMENTUM]=
                            supernova_momentum(star);
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_LUMINOSITY]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_IA_LUMINOSITY]=
                            supernova_luminous_energy(star);
                        break;
                    case SN_IA_CHAND_Coal:
                        ensemble_weight[NUCSYN_ENSEMBLE_SN_IA_CHAND_Coal]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_KINETIC_POWER]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_IA_KINETIC_POWER]=
                            supernova_kinetic_energy(star);
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_MOMENTUM]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_IA_MOMENTUM]=
                            supernova_momentum(star);
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_LUMINOSITY]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_IA_LUMINOSITY]=
                            supernova_luminous_energy(star);
                        break;
                    case SN_IBC:
                        ensemble_weight[NUCSYN_ENSEMBLE_SN_IBC]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_KINETIC_POWER]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_CORE_COLLAPSE_KINETIC_POWER]=
                            supernova_kinetic_energy(star);
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_MOMENTUM]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_CORE_COLLAPSE_MOMENTUM]=
                            supernova_momentum(star);
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_LUMINOSITY]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_CORE_COLLAPSE_LUMINOSITY]=
                            supernova_luminous_energy(star);
                        break;
                    case SN_II:
                        ensemble_weight[NUCSYN_ENSEMBLE_SN_II]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_KINETIC_POWER]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_CORE_COLLAPSE_KINETIC_POWER]=
                            supernova_kinetic_energy(star);
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_MOMENTUM]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_CORE_COLLAPSE_MOMENTUM]=
                            supernova_momentum(star);                        
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_LUMINOSITY]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_CORE_COLLAPSE_LUMINOSITY]=
                            supernova_luminous_energy(star);
                        break;
                    case SN_IIa:
                        ensemble_weight[NUCSYN_ENSEMBLE_SN_IIa]=1.0;
                        break;
                    case SN_NS_NS:
                        ensemble_weight[NUCSYN_ENSEMBLE_GRB_NS_NS]=1.0;
                        break;
                    case SN_GRB_COLLAPSAR:
                        /*
                         * A collapsar is also a SN 1bc, so set SN_IBC as well
                         */
                        ensemble_weight[NUCSYN_ENSEMBLE_GRB_COLLAPSAR]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_SN_IBC]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_KINETIC_POWER]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_CORE_COLLAPSE_KINETIC_POWER]=
                            supernova_kinetic_energy(star);
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_MOMENTUM]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_CORE_COLLAPSE_MOMENTUM]=
                            supernova_momentum(star);
                        ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_LUMINOSITY]=
                            ensemble_weight[NUCSYN_ENSEMBLE_SUPERNOVA_CORE_COLLAPSE_LUMINOSITY]=
                            supernova_luminous_energy(star);
                        break;
                    case SN_TZ:
                        /* 
                         * Thorne-Zytkow object
                         */
                        ensemble_weight[NUCSYN_ENSEMBLE_SN_TZ]=1.0;

                        /*
                         * TZ mass distribution
                         * 
                         * We use the previous timestep's mass as the TZO mass.
                         * This neglects mass that is lost in the common envelope 
                         * merger, so is likely only accurate to ~10%
                         */
                        const double MTZ = previous_star->mass + previous_other_star->mass;
                        const Ensemble_index tzi =
                            MTZ < 8.0 ? NUCSYN_ENSEMBLE_TZ_M1_LT8 :
                            MTZ > 30.0 ? NUCSYN_ENSEMBLE_TZ_M1_GT30 :
                            (NUCSYN_ENSEMBLE_TZ_M1_8_9 +
                             (Ensemble_index)( Max(0.0, Min(30.0 - 8.0,MTZ - 8.0 ))));
                        
                        Dprint("TZ MASSES : PREV %g %g : NOW %g %g : MTZ = %g -> tz index %u\n",
                               previous_star->mass,
                               previous_other_star->mass,
                               star->mass,
                               other_star->mass,
                               MTZ,
                               tzi);
                        ensemble_weight[tzi] = 1.0;
                        break;
                    case SN_AIC_BH:
                        ensemble_weight[NUCSYN_ENSEMBLE_SN_AIC_BH]=1.0;
                        break;
                    case SN_BH_NS:
                        ensemble_weight[NUCSYN_ENSEMBLE_SN_BH_NS]=1.0;
                        break;
                    case SN_BH_BH:
                        ensemble_weight[NUCSYN_ENSEMBLE_SN_BH_BH]=1.0;
                        break;
                    }

                    // reset sn_last_time
                    star->sn_last_time=-1;
                }

                Dprint("Xray binary checks");       
                /*
                 * X-ray binaries: luminosity is calculated in log_xray_binary 
                 */
                if(star->Xray_luminosity > TINY)
                {
                    if(ON_MAIN_SEQUENCE(other_star->stellar_type)&&
                       (other_star->mass<=2.0))
                    {
                        /*
                         * LMXB
                         */
                        ensemble_weight[NUCSYN_ENSEMBLE_XRAY_LMXB]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_XRAY_LMXB_LUM]=star->Xray_luminosity;
                    }
                    else if(WHITE_DWARF(other_star->stellar_type))
                    {
                        /*
                         * WDXB
                         */
                        ensemble_weight[NUCSYN_ENSEMBLE_XRAY_WDXB]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_XRAY_WDXB_LUM]=star->Xray_luminosity;
                    }
                    else
                    {
                        /*
                         * HMXB
                         */
                        ensemble_weight[NUCSYN_ENSEMBLE_XRAY_HMXB]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_XRAY_HMXB_LUM]=star->Xray_luminosity;
                    }
                }

                Dprint("Stellar colours");

                if(star->stellar_type<NEUTRON_STAR)
                {
                    /* 
                     * The eldridge2015_colours and gaia_magnitudes functions
                     * expect a pointer to an array of size NUMBER_OF_STELLAR_MAGNITUDES.
                     * We can set these magnitudes directly into the ensemble array, which
                     * is set up to handle the magnitudes by making sure that 
                     * NUCSYN_ENSEMBLE_MAGNITUDE_U to NUCSYN_ENSEMBLE_MAGNITUDE_GAIA_GRVS
                     * match (with an offset) the macros STELLAR_MAGNITUDE_* in
                     * src/stellar_magnitudes/stellar_colour_macros.h
                     *
                     * We can use the stellar_magnitudes wrapper function to 
                     * call the subordinate magnitude functions.
                     */
                    
                    double ** stellar_mags =
                        stardata->tmpstore->stellar_magnitudes;
                    stellar_magnitudes(stardata,star,&stellar_mags[star->starnum]);

                    Stellar_magnitude_index m;
                    for(m=0; m<NUMBER_OF_STELLAR_MAGNITUDES; m++)
                    {
                        /* 
                         * The ensemble weight should be a luminosity, not a magnitude,
                         * because we can weight and add luminosities, but not magnitudes.
                         * Conversion is done on output. 
                         */
                        int n = NUCSYN_ENSEMBLE_MAGNITUDE_U+m;      
                        ensemble_weight[n] = luminosity_from_magnitude(stellar_mags[star->starnum][m]);
                    }
                }
                Dprint("Spectral types");
                /* 
                 * Spectral types from Jaschek and Jaschek based on
                 * Lang (1992) "Astrophysical Data: Planets and Stars (Springer)
                 */
                if((star->stellar_type>2)&&
                   (star->stellar_type<10))
                {
                    if(l>4.9)
                    {
                        // supergiant?
                        if(teff>log10(26000.0))
                        {
                            // O
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_O]=1.0;
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_O_LUM]=star->luminosity;
                        }
                        else if(teff>log10(9730.0))
                        {
                            // B
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_B]=1.0;
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_B_LUM]=star->luminosity;
                        }
                        else if(teff>log10(7700.0))
                        {
                            // A
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_A]=1.0;
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_A_LUM]=star->luminosity;
                        }
                        else if(teff>log10(6550.0))
                        {
                            // F
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_F]=1.0;
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_F_LUM]=star->luminosity;
                        }
                        else if(teff>log10(4420.0))
                        {
                            // G
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_G]=1.0;
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_G_LUM]=star->luminosity;
                        }
                        else if(teff>log10(3650.0))
                        {
                            // K
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_K]=1.0;
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_K_LUM]=star->luminosity;
                        }
                        else
                        {
                            // M
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_M]=1.0;
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_M_LUM]=star->luminosity;
                        }
                    }
                    else
                    {
                        // giant
                        if(teff>log10(29000.0))
                        {
                            // O
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_O]=1.0;
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_O_LUM]=star->luminosity;
                        }
                        else if(teff>log10(10100.0))
                        {
                            // B
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_B]=1.0;
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_B_LUM]=star->luminosity;
                        }
                        else if(teff>log10(7150.0))
                        {
                            // A
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_A]=1.0;
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_A_LUM]=star->luminosity;
                        }
                        else if(teff>log10(5850.0))
                        {
                            // F
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_F]=1.0;
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_F_LUM]=star->luminosity;
                        }
                        else if(teff>log10(4750.0))
                        {
                            // G
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_G]=1.0;
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_G_LUM]=star->luminosity;
                        }
                        else if(teff>log10(3800.0))
                        {
                            // K
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_K]=1.0;
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_K_LUM]=star->luminosity;
                        }
                        else
                        {
                            // M
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_M]=1.0;
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_M_LUM]=star->luminosity;
                        }
                    }
                }
                else if(star->stellar_type<14)
                {
                    // everything else (not a BH or massless remnant) is a dwarf
                    if(teff>log10(30000.0))
                    {
                        // O
                        ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_O]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_O_LUM]=star->luminosity;
                    }
                    else if(teff>log10(9520.0))
                    {
                        // B
                        ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_B]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_B_LUM]=star->luminosity;
                    }
                    else if(teff>log10(7200.0))
                    {
                        // A
                        ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_A]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_A_LUM]=star->luminosity;
                    }
                    else if(teff>log10(6030.0))
                    {
                        // F
                        ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_F]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_F_LUM]=star->luminosity;
                    }
                    else if(teff>log10(5250.0))
                    {
                        // G
                        ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_G]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_G_LUM]=star->luminosity;
                    }
                    else if(teff>log10(3850.0))
                    {
                        // K
                        ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_K]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_K_LUM]=star->luminosity;
                    }
                    else
                    {
                        // M
                        ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_M]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_M_LUM]=star->luminosity;
                    }
                }

                /* C/O by number */
                Abundance C_over_O =
                    nucsyn_elemental_abundance("C",star->Xenv,stardata,stardata->store)/
                    nucsyn_elemental_abundance("O",star->Xenv,stardata,stardata->store);
             
                /* hydrogen-rich giant */
                Boolean Hgiant = Boolean_(star->stellar_type>=GIANT_BRANCH && star->stellar_type<HeMS);

                if(Hgiant==TRUE)
                {
                    if(Fequal(ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_M],1.0))
                    {
                        /*
                         * M-type stars can be C stars
                         */
                        if(C_over_O > 1.05)
                        {
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_C]=1.0;
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_C_LUM]=star->luminosity;
        
                            /*
                             * Which might also be J-stars is C12/C13 < 12
                             */
                            if(((star->Xenv[XC12]/12.0)/
                                (star->Xenv[XC13]/13.0))<12.0)
                            {
                                ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_J]=1.0;
                                ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_J_LUM]=star->luminosity;
                            }
                  
                        }
                        else if(C_over_O > 0.95)
                        {
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_S]=1.0;
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_S_LUM]=star->luminosity;
                        }
                    }
                    else if(Fequal(ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_K],1.0)&&
                            ((star->stellar_type==GIANT_BRANCH) ||
                             (star->stellar_type==CHeB))
                        )
                    {
                        /*
                         * K-type giants which are C-rich are assumed to be R-stars
                         */
                        if(C_over_O > 1.0)
                        {
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_R]=1.0;
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_R_LUM]=star->luminosity;
                        }
                    }

                    /*
                     * Red G/K type giants which are barium rich 
                     */
                    if((Fequal(ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_G],1.0) ||
                        Fequal(ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_K],1.0)) &&
                       (star->stellar_type>=GIANT_BRANCH)&&
                       (star->stellar_type<HeMS)&&
                       (nucsyn_elemental_abundance("Ba",star->Xenv,stardata,stardata->store)/
                        nucsyn_elemental_abundance("Ba",stardata->common.XZAMS,stardata,stardata->store)>2.0)
                        )
                    {
                        ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_Ba]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_Ba_LUM]=star->luminosity;
                    }
        
                    /*
                     * Look for Tc-rich stars which are GB/CHeB
                     */
                    if(((star->stellar_type==GIANT_BRANCH)||
                        (star->stellar_type==CHeB)) &&
                       ((nucsyn_elemental_abundance("Tc",star->Xenv,stardata,stardata->store)/
                         nucsyn_elemental_abundance("Tc",stardata->common.XZAMS,stardata,stardata->store)>2.0)))
                    {
                        ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_Tc]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_Tc_LUM]=star->luminosity;
                    }
                }

                /*
                 * Look for extrinsic carbon stars:
                 * Must be pre-1TP, C/O>1 and have WD companion so we can 
                 * definitely associate them with AGB mass transfer
                 */
                if((star->stellar_type<=TPAGB) &&
                   (C_over_O>1.0) &&
                   (WHITE_DWARF(other_star->stellar_type)))
                {
                    if(ON_MAIN_SEQUENCE(star->stellar_type))
                    {
                        ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_EXTRINSIC_DWARF_C]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_EXTRINSIC_DWARF_C_LUM]=star->luminosity;
                    }
                    else
                    {
                        /*
                         * C-rich pre-TPAGB stars are "extrinsic"
                         */
                        if(star->num_thermal_pulses_since_mcmin<0.5)
                        {
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_EXTRINSIC_GIANT_C]=1.0;
                            ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_EXTRINSIC_GIANT_C_LUM]=star->luminosity;
                        }
                    }
                }

                /*
                 * Classify WR stars by Hurley Pols Tout 2002 prescription:
                 * Assume WDs cannot be WR stars (dubious?)
                 */
                if((star->stellar_type>1)&&
                   (star->stellar_type<10))
                {
                    mu = ((star->mass-star->core_mass)/star->mass)*Min(5.0,Max(1.20,pow((star->luminosity/LUM0),KAP)));
                    if(mu<1.0)
                    {
                        ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_WR]=1.0;
                        ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_WR_LUM]=star->luminosity;
                    }
                }

                /*
                 * Luminous blue variable
                 */
                if((star->luminosity>6e5)&&
                   (1e-5*star->radius*sqrt(star->luminosity)>1.0))
                {
                    ensemble_weight[NUCSYN_ENSEMBLE_LBV]=1.0;
                    ensemble_weight[NUCSYN_ENSEMBLE_LBV_LUM]=star->luminosity;
                }

                /*
                 * FK Comae stars :
                 * 
                 * These are G and K giants that are rapidly rotating.
                 * What does this mean? For some reason I had v>20km/s in here,
                 * but perhaps > 50% critical rotation is more useful?
                 *
                 * Korhonen et al. 1999 have v sin i = 155km/s and G5 III type
                 * for FK Com. They also define FK Com stars as single systems
                 * rather than spun-up binaries (radial velocity variations are
                 * < 3km/s [Heuenemoerder+ 1993] or 5km/s [McCarthy and Ramsey 1984].
                 * Korhonen provide a nice review of the class definition.
                 * 
                 * The class includes FK Com, HD 199178, possibly HD 32918, I-1 in NGC188,
                 * and 1E 1751+7046.
                 */
                if(
                    (Fequal(ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_G],1.0) ||
                     Fequal(ensemble_weight[NUCSYN_ENSEMBLE_SPECTRAL_TYPE_K],1.0)) &&
                    GIANT_LIKE_STAR(star->stellar_type) &&
                    star->v_eq_ratio > 0.5 &&
                    stardata->model.sgl == TRUE
                    )
                {
                    ensemble_weight[NUCSYN_ENSEMBLE_FK_COMAE]=1.0;
                }
            
                Dprint("Special binaries"); 
                /*
                 * Special binary stellar systems
                 */
                if(i==0) // only check when i==0 to avoid double counting
                {
                    /*
                     * Check for double degenerates
                     */
                    if(WHITE_DWARF(star->stellar_type))
                    {
                        /*
                         * Primary is a white dwarf
                         */
                        if(WHITE_DWARF(other_star->stellar_type))
                        {
                            /*
                             * WD WD system
                             */
                            ensemble_weight[NUCSYN_ENSEMBLE_WDWD_DD]=1.0;
                        }
                        else if(other_star->stellar_type==NS)
                        {
                            ensemble_weight[NUCSYN_ENSEMBLE_WDNS_DD]=1.0;
                        }
                        else if(other_star->stellar_type==BH)
                        {
                            ensemble_weight[NUCSYN_ENSEMBLE_WDBH_DD]=1.0;
                        }
                    }
                    else if(star->stellar_type==NS)
                    {
                        if(other_star->stellar_type==NS)
                        {
                            ensemble_weight[NUCSYN_ENSEMBLE_NSNS_DD]=1.0;
                        }
                        else if(other_star->stellar_type==BH)
                        {
                            ensemble_weight[NUCSYN_ENSEMBLE_NSBH_DD]=1.0;
                        }
                    }
                    else if(star->stellar_type==BH)
                    {
                        if(other_star->stellar_type==BH)
                        {
                            ensemble_weight[NUCSYN_ENSEMBLE_BHBH_DD]=1.0;
                        }
                    }
                }
          
                /*
                 * Cataclysmic variables
                 */
                Dprint("Cataclysmic variables");
                if(WHITE_DWARF(star->stellar_type)&&
                   (other_star->radius > other_star->roche_radius)&&
                   (other_star->stellar_type<COWD))
                {
                    /*
                     * CV
                     */
                    if(other_star->stellar_type<=MAIN_SEQUENCE)
                    {
                        /*
                         * Donor on MS: classical CV
                         */
                        ensemble_weight[NUCSYN_ENSEMBLE_CLASSICAL_CV]=1.0;
                    }
                    else if(other_star->stellar_type==HERTZSPRUNG_GAP)
                    {
                        /*
                         * Donor is HG star: GK Per CV
                         */
                        ensemble_weight[NUCSYN_ENSEMBLE_GK_PER_CV]=1.0;
                    }
                    else if((other_star->stellar_type>=GIANT_BRANCH)&&
                            (other_star->stellar_type<=TPAGB))
                    {
                        /*
                         * Donor is helium-burning: Symbiotic star
                         */
                        ensemble_weight[NUCSYN_ENSEMBLE_SYMBIOTIC_CV]=1.0;
                    }
                    else if(NAKED_HELIUM_STAR(other_star->stellar_type))
                    {
                        /*
                         * AM CVn systems
                         */
                        ensemble_weight[NUCSYN_ENSEMBLE_AMCVN]=1.0;
                    }
                }

                /* 
                 * Hot subdwarfs:
                 * these are either low-mass helium stars
                 * or 
                 * very stripped CHeB stars
                 */
                if(
                    (NAKED_HELIUM_STAR(star->stellar_type) ||
                     (star->stellar_type==CHeB &&
                      star->mass - star->core_mass < 0.01))
                    &&
                    star->mass < 1.4
                    )
                {
                    /*
                     * sdB : 20-40kK
                     * sdO : 40-100kK
                     */
                    if(teff > log10(20e3) && teff < log10(40e3))
                    {
                        ensemble_weight[NUCSYN_ENSEMBLE_SDB]=1.0;
                    }
                    else if(teff > log10(100e3))
                    {
                        ensemble_weight[NUCSYN_ENSEMBLE_SDO]=1.0;
                    }
                }
                

                /*
                 * The various types of Algol systems
                 */
                Dprint("Algols");
                if(ON_MAIN_SEQUENCE(star->stellar_type)&& 
                   (other_star->radius > other_star->roche_radius))
                {
                    if(other_star->mass/star->mass > 1.0)
                    {
                        ensemble_weight[NUCSYN_ENSEMBLE_PRE_ALGOL]=1.0;
                    }
                    else
                    {
                        if(ON_MAIN_SEQUENCE(other_star->stellar_type))
                        {
                            ensemble_weight[NUCSYN_ENSEMBLE_MS_ALGOL]=1.0;
                        }
                        else
                        {
                            if(star->mass<=1.25)
                            {
                                ensemble_weight[NUCSYN_ENSEMBLE_COLD_ALGOL]=1.0;
                            }
                            else
                            {
                                ensemble_weight[NUCSYN_ENSEMBLE_HOT_ALGOL]=1.0;
                            }
                        }
                    }
                }

                Dprint("Blue stragglers");
                /*
                 * Blue stragglers (set in acquire_stellar_parameters...)
                 */
                if((stardata->star[0].blue_straggler==TRUE ||
                    stardata->star[1].blue_straggler==TRUE)&&(i==0))
                {
                    ensemble_weight[NUCSYN_ENSEMBLE_BLUE_STRAGGLER]=1.0;
            
                    if(Observable_binary)
                    {
                        ensemble_weight[NUCSYN_ENSEMBLE_BINARY_BLUE_STRAGGLER]=1.0;
                    }
                    else
                    {
                        ensemble_weight[NUCSYN_ENSEMBLE_SINGLE_BLUE_STRAGGLER]=1.0;
                    }
                }

                Dprint("Low-mass WDs");
                /*
                 * Low-mass white dwarfs
                 */
                if(star->mass<0.5)
                {
                    if(star->stellar_type==HeWD)
                    {
                        ensemble_weight[NUCSYN_ENSEMBLE_LOW_MASS_He_WD]=1.0;
                    }
                    else if(star->stellar_type==COWD)
                    {
                        ensemble_weight[NUCSYN_ENSEMBLE_LOW_MASS_CO_WD]=1.0;
                    }
                }

                Dprint("Symbiotic stars");
                /*
                 * Symbiotic stars
                 */
            
                if(star->stellar_type<=ONeWD)
                {
            
                    if((star->accretion_luminosity>10.0)||
                       (star->accretion_luminosity>0.01*other_star->luminosity))
                    {
                        if(other_star->stellar_type==TPAGB)
                        {
                            /*
                             * D-type symbiotic
                             */
                            ensemble_weight[NUCSYN_ENSEMBLE_D_TYPE_SYMBIOTIC]=1.0;
                        }
                        else if(other_star->stellar_type<TPAGB)
                        {
                            /*
                             * S-type symbiotic
                             */
                            ensemble_weight[NUCSYN_ENSEMBLE_S_TYPE_SYMBIOTIC]=1.0;
                        }
                    }
                }

                if(i==0)
                {
                    /*
                     * Mergers and post-common envelope systems
                     */
                    if(stardata->model.comenv_type != 0)
                    {
                        /*
                         * If stardata->model.comenv_type < 0, then the
                         * type is negative, meaning there was a merger.
                         * Otherwise, the post-CE system is detached.
                         */
                
                        int type = stardata->model.comenv_type;
                        if(type<0)
                        {
                            /* merger */
                            type = - stardata->model.comenv_type;
                            ensemble_weight[NUCSYN_ENSEMBLE_COMENV_MERGER] = 1.0;
                        }
                        else
                        {
                            /* detached */
                            type = stardata->model.comenv_type;
                            ensemble_weight[NUCSYN_ENSEMBLE_COMENV_DETACHED] = 1.0;
                        }
                        ensemble_weight[type]=1.0;                
                    }
                }

                /* mergers */
                Boolean merged = 
                    /* one of the flags should be set */
                    (
                        stardata->model.merged == TRUE || 
                        stardata->model.coalesce == TRUE || 
                        stardata->model.comenv_type !=0
                        ) &&

                    /* must change from binary to single */
                    stardata->model.sgl==TRUE &&
                    previous_stardata->model.sgl==FALSE &&
                
                    /* companion must be a massless remnant and wasn't last time */
                    other_star->stellar_type==MASSLESS_REMNANT &&
                    previous_other_star->stellar_type!=MASSLESS_REMNANT &&
                    Is_zero(other_star->mass) &&
                    Is_not_zero(previous_other_star->mass);
                
                if(merged && stardata->star[i].stellar_type>=0)
                {
                    /* 
                     * The stars merged into star i and 
                     * the code in here is called only once per 
                     * timestep, so if you want to check 
                     * symmetric mergers, use a and b below.
                     */
                    const Ensemble_index merge_index = NUCSYN_ENSEMBLE_MERGER_INTO_LMMS +
                        (Ensemble_index)stardata->star[i].stellar_type;

                    Dprint("merge index %u from %d,%d -> matrix %u\n",
                           merge_index,
                           previous_star->stellar_type,
                           previous_other_star->stellar_type,
                           merger_matrix[previous_star->stellar_type][previous_other_star->stellar_type] 
                        );
                    
                    /* log the merged star's stellar type */
                    ensemble_weight[merge_index] = 1.0;
                
                    /* log the merging stars' types */
                    ensemble_weight[merger_matrix[previous_star->stellar_type][previous_other_star->stellar_type]] = 1.0;

                    /*
                     * Special case : Hybrid COWD mergers
                     */
                    if(previous_star->hybrid_HeCOWD == TRUE ||
                       previous_other_star->hybrid_HeCOWD == TRUE)
                    {
                        Ensemble_index hybrid_merge_index;
                        
                        if(previous_star->hybrid_HeCOWD == TRUE &&
                           previous_other_star->hybrid_HeCOWD == TRUE)
                        {
                            /*
                             * Both stars are hybrids
                             */
                            hybrid_merge_index = NUCSYN_ENSEMBLE_MERGER_Hybrid_HeCOWD_Hybrid_HeCOWD;
                        }
                        else
                        {
                            /* 
                             * Identify the non-hybrid
                             */
                            struct star_t * non_hybrid =
                                previous_star->hybrid_HeCOWD == FALSE ? previous_star : previous_other_star;

                            /*
                             * Hence the merge index
                             */
                            hybrid_merge_index = NUCSYN_ENSEMBLE_MERGER_LMMS_Hybrid_HeCOWD +
                                non_hybrid->stellar_type;
                        }
                        
                        ensemble_weight[hybrid_merge_index] = 1.0;
                    }

                    
                    /*
                     * Now do logging for stars a and b, where 
                     * the two are previous stars 0,1 and then 1,0.
                     */
                    struct star_t *a,*b;
                    int kk;
                    for(kk=0; kk<2; kk++)
                    {
                        /*
                         * Set stars a and b, note that kk is 0 or 1
                         */
                        a = kk == 0 ? previous_star : previous_other_star;
                        b = kk == 0 ? previous_other_star : previous_star;
                                        
                        /*
                         * Special case for Simon Jeffery 
                         * HeWD + COWD mergers in which the HeWD is more
                         * massive than the COWD
                         */
                        if(((a)->stellar_type == HeWD &&
                            (b)->stellar_type == COWD &&
                            (a)->mass > (b)->mass))
                        {                    
                            ensemble_weight[NUCSYN_ENSEMBLE_MERGER_HeWD_gt_COWD] = 1.0;
                        }
                    
                        /* 
                         * Hagai Perets' HeWD + CO/ONeWD mergers
                         */
                        if(a->stellar_type==HeWD)
                        {
                            if(b->stellar_type==COWD)
                            {
                                /* cases with no HeWD mass cuts */ 
                                if(b->mass < 0.8)
                                {
                                    ensemble_weight[NUCSYN_ENSEMBLE_PERETS_HeWD_COWD_LT0p8] = 1.0;
                                }
                                else if(b->mass < 0.7)
                                {
                                    ensemble_weight[NUCSYN_ENSEMBLE_PERETS_HeWD_COWD_LT0p7] = 1.0;
                                }
                                else if(b->mass < 0.6)
                                {
                                    ensemble_weight[NUCSYN_ENSEMBLE_PERETS_HeWD_COWD_LT0p6] = 1.0;
                                }
                                else if(b->mass < 0.5)
                                {
                                    ensemble_weight[NUCSYN_ENSEMBLE_PERETS_HeWD_COWD_LT0p5] = 1.0;
                                }

                                /* as above with mass cuts */
                                if(a->mass>0.3)
                                {
                                    if(b->mass < 0.8)
                                    {
                                        ensemble_weight[NUCSYN_ENSEMBLE_PERETS_HeWD_GT0p3_COWD_LT0p8] = 1.0;
                                    }
                                    else if(b->mass < 0.7)
                                    {
                                        ensemble_weight[NUCSYN_ENSEMBLE_PERETS_HeWD_GT0p3_COWD_LT0p7] = 1.0;
                                    }
                                    else if(b->mass < 0.6)
                                    {
                                        ensemble_weight[NUCSYN_ENSEMBLE_PERETS_HeWD_GT0p3_COWD_LT0p6] = 1.0;
                                    }
                                    else if(b->mass < 0.5)
                                    {
                                        ensemble_weight[NUCSYN_ENSEMBLE_PERETS_HeWD_GT0p3_COWD_LT0p5] = 1.0;
                                    }
                                }

                                if(a->mass<0.25)
                                {
                                    if(b->mass < 0.8)
                                    {
                                        ensemble_weight[NUCSYN_ENSEMBLE_PERETS_HeWD_LT0p25_COWD_LT0p8] = 1.0;
                                    }
                                    else if(b->mass < 0.7)
                                    {
                                        ensemble_weight[NUCSYN_ENSEMBLE_PERETS_HeWD_LT0p25_COWD_LT0p7] = 1.0;
                                    }
                                    else if(b->mass < 0.6)
                                    {
                                        ensemble_weight[NUCSYN_ENSEMBLE_PERETS_HeWD_LT0p25_COWD_LT0p6] = 1.0;
                                    }
                                    else if(b->mass < 0.5)
                                    {
                                        ensemble_weight[NUCSYN_ENSEMBLE_PERETS_HeWD_LT0p25_COWD_LT0p5] = 1.0;
                                    }
                                }
                            }
                            else if(b->stellar_type==ONeWD)
                            {
                                ensemble_weight[NUCSYN_ENSEMBLE_PERETS_HeWD_ONeWD] = 1.0;

                                if(a->mass>0.3)
                                {
                                    ensemble_weight[NUCSYN_ENSEMBLE_PERETS_HeWD_GT0p3_ONeWD] = 1.0;
                                }
                                else if(a->mass<0.25)
                                {
                                    ensemble_weight[NUCSYN_ENSEMBLE_PERETS_HeWD_LT0p25_ONeWD] = 1.0;
                                }
                            }
                        }
                    }
                } // end loop kk
                
                /*
                 * Output
                 */
                Dprint("Output");

#ifdef NUCSYN_STELLAR_POPULATIONS_ENSEMBLE_SPARSE
                if(Is_zero(stardata->model.time))
                {
                    /*
                     * First timestep
                     */
                    lastensembleouttime =
                        stardata->preferences->yields_logtimes==TRUE ?
                        log10(stardata->preferences->yields_startlogtime) :
                        0.0;
                }
            
                /* add up the ensemble store */
                double * es = ensemble_store[i];

                /* 
                 * modulate by the probability now,
                 * NOT when we output, because the probability
                 * may change.
                 *
                 * Note that dtp is the linear timestep * probability
                 */
                const double p = stardata->model.probability;
                const double dtp = lineardt * p;
                
                for(k=0;k<NUCSYN_ENSEMBLE_N;k++)
                {
                    es[k] +=

                        ensemble_weight[k] * 
                    
                        (
                            ensemble_type[k]==ENSEMBLE_TYPE_RATE ?
                        
                            /* this is a rate: just count probabilities */
                            p :
                        
                            /* everything else: weight by timestep and probability */
                            dtp
                            );
                    
                }

                
                /*******************************
                 *
                 * Ensemble output
                 *
                 *******************************/
                if(timestep_fixed_trigger(stardata,FIXED_TIMESTEP_YIELDS))
                {
                    double T = Yields_time(stardata->model.time);
                    if(unlikely(fabs(T)<1e-15)) T = 0.0;
                    
#ifndef NUCSYN_GCE_DISABLE_ENSEMBLE_OUTPUT
                    Printf("STELLARPOPENSEMBLE%d__ %g ",
                           i,
                           T);
#endif // !NUCSYN_GCE_DISABLE_ENSEMBLE_OUTPUT
                
                    /*
                     * We want to output the value of x taken as a
                     * time-averaged mean over the time bin.
                     *
                     * The values which are in ensemble_store are 
                     * 
                     *  sum(x * delta t)
                     *
                     * where delta t is the each *code* timestep.
                     * 
                     * x is the variable in question (=1 for simple counts, but
                     * possibly a true stellar variable)
                     * 
                     * We must now modulate by the probability to get the true
                     * 'dtp' contribution to each time *bin* (given by 
                     * Yields_timestep), then divide by the 
                     * time since the last log to get the time-average dp.
                     *
                     * Rates are treated identically except that they are counted as
                     * 
                     * sum = # of events
                     *
                     * i.e. there is no 'delta t' term, hence the rate, or amount in the 
                     * "timestep", is output directly.
                     */
                    double fac = 1.0/linear_dt_since_last;

#ifndef NUCSYN_GCE_DISABLE_ENSEMBLE_OUTPUT
#ifdef COMPRESS_YIELD_AND_ENSEMBLE_OUTPUT
                    int zerocount = 0;
#endif // COMPRESS_YIELD_AND_ENSEMBLE_OUTPUT
                    for(k=0;k<NUCSYN_ENSEMBLE_N;k++)
                    {
                        /* output */
#ifdef COMPRESS_YIELD_AND_ENSEMBLE_OUTPUT
                        double x = fac*ensemble_store[i][k];
                        if(Is_zero(x))
                        {
                            // found a zero, do not output yet
                            zerocount++;
                        }
                        else
                        {
                            if(zerocount!=0)
                            {
                                ENSEMBLE_Zeros(zerocount);
                                zerocount=0;
                            }

                            Ensemble_print(k,x);
                        }
#else //COMPRESS_YIELD_AND_ENSEMBLE_OUTPUT
                        Ensemble_print(k,fac*ensemble_store[i][k]);
#endif //COMPRESS_YIELD_AND_ENSEMBLE_OUTPUT

                        /* reset */
                        ensemble_store[i][k]=0.0;
                    }

#ifdef COMPRESS_YIELD_AND_ENSEMBLE_OUTPUT
                    if(zerocount!=0)
                    {
                        ENSEMBLE_Zeros(zerocount);
                    }
#endif // COMPRESS_YIELD_AND_ENSEMBLE_OUTPUT

                    Printf("\n");
#endif // ! NUCSYN_GCE_DISABLE_ENSEMBLE_OUTPUT
                    reset_time=TRUE;
                }
#else
                printf("This is deprecated!\n");exit(0);
            
                Printf("STELLARPOPENSEMBLE%d__ %6.6e %3.3e ",
                       i,
                       Yields_time(stardata->model.time),
                       stardata->model.probability);
            
                for(k=0;k<NUCSYN_ENSEMBLE_N;k++)
                {
                    Ensemble_print(k,ensemble_weight[k]);
                }
                Printf("\n");
#endif // NUCSYN_STELLAR_POPULATIONS_ENSEMBLE_SPARSE
            }
        } // i loop  over stars

        if(reset_time==TRUE)
        {
            lastensembleouttime = Yields_time(stardata->model.time);
        }
#endif // ENSEMBLE

    }
}

#endif // NUCSYN && NUCSYN_GCE
