#include "nucsyn.h"
#ifdef NUCSYN

Abundance_ratio nucsyn_elemental_square_bracket(const char * const top,
                                                const char * const bottom,
                                                const Abundance * RESTRICT const X,
                                                const Abundance * RESTRICT const Xsolar,
                                                struct stardata_t * RESTRICT const stardata)
{
    Dprint("Square bracket [%s/%s]\n",top,bottom);
    /*
     *
     * Square bracket notation:
     * [A/B] = log(N_A/N_B) - log(N_A_solar/N_B_solar)
     *       = log(X_A/X_B) - log(X_A_solar/X_B_solar)
     * where log=log10
     *
     * A=top, B=bottom. 
     *
     * This function is for *elements* : see nucsyn_square_bracket
     * and nucsyn_square_multibracket for isotopic alternatives.
     */

    Abundance X_top = nucsyn_elemental_abundance(top,X,stardata,stardata->store);
    Abundance solar_top = nucsyn_elemental_abundance(top,Xsolar,stardata,stardata->store);
    Abundance X_bottom = nucsyn_elemental_abundance(bottom,X,stardata,stardata->store);
    Abundance solar_bottom = nucsyn_elemental_abundance(bottom,Xsolar,stardata,stardata->store);

    double sqb=log10((X_top+REALLY_TINY)/(solar_top+REALLY_TINY))-
        log10((X_bottom+REALLY_TINY)/(solar_bottom+REALLY_TINY));
    /*
     * Limit sqb: in the case where abundances are really close to solar,
     * the REALLY_TINY terms will give you a result around 10^-6, so just truncate
     * at 1e-5, which is 4 orders of magnitude better than stellar observations
     * (except with grains... perhaps...:)
     */
  
    if(Abs_less_than(sqb,1e-5)) sqb=0.0;
    Dprint("sqb[%s/%s] = log10(%g/%g)[star] - log10(%g/%g)[solar] = log10(%g) - log10(%g) = %g - %g =%g\n",
	   top,bottom,
	   X_top,solar_top,X_bottom,solar_bottom,
	   (X_top+REALLY_TINY)/(solar_top+REALLY_TINY),
	   (X_bottom+REALLY_TINY)/(solar_bottom+REALLY_TINY),
	   log10((X_top+REALLY_TINY)/(solar_top+REALLY_TINY)),
	   log10((X_bottom+REALLY_TINY)/(solar_bottom+REALLY_TINY)),
	   sqb);

    return(sqb);
}


#endif
