#include "../binary_c.h"

#ifdef NUCSYN

static double Hot_function Pure_function nucleon_density(struct stardata_t * RESTRICT const stardata,
                                                         const Abundance * RESTRICT const N,
                                                         const int * RESTRICT const A,
                                                         const Boolean inout) ;


void nucsyn_burning_cycles(Abundance * RESTRICT const N,
                           const Reaction_rate * RESTRICT const sigmav,
                           const double dt,
                           struct stardata_t * RESTRICT stardata)
{

    Dprint("nucsyn_burning_cycles...\n");

    /*
     * Given abundances (by number) N, reaction rates sigmav,
     * burn for a time dt.
     */
    //#undef NUCSYN_NORMALIZE_NUCLEONS
#ifdef NUCSYN_NORMALIZE_NUCLEONS
    /* count nucleons */
    double nucleon_density_in=
        nucleon_density(stardata,N,stardata->store->nucleon_number,0);
#endif
  
    /*
     * These are the initial guesses for the timestep.
     * nucsyn_network_burn should return a newer, better guess
     */
    double *dtguess = stardata->common.dtguess;

#if (DEBUG==1)
    {
        Reaction_network i;
        for(i=0;i<5;i++)
        {
            Dprint("dtguess[%u]=%g\n",i,dtguess[i]);
        }
    }
#endif

    /* burn PP */
#if (defined NUCSYN_NETWORK_PP || defined NUCSYN_NETWORK_PPfast) \
    && !defined LITHIUM_TABLES
    Dprint("call pp burn\n");

    //dtguess[NUCSYN_NETWORK_PP]=nucsyn_network_burn(N,sigmav,dt,&nucsyn_burn_pp,NULL,dtguess[NUCSYN_NETWORK_PP],stardata);
  
    /*
     * ppfast network instead
     * :increase Be7 e-capture rate by electron density: saves putting it in the matrices 
     */ 
    const double edens=nucsyn_free_electron_density(N,stardata->store->atomic_number);
    sigmav[SIGMAV_Be7_e]*=edens;

#ifdef NUCSYN_NETWORK_PPfast
    dtguess[NUCSYN_NETWORK_PP]=nucsyn_network_burn(N,sigmav,dt,&nucsyn_burn_ppfast,NULL,dtguess[NUCSYN_NETWORK_PP],stardata);
#endif
#ifdef NUCSYN_NETWORK_PP
    dtguess[NUCSYN_NETWORK_PP]=nucsyn_network_burn(N,sigmav,dt,&nucsyn_burn_pp,NULL,dtguess[NUCSYN_NETWORK_PP],stardata);
#endif

    sigmav[SIGMAV_Be7_e]/=edens;
#ifdef NANCHECKS
    {
        Isotope i;
        Isotope_loop(i)
        {
            if(isnan(N[i])!=0)
            {
                Exit_binary_c(BINARY_C_EXIT_NAN,"nucsyn_burning_cycles (post-pp) isotope %d/%d () is %g\n",
                              i,
                              ISOTOPE_ARRAY_SIZE,
                              N[i]);
            }
        }
    }
#endif
#endif

    /* burn CNO, either cold or hot */
#ifdef NUCSYN_NETWORK_COLDCNO
    Dprint("call cold CNO burn\n");
  
    dtguess[NUCSYN_NETWORK_COLDCNO]=nucsyn_network_burn(N,sigmav,dt,&nucsyn_burn_coldCNO,NULL,dtguess[NUCSYN_NETWORK_COLDCNO],stardata);

#ifdef NANCHECKS
    {
        Isotope i;
        Isotope_loop(i)
        {
            if(isnan(N[i])!=0)
            {
                Exit_binary_c(BINARY_C_EXIT_NAN,"nucsyn_burning_cycles (post-coldCNO) isotope %d/%d () is %g\n",
                              i,
                              ISOTOPE_ARRAY_SIZE,
                              N[i]);
            }
        }
    }
#endif

#endif

#ifdef NUCSYN_NETWORK_HOTCNO
    Dprint("call hot CNO burn\n");

    dtguess[NUCSYN_NETWORK_HOTCNO]=nucsyn_network_burn(N,sigmav,dt,&nucsyn_burn_hotCNO,NULL,dtguess[NUCSYN_NETWORK_HOTCNO],stardata);
#ifdef NANCHECKS
    {
        Isotope i;
        Isotope_loop(i)
        {
            if(isnan(N[i])!=0)
            {
                Exit_binary_c(BINARY_C_EXIT_NAN,"nucsyn_burning_cycles (post-hotCNO) isotope %d/%d () is %g\n",
                              i,
                              ISOTOPE_ARRAY_SIZE,
                              N[i]);
            }
        }
    }
#endif

#endif

    /* burn NeNaMgAl(Si28) */
#ifdef NUCSYN_NETWORK_NeNaMgAl
    Dprint("call NeNaMgAl burn\n");

    dtguess[NUCSYN_NETWORK_NeNaMgAl]=nucsyn_network_burn(N,sigmav,dt,&nucsyn_burn_NeNaMgAl,NULL,dtguess[NUCSYN_NETWORK_NeNaMgAl],stardata);


#ifdef NANCHECKS
    {
        Isotope i;
        Isotope_loop(i)
        {
            if(isnan(N[i])!=0)
            {
                Exit_binary_c(BINARY_C_EXIT_NAN,"nucsyn_burning_cycles (post-NeNaMgAl) isotope %d/%d () is %g\n",
                              i,
                              ISOTOPE_ARRAY_SIZE,
                              N[i]);
            }
        }
    }
#endif
#endif

#ifdef NANCHECKS
    {
        Isotope i;
        Isotope_loop(i)
        {
            if(isnan(N[i])!=0)
            {
                Exit_binary_c(BINARY_C_EXIT_NAN,"nucsyn_burning_cycles (post-burning) isotope %d/%d () is %g\n",
                              i,
                              ISOTOPE_ARRAY_SIZE,
                              N[i]);
            }
        }
    }
#endif

#ifdef NUCSYN_NORMALIZE_NUCLEONS
    double nucleon_density_out=
        nucleon_density(stardata,
                        N,
                        stardata->store->nucleon_number,
                        1);
    
    if(Is_not_zero(nucleon_density_out) &&
       Is_not_zero(nucleon_density_in))
    {
        /* normalize: preserve nucleons! */
        double fnucleon = nucleon_density_in/nucleon_density_out;
        Xmult(N,fnucleon);
#ifdef NANCHECKS
        {
            Isotope i;
            Isotope_loop(i)
            {
                if(isnan(N[i])!=0 || !isfinite(N[i]))
                {
                    Exit_binary_c(BINARY_C_EXIT_NAN,"nucsyn_burning_cycles (post-burning and normalization) isotope %d/%d is %g (fnucleon = %g from nucleon density in = %g and out = %g)\n",
                                  i,
                                  ISOTOPE_ARRAY_SIZE,
                                  N[i], 
                                  fnucleon,
                                  nucleon_density_in,
                                  nucleon_density_out
                        );
                }
            }
        }
    
#endif // NANCHECKS
    }
#endif // NUCSYN_NORMALIZE_NUCLEONS
}

static double Pure_function nucleon_density(struct stardata_t * const stardata,
                                            const Abundance * const N,
                                            const int * const A, 
                                            const Boolean inout) 
{
    double n=0.0;
    Isotope i;
    Isotope_loop(i)
    {
        n += N[i] ? (N[i] * (double)(A[i])) : 0.0;
#ifdef NANCHECKS
        if(isnan(n))
        {
            Exit_binary_c(BINARY_C_EXIT_NAN,"nucsyn_burning_cycles at %d isotope %d N[i]=%g A[i]=%d",
                          inout,
                          i,
                          N[i], 
                          A[i]);
        }
#endif
    }
    return(n);
}


#endif

